﻿/*---------------------------------------------------------------------------*
  Copyright (C) 2016 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/

#include "jira-regression-tests.h"
#define PRINT_TEST_WARN 0
#include "ntd-test-libc.h"
#include <stdio.h>
#include <math.h>
#include <fenv.h>
double __nnmusl_force_sub(double x, double y);

static int set_and_test_rounding_mode(int round_mode)
{
    int err = fesetround(round_mode);
    TESTWARN("fesetround(%x) = %d", round_mode, err);
    TESTCASE_MESSAGE(err == 0, "fesetround(%x) returned %d expected 0", round_mode, err);
    if (err) return err;

    int round_state = fegetround();
    TESTWARN("fegetround() = %x", round_state);
    TESTCASE_MESSAGE(round_state == round_mode, "fegetround() returned %x expected %x", round_state, round_mode);
    return err;
}

static double subtraction(double a, double b)
{
    return (a - b);
}

static
__attribute__((noinline))
void test_sub(int round_mode, double a, double b, bool expect_zero, const char *expect_string)
{

    int save_round_state = fegetround();
    TESTWARN("fegetround() = %x", save_round_state);

    int err = set_and_test_rounding_mode(round_mode);
    if (err) return;

    double result = a - b;

    // restore rounding mode before printing to avoid possible problems with FTZ and printing the value
    err = set_and_test_rounding_mode(save_round_state);

    char result_string[80];

    snprintf(result_string, sizeof(result_string), "%a - %a = %a", a, b, result);

    TESTWARN("%s", result_string);
    bool success = (expect_zero && result == 0.0) || (!expect_zero && result != 0.0);
    TESTCASE_MESSAGE(success, "%a - %a = %a expected %s 0.0", a, b, result, expect_zero ? "==" : "!=");
    TESTCASE_STRINGS_MATCH(result_string, expect_string);
}

static double inputs[8] = {
    0x1.1p-1023, 0x1.0p-1023,
    0x1.3p-1023, 0x1.4p-1023,
    0x1.01p-1023, 0x1.02p-1023,
    0x1.04p-1023, 0x1.01p-1023,
};

static double results[8] = {
    0x1p-1027, -0x1p-1027,
    -0x1p-1027, 0x1p-1027,
    -0x1p-1031, 0x1p-1031,
    0x1.8p-1030, -0x1.8p-1030,
};

#ifdef FE_SET_FTZ
static double ftz_results[8] = {
    0x0p+0, 0x0p+0,
    0x0p+0, 0x0p+0,
    0x0p+0, 0x0p+0,
    0x0p+0, 0x0p+0
};

static
__attribute__((noinline))
void test_simple_sub(double a, double b, double expected_result)
{

    double result = a - b;

    TESTCASE_MESSAGE(result == expected_result, "%a - %a = %a expected %a", a, b, result, expected_result);
}

static void test_ftz()
{
    int save_round= fegetround();

    int err = fesetround(FE_SET_FTZ|FE_TONEAREST);
    if (err) return;

    for (int i=0; i<8; i=i+2)
    {
        test_simple_sub(inputs[i], inputs[i+1], ftz_results[i]);
        test_simple_sub(inputs[i+1], inputs[i], ftz_results[i+1]);
    }

    test_simple_sub(0x1.4p-1023, 0x1.3p-1023, 0x0p+0);
    TESTCASE_MESSAGE(FLT_ROUNDS == 1, "FLT_ROUNDS = %d, expected 1", FLT_ROUNDS);

#ifdef FE_DOWNWARD
    fesetround(FE_SET_FTZ|FE_DOWNWARD);
    TESTCASE_MESSAGE(FLT_ROUNDS == 3, "FLT_ROUNDS = %d, expected 3", FLT_ROUNDS);
#endif

#ifdef FE_UPWARD
    fesetround(FE_SET_FTZ|FE_UPWARD);
    TESTCASE_MESSAGE(FLT_ROUNDS == 2, "FLT_ROUNDS = %d, expected 2", FLT_ROUNDS);
#endif

#ifdef FE_TOWARDZERO
    fesetround(FE_SET_FTZ|FE_TOWARDZERO);
    TESTCASE_MESSAGE(FLT_ROUNDS == 0, "FLT_ROUNDS = %d, expected 0", FLT_ROUNDS);
#endif

    err = fesetround(save_round);
    if (err) return;
}
#endif

#define ACOSF_ZERO        1.57079637050628662109375F
#define ACOS_ZERO         1.5707963267948965579989817342720925807952880859375
#define ACOSF_ZERO_EXPECT 1.57079637

void rynda_668()
{
    int save_round= fegetround(), rounding_mode = save_round;

    float acosf_value = acosf(0.0f);
    double acos_value = acos(0.0f);
    TESTCASE_MESSAGE(save_round == FE_TONEAREST, "rounding mode = %d, expected FE_TONEAREST", rounding_mode);
    TESTCASE_MESSAGE(acosf_value == ACOSF_ZERO, "acosf(0.0f) = %.8f, expected %.8f", acosf_value, ACOSF_ZERO);
    TESTCASE_MESSAGE(acos_value == ACOS_ZERO, "acos(0.0f) = %.8f, expected %.8f", acos_value, ACOS_ZERO);

#ifdef FE_DOWNWARD
    int err = fesetround(FE_DOWNWARD);
    if (err) return;
    rounding_mode = fegetround();
    acosf_value = acosf(0.0f);
    acos_value = acos(0.0f);
    TESTCASE_MESSAGE(rounding_mode == FE_DOWNWARD, "rounding mode = %d, expected FE_DOWNWARD", rounding_mode);
    TESTCASE_MESSAGE(acosf_value == ACOSF_ZERO, "acosf(0.0f) = %.8f, expected %.8f", acosf_value, ACOSF_ZERO);
    TESTCASE_MESSAGE(acos_value == ACOS_ZERO, "acos(0.0f) = %.8f, expected %.8f", acos_value, ACOS_ZERO);
#endif

#ifdef FE_UPWARD
    err = fesetround(FE_UPWARD);
    if (err) return;
    rounding_mode= fegetround();
    acosf_value = acosf(0.0f);
    acos_value = acos(0.0f);
    TESTCASE_MESSAGE(rounding_mode == FE_UPWARD, "rounding mode = %d, expected FE_UPWARD", rounding_mode);
    TESTCASE_MESSAGE(acosf_value == ACOSF_ZERO, "acosf(0.0f) = %.8f, expected %.8f", acosf_value, ACOSF_ZERO);
    TESTCASE_MESSAGE(acos_value == ACOS_ZERO, "acos(0.0f) = %.8f, expected %.8f", acos_value, ACOS_ZERO);
#endif

#ifdef FE_TOWARDZERO
    err = fesetround(FE_TOWARDZERO);
    if (err) return;
    rounding_mode= fegetround();
    acosf_value = acosf(0.0f);
    acos_value = acos(0.0f);
    TESTCASE_MESSAGE(rounding_mode == FE_TOWARDZERO, "rounding mode = %d, expected FE_TOWARDZERO", rounding_mode);
    TESTCASE_MESSAGE(acosf_value == ACOSF_ZERO, "acosf(0.0f) = %.8f, expected %.8f", acosf_value, ACOSF_ZERO);
    TESTCASE_MESSAGE(acos_value == ACOS_ZERO, "acos(0.0f) = %.8f, expected %.8f", acos_value, ACOS_ZERO);
#endif

    err = fesetround(save_round);
    if (err) return;
}

__attribute__((noinline))
void rynda_525()
{
    NTD_TEST_GROUP_START("RYNDA-525", 1);
    double  one = 1.0;
    int round = 0;

    test_sub(FE_TONEAREST,            0x1.1p-1023, 0x1.0p-1023, false, "0x1.1p-1023 - 0x1p-1023 = 0x1p-1027");

    round = fegetround();

    fesetround(FE_TONEAREST);

    for (int i=0; i<8; i=i+2)
    {
        test_simple_sub(inputs[i], inputs[i+1], results[i]);
        test_simple_sub(inputs[i+1], inputs[i], results[i+1]);
    }

#ifdef FE_SET_FTZ
    test_ftz();
#endif

    rynda_668();

    fesetround(round);

    NTD_TEST_GROUP_END("RYNDA-525", 1);
}
