﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nnt/gtest/detail/gtest-heap.h>
#include <nnt/gtest/detail/gtest-map.h>
#include <nnt/gtest/detail/gtest-utility.h>

namespace nnt { namespace testing { namespace detail {

template <typename KeyT, typename ComparatorT = Less<KeyT>>
class Set final
{
public:
    NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

public:
    typedef KeyT value_type;

private:
    typedef Map<KeyT, int8_t, ComparatorT> MapT;
    typedef typename MapT::iterator IteratorT;
    typedef typename MapT::const_iterator ConstIteratorT;

private:
    MapT m_Map;

public:
    class const_iterator final
    {
    private:
        ConstIteratorT m_Iter;

    public:
        explicit const_iterator(const ConstIteratorT& iter) NN_NOEXCEPT
            : m_Iter(iter) {}

        const KeyT& operator*() const NN_NOEXCEPT
        {
            return (*m_Iter).first;
        }

        const KeyT* operator->() const NN_NOEXCEPT
        {
            return &m_Iter->first;
        }

        const_iterator& operator++() NN_NOEXCEPT
        {
            ++m_Iter;
            return *this;
        }

        bool operator==(const const_iterator& other) const NN_NOEXCEPT
        {
            return m_Iter == other.m_Iter;
        }

        bool operator!=(const const_iterator& other) const NN_NOEXCEPT
        {
            return m_Iter != other.m_Iter;
        }
    };

    class iterator final
    {
    private:
        IteratorT m_Iter;

    public:
        explicit iterator(const IteratorT& iter) NN_NOEXCEPT
            : m_Iter(iter) {}
    };

public:
    size_t size() const NN_NOEXCEPT
    {
        return m_Map.size();
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_Map.empty();
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        return const_iterator(m_Map.begin());
    }

    const_iterator end() const NN_NOEXCEPT
    {
        return const_iterator(m_Map.end());
    }

    bool operator==(const Set& other) const NN_NOEXCEPT
    {
        return m_Map == other.m_Map;
    }

    size_t count(const KeyT& key) const NN_NOEXCEPT
    {
        return m_Map.count(key);
    }

    Pair<iterator, bool> insert(const KeyT& key) NN_NOEXCEPT
    {
        Pair<IteratorT, bool> pair = m_Map.insert(Pair<KeyT, int8_t>(key, 0));

        return Pair<iterator, bool>(iterator(pair.first), pair.second);
    }

    size_t erase(const KeyT& key) NN_NOEXCEPT
    {
        return m_Map.erase(key);
    }
};

}}} // namespace nnt::testing::detail
