﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nnt/gtest/detail/gtest-functional.h>
#include <nnt/gtest/detail/gtest-heap.h>
#include <nnt/gtest/detail/gtest-utility.h>

namespace nnt { namespace testing { namespace detail {

template <typename KeyT, typename ValueT, typename ComparatorT = Less<KeyT>>
class Map final
{
public:
    NNT_TESTING_DETAIL_HEAP_IS_ALLOCATABLE();

private:
    enum class Color
    {
        Black,
        Red,
    };

    struct Node
    {
        Color color;
        Pair<KeyT, ValueT>* pData;
        Node* pParent;
        Node* pLhs;
        Node* pRhs;
    };

public:
    typedef size_t size_type;
    typedef Pair<KeyT, ValueT> value_type;

public:
    class const_iterator
    {
    protected:
        Node* m_pNode;

    public:
        explicit const_iterator(Node* pNode) NN_NOEXCEPT : m_pNode(pNode)
        {
            NN_SDK_ASSERT_NOT_NULL(pNode);
        }

        const Pair<KeyT, ValueT>& operator*() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pNode);
            NN_SDK_REQUIRES_NOT_NULL(m_pNode->pData);
            return *(m_pNode->pData);
        }

        const Pair<KeyT, ValueT>* operator->() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pNode);
            return m_pNode->pData;
        }

        const_iterator& operator++() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pNode);

            m_pNode = Map::GetFollowing(m_pNode);

            return *this;
        }

        bool operator==(const const_iterator& other) const NN_NOEXCEPT
        {
            return m_pNode == other.m_pNode;
        }

        bool operator!=(const const_iterator& other) const NN_NOEXCEPT
        {
            return !(*this == other);
        }
    };

    class iterator final : public const_iterator
    {
    public:
        explicit iterator(Node* pNode) NN_NOEXCEPT : const_iterator(pNode) {}

        Pair<KeyT, ValueT>* operator->() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(this->m_pNode);
            return this->m_pNode->pData;
        }

        bool operator==(const iterator& other) const NN_NOEXCEPT
        {
            return const_iterator::operator==(other);
        }

        bool operator!=(const iterator& other) const NN_NOEXCEPT
        {
            return const_iterator::operator!=(other);
        }
    };

private:
    Node* m_pNode;

    size_t m_Count;

public:
    Map() NN_NOEXCEPT : m_pNode(Map::Allocate()), m_Count(0)
    {
        m_pNode->color = Color::Black;
    }

    Map(const Map& other) NN_NOEXCEPT
    {
        Map::Clone(&m_pNode, other.m_pNode);

        m_Count = other.m_Count;
    }

    ~Map() NN_NOEXCEPT { Map::Free(m_pNode); }

    Map& operator=(const Map& other) NN_NOEXCEPT
    {
        Map::Free(m_pNode);

        Map::Clone(&m_pNode, other.m_pNode);

        m_Count = other.m_Count;

        return *this;
    }

    size_t size() const NN_NOEXCEPT
    {
        return m_Count;
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_Count == 0;
    }

    const_iterator begin() const NN_NOEXCEPT
    {
        Node* pNode = m_pNode;
        while (pNode->pLhs != nullptr) { pNode = pNode->pLhs; }
        return const_iterator(pNode);
    }

    iterator begin() NN_NOEXCEPT
    {
        Node* pNode = m_pNode;
        while (pNode->pLhs != nullptr) { pNode = pNode->pLhs; }
        return iterator(pNode);
    }

    const_iterator end() const NN_NOEXCEPT
    {
        Node* pNode = m_pNode;
        while (pNode->pRhs != nullptr) { pNode = pNode->pRhs; }
        return const_iterator(pNode);
    }

    iterator end() NN_NOEXCEPT
    {
        Node* pNode = m_pNode;
        while (pNode->pRhs != nullptr) { pNode = pNode->pRhs; }
        return iterator(pNode);
    }

    bool operator==(const Map& other) const NN_NOEXCEPT
    {
        if (this->size() != other.size())
        {
            return false;
        }

        const_iterator head = this->begin();
        const_iterator tail = this->end();
        const_iterator peer = other.begin();

        while (head != tail)
        {
            if (*head != *peer)
            {
                return false;
            }

            ++head;
            ++peer;
        }

        return true;
    }

    ValueT& operator[](const KeyT& key) NN_NOEXCEPT
    {
        Node* pNode = nullptr;
        bool flag = false;

        Map::Emplace(&pNode, &flag, &m_pNode, key);
        NN_SDK_ASSERT_NOT_NULL(pNode);
        NN_SDK_ASSERT_NOT_NULL(pNode->pData);

        if (flag)
        {
            m_pNode->color = Color::Black;

            ++m_Count;
        }

        return pNode->pData->second;
    }

    const_iterator find(const KeyT& key) const NN_NOEXCEPT
    {
        Node* pNode = Map::Find(m_pNode, key);

        return (pNode == nullptr) ? this->end() : const_iterator(pNode);
    }

    iterator find(const KeyT& key) NN_NOEXCEPT
    {
        Node* pNode = Map::Find(m_pNode, key);

        return (pNode == nullptr) ? this->end() : iterator(pNode);
    }

    size_t count(const KeyT& key) const NN_NOEXCEPT
    {
        return (this->find(key) == this->end()) ? 0 : 1;
    }

    Pair<iterator, bool> insert(const Pair<KeyT, ValueT>& pair) NN_NOEXCEPT
    {
        Node* pNode = nullptr;
        bool flag = false;

        Map::Emplace(&pNode, &flag, &m_pNode, pair.first);
        NN_SDK_ASSERT_NOT_NULL(pNode);
        NN_SDK_ASSERT_NOT_NULL(pNode->pData);

        if (flag)
        {
            pNode->pData->second = pair.second;

            m_pNode->color = Color::Black;

            ++m_Count;
        }

        return Pair<iterator, bool>(iterator(pNode), flag);
    }

    iterator erase(const_iterator iter) NN_NOEXCEPT
    {
        Node* pNode = nullptr;
        bool flag = false;
        Map::Remove(&pNode, &flag, &m_pNode, iter->first);

        if (flag)
        {
            m_pNode->color = Color::Black;

            --m_Count;
        }

        return iterator(pNode);
    }

    size_t erase(const KeyT& key) NN_NOEXCEPT
    {
        Node* pNode = nullptr;
        bool flag = false;
        Map::Remove(&pNode, &flag, &m_pNode, key);

        if (flag)
        {
            m_pNode->color = Color::Black;

            --m_Count;
        }

        return flag ? 1 : 0;
    }

private:
    static Node* Allocate() NN_NOEXCEPT
    {
        auto pNode = static_cast<Node*>(Heap::Allocate(sizeof(Node)));
        NN_SDK_ASSERT_NOT_NULL(pNode);
        pNode->color = Color::Red;
        pNode->pData = nullptr;
        pNode->pParent = nullptr;
        pNode->pLhs = nullptr;
        pNode->pRhs = nullptr;
        return pNode;
    }

    static void Free(Node* pNode) NN_NOEXCEPT
    {
        if (pNode != nullptr)
        {
            if (pNode->pData != nullptr) { delete pNode->pData; }
            Map::Free(pNode->pLhs);
            Map::Free(pNode->pRhs);
            Heap::Free(pNode);
        }
    }

    static void Clone(Node** ppOutNode, const Node* pNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppOutNode);
        NN_SDK_REQUIRES_NOT_NULL(pNode);

        Node*& pOutNode = *ppOutNode;

        pOutNode = Map::Allocate();

        pOutNode->color = pNode->color;

        if (pNode->pData != nullptr)
        {
            pOutNode->pData = new Pair<KeyT, ValueT>(*(pNode->pData));
        }

        if (pNode->pLhs != nullptr)
        {
            Map::Clone(&pOutNode->pLhs, pNode->pLhs);

            pOutNode->pLhs->pParent = pOutNode;
        }

        if (pNode->pRhs != nullptr)
        {
            Map::Clone(&pOutNode->pRhs, pNode->pRhs);

            pOutNode->pRhs->pParent = pOutNode;
        }
    }

    static Node* GetFollowing(Node* pNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pNode);

        if (pNode->pRhs != nullptr)
        {
            pNode = pNode->pRhs;

            while (pNode->pLhs != nullptr) { pNode = pNode->pLhs; }

            return pNode;
        }
        else
        {
            while (pNode->pParent != nullptr && pNode->pParent->pLhs != pNode)
            {
                pNode = pNode->pParent;
            }

            NN_SDK_ASSERT_NOT_NULL(pNode->pParent);

            return pNode->pParent;
        }
    }

    static Node* Find(Node* pNode, const KeyT& key) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pNode);

        while (pNode != nullptr)
        {
            if (pNode->pData == nullptr)
            {
                pNode = pNode->pLhs;
                continue;
            }

            const Pair<KeyT, ValueT>& data = *(pNode->pData);

            ComparatorT compare;

            if (compare(key, data.first))
            {
                pNode = pNode->pLhs;
                continue;
            }

            if (compare(data.first, key))
            {
                pNode = pNode->pRhs;
                continue;
            }

            break;
        }

        return pNode;
    }

    static bool Emplace(
        Node** ppOutNode, bool* pOutFlag, Node** ppNode, const KeyT& key
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppOutNode);
        NN_SDK_REQUIRES_NOT_NULL(pOutFlag);
        NN_SDK_REQUIRES_NOT_NULL(ppNode);

        Node*& pOutNode = *ppOutNode;
        bool& outFlag = *pOutFlag;
        Node*& pNode = *ppNode;

        ComparatorT compare;

        if (pNode == nullptr)
        {
            pOutNode = Map::Allocate();
            pOutNode->pData = new Pair<KeyT, ValueT>(key, ValueT());
            outFlag = true;
            pNode = pOutNode;
            return true;
        }
        else if (pNode->pData == nullptr || compare(key, pNode->pData->first))
        {
            if (!Map::Emplace(&pOutNode, &outFlag, &pNode->pLhs, key))
            {
                return false;
            }
            else
            {
                pNode->pLhs->pParent = pNode;
                return Map::BalanceEmplacementL(&pNode);
            }
        }
        else if (compare(pNode->pData->first, key))
        {
            if (!Map::Emplace(&pOutNode, &outFlag, &pNode->pRhs, key))
            {
                return false;
            }
            else
            {
                pNode->pRhs->pParent = pNode;
                return Map::BalanceEmplacementR(&pNode);
            }
        }
        else
        {
            pOutNode = pNode;
            outFlag = false;
            return false;
        }
    }

    static bool Remove(
        Node** ppOutNode, bool* pOutFlag, Node** ppNode, const KeyT& key
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppOutNode);
        NN_SDK_REQUIRES_NOT_NULL(pOutFlag);
        NN_SDK_REQUIRES_NOT_NULL(ppNode);

        Node*& pOutNode = *ppOutNode;
        bool& outFlag = *pOutFlag;
        Node*& pNode = *ppNode;

        ComparatorT compare;

        if (pNode == nullptr)
        {
            pOutNode = nullptr;
            outFlag = false;
            return false;
        }
        else if (pNode->pData == nullptr || compare(key, pNode->pData->first))
        {
            return Map::Remove(&pOutNode, &outFlag, &pNode->pLhs, key) ?
                Map::BalanceRemovalL(&pNode) : false;
        }
        else if (compare(pNode->pData->first, key))
        {
            return Map::Remove(&pOutNode, &outFlag, &pNode->pRhs, key) ?
                Map::BalanceRemovalR(&pNode) : false;
        }
        else if (pNode->pLhs == nullptr && pNode->pRhs == nullptr)
        {
            pOutNode = Map::GetFollowing(pNode);
            outFlag = true;
            const bool unbalance = pNode->color == Color::Black;
            Map::Free(pNode);
            pNode = nullptr;
            return unbalance;
        }
        else if (pNode->pRhs == nullptr)
        {
            pOutNode = Map::GetFollowing(pNode);
            outFlag = true;
            Node* pLhs = pNode->pLhs;
            pLhs->color = Color::Black;
            pLhs->pParent = pNode->pParent;
            pNode->pLhs = nullptr;
            Map::Free(pNode);
            pNode = pLhs;
            return false;
        }
        else if (pNode->pLhs == nullptr)
        {
            pOutNode = Map::GetFollowing(pNode);
            outFlag = true;
            Node* pRhs = pNode->pRhs;
            pRhs->color = Color::Black;
            pRhs->pParent = pNode->pParent;
            pNode->pRhs = nullptr;
            Map::Free(pNode);
            pNode = pRhs;
            return false;
        }
        else
        {
            Node* pRhs = pNode->pRhs;
            while (pRhs->pLhs != nullptr) { pRhs = pRhs->pLhs; }
            Pair<KeyT, ValueT>* pData = pNode->pData;
            pNode->pData = pRhs->pData;
            pRhs->pData = pData;

            auto removeTail = [](Node** ppTail) NN_NOEXCEPT -> bool
            {
                const Color color = (*ppTail)->color;
                Map::Free(*ppTail);
                *ppTail = nullptr;
                return color == Color::Black;
            };

            const bool unbalance = (pNode->pData == nullptr)
                ? removeTail(&pNode->pRhs)
                : Map::Remove(&pOutNode, &outFlag, &pNode->pRhs, key);

            pOutNode = pNode;
            outFlag = true;

            return unbalance ? Map::BalanceRemovalR(&pNode) : false;
        }
    }

    static bool IsBlack(const Node* const pNode) NN_NOEXCEPT
    {
        return pNode == nullptr || pNode->color == Color::Black;
    }

    static bool IsRed(const Node* const pNode) NN_NOEXCEPT
    {
        return pNode != nullptr && pNode->color == Color::Red;
    }

    static void RotateL(Node** ppNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppNode);
        NN_SDK_REQUIRES_NOT_NULL((*ppNode)->pRhs);

        Node*& pNode = *ppNode;
        Node* pRhs = pNode->pRhs;

        pNode->pRhs = pRhs->pLhs;
        pRhs->pLhs = pNode;
        pRhs->color = pNode->color;
        pNode->color = Color::Red;
        pNode = pRhs;
        pNode->pParent = pNode->pLhs->pParent;
        pNode->pLhs->pParent = pNode;
        if (pNode->pLhs->pRhs != nullptr)
        {
            pNode->pLhs->pRhs->pParent = pNode->pLhs;
        }
    }

    static void RotateR(Node** ppNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppNode);
        NN_SDK_REQUIRES_NOT_NULL((*ppNode)->pLhs);

        Node*& pNode = *ppNode;
        Node* pLhs = pNode->pLhs;

        pNode->pLhs = pLhs->pRhs;
        pLhs->pRhs = pNode;
        pLhs->color = pNode->color;
        pNode->color = Color::Red;
        pNode = pLhs;
        pNode->pParent = pNode->pRhs->pParent;
        pNode->pRhs->pParent = pNode;
        if (pNode->pRhs->pLhs != nullptr)
        {
            pNode->pRhs->pLhs->pParent = pNode->pRhs;
        }
    }

    static bool BalanceEmplacementL(Node** ppNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppNode);
        NN_SDK_REQUIRES_NOT_NULL((*ppNode)->pLhs);

        Node*& pNode = *ppNode;

        if (Map::IsRed(pNode))
        {
            return true;
        }
        else
        {
            if (Map::IsRed(pNode->pLhs->pRhs))
            {
                Map::RotateL(&pNode->pLhs);
            }

            if (Map::IsBlack(pNode->pLhs->pLhs))
            {
                return false;
            }
            else
            {
                if (Map::IsRed(pNode->pRhs))
                {
                    pNode->color = Color::Red;
                    pNode->pLhs->color = Color::Black;
                    pNode->pRhs->color = Color::Black;
                    return true;
                }
                else
                {
                    Map::RotateR(&pNode);
                    return false;
                }
            }
        }
    }

    static bool BalanceEmplacementR(Node** ppNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppNode);
        NN_SDK_REQUIRES_NOT_NULL((*ppNode)->pRhs);

        Node*& pNode = *ppNode;

        if (Map::IsRed(pNode))
        {
            return true;
        }
        else
        {
            if (Map::IsRed(pNode->pRhs->pLhs))
            {
                Map::RotateR(&pNode->pRhs);
            }

            if (Map::IsBlack(pNode->pRhs->pRhs))
            {
                return false;
            }
            else
            {
                if (Map::IsRed(pNode->pLhs))
                {
                    pNode->color = Color::Red;
                    pNode->pLhs->color = Color::Black;
                    pNode->pRhs->color = Color::Black;
                    return true;
                }
                else
                {
                    Map::RotateL(&pNode);
                    return false;
                }
            }
        }
    }

    static bool BalanceRemovalL(Node** ppNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppNode);
        NN_SDK_REQUIRES_NOT_NULL((*ppNode)->pRhs);

        Node*& pNode = *ppNode;

        if (Map::IsRed(pNode->pRhs->pLhs))
        {
            Map::RotateR(&pNode->pRhs);
            Map::RotateL(&pNode);
            pNode->pLhs->color = Color::Black;
            pNode->pRhs->color = Color::Black;
            return false;
        }
        else if (Map::IsRed(pNode->pRhs->pRhs))
        {
            Map::RotateL(&pNode);
            pNode->pLhs->color = Color::Black;
            pNode->pRhs->color = Color::Black;
            return false;
        }
        else if (Map::IsRed(pNode->pRhs))
        {
            Map::RotateL(&pNode);
            Map::BalanceRemovalL(&pNode->pLhs);
            return false;
        }
        else
        {
            pNode->pRhs->color = Color::Red;

            if (Map::IsBlack(pNode))
            {
                return true;
            }
            else
            {
                pNode->color = Color::Black;
                return false;
            }
        }
    }

    static bool BalanceRemovalR(Node** ppNode) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(ppNode);
        NN_SDK_REQUIRES_NOT_NULL((*ppNode)->pLhs);

        Node*& pNode = *ppNode;

        if (Map::IsRed(pNode->pLhs->pRhs))
        {
            Map::RotateL(&pNode->pLhs);
            Map::RotateR(&pNode);
            pNode->pLhs->color = Color::Black;
            pNode->pRhs->color = Color::Black;
            return false;
        }
        else if (Map::IsRed(pNode->pLhs->pLhs))
        {
            Map::RotateR(&pNode);
            pNode->pLhs->color = Color::Black;
            pNode->pRhs->color = Color::Black;
            return false;
        }
        else if (Map::IsRed(pNode->pLhs))
        {
            Map::RotateR(&pNode);
            Map::BalanceRemovalR(&pNode->pRhs);
            return false;
        }
        else
        {
            pNode->pLhs->color = Color::Red;

            if (Map::IsBlack(pNode))
            {
                return true;
            }
            else
            {
                pNode->color = Color::Black;
                return false;
            }
        }
    }
};

}}} // namespace nnt::testing::detail
