﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Text.RegularExpressions;
using System.Globalization;
using System.Threading;
using UtilExtension;

namespace Util
{
    partial class BuiltBinaryPathTemplate
    {
        public List<NspInfo> Applets { get; set; }
        public List<NspInfo> Processes { get; set; }
        public List<NspInfo> SystemData { get; set; }
        public List<NspInfo> AppletSystemData { get; set; }

        public BuiltBinaryPathTemplate()
        {
            this.Applets = new List<NspInfo>();
            this.Processes = new List<NspInfo>();
            this.SystemData = new List<NspInfo>();
            this.AppletSystemData = new List<NspInfo>();
        }
    }

    public class RuleGenerator
    {
        #region Ocean リスト
        private static readonly string[] AppletList =
        {
            "Auth",
            "Cabinet",
            "Controller",
            "DataErase",
            "DummyECApplet",
            "Error",
            "LoginShare",
            "Maintenance",
            "MiiEditN",
            "MyPage",
            "NetConnect",
            "OfflineWeb",
            "OfflineWebDev",
            "OverlayDisp",
            "PhotoViewer",
            "PlayerSelect",
            "QLaunch",
            "Shop",
            "Starter",
            "Swkbd",
            "Web",
            "WebDev",
            "WifiWebAuth",
        };

        private static readonly string[] AppletSystemDataList =
        {
            "AccountIconParts",
            "BlackList",
            "CruiserSharedData",
            "Dictionary",
            "DictionaryCKT",
            "ErrorMessage",
            "Eula",
            "LocalNews",
            "OceanShared",
            "SettingsHtdocs",
        };
        #endregion

        public static string Generate(string RootDir)
        {
            var binaryPathRule = new BuiltBinaryPathTemplate();

            var appletDir = Path.Combine(RootDir, "Applets");
            var processDir = Path.Combine(RootDir, "Processes", "Release");
            var systemDataDir = Path.Combine(RootDir, "SystemData");

            {   // Applet

                var applets = Directory.EnumerateFiles(appletDir, "*")
                                .Select(file => Path.GetFileName(file))
                                .Distinct(e => e.Split('_')[0], DistinctExtension.DistinctBy.Last);

                var nspInfo = from appletName in AppletList
                              from nsp in applets
                              where nsp.ToLower().StartsWith(appletName.ToLower() + "_")
                              select GetNspInfo(nsp, NspType.Applet, appletName);

                binaryPathRule.Applets = nspInfo.ToList();
            }

            {   //SystemData
                var appletSystemData = Directory.EnumerateFiles(systemDataDir, "data_*")
                                        .Select(file => Path.GetFileName(file))
                                        .Distinct(e => e.Split('_')[1], DistinctExtension.DistinctBy.Last);

                var nspInfo = from dataName in AppletSystemDataList
                              from nsp in appletSystemData
                              where nsp.ToLower().StartsWith("data_" + dataName.ToLower() + "_")
                              select GetNspInfo(nsp, NspType.AppletSystemData, dataName);

                binaryPathRule.AppletSystemData = nspInfo.ToList();


                var nspInfo2 = from nsp in Directory.EnumerateFiles(systemDataDir, "*").Select(file => Path.GetFileName(file))
                               where !nsp.StartsWith("data_")
                               select GetNspInfo(nsp, NspType.SystemData);

                binaryPathRule.SystemData = nspInfo2.ToList();
            }

            {   // Process
                var nspInfo = from nsp in Directory.EnumerateFiles(processDir, "*").Select(file => Path.GetFileName(file))
                              select GetNspInfo(nsp, NspType.Process);

                binaryPathRule.Processes = nspInfo.ToList();
            }

            return binaryPathRule.TransformText();
        }

        private static NspInfo GetNspInfo(string FileName, NspType Type, string NspName = "")
        {
            switch(Type)
            {
                case NspType.Applet:
                case NspType.AppletSystemData:
                    if(NspName == "")
                    {
                        throw new ArgumentException(string.Format("NspName must be specified for {0}.", Type));
                    }
                    return new NspInfo()
                    {
                        Name = NspName,
                        Version = NspInfo.FileNameFormat.Match(FileName).Groups["version"].ToString(),
                        FileName = NspInfo.FileNameFormat.Match(FileName).Groups["filename"].ToString(),
                    };

                case NspType.SystemData:
                case NspType.Process:
                    return new NspInfo()
                    {
                        Name = NspInfo.FileNameWithRevisionFormat.Match(FileName).Groups["name"].ToString(),
                        Version = NspInfo.FileNameWithRevisionFormat.Match(FileName).Groups["version"].ToString(),
                        FileName = NspInfo.FileNameWithRevisionFormat.Match(FileName).Groups["filename"].ToString(),
                    };

                default:
                    throw new ArgumentException(string.Format("Unkwnon type: {0}", Type));
            }
        }

        enum NspType
        {
            Applet,
            AppletSystemData,
            SystemData,
            Process
        }
    }
}
