﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UniformBufferManager.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that manages memory
 *  for uniform buffers by using multiple chunks of
 *  of GPU memory. The active chunk is written to and
 *  submitted to the GPU. At the end of the frame,
 *  it is swapped with the next chunk which then becomes
 *  active and writable for the next frame. This is done to
 *  avoid have to wait too much time for a sync object to
 *  signal that the memory is done being consumed by the
 *  GPU.
 *
 *  Requesting a single chunk for the memory pool will provide
 *  an undivided block of memory for uniform values that don't
 *  necessarily need to be updated at all or just infrequently,
 *  though the user must now handle proper syncing if they
 *  do wish to update the memory.
 */

#include <nn/nn_Assert.h>
#include <nvntutorial/UniformBufferManager.h>

/*
 * UniformBufferManager Constructor
 * --------------------------------
 * Initialize the managers memory pool with the proper
 * size and set initial values for the manager.
 */
UniformBufferManager::UniformBufferManager(NVNdevice* pDevice, size_t chunkSize, int numChunks /*= 2*/) :
    m_NumChunks(numChunks),
    m_WriteIndex(0),
    m_CurrentChunk(0),
    m_BufferAddress(0),
    m_pMappedBuffer(NULL),
    m_pDevice(pDevice),
    m_bufferAlignment(0)
{
    NN_ASSERT(m_NumChunks > 0, "Zero or negative number of chunks requested of uniform buffer manager\n");

        /* Requested size. */
    size_t size = chunkSize * m_NumChunks;

        /* Align the size properly. */
    if (size < g_MinimumPoolSize)
    {
        m_PoolSize = g_MinimumPoolSize;
    }
    else
    {
        m_PoolSize = Align(size, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    }

        /* Create the memory pool. */
    m_Pool.Init(NULL, m_PoolSize, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, m_pDevice);

        /* Create an NVN buffer to provide memory from. */
    NVNbufferBuilder bufferBuilder;
    nvnBufferBuilderSetDevice(&bufferBuilder, m_pDevice);
    nvnBufferBuilderSetDefaults(&bufferBuilder);
    nvnBufferBuilderSetStorage(&bufferBuilder, m_Pool.GetMemoryPool(), 0, m_PoolSize);

    if (!nvnBufferInitialize(&m_Buffer, &bufferBuilder))
    {
        NN_ASSERT(0, "Failed to initialize uniform block's buffer");
    }

        /* Get the mapped buffer pointer and NVNbufferAddress. */
    m_pMappedBuffer = reinterpret_cast<char*>(nvnBufferMap(&m_Buffer));
    m_BufferAddress = nvnBufferGetAddress(&m_Buffer);

        /* Get the size of the per frame chunks. */
    int bufferAlignment = 0;
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &bufferAlignment);
    m_bufferAlignment = bufferAlignment;

    m_ChunkSize = Align(m_PoolSize / m_NumChunks, m_bufferAlignment);
}

/*
 * UniformBufferManager Destructor
 * -------------------------------
 * Clean up NVN data and memory allocations.
 */
UniformBufferManager::~UniformBufferManager()
{
    for (unsigned i = 0; i < m_ManagedBuffers.size(); ++i)
    {
        if (m_ManagedBuffers[i])
        {
            delete m_ManagedBuffers[i];
            m_ManagedBuffers[i] = NULL;
        }
    }

    nvnBufferFinalize(&m_Buffer);

    m_Pool.Shutdown();
}

/*
 * UniformBufferManager::CreateUniformBuffer
 * -----------------------------------------
 * Create a ManagedUniformBuffer of the given size from the pool.
 */
ManagedUniformBuffer* UniformBufferManager::CreateUniformBuffer(size_t bufferSize)
{
    ManagedUniformBuffer* buffer = new ManagedUniformBuffer(m_pDevice, bufferSize, m_BufferAddress, m_pMappedBuffer, this);
    m_ManagedBuffers.push_back(buffer);
    return buffer;
}

/*
 * UniformBufferManager::SwapPools
 * -------------------------------
 * Swap the memory memory pools so that memory written to
 * in the previous frame isn't written to until a later frame
 *
 * NOTE: This function should only be called if you can guarantee
 *       that the memory in the next chunk is safe to start writing
 *       to.  This can be done by registering an instance of this
 *       class with an instance of the FrameBufferedSyncManager and
 *       inserting a sync through InsertFence().
 */
void UniformBufferManager::SwapPools()
{
        /* If there is only undivided chunk in the memory pool, then there is nothing to swap */
    if (m_NumChunks == 1)
    {
        return;
    }

    m_CurrentChunk = (m_CurrentChunk + 1) % m_NumChunks;
    m_WriteIndex = m_CurrentChunk * m_ChunkSize;

    for(unsigned i = 0; i < m_ManagedBuffers.size(); ++i)
    {
        m_ManagedBuffers[i]->Reset();
    }
}

/*
 * UniformBufferManager::GetNumChunks
 * ----------------------------------
 * Retrieve number of chunks uniform buffer manager is using
 */
int UniformBufferManager::GetNumChunks()
{
    return m_NumChunks;
}

/*
 * UniformBufferManager::GetMemoryChunk
 * ------------------------------------
 * Get the offset into the pool of a writable chunk of memory
 * of the given size.
 */
ptrdiff_t UniformBufferManager::GetMemoryChunk(size_t size)
{
    ptrdiff_t val = m_WriteIndex.fetch_add(Align(size, m_bufferAlignment), std::memory_order_relaxed);

        /* Check for overrunning buffer. */
    NN_ASSERT(((m_WriteIndex - (m_CurrentChunk * m_ChunkSize)) < m_ChunkSize) && "Bytes written to uniform buffer pool exceeds buffer limits");

    return val;
}
