﻿//==============================================================================
// Utils.cpp
//==============================================================================

#include <stdio.h>
#include <stdlib.h>
#include <cstdarg>
#include <cstdlib>
#include <string.h>
#ifdef NN_NINTENDO_SDK
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/htcs.h>
#else
#include <WinSock2.h>
#endif
#include "Utils.h"

#if __NX__
#define STRCAT( Dest, Source ) strcat( Dest, Source )
#define STRCPY( Dest, Source ) strcpy( Dest, Source )
#define STRNCMP( Comp1, Comp2, Length ) strncmp( Comp1, Comp2, Length)
#else
#define STRCAT( Dest, Source ) strcat( Dest, Source )
#define STRCPY( Dest, Source ) strcpy_s( Dest, 256, Source )
#define STRNCMP( Comp1, Comp2, Length ) strncmp( Comp1, Comp2, Length)
#ifndef NN_UNUSED
#define NN_UNUSED(X) (void)X
#endif
#endif

#define LISTENQ                     1024

void* Allocate(size_t size)
{
    return malloc(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    free(p);
}

//==============================================================================

void InitSystem( void )
{
#ifdef NN_NINTENDO_SDK
    //tma::Initialize();
    nn::htcs::Initialize(Allocate, Deallocate);
    //nn::htcs2::Initialize();
    srand( 0x4356 );
#else
    //==============================================================================
     // Initialize Winsock
    WSADATA wsaData;
    if( WSAStartup(MAKEWORD(2,2), &wsaData) != 0 )
    {
        printf("EchoClient: WSAStartup failed.\n");
        exit(EXIT_FAILURE);
    }
    srand( GetTickCount() );
#endif
}

//==============================================================================

void CloseSystem( void )
{
#ifdef NN_NINTENDO_SDK
    //tma::Finalize();
    nn::htcs::Finalize();
#else
    WSACleanup();
#endif
}

//===============================================================================================
//  Creates a listening socket for our server.  Returns the socket or < 0 if there was an error.
#ifdef NN_NINTENDO_SDK
int CreateListeningSocket( const char* pServerName, const char* pPortName )
#else
int CreateListeningSocket( int PortNumber )
#endif
{
    //==============================================================================
    //  Create our socket
#ifdef NN_NINTENDO_SDK
    int Ret = ::nn::htcs::Socket();
#else
    SOCKET Ret = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
#endif
    if( Ret < 0 )
    {
#ifdef NN_NINTENDO_SDK
        int Error = ::nn::htcs::GetLastError();
        REPORT( "nn::htcs2::Socket() returned %d\n", Error );
#else
        int Error = GetLastError();
        REPORT( "GetLastError returned %d\n", Error );
#endif
        return Ret;
    }

    //==============================================================================
    // Get our address ready for the bind.
#ifdef NN_NINTENDO_SDK
    nn::htcs::SockAddrHtcs ServerAddress;
    memset( &ServerAddress, 0, sizeof(ServerAddress) );
    ServerAddress.family = nn::htcs::HTCS_AF_HTCS;
    memcpy(ServerAddress.peerName.name, pServerName, sizeof(ServerAddress.peerName.name) );
    memcpy(ServerAddress.portName.name, pPortName, sizeof(ServerAddress.portName.name) );
#else
    struct sockaddr_in ServerAddress;

    ServerAddress.sin_family = AF_INET;
    ServerAddress.sin_addr.s_addr = INADDR_ANY;
    ServerAddress.sin_port = htons( PortNumber );
#endif

    //==============================================================================
    // Bind our socket addresss to the  listening socket
#ifdef NN_NINTENDO_SDK
    int Res = ::nn::htcs::Bind( Ret, &ServerAddress );
#else
    int Res = bind( Ret, (struct sockaddr *)&ServerAddress, sizeof(ServerAddress) );
#endif
    if( Res >= 0 )
    {
        //==============================================================================
        // Call listen()
#ifdef NN_NINTENDO_SDK
        Res = nn::htcs::Listen( Ret, LISTENQ );
#else
        Res = listen( Ret, LISTENQ );
#endif
        if( Res >= 0 )
        {
            //=========================================
            // No problems, so return the socket.
            return Ret;
        }
        else
        {
#ifdef NN_NINTENDO_SDK
            int Error = ::nn::htcs::GetLastError();
#else
            int Error = GetLastError();
#endif
            REPORT( "Listen() returned %d\n", Error );
        }
    }
    else
    {
#ifdef NN_NINTENDO_SDK
        int Error = ::nn::htcs::GetLastError();
#else
        int Error = GetLastError();
#endif
        REPORT( "Bind() returned %d\n", Error );
    }

    //=================================================// Problem creating our socket, so close it and
    // return the error code.
    //=================================================
#ifdef NN_NINTENDO_SDK
    nn::htcs::Close( Ret );
//    ::nn::htcs::Finalize();
#else
    closesocket( Ret );
#endif

    return Res;
}

//============================================================================================
//  Creates a connecting socket to the passed server.  Returns < 0 if there's an error.
#ifdef NN_NINTENDO_SDK
int CreateConnectSocket( const char* pHTCSServerName, const char* pHTCSServerPortName )
#else
int CreateConnectSocket( char* pIPAddress, int Port )
#endif
{
    //==============================================================================
    //  Create our socket
#ifdef NN_NINTENDO_SDK
    int Ret = ::nn::htcs::Socket();
#else
    SOCKET Ret = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
#endif
    if( Ret < 0 )
    {
#ifdef NN_NINTENDO_SDK
        int Error = ::nn::htcs::GetLastError();
        REPORT( "nn::htcs2::Socket() returned %d\n", Error );
#else
        int Error = GetLastError();
        REPORT( "GetLastError returned %d\n", Error );
#endif
        return Ret;
    }

    //==============================================================================
    // Get our address ready for the connect call.
#ifdef NN_NINTENDO_SDK

    nn::htcs::SockAddrHtcs ServerAddress;
    memset( &ServerAddress, 0, sizeof(ServerAddress) );
    ServerAddress.family = nn::htcs::HTCS_AF_HTCS;
    memcpy(ServerAddress.peerName.name, pHTCSServerName, sizeof(ServerAddress.peerName.name) );
    memcpy(ServerAddress.portName.name, pHTCSServerPortName, sizeof(ServerAddress.portName.name) );
#else
    struct sockaddr_in ServerAddress;

    ServerAddress.sin_family = AF_INET;
    InetPton( AF_INET, pIPAddress, &ServerAddress.sin_addr.s_addr );
    ServerAddress.sin_port = htons( Port );
#endif

    //==============================================================================
    // Connect
#ifdef NN_NINTENDO_SDK
    int Res = nn::htcs::Connect( Ret, &ServerAddress );
#else
    int Res = connect( Ret, (struct sockaddr *) &ServerAddress, sizeof(ServerAddress) );
#endif
    if( Res >= 0  )
    {
        //=========================================
        // If no problems, then return the socket.
        return Ret;
    }

    //=================================================
    // Problem creating our socket, so close it and
    // return the error code.
    //=================================================
#ifdef NN_NINTENDO_SDK
    nn::htcs::Close( Ret );
#else
    closesocket( Ret );
#endif
#ifdef NN_NINTENDO_SDK
    int Error = ::nn::htcs::GetLastError();
#else
    int Error = GetLastError();
#endif
    REPORT( "CreateConnectSocket failed, GetLastError returned %d\n", Error );

    return Res;
}

//========================================================================================
// Thin platform-agnostic layer to the Recv call.  Returns the value that recv returns.
int Recv( int Socket, char* pDataBuffer, size_t ReadAmount, int Flags )
{
#ifdef NN_NINTENDO_SDK
    return nn::htcs::Recv( Socket, pDataBuffer, ReadAmount, Flags );
#else
    return recv( Socket, pDataBuffer, ReadAmount, Flags );
#endif
}

//========================================================================================
// Thin platform-agnostic layer to the Send call.  Returns the value that send returns.
int Send( int Socket, const char* pData, size_t SizeOfData, int Flags )
{
#ifdef NN_NINTENDO_SDK
    return nn::htcs::Send( Socket, pData, SizeOfData, Flags );
#else
    return send( Socket, pData, SizeOfData, Flags );
#endif
}

//========================================================================================
// Thin platform-agnostic layer to the close call.  Returns the value that close returns.
int CloseSocket( int Socket )
{
    int Ret = -1;
#ifdef NN_NINTENDO_SDK
    Ret = nn::htcs::Close( Socket );
#else
    Ret = closesocket( Socket );
#endif
    if( Ret < 0 )
    {
#ifdef NN_NINTENDO_SDK
        int Error = ::nn::htcs::GetLastError();
#else
        int Error = GetLastError();
#endif
        REPORT( "CloseSocket() error:  %d\n", Error );
    }

    return Ret;
}

//==============================================================================

bool GetArgValue( char* pArg, char* pArgValue )
{
    char* pEquals = strchr(pArg, '=');
    if( pEquals != NULL )
    {
        STRCPY( pArgValue, pEquals + 1 );
        return true;
    }

    return false;
}

//==============================================================================

bool GetArg( int NumArgs, char** pArgs, const char* pFindArg, char* pArgValue )
{
    for( int ArgIndex = 1; ArgIndex < NumArgs; ArgIndex += 1)
    {
        char* pArg = pArgs[ArgIndex];
        if(pArg != NULL)
        {
            if(STRNCMP( pFindArg, pArg, strlen(pFindArg) ) == 0)
            {
                return GetArgValue( pArg, pArgValue );
            }
        }
    }

    return false;
}

//==============================================================================

int GetRandom( int Min, int Max )
{
    int Rand = rand();

    int Diff = Max - Min;
    if( Diff )
    {
        Rand = Rand % Diff;
    }

    return (Min + Rand);
}


//==============================================================================
// Thread functionality.

bool StartThread( THREAD_FUNCTION* pfnThread, const char* pName, void* pStack, size_t StackSize, void* pArg, ThreadId* pThread )
{
    memset( pThread, 0, sizeof(ThreadId) );

#ifdef __NX__
    nn::os::CreateThread( pThread, (nn::os::ThreadFunction)pfnThread, (void*)pArg, pStack, StackSize, nn::os::HighestThreadPriority );
    nn::os::StartThread( pThread );
    nn::os::SetThreadName(  pThread, pName );
#else
    pfnThread( pArg );
#endif
    return 1;
}

//==============================================================================

void JoinThread( ThreadId* pThread )
{
#ifdef __NX__
    nn::os::WaitThread( pThread );
    nn::os::DestroyThread( pThread );
#else
#endif
}

//==============================================================================
