﻿/*--------------------------------------------------------------------------------*
  Copyright (C) Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ro.h>
#include <thread>
#include "MainModuleExterns.h"

const char*  NRR_FILE_PATH  = "rom:/.nrr/MultiModuleTest.nrr";
const char*  NRO1_FILE_PATH = "rom:/nro/NroModule1.nro";
const char*  NRO2_FILE_PATH = "rom:/nro/NroModule2.nro";
const char*  NRO3_FILE_PATH = "rom:/nro/NroModule3.nro";
const size_t MAX_FILE_SIZE  = 0x400000;

char*                   gCacheBuffer    = nullptr;
thread_local int        gMainTlsInt     = 0xFEED;
std::mutex              gCoutMutex;
extern thread_local int gMainModuleExternTlsInt;

// Lib and NRO functions.
void NroPrint1();
void NroPrint2();
void NroPrint3();
void LibPrint1();
void LibPrint2();

size_t ReadFromFs(void* pOut, size_t bufferSize, const char* path)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, pOut, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);

    nn::fs::CloseFile(file);

    return readSize;
}

nn::Result LoadModule( const char* filePath, nn::ro::Module& module )
{
    void*       nro;
    void*       bss;
    size_t      imageSize;
    size_t      bufferSize;
    nn::Result  result;

    nro = aligned_alloc( nn::os::MemoryPageSize, MAX_FILE_SIZE );
    imageSize = ReadFromFs( nro, MAX_FILE_SIZE, filePath );

    result = nn::ro::GetBufferSize( &bufferSize, nro );
    NN_ASSERT( result.IsSuccess() );
    if( bufferSize != 0 )
    {
        bss = aligned_alloc( nn::os::MemoryPageSize, bufferSize );
    }
    else
    {
        bss = 0;
    }

    result = nn::ro::LoadModule( &module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy );
    NN_ASSERT( result.IsSuccess() );

    return result;
}

nn::Result ReadNRR( const char* filePath, nn::ro::RegistrationInfo& info )
{
    nn::Result  result;
    void*       nrr;
    size_t      nrrSize;
    size_t      cacheSize = 0;

    nn::ro::Initialize();

    result = nn::fs::QueryMountRomCacheSize( &cacheSize );
    NN_ASSERT( result.IsSuccess() );

    char* gCacheBuffer = new( std::nothrow ) char[cacheSize];
    NN_ASSERT_NOT_NULL( gCacheBuffer );

    result = nn::fs::MountRom( "rom", gCacheBuffer, cacheSize );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    // Read this apps NRR for module information.
    nrr = aligned_alloc( nn::os::MemoryPageSize, MAX_FILE_SIZE );
    nrrSize = ReadFromFs( nrr, MAX_FILE_SIZE, filePath );

    result = nn::ro::RegisterModuleInfo( &info, nrr );
    NN_ASSERT( result.IsSuccess() );

    return result;
}

extern "C" void nnMain()
{
    nn::Result                  result;
    nn::ro::Module              module1;
    nn::ro::Module              module2;
    nn::ro::Module              module3;
    nn::ro::RegistrationInfo    info;
    thread_local int            mainTlsInt = 0xACE;

    result = ReadNRR( NRR_FILE_PATH, info );
    result = LoadModule( NRO1_FILE_PATH, module1 );
    result = LoadModule( NRO2_FILE_PATH, module2 );

    // Create a few threads and verify each threads TLS variables are not shared
    // between the threads. Each TLS variable should have a unique address.
// DISABLING UNTIL NRO SUPPORT IS COMPLETED.
    //std::thread  thread1( &NroPrint1 );
    //std::thread* thread2 = new std::thread( &NroPrint1 );
    //std::thread  thread3( &NroPrint2 );
    //std::thread* thread4 = new std::thread( &NroPrint2 );
    //std::thread  thread5( &LibPrint1 );
    //std::thread* thread6 = new std::thread( &LibPrint1 );
    //std::thread  thread7( &LibPrint2 );
    //std::thread* thread8 = new std::thread( &LibPrint2 );

    gCoutMutex.lock();
    NN_LOG( "nnMain()\n" );
    NN_LOG( "nnMain: mainTlsInt:  (0x%p): %x\n", &mainTlsInt, mainTlsInt );
    NN_LOG( "\n" );
    gCoutMutex.unlock();

    IncrementExternTlsVariables();
    mainTlsInt++; // TLSVariablesTestBP03
    mainTlsInt--;

    // Call the module functions.
// DISABLING UNTIL NRO SUPPORT IS COMPLETED.
    //NroPrint2();
    //NroPrint1();
    LibPrint1();
    LibPrint2();

// DISABLING UNTIL NRO SUPPORT IS COMPLETED.
    //thread1.join();
    //thread2->join();
    //thread3.join();
    //thread4->join();
    //thread5.join();
    //thread6->join();
    //thread7.join();
    //thread8->join();

    // Unload the modules, then reload in reverse order.
    nn::ro::UnloadModule( &module1 );

// DISABLING UNTIL NRO SUPPORT IS COMPLETED.
    //result = LoadModule( NRO3_FILE_PATH, module3 );
    //NroPrint3();
    //NroPrint2();

    //nn::ro::UnloadModule( &module3 );
    nn::ro::UnloadModule( &module2 );

    result = LoadModule( NRO2_FILE_PATH, module2 );
    result = LoadModule( NRO1_FILE_PATH, module1 );

    // Call the module functions.
// DISABLING UNTIL NRO SUPPORT IS COMPLETED.
    //NroPrint2();
    //NroPrint1();
    LibPrint1();
    LibPrint2();

    nn::ro::UnloadModule( &module2 );
    nn::ro::UnloadModule( &module1 );

// DISABLING UNTIL NRO SUPPORT IS COMPLETED.
    //delete thread2;
    //delete thread4;
    //delete thread6;
    //delete thread8;

    nn::ro::UnregisterModuleInfo(&info);
    nn::ro::Finalize();

    nn::fs::Unmount("rom");

    if( gCacheBuffer != nullptr )
    {
        delete[] gCacheBuffer;
        gCacheBuffer = nullptr;
    }

    return;
}
