﻿//==============================================================================
//  Copyright (C) DejaTools, LLC.  All rights reserved.
//==============================================================================

#include <stdio.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#ifdef _WIN32
#include <Windows.h>
#include "DejaInsight/include/DejaLib.h"
#else
#define DEJA_CONTEXT(...)
#endif

#include "PXMalloc.h"

#ifdef WIN32

#ifndef MEM_ENABLE_DEJA_MALLOC
#define MEM_ENABLE_DEJA_MALLOC 1
#endif


#ifndef MEM_ENABLE_DEJA_SYS_MALLOC
#define MEM_ENABLE_DEJA_SYS_MALLOC 0
#endif

#endif

#ifdef _WIN32
bool __memory_assert()
{
    __debugbreak();
    return true;
}
#else
bool __memory_assert()
{
    __builtin_trap();
    return true;
}
#endif

#if MEM_ENABLE_DEJA_MALLOC
#define MEM_DEJA_LOG_MALLOC DEJA_LOG_MALLOC
#define MEM_DEJA_LOG_FREE   DEJA_LOG_FREE
#else
#define MEM_DEJA_LOG_MALLOC(...)
#define MEM_DEJA_LOG_FREE(...)
#endif


#if MEM_ENABLE_DEJA_SYS_MALLOC
#define MEM_DEJA_LOG_SYS_MALLOC DEJA_LOG_MALLOC
#define MEM_DEJA_LOG_SYS_FREE   DEJA_LOG_FREE
#else
#define MEM_DEJA_LOG_SYS_MALLOC(...)
#define MEM_DEJA_LOG_SYS_FREE(...)
#endif

#define AlignUp(val, align)  (((val)+(align-1))&~((align)-1))

namespace px
{
namespace Memory
{
    void Allocator::Shutdown()
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualShutdown");

        uintptr_t FirstPage = VirtualGetPageIndex(mVirtualBaseAddress);
        uintptr_t LastPage = VirtualGetPageIndex(mVirtualBaseAddress + mVirtualMemorySize - 1);

        for (uintptr_t i = FirstPage; i <= LastPage; i++)
        {
            MEM_DEJA_LOG_SYS_FREE((void*)(mVirtualBaseAddress + i * kVirtualPageSize), __FILE__, __LINE__);
            SystemVirtualDecommit(
                (void*)(mVirtualBaseAddress + i * kVirtualPageSize),
                kVirtualPageSize);
            mMemoryStatus.VirtualPagesComitted.Decrement();
            mMemoryStatus.VirtualPagesUnused.Increment();
            mMemoryStatus.VirtualBytesComitted.Subtract(kVirtualPageSize);
        }

        SystemVirtualFree((void*)mVirtualBaseAddress, mVirtualMemorySize);
        Construct();
    }

    bool Allocator::Initialize(uintptr_t VirtualAddressRange, int VirtualHashTableSize)
    {
        // DEJA_CONTEXT("px::Memory::Allocator::Initialize");
        LockGuard VirtualGuard(mVirtualMutex);
        MEM_ASSERT(VirtualHashTableSize != 0);
        MEM_ASSERT(VirtualAddressRange >= kVirtualPageSize);

        // Init bins
        for (int i = 0; i < kVirtualBinCount; i++)
            mVirtualFreeBins[i].pFreeList = 0;

        // Setup the initial hash table
        mVirtualHashTable = mVirtualInitialHashTable;
        mVirtualHashTableSize = sizeof(mVirtualInitialHashTable) / sizeof(mVirtualInitialHashTable[0]);
        memset(mVirtualInitialHashTable, 0, sizeof(mVirtualInitialHashTable));

        // Setup the initial block pool
        int ClusterSize = sizeof(mVirtualInitialBlockPool)/ sizeof(mVirtualInitialBlockPool[0]) - 1;
        int x = ClusterSize - 1;
        for (int i = 0; i < x; i++)
            mVirtualInitialBlockPool[i].pNextFree = &mVirtualInitialBlockPool[i+1];
        mVirtualInitialBlockPool[x].pNextFree = NULL;
        mpVirtualFreeBlockPool = mVirtualInitialBlockPool;
        mVirtualFreeBlockCount = ClusterSize;
        mMemoryStatus.VirtualBlocksReserved.Set(ClusterSize);
        mMemoryStatus.VirtualBlocksUnused.Set(ClusterSize);

        // Setup the initial free block
        mVirtualBaseAddress = (uintptr_t) SystemVirtualReserve(VirtualAddressRange);
        MEM_ASSERT(mVirtualBaseAddress != 0 && "Failed to reserve virtual address space.");
        if (mVirtualBaseAddress == 0)
            return false;
        mVirtualMemorySize = VirtualAddressRange;

        uintptr_t PageCount = VirtualAddressRange / kVirtualPageSize;
        uintptr_t BufferSize = sizeof(VirtualPage) * PageCount;
        uintptr_t BlockPtr = mVirtualBaseAddress;
        uintptr_t BlockSize = mVirtualMemorySize;

        if (mpInternalAllocator == this)
        {
            uintptr_t AlignedSize = AlignUp(BufferSize, kVirtualPageSize);
            MEM_ASSERT(AlignedSize >= BufferSize);

            mpVirtualPageArray = (VirtualPage*) SystemVirtualCommit((void*)mVirtualBaseAddress, AlignedSize);
            MEM_ASSERT(mpVirtualPageArray != NULL && "Unable to map virtual page array.");
            if (mpVirtualPageArray == NULL)
                return false;
            mVirtualPageCount = PageCount;
            memset(mpVirtualPageArray, 0, AlignedSize);
            MEM_DEJA_LOG_MALLOC(mpVirtualPageArray, AlignedSize, __FILE__, __LINE__);

            BlockPtr = mVirtualBaseAddress + AlignedSize;
            BlockSize = mVirtualMemorySize - AlignedSize;
        }
        else
        {
            mpVirtualPageArray = (VirtualPage*) mpInternalAllocator->VirtualAlloc(BufferSize, 8, "System/Memory/PageTable");
            MEM_ASSERT(mpVirtualPageArray != NULL && "Unable to allocate virtual page array.");
            if (mpVirtualPageArray == NULL)
                return false;
            mVirtualPageCount = PageCount;
            memset(mpVirtualPageArray, 0, BufferSize);
        }

        VirtualBlock* pBlock = VirtualCreateBlock(BlockPtr, BlockSize, BLOCK_FREE);
        mMemoryStatus.VirtualPagesReserved.Add(PageCount);
        mMemoryStatus.VirtualPagesUnused.Add(PageCount);
        mMemoryStatus.VirtualBytesReserved.Add(BlockSize);
        mMemoryStatus.VirtualBlockFragments.Increment();
        x = VirtualGetBinIndex(BlockSize);
        mVirtualFreeBins[x].pFreeList = pBlock;

        // Setup the primary hash table
        BufferSize = sizeof(VirtualBlock*) * VirtualHashTableSize;
        VirtualBlock** VirtualHashTable = (VirtualBlock**) mpInternalAllocator->VirtualAlloc(BufferSize, 8, "System/Memory/HashTable");
        MEM_ASSERT(VirtualHashTable && "Failed to allocate memory for the hash table.");
        if (VirtualHashTable == NULL)
            return false;
        memset(VirtualHashTable, 0, BufferSize);

        int c = sizeof(mVirtualInitialHashTable)/sizeof(mVirtualInitialHashTable[0]);
        for (int i = 0; i < c; i++)
        {
            if (mVirtualInitialHashTable[i])
                VirtualHashRemove(mVirtualInitialHashTable[i]->MemoryPtr);
        }

        mVirtualHashTable = VirtualHashTable;
        mVirtualHashTableSize = VirtualHashTableSize;

        for (int i = 0; i < c; i++)
        {
            if (mVirtualInitialHashTable[i])
                VirtualHashInsert(mVirtualInitialHashTable[i]);
        }

        // Ready!
        return true;
    }

    void* Allocator::VirtualAlloc(uintptr_t Size, uint_t Alignment, const char* pName)
    {
        // DEJA_CONTEXT("px::Memory::Allocator::VirtualAlloc");

        if (Size == 0)
            Size = 1;

        // Allocation size is always rounded up to nearest multiple of SmallBlock.
        Size = AlignUp(Size, sizeof(SmallBlock));
        void* pMemory = NULL;
        if (Size <= kVirtualSmallBlockSize && Alignment <= 16)
        {
            if (Size < Alignment)
                Size = Alignment;
            pMemory = VirtualAllocSmallBlock((uint_t)Size, Alignment);
        }

        if (pMemory == NULL)
        {
            mVirtualMutex.Lock();
            pMemory = VirtualAllocInternal(Size, Alignment);
            mVirtualMutex.Unlock();
        }

        MEM_DEJA_LOG_MALLOC(pMemory, Size, pName, 0);
#ifdef _DEBUG
        memset(pMemory, 0xcd, Size);
#endif

        MEM_ASSERT(pMemory != NULL && "Memory allocation failed.");
        MEM_ASSERT((uintptr_t(pMemory) & Alignment - 1) == 0 && "Memory not aligned.");
        return pMemory;
    }

    void* Allocator::VirtualRealloc(void* MemoryPtr, uintptr_t NewSize, uint_t Alignment, const char* pName)
    {
        // DEJA_CONTEXT("px::Memory::Allocator::VirtualRealloc");
        // To consider: realloc in place
        void* NewMemory = VirtualAlloc(NewSize, Alignment, pName);
        if (NewMemory != NULL)
        {
            uintptr_t PageIndex = VirtualGetPageIndex((uintptr_t)MemoryPtr);
            MEM_ASSERT(PageIndex < mVirtualPageCount);
            VirtualPage &page = mpVirtualPageArray[PageIndex];
            uintptr_t CopySize = page.BlockSize;
            if (CopySize == 0)
            {
                mVirtualMutex.Lock();
                VirtualBlock* pBlock = VirtualHashLookup((uintptr_t)MemoryPtr);
                mVirtualMutex.Unlock();
                CopySize = pBlock->GetMemorySize();
            }

            if (NewSize < CopySize)
                CopySize = NewSize;
            memcpy(NewMemory, MemoryPtr, CopySize);
            VirtualFree(MemoryPtr);
        }
        return NewMemory;
    }

    void* Allocator::VirtualResize(void* MemoryPtr, uintptr_t NewSize, uint_t Alignment, const char* pName)
    {
        // This method resizes the block if possible (shrinking should always succeed unless its a small block).
        uintptr_t BlockPtr = (uintptr_t) MemoryPtr;
        if (BlockPtr < mVirtualBaseAddress ||
            BlockPtr >= mVirtualBaseAddress + mVirtualMemorySize)
        {
            // The specified memory address isn't part of our heap; probably belongs to another
            return NULL;
        }

        // Blocks served from the small block allocator cannot be resized.
        uintptr_t PageIndex = VirtualGetPageIndex((uintptr_t)MemoryPtr);
        MEM_ASSERT(PageIndex < mVirtualPageCount);
        VirtualPage &page = mpVirtualPageArray[PageIndex];
        if (page.BlockSize != 0)
            return NULL;

        LockGuard VirtualLock(mVirtualMutex);
        VirtualBlock* pBlock = VirtualHashLookup(BlockPtr);
        MEM_ASSERT(pBlock && "Failed to a locate block header for the specified memory address.");

        // Growing or shrinking?
        uintptr_t OldSize = pBlock->GetMemorySize();
        if (NewSize <= OldSize)
        {
            if (NewSize == OldSize)
                return MemoryPtr; // no change

            MEM_DEJA_LOG_FREE(MemoryPtr);
            MEM_DEJA_LOG_MALLOC(MemoryPtr, NewSize, pName, 0);

            // Shrink the current block and put the remainder in the free list
            pBlock->SetMemorySize(NewSize);
            uintptr_t RemainderPtr = BlockPtr + NewSize;
            uintptr_t RemainderSize = OldSize - NewSize;
            MEM_ASSERT(RemainderSize <= OldSize && "Overflow!");
            VirtualBlock* pOtherBlock = VirtualCreateBlock(RemainderPtr, RemainderSize, BLOCK_FREE, pBlock);
            uint_t OtherBin = VirtualGetBinIndex(RemainderSize);
            VirtualInsertFreeBlock(pOtherBlock, OtherBin);

            VirtualMapPages(BlockPtr, NewSize);
            VirtualUnmapPages(BlockPtr, OldSize);

#if MEM_VALIDATION_ON
            uintptr_t PageIndex = VirtualGetPageIndex(BlockPtr);
            VirtualPage &page = mpVirtualPageArray[PageIndex];
            MEM_ASSERT(page.BlockCount != 0);
            MEM_ASSERT(page.BlockSize == 0);
#endif

            mMemoryStatus.VirtualBlocksUnused.Increment();
            mMemoryStatus.VirtualBytesAllocated.Subtract(RemainderSize);
            mMemoryStatus.VirtualBytesUnused.Diff(mMemoryStatus.VirtualBytesComitted, mMemoryStatus.VirtualBytesAllocated);

            // All done!
            return MemoryPtr;
        }


        // Growing is not yet supported
        MEM_ASSERT_ONCE(0 && "Unsupported; memory blocks cannot resized in place.");
        return NULL;
    }

    bool Allocator::VirtualFree(void* MemoryPtr)
    {
        // DEJA_CONTEXT("px::Memory::Allocator::VirtualFree");
        uintptr_t MemoryAddress = (uintptr_t) MemoryPtr;
        if (MemoryAddress < mVirtualBaseAddress ||
            MemoryAddress >= mVirtualBaseAddress + mVirtualMemorySize)
        {
            // Reminder: it is valid C++ to 'free' nullptr
            if (MemoryPtr == NULL)
                return true;

            // return false to indicate that the memory couldn't
            // be deallocated because the memory isn't mapped to
            // this allocator.
            return false;
        }

        MEM_DEJA_LOG_FREE(MemoryPtr);

        if (!VirtualFreeSmallBlock(MemoryPtr))
        {
            mVirtualMutex.Lock();
            VirtualFreeInternal(MemoryPtr);
            mVirtualMutex.Unlock();
        }

        return true;
    }

    bool Allocator::VirtualValidateHeap(void)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualValidateHeap");

        {
            LockGuard VirtualLock(mVirtualMutex);

            //
            // Validate the global list
            //

            VirtualBlock* pBlock = mpVirtualBlockList->pGlobalNext;
            VirtualBlock* pPrev = pBlock->pGlobalPrev;
            VirtualBlock* pNext = pBlock->pGlobalNext;
            while (pNext != NULL)
            {
                // The next block must point back to us.
                MEM_ASSERT(pNext->pGlobalPrev == pBlock);
                if (!(pNext->pGlobalPrev == pBlock))
                    return false;


                // The previous block must point to us.
                MEM_ASSERT(pPrev->pGlobalNext == pBlock);
                if (!(pPrev->pGlobalNext == pBlock))
                    return false;


                // The end of the previous block must also be the start of this block
                MEM_ASSERT(pPrev->MemoryPtr + pPrev->GetMemorySize() == pBlock->MemoryPtr);
                if (!(pPrev->MemoryPtr + pPrev->GetMemorySize() == pBlock->MemoryPtr))
                    return false;


                // If this block is free, then...
                if (pBlock->Flags & BLOCK_FREE)
                {
                    // The previous block must not be free.
                    MEM_ASSERT(!(pPrev->Flags & BLOCK_FREE));
                    if (pPrev->Flags & BLOCK_FREE)
                        return false;

                    // The next block must not be free.
                    MEM_ASSERT(!(pNext->Flags & BLOCK_FREE));
                    if (pPrev->Flags & BLOCK_FREE)
                        return false;
                }

                // Advance to the next block.
                pBlock = pNext;
                pNext = pNext->pGlobalNext;
                pPrev = pPrev->pGlobalNext;
            }

            //
            // Validate the bins
            //

            for (uint_t i = 0; i < kVirtualBinCount; i++)
            {
                VirtualBlock* pBlock = mVirtualFreeBins[i].pFreeList;
                if (pBlock)
                {
                    MEM_ASSERT(pBlock->pBinPrev == NULL);
                    if (pBlock->pBinPrev != NULL)
                        return false;
                }

                while (pBlock != NULL)
                {
                    int BinIndex = VirtualGetBinIndex(pBlock->GetMemorySize());
                    VirtualBlock* pBinNext = pBlock->pBinNext;
                    VirtualBlock* pBinPrev = pBlock->pBinPrev;

                    MEM_ASSERT(i == BinIndex);
                    if (!(i == BinIndex))
                        return false;

                    if (pBinNext)
                    {
                        MEM_ASSERT(pBinNext->pBinPrev == pBlock);
                        if (pBinNext->pBinPrev != pBlock)
                            return false;
                    }

                    if (pBinPrev)
                    {
                        MEM_ASSERT(pBinPrev->pBinNext == pBlock);
                        if (pBinPrev->pBinNext != pBlock)
                            return false;
                    }

                    pBlock = pBinNext;
                }
            }

            //
            // Validate the page headers
            //

            for (uint_t i = 0; i < mVirtualPageCount; i++)
            {
                VirtualPage &page = mpVirtualPageArray[i];
                MEM_ASSERT(!(page.BlockSize > kVirtualSmallBlockSize));
                if (page.BlockSize > kVirtualSmallBlockSize)
                    return false;


                MEM_ASSERT(!(page.BlockCount > (kVirtualPageSize / 4)));
                if (page.BlockCount > (kVirtualPageSize / 4))
                    return false;
            }

            //
            // Validate the free block headers
            //

            int NumberOfFreeBlockHeaders = 0;
            pBlock = mpVirtualFreeBlockPool;
            while (pBlock != NULL)
            {
                pBlock = pBlock->pNextFree;
                NumberOfFreeBlockHeaders++;
            }

            MEM_ASSERT(NumberOfFreeBlockHeaders == mVirtualFreeBlockCount);
        }

        //
        // Validate the small block bins
        //

        for (uint_t i = 0; i < kVirtualSmallBinCount; i++)
        {
            LockGuard Lock3(mVirtualSmallBins[i].mMutex);
            SmallBlock* pSmallBlock = mVirtualSmallBins[i].pFreeList;

            if (pSmallBlock != NULL)
            {
                MEM_ASSERT(pSmallBlock->pPrev == NULL);
                if (pSmallBlock->pPrev != NULL)
                    return false;
            }

            while (pSmallBlock)
            {
                uintptr_t PageIndex = VirtualGetPageIndex((uintptr_t) pSmallBlock);
                MEM_ASSERT(PageIndex < mVirtualPageCount);
                if (!(PageIndex < mVirtualPageCount))
                    return false;


                VirtualPage &page = mpVirtualPageArray[PageIndex];
                int BinIndex = VirtualGetSmallBinIndex(page.BlockSize);
                MEM_ASSERT(i == BinIndex);
                if (!(i == BinIndex))
                    return false;


                MEM_ASSERT(!(page.BlockCount == 0));
                if (page.BlockCount == 0)
                    return false;


                if (pSmallBlock->pNext)
                    MEM_ASSERT(pSmallBlock->pNext->pPrev == pSmallBlock);
                if (pSmallBlock->pPrev)
                    MEM_ASSERT(pSmallBlock->pPrev->pNext == pSmallBlock);


                pSmallBlock = pSmallBlock->pNext;
            }
        }

        return true;
    }

    void* Allocator::VirtualGetBaseAddress()
    {
        return (void*) mVirtualBaseAddress;
    }

    uintptr_t Allocator::VirtualGetAddressRange()
    {
        return mVirtualMemorySize;
    }

    Allocator::Allocator(Allocator* pInternalAllocator)
        : mpInternalAllocator(pInternalAllocator)
    {
        if (mpInternalAllocator == NULL)
            mpInternalAllocator = this;
        Construct();
    }

    Allocator::~Allocator()
    {
        Shutdown();
    }

    void Allocator::GetMemoryStatus(MemoryStatus &st)
    {
        st = mMemoryStatus;
    }

    void Allocator::Construct()
    {
        mVirtualBaseAddress             = 0;
        mVirtualMemorySize              = 0;
        mpVirtualPageArray              = NULL;
        mVirtualPageCount               = 0;
        mpVirtualBlockList              = NULL;
        mVirtualHashTable               = NULL;
        mVirtualHashTableSize           = 0;
        mpVirtualBlockClusterChain      = NULL;
        mpVirtualFreeBlockPool          = NULL;
        mVirtualFreeBlockCount          = 0;

        for (int i = 0; i < kVirtualSmallBinCount; i++)
            mVirtualSmallBins[i].pFreeList = NULL;

        for (int i = 0; i < kVirtualBinCount; i++)
            mVirtualFreeBins[i].pFreeList = NULL;

        memset(&mMemoryStatus, 0, sizeof(mMemoryStatus));
    }

    void* Allocator::VirtualAllocSmallBlock(uint_t Size, uint_t Alignment)
    {
        // DEJA_CONTEXT("px::Memory::Allocator::VirtualAllocSmallBlock");
        MEM_ASSERT(Alignment == 4 || Alignment == 8 || Alignment == 16);
        MEM_ASSERT(Size <= kVirtualSmallBlockSize);
        int BinIndex = VirtualGetSmallBinIndex(Size);
        MEM_ASSERT(BinIndex < kVirtualSmallBinCount);

        SmallBlockBin &Bin = mVirtualSmallBins[BinIndex];
        Bin.mMutex.Lock();
        SmallBlock* pSmallBlock = Bin.pFreeList;
        while ((uintptr_t(pSmallBlock) & (Alignment - 1)) != 0)
            pSmallBlock = pSmallBlock->pNext;
        if (pSmallBlock == NULL)
        {
            DEJA_CONTEXT("px::Memory::Allocator::VirtualAllocSmallBlock >> Initializing Page");

            // The CRT will always guarantee alignment of 16 bytes and we must also honor that alignment
            uint32_t BlockSize = AlignUp(Size, 16);

            // This is an optimization that applies only to 32-bit systems
            // The 8-byte and 16-byte bins are separated for efficiency because we presume
            // an 8-byte allocation is not required to be 16-byte aligned.

            if ((sizeof(SmallBlock) == 8) && Size <= 16)
                BlockSize = AlignUp(Size, 8);

            mVirtualMutex.Lock();
            uintptr_t MemoryPtr = (uintptr_t) VirtualAllocInternal(kVirtualPageSize, kVirtualPageSize);
            uint_t Count = kVirtualPageSize / BlockSize;
            mMemoryStatus.VirtualSmallBlockPagesAllocated.Increment();
            mMemoryStatus.VirtualSmallBlockBlocksReserved.Add(Count);
            mMemoryStatus.VirtualSmallBlockBlocksUnused.Add(Count);
            mMemoryStatus.VirtualSmallBlockBytesReserved.Add(kVirtualPageSize);
            mMemoryStatus.VirtualSmallBlockBytesUnused.Add(kVirtualPageSize);
            uintptr_t PageIndex = VirtualGetPageIndex(MemoryPtr);
            VirtualPage &page = mpVirtualPageArray[PageIndex];
            MEM_ASSERT(page.BlockCount == 1);
            MEM_ASSERT(page.BlockSize == 0);
            page.BlockSize = BlockSize;
            mVirtualMutex.Unlock();

            SmallBlock* pBlockList = NULL;
            uintptr_t LastPtr = MemoryPtr + kVirtualPageSize - BlockSize;
            Bin.pFreeList = (SmallBlock*) MemoryPtr;

            while (MemoryPtr <= LastPtr)
            {
                pSmallBlock = (SmallBlock*) MemoryPtr;
                pSmallBlock->pPrev = pBlockList;
                pSmallBlock->pNext = (SmallBlock*) (MemoryPtr + BlockSize);
                pBlockList = pSmallBlock;
                MemoryPtr += BlockSize;
            }

            pSmallBlock->pNext = NULL;
            pSmallBlock =  Bin.pFreeList;
            MEM_ASSERT(pSmallBlock != NULL);
        }

        uintptr_t PageIndex = VirtualGetPageIndex((uintptr_t) pSmallBlock);
        VirtualPage &page = mpVirtualPageArray[PageIndex];

        MEM_ASSERT(page.BlockCount != 0);
        MEM_ASSERT(page.BlockSize >= Size);
        page.BlockCount++;

        if (pSmallBlock == Bin.pFreeList)
        {
            MEM_ASSERT(pSmallBlock->pPrev == NULL);
            Bin.pFreeList = pSmallBlock->pNext;
        }
        else
        {
            MEM_ASSERT(pSmallBlock->pPrev != NULL);
            MEM_ASSERT(pSmallBlock->pPrev->pNext == pSmallBlock);
            pSmallBlock->pPrev->pNext = pSmallBlock->pNext;
        }

        if (pSmallBlock->pNext)
        {
            MEM_ASSERT(pSmallBlock->pNext->pPrev != NULL);
            MEM_ASSERT(pSmallBlock->pNext->pPrev == pSmallBlock);
            pSmallBlock->pNext->pPrev = pSmallBlock->pPrev;
        }

        Bin.mMutex.Unlock();

        mMemoryStatus.VirtualSmallBlockBlocksAllocated.Increment();
        mMemoryStatus.VirtualSmallBlockBlocksUnused.Decrement();
        mMemoryStatus.VirtualSmallBlockBytesAllocated.Add(Size);
        mMemoryStatus.VirtualSmallBlockBytesUnused.Subtract(Size);

        return pSmallBlock;
    }

    void* Allocator::VirtualAllocInternal(uintptr_t Size, uint_t Alignment)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualAllocInternal");
        MEM_ASSERT(mVirtualMutex.Locked());

        if (mVirtualFreeBlockCount <= 5)
        {
            mVirtualFreeBlockCount += 0x80000000; // Lazy hack to get around recursion
            VirtualBlockCluster* pNewCluster =
                (VirtualBlockCluster*) mpInternalAllocator->VirtualAlloc(kVirtualPageSize, 8, "System/Memory/Blocks");
            mVirtualFreeBlockCount -= 0x80000000; // Put it back when you're done..
            MEM_DEJA_LOG_MALLOC(pNewCluster, kVirtualPageSize, __FILE__, __LINE__);
            pNewCluster->pNext = mpVirtualBlockClusterChain;
            mpVirtualBlockClusterChain = pNewCluster;
            const uint_t ClusterSize
                = (kVirtualPageSize
                - sizeof(VirtualBlockCluster)
                +sizeof(VirtualBlock))
                / sizeof(VirtualBlock);
            const uint_t x = ClusterSize - 1;
            for (uint_t i = 0; i < x; i++)
                pNewCluster->Blocks[i].pNextFree = &pNewCluster->Blocks[i + 1];
            pNewCluster->Blocks[x].pNextFree = mpVirtualFreeBlockPool;
            mpVirtualFreeBlockPool = &pNewCluster->Blocks[0];
            mVirtualFreeBlockCount += ClusterSize;
            mMemoryStatus.SysMemBytesAllocated.Add(kVirtualPageSize);
            mMemoryStatus.VirtualBlocksReserved.Add(ClusterSize);
            mMemoryStatus.VirtualBlocksUnused.Add(ClusterSize);
        }

        MEM_ASSERT(Alignment >= 4 && "Alignment cannot be less than 4.");
        MEM_ASSERT((!(Alignment & (Alignment - 1))) && "Alignment is not a power of 2.");
        int x = VirtualGetBinIndex(Size);
        VirtualBlock* pBlock = mVirtualFreeBins[x].pFreeList;
        uintptr_t AlignmentMask = Alignment - 1;
        uintptr_t AlternateSize = Size + Alignment;
        {
            DEJA_CONTEXT("px::Memory::Allocator::VirtualAllocInternal >> Searching");
            // Skip past any block which is too small.
            while (pBlock && pBlock->GetMemorySize() < Size)
                pBlock = pBlock->pBinNext;

            while (pBlock == NULL)
            {
                // Try a higher bin (larger blocks)
                if (++x == kVirtualBinCount)
                    break;
                pBlock = mVirtualFreeBins[x].pFreeList;
            }

            while (pBlock)
            {
                uintptr_t BlockSize = pBlock->GetMemorySize();
                uintptr_t BlockPtr  = pBlock->MemoryPtr;
                MEM_ASSERT(Size <= BlockSize);

                // Block is already aligned just right
                if (0 == (BlockPtr & AlignmentMask))
                {
                    MEM_ASSERT(Size <= BlockSize);
                    break;
                }

                // Block is large enough to do alignment
                if (BlockSize >= AlternateSize)
                    break;

                // Block can be aligned but just barely
                if (Alignment - (BlockPtr & AlignmentMask) <= (BlockSize - Size))
                    break;

                pBlock = pBlock->pBinNext;
                while (pBlock == NULL)
                {
                    // Try a higher bin (larger blocks)
                    if (++x == kVirtualBinCount)
                        break;
                    pBlock = mVirtualFreeBins[x].pFreeList;
                }
            }

            if (pBlock == NULL)
            {
                __memory_assert(); //Moved Up as Dump Causes a crash on NX
                //Dump("Out of memory.");
                //__memory_assert();
            }
        }

        MEM_ASSERT(pBlock->Flags & BLOCK_FREE);
        uintptr_t BlockPtr = pBlock->MemoryPtr;
        uintptr_t BlockSize = pBlock->GetMemorySize();
        MEM_ASSERT(BlockSize >= Size);
        VirtualRemoveFreeBlock(pBlock, x);
        pBlock->Flags &= ~BLOCK_FREE;

        if (0 != (BlockPtr & AlignmentMask)) // Not aligned
        {
            uintptr_t AlignedBlockPtr = (pBlock->MemoryPtr + AlignmentMask) & ~AlignmentMask;
            uintptr_t AlignedBlockSize = BlockSize - (AlignedBlockPtr - BlockPtr);
            MEM_ASSERT(Size <= AlignedBlockSize);
            pBlock->SetMemorySize(AlignedBlockPtr - BlockPtr);
            pBlock->Flags |= BLOCK_FREE;
            VirtualBlock* pAlignedBlock = VirtualCreateBlock(AlignedBlockPtr, AlignedBlockSize, 0, pBlock);
            int k = VirtualGetBinIndex(pBlock->GetMemorySize());
            VirtualInsertFreeBlock(pBlock, k);

            pBlock    = pAlignedBlock;
            BlockPtr  = AlignedBlockPtr;
            BlockSize = AlignedBlockSize;
            x = VirtualGetBinIndex(BlockSize);
        }

        if (BlockSize != Size) // Not the right size
        {
            pBlock->SetMemorySize(Size);
            uintptr_t RemainderPtr = BlockPtr + Size;
            uintptr_t RemainderSize = BlockSize - Size;
            MEM_ASSERT(RemainderSize <= BlockSize && "Overflow!");
            VirtualBlock* pOtherBlock = VirtualCreateBlock(RemainderPtr, RemainderSize, BLOCK_FREE, pBlock);
            uint_t OtherBin = VirtualGetBinIndex(RemainderSize);
            VirtualInsertFreeBlock(pOtherBlock, OtherBin);
        }

        VirtualHashInsert(pBlock);
        MEM_ASSERT(pBlock->MemoryPtr == BlockPtr);
        MEM_ASSERT((pBlock->MemoryPtr & AlignmentMask) == 0);
        VirtualMapPages(BlockPtr, Size);

        mMemoryStatus.VirtualBytesAllocated.Add(Size);
        mMemoryStatus.VirtualBytesUnused.Diff(mMemoryStatus.VirtualBytesComitted, mMemoryStatus.VirtualBytesAllocated);

        return (void*) BlockPtr;
    }

    bool Allocator::VirtualFreeSmallBlock(void* MemoryPtr)
    {
        // DEJA_CONTEXT("px::Memory::Allocator::VirtualFreeSmall");
        uintptr_t PageIndex = VirtualGetPageIndex((uintptr_t)MemoryPtr);
        MEM_ASSERT(PageIndex < mVirtualPageCount); // Its not a virtual address if the assert trips.
        VirtualPage &page = mpVirtualPageArray[PageIndex];
        if (page.BlockSize == 0)
            return false; // Not one of mine.

        // We own it; free it
        MEM_ASSERT(page.BlockSize >= sizeof(SmallBlock));
        uint_t x = VirtualGetSmallBinIndex(page.BlockSize);
        SmallBlockBin& bin = mVirtualSmallBins[x];
        bin.mMutex.Lock();
        SmallBlock* pSmallBlock = (SmallBlock*) MemoryPtr;
        pSmallBlock->pPrev = NULL;
        pSmallBlock->pNext = bin.pFreeList;
        if (bin.pFreeList != NULL)
            bin.pFreeList->pPrev = pSmallBlock;
        bin.pFreeList = pSmallBlock;

        mMemoryStatus.VirtualSmallBlockBlocksAllocated.Decrement();
        mMemoryStatus.VirtualSmallBlockBlocksUnused.Increment();
        mMemoryStatus.VirtualSmallBlockBytesAllocated.Subtract(page.BlockSize);
        mMemoryStatus.VirtualSmallBlockBytesUnused.Add(page.BlockSize);

        page.BlockCount--;
        if (page.BlockCount == 1)
        {
            // TODO: this is a major performance bottleneck right now

            DEJA_CONTEXT("px::Memory::Allocator::VirtualFreeSmallBlock >> Removing Page");

            uintptr_t PagePtr = VIRTUAL_PAGE_MASK((uintptr_t)MemoryPtr);
            uintptr_t BlockPtr = PagePtr;
            int BlockCount = kVirtualPageSize / page.BlockSize;
            int BlockSize = page.BlockSize;

            for (int i = 0; i < BlockCount; i++)
            {
                pSmallBlock = (SmallBlock*) BlockPtr;
                BlockPtr += BlockSize;
                if (pSmallBlock == bin.pFreeList)
                {
                    MEM_ASSERT(pSmallBlock->pPrev == NULL);
                    bin.pFreeList = pSmallBlock->pNext;
                }
                else
                {
                    MEM_ASSERT(pSmallBlock->pPrev != NULL);
                    MEM_ASSERT(pSmallBlock->pPrev->pNext == pSmallBlock);
                    pSmallBlock->pPrev->pNext = pSmallBlock->pNext;
                }

                if (pSmallBlock->pNext)
                {
                    MEM_ASSERT(pSmallBlock->pNext->pPrev == pSmallBlock);
                    pSmallBlock->pNext->pPrev = pSmallBlock->pPrev;
                }

                pSmallBlock->pPrev = NULL;
                pSmallBlock->pNext = NULL;
            }

            mMemoryStatus.VirtualSmallBlockPagesAllocated.Decrement();
            mMemoryStatus.VirtualSmallBlockBytesReserved.Subtract(kVirtualPageSize);
            mMemoryStatus.VirtualSmallBlockBytesUnused.Subtract(kVirtualPageSize);
            mMemoryStatus.VirtualSmallBlockBlocksReserved.Subtract(BlockCount);
            mMemoryStatus.VirtualSmallBlockBlocksUnused.Subtract(BlockCount);

            // Free the whole page!
            mVirtualMutex.Lock();
            VirtualFreeInternal((void*)PagePtr);
            mVirtualMutex.Unlock();
        }

        bin.mMutex.Unlock();
        return true;
    }



    void Allocator::VirtualFreeInternal(void* MemoryPtr)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualFreeInternal");
        MEM_ASSERT(mVirtualMutex.Locked());
        uintptr_t BlockPtr = (uintptr_t) MemoryPtr;
        VirtualBlock* pBlock = VirtualHashRemove(BlockPtr);
        uintptr_t BlockSize = pBlock->GetMemorySize();
        mMemoryStatus.VirtualBytesAllocated.Subtract(BlockSize);
        mMemoryStatus.VirtualBytesUnused.Diff(mMemoryStatus.VirtualBytesComitted, mMemoryStatus.VirtualBytesAllocated);
        MEM_ASSERT(pBlock->MemoryPtr == (uintptr_t) MemoryPtr);
        unsigned int BinIndex = VirtualGetBinIndex(BlockSize);
        MEM_ASSERT(BinIndex < kVirtualBinCount);
        pBlock->Flags |= BLOCK_FREE;

        VirtualBlock* pPrevBlock = pBlock->pGlobalPrev;
        if (pPrevBlock && pPrevBlock->Flags & BLOCK_FREE)
        {
            uintptr_t OtherSize = pPrevBlock->GetMemorySize();
            int OtherBin = VirtualGetBinIndex(OtherSize);
            VirtualRemoveFreeBlock(pPrevBlock, OtherBin);
            pPrevBlock->SetMemorySize(OtherSize + pBlock->GetMemorySize());
            pPrevBlock->pGlobalNext = pBlock->pGlobalNext;
            if (pBlock->pGlobalNext)
                pBlock->pGlobalNext->pGlobalPrev = pPrevBlock;
            VirtualDestroyBlock(pBlock);
            pBlock = pPrevBlock;
        }

        VirtualBlock* pNextBlock = pBlock->pGlobalNext;
        if (pNextBlock && pNextBlock->Flags & BLOCK_FREE)
        {
            uintptr_t OtherSize = pNextBlock->GetMemorySize();
            uint32_t OtherBin = VirtualGetBinIndex(OtherSize);
            VirtualRemoveFreeBlock(pNextBlock, OtherBin);
            pNextBlock->SetMemorySize(OtherSize + pBlock->GetMemorySize());
            pNextBlock->MemoryPtr = pBlock->MemoryPtr;
            pNextBlock->pGlobalPrev = pBlock->pGlobalPrev;
            if (pBlock->pGlobalPrev)
                pBlock->pGlobalPrev->pGlobalNext = pNextBlock;
            if (mpVirtualBlockList == pBlock)
                mpVirtualBlockList = pNextBlock;
            VirtualDestroyBlock(pBlock);
            pBlock = pNextBlock;
        }

        BinIndex = VirtualGetBinIndex(pBlock->GetMemorySize());
        MEM_ASSERT(pBlock->Flags & BLOCK_FREE);
        VirtualInsertFreeBlock(pBlock, BinIndex);
        VirtualUnmapPages(BlockPtr, BlockSize);
    }

    Allocator::VirtualBlock* Allocator::VirtualCreateBlock(uintptr_t BlockPtr, uintptr_t BlockSize, uint_t Flags, VirtualBlock* pOtherBlock)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualCreateBlock");
        MEM_ASSERT(mVirtualMutex.Locked());
        MEM_ASSERT(mpVirtualFreeBlockPool != 0 && "Out of block headers.");

        VirtualBlock* pBlock = mpVirtualFreeBlockPool;
        mpVirtualFreeBlockPool = pBlock->pNextFree;
        mVirtualFreeBlockCount--;
        MEM_ASSERT(mVirtualFreeBlockCount != 0); // ruh-roh!
        pBlock->MemoryPtr = BlockPtr;
        pBlock->SetMemorySize(BlockSize);
        pBlock->pBinPrev = NULL;
        pBlock->pBinNext = NULL;
        pBlock->Flags = Flags;

        mMemoryStatus.VirtualBlocksAllocated.Increment();
        mMemoryStatus.VirtualBlocksUnused.Decrement();

        // Insert sorted by address into global list
        if (pOtherBlock == NULL)
            pOtherBlock = mpVirtualBlockList;
        if (pOtherBlock)
        {
            while (pOtherBlock->MemoryPtr < BlockPtr)
            {
                if (pOtherBlock->pGlobalNext == NULL)
                    break;
                pOtherBlock = pOtherBlock->pGlobalNext;
            }
            while (pOtherBlock && pOtherBlock->MemoryPtr >= BlockPtr)
                pOtherBlock = pOtherBlock->pGlobalPrev;
            pBlock->pGlobalPrev = pOtherBlock;
        }
        if (pOtherBlock)
        {
            pBlock->pGlobalNext = pOtherBlock->pGlobalNext;
            if (pBlock->pGlobalNext)
                pBlock->pGlobalNext->pGlobalPrev = pBlock;
            pOtherBlock->pGlobalNext = pBlock;
        }
        else
        {
            pBlock->pGlobalNext = mpVirtualBlockList;
            MEM_ASSERT(mpVirtualBlockList == NULL);
            mpVirtualBlockList = pBlock;
        }
        MEM_ASSERT(mpVirtualBlockList != NULL);
        return pBlock;
    }

    void Allocator::VirtualDestroyBlock(VirtualBlock* pBlock)
    {
        MEM_ASSERT(mVirtualMutex.Locked());
#ifdef _DEBUG
        memset(pBlock, 0xfe, sizeof(VirtualBlock));
#endif
        mMemoryStatus.VirtualBlocksUnused.Increment();
        mMemoryStatus.VirtualBlocksAllocated.Decrement();
        pBlock->pNextFree = mpVirtualFreeBlockPool;
        mpVirtualFreeBlockPool = pBlock;
        mVirtualFreeBlockCount++;
    }

    void Allocator::VirtualInsertFreeBlock(VirtualBlock* pBlock, int BinIndex)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualInsertFreeBlock");
        MEM_ASSERT(mVirtualMutex.Locked());
        MEM_ASSERT(pBlock->Flags & BLOCK_FREE);
        VirtualBin &Bin = mVirtualFreeBins[BinIndex];
        const uintptr_t BlockSize = pBlock->GetMemorySize();
        VirtualBlock* pNextBlock = Bin.pFreeList;
        VirtualBlock* pPrevBlock = NULL;
        while (pNextBlock && BlockSize > pNextBlock->GetMemorySize())
        {
            pPrevBlock = pNextBlock;
            pNextBlock = pNextBlock->pBinNext;
        }
        pBlock->pBinPrev = pPrevBlock;
        pBlock->pBinNext = pNextBlock;
        if (pNextBlock)
            pNextBlock->pBinPrev = pBlock;
        if (pPrevBlock)
            pPrevBlock->pBinNext = pBlock;
        if (pPrevBlock == NULL)
            Bin.pFreeList = pBlock;
        MEM_ASSERT(Bin.pFreeList != NULL);
        mMemoryStatus.VirtualBlockFragments.Increment();
    }

    void Allocator::VirtualRemoveFreeBlock(VirtualBlock* pBlock, int BinIndex)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualRemoveFreeBlock");
        MEM_ASSERT(mVirtualMutex.Locked());
        VirtualBin &Bin = mVirtualFreeBins[BinIndex];
        MEM_ASSERT(Bin.pFreeList != NULL);
        if (pBlock->pBinPrev)
            pBlock->pBinPrev->pBinNext = pBlock->pBinNext;
        if (pBlock->pBinNext)
            pBlock->pBinNext->pBinPrev = pBlock->pBinPrev;
        if (Bin.pFreeList == pBlock)
            Bin.pFreeList = pBlock->pBinNext;
        pBlock->pBinPrev = NULL;
        pBlock->pBinNext = NULL;
        mMemoryStatus.VirtualBlockFragments.Decrement();
    }

    void Allocator::VirtualHashInsert(VirtualBlock* pBlock)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualHashInsert");
        MEM_ASSERT(mVirtualMutex.Locked());
        MEM_ASSERT(pBlock->pBinPrev == NULL);
        MEM_ASSERT(pBlock->pBinNext == NULL);
        const uintptr_t BlockPtr = pBlock->MemoryPtr;
        int HashValue = VirtualGetHash(BlockPtr);
        VirtualBlock* pOtherBlock = mVirtualHashTable[HashValue];
        mVirtualHashTable[HashValue] = pBlock;
        if (pOtherBlock)
            pOtherBlock->pBinPrev = pBlock;
        pBlock->pBinNext = pOtherBlock;
    }


    Allocator::VirtualBlock* Allocator::VirtualHashRemove(uintptr_t BlockPtr)
    {
        DEJA_CONTEXT("px::Memory::Allocator::VirtualHashRemove");
        MEM_ASSERT(mVirtualMutex.Locked());
        int HashValue = VirtualGetHash(BlockPtr);
        VirtualBlock* pBlock = mVirtualHashTable[HashValue];
        while (BlockPtr != pBlock->MemoryPtr)
            pBlock = pBlock->pBinNext;
        VirtualBlock* pPrev = pBlock->pBinPrev;
        pBlock->pBinPrev = NULL;
        VirtualBlock* pNext = pBlock->pBinNext;
        pBlock->pBinNext = NULL;

        if (pNext != NULL)
        {
            MEM_ASSERT(pNext->pBinPrev == pBlock);
            pNext->pBinPrev = pPrev;
        }

        if (mVirtualHashTable[HashValue] == pBlock)
        {
            MEM_ASSERT(pPrev == NULL);
            mVirtualHashTable[HashValue] = pNext;
        }
        else
        {
            MEM_ASSERT(pPrev != NULL);
            MEM_ASSERT(pPrev->pBinNext == pBlock);
            pPrev->pBinNext = pNext;
        }

        return pBlock;
    }

    Allocator::VirtualBlock* Allocator::VirtualHashLookup(uintptr_t BlockPtr)
    {
        MEM_ASSERT(mVirtualMutex.Locked());
        VirtualBlock* pBlock = mVirtualHashTable[VirtualGetHash(BlockPtr)];
        while (pBlock)
        {
            if (pBlock->MemoryPtr == BlockPtr)
                return pBlock;
            pBlock = pBlock->pBinNext;
        }
        MEM_ASSERT(0 && "The specified memory address is invalid!");
        return NULL;
    }

    void Allocator::VirtualMapPages(uintptr_t BlockPtr, uintptr_t BlockSize)
    {
        uintptr_t FirstPage = VirtualGetPageIndex(BlockPtr);
        uintptr_t LastPage = VirtualGetPageIndex(BlockPtr + BlockSize - 1);

        for (uintptr_t i = FirstPage; i <= LastPage; i++)
        {
            if (mpVirtualPageArray[i].BlockCount == 0)
            {
                void* Result = SystemVirtualCommit(
                    (void*)(mVirtualBaseAddress + i * kVirtualPageSize),
                    kVirtualPageSize);
                MEM_ASSERT(Result != NULL && "Failed to commit page.");
                MEM_DEJA_LOG_SYS_MALLOC((void*)(mVirtualBaseAddress + i * kVirtualPageSize), kVirtualPageSize, __FILE__, __LINE__);
                mpVirtualPageArray[i].BlockSize = 0;
                mMemoryStatus.VirtualPagesComitted.Increment();
                mMemoryStatus.VirtualPagesUnused.Decrement();
                mMemoryStatus.VirtualBytesComitted.Add(kVirtualPageSize);
            }
            MEM_ASSERT(mpVirtualPageArray[i].BlockCount != 65535);
            mpVirtualPageArray[i].BlockCount++;
        }
    }

    void Allocator::VirtualUnmapPages(uintptr_t BlockPtr, uintptr_t BlockSize)
    {
        uintptr_t FirstPage = VirtualGetPageIndex(BlockPtr);
        uintptr_t LastPage = VirtualGetPageIndex(BlockPtr + BlockSize - 1);
        for (uintptr_t i = FirstPage; i <= LastPage; i++)
        {
            MEM_ASSERT(mpVirtualPageArray[i].BlockCount != 0);
            mpVirtualPageArray[i].BlockCount--;
            if (mpVirtualPageArray[i].BlockCount == 0)
            {
                MEM_DEJA_LOG_SYS_FREE((void*)(mVirtualBaseAddress + i * kVirtualPageSize), __FILE__, __LINE__);
                SystemVirtualDecommit(
                    (void*)(mVirtualBaseAddress + i * kVirtualPageSize),
                    kVirtualPageSize);
                mMemoryStatus.VirtualPagesComitted.Decrement();
                mMemoryStatus.VirtualPagesUnused.Increment();
                mMemoryStatus.VirtualBytesComitted.Subtract(kVirtualPageSize);
            }
        }
    }

    void Allocator::Dump(const char* msg)
    {
        if (msg) printf("%s\n", msg);

        VirtualBlock* pBlock = mpVirtualBlockList;
        VirtualBlock* pLargestFree = mpVirtualBlockList;
        uintptr_t AvailableFreeMemory = 0;
        uintptr_t LargestFreeBlock = 0;

        for (int i = 0; pBlock != NULL; i++)
        {
            char AllocState = 'A';
            if (pBlock->Flags & BLOCK_FREE)
            {
                AvailableFreeMemory += pBlock->GetMemorySize();
                if (pBlock->GetMemorySize() > pLargestFree->GetMemorySize())
                    pLargestFree = pBlock;
                AllocState = 'F';
            }

            printf("%08d: %c %08x %10d\n", i,
                AllocState,
                pBlock->MemoryPtr,
                pBlock->GetMemorySize());

            pBlock = pBlock->pGlobalNext;
        }

        if (pLargestFree != NULL)
            LargestFreeBlock = pLargestFree->GetMemorySize();

        printf("\n");

        printf(
            "Heap information:\n"
            "    Size       %d bytes\n"
            "    Address    %08x\n"
            "    Free       %d bytes\n"
            "    LFB        %d bytes\n",
            mVirtualMemorySize,
            mVirtualBaseAddress,
            AvailableFreeMemory,
            LargestFreeBlock);

        printf("\n");
    }

} // namespace Memory
} // namespace px

//==============================================================================
