﻿/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     sleep.cpp

  Copyright (C)2010 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 25650 $
 *---------------------------------------------------------------------------*/
#include "sys_Sleep.h"
        

namespace uji {
namespace sys {
    

/*------------------------------------------------------------------------*
        本体スリープに関連する処理
 *------------------------------------------------------------------------*/
nn::os::LightEvent  SleepHandler::s_AwakeEvent;
bool                SleepHandler::s_IsInitialized = false;

/*------------------------------------------------------------------------*/
void SleepHandler::Initialize( void )
{
    // スリープ関連のコールバック設定
    nn::applet::SetSleepQueryCallback(SleepQueryCallback, 0);
    nn::applet::SetAwakeCallback(AwakeCallback, 0);

    s_AwakeEvent.Initialize(false);

    // 起床中は常に Signal 状態にしておく
    s_AwakeEvent.Signal();

    // アプレットライブラリに関する機能を有効にします。
    // 引数を false にしておくと、nn::applet::EnableSleep() を呼ぶまでは
    // 自動的にスリープ拒否を行います。
    nn::applet::Enable(false);
    
    s_IsInitialized = true;
}

/*------------------------------------------------------------------------*/
void SleepHandler::Finalize( void )
{
    nn::applet::DisableSleep();    
    s_AwakeEvent.Finalize();
    nn::applet::SetSleepQueryCallback(NULL, 0);
    nn::applet::SetAwakeCallback(NULL, 0);
    s_IsInitialized = false;
}

/*------------------------------------------------------------------------*/
void SleepHandler::EnableSleep()
{
    // スリープ要求に対する返答を有効にします。
    // 蓋の状態チェックを行い蓋が閉じられているならスリープ要求が発生します。
    nn::applet::EnableSleep(true);
}

/*------------------------------------------------------------------------*/
void SleepHandler::DisableSleep()
{
    // スリープ要求に対する返答を無効にします。
    nn::applet::DisableSleep(true);
}

/*------------------------------------------------------------------------*
    システムスリープの指示が来ているかどうかを返します。
 *------------------------------------------------------------------------*/
bool SleepHandler::IsSleepRequested( void )
{
    NN_TASSERT_(s_IsInitialized);
    
    if ( nn::applet::IsExpectedToReplySleepQuery() )
    {
        return true;
    }
    return false;
}

/*------------------------------------------------------------------------*
    本体スリープ状態に移行させるための処理を行います。
    グラフィックスなどの処理がスリープに入っても問題ない
    適切なタイミングで呼び出す必要があります。
 *------------------------------------------------------------------------*/
void SleepHandler::SleepSystem( void )
{
    // 指示が来ていない場合は何もせず戻る。
    if ( !IsSleepRequested() )
    {
        return;
    }
    
    nn::applet::ReplySleepQuery(nn::applet::REPLY_ACCEPT);
    
    s_AwakeEvent.Wait();    
}

/*------------------------------------------------------------------------*
    スリープ問い合わせ時に呼ばれるコールバック
 *------------------------------------------------------------------------*/
AppletQueryReply SleepHandler::SleepQueryCallback( uptr arg )
{
    NN_UNUSED_VAR(arg);
    
    s_AwakeEvent.ClearSignal();
    
#ifdef EVA_ENABLE_SLEEP            
    if ( s_IsInitialized )
    {
        return nn::applet::REPLY_LATER;
    }
    else
#endif        
    {
        return nn::applet::REPLY_REJECT;
    }
}

/*------------------------------------------------------------------------*
    スリープ復帰時に呼ばれるコールバック
 *------------------------------------------------------------------------*/
void SleepHandler::AwakeCallback( uptr arg )
{
    NN_UNUSED_VAR(arg);
    s_AwakeEvent.Signal();
}


}   // namespace
}
