/*---------------------------------------------------------------------------*
  Project:  
  File: sys_GraphicsDrawing.cpp


 *---------------------------------------------------------------------------*/
#include "sys_GraphicsDrawing.h"
//#include "sys_App.h"
//#include "sys_Memory.h"    

#include <nn/gx.h>
#include <nn/os.h>
#include <nn/fnd.h>
//#include <nn/fs.h>
//#include <nw/demo/demo_Utility.h>

namespace uji {
namespace sys {
    

// カラーバッファのクリアカラーデフォルト値
const f32 GraphicsDrawing::DEFAULT_CLEAR_COLOR_RED      = 0.5f;
const f32 GraphicsDrawing::DEFAULT_CLEAR_COLOR_GREEN    = 0.5f;
const f32 GraphicsDrawing::DEFAULT_CLEAR_COLOR_BLUE     = 0.5f;
const f32 GraphicsDrawing::DEFAULT_CLEAR_COLOR_ALPHA    = 1.0f;

GraphicsDrawing* GraphicsDrawing::m_pInstance = 0;

/*---------------------------------------------------------------------------
  Desc: 初期化処理
---------------------------------------------------------------------------*/
int GraphicsDrawing::Initialize(void)
{
    // アロケータを引数とした後削除予定
    uji::sys::SysApp* sysApp = uji::sys::SysApp::GetInstance();
    
    m_FcramAddress = reinterpret_cast<uptr>(AllocDeviceMemory(GRAPHICS_MEMORY_SIZE, 4));
    m_DrawFramework->Initialize(m_FcramAddress, GRAPHICS_MEMORY_SIZE);
    nngxCmdlistStorage(0x200000, 128);
    
    m_ShapePgID = glCreateProgram();
    m_ShapeShID = glCreateShader(GL_VERTEX_SHADER);

    // ファイルシステムによるシェーダバイナリの読み込み
    nn::fs::FileReader shaderReader(L"rom:/sys/shader.shbin");
    size_t size = shaderReader.GetSize();
    m_pShapeShader = sysApp->AllocateDeviceMemory(size);
    s32 read = shaderReader.Read(m_pShapeShader, size);
    NN_ASSERT(read == size);
    glShaderBinary(1, &m_ShapeShID, GL_PLATFORM_BINARY_DMP, m_pShapeShader, read);

    glAttachShader(m_ShapePgID, m_ShapeShID);
    glAttachShader(m_ShapePgID, GL_DMP_FRAGMENT_SHADER_DMP);

    glBindAttribLocation(m_ShapePgID, 0, "aPosition");
    glBindAttribLocation(m_ShapePgID, 1, "aColor");

    glLinkProgram(m_ShapePgID);
    glValidateProgram(m_ShapePgID); 
    glUseProgram(m_ShapePgID);

    glClearDepthf(1.f);
    m_DrawFramework->SetClearColor(NN_GX_DISPLAY0, 
        DEFAULT_CLEAR_COLOR_RED, 
        DEFAULT_CLEAR_COLOR_GREEN, 
        DEFAULT_CLEAR_COLOR_BLUE, 
        DEFAULT_CLEAR_COLOR_ALPHA);
    m_DrawFramework->SetClearColor(NN_GX_DISPLAY1,
        DEFAULT_CLEAR_COLOR_RED, 
        DEFAULT_CLEAR_COLOR_GREEN, 
        DEFAULT_CLEAR_COLOR_BLUE, 
        DEFAULT_CLEAR_COLOR_ALPHA);
    
    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_STENCIL_TEST);
 
    glUniform3i(glGetUniformLocation(m_ShapePgID, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
    glUniform3i(glGetUniformLocation(m_ShapePgID, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

    // フォントの構築
    bool bSuccess = InitFont(NW_DEMO_FILE_PATH(L"sys/ipa.bcfnt"));
    NN_ASSERTMSG(bSuccess, "Fail to load ResFont.");
    InitFontShaders();       

    m_TextWriter.SetFont(&m_Font);    
    
    // 描画用バッファの確保
    m_pDrawStringBuf = AllocDispStringBuffer(67*24);
    
    return 0;
}

int GraphicsDrawing::Finalize(void)
{
    // TODO: アロケータを引数とした後に削除
    uji::sys::SysApp* sysApp = uji::sys::SysApp::GetInstance();    
    
    m_RectDrawer.Finalize();
    
    m_DrawFramework->Finalize();
    Free(reinterpret_cast<void*>(m_FcramAddress));
    
    sysApp->Free(m_pShapeShader);
    sysApp->Free(m_pDrawStringBuf);
    
    return 0;
}
   

//============================================================================
//============================================================================
//                              シェイプ描画
//============================================================================
//============================================================================

const GLushort s_SquarVertexIndexs[] =
{
    1, 0, 2, 3
};

/*---------------------------------------------------------------------------
  Desc: GraphicsDrawing のシェイプレンダリングの開始関数
---------------------------------------------------------------------------*/
void GraphicsDrawing::BeginDrawingShape()
{    
    glUseProgram(m_ShapePgID);
    
    glBindTexture(GL_TEXTURE_2D, 0);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_STENCIL_TEST);
    
    glFrontFace(GL_CCW);
    
    // ソースアルファでブレンド
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);    
    glBlendEquation(GL_FUNC_ADD);
    
    // バッファオブジェクト無効化
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);   
    
    this->SetupTexEnv();
    this->SendMatrix();
}

/*---------------------------------------------------------------------------
  Desc: GraphicsDrawing のシェイプレンダリングの開始関数
---------------------------------------------------------------------------*/
void GraphicsDrawing::SetupTexEnv()
{
#define TEXENV_LAST_STAGE "dmp_TexEnv[2]"
    
    // プライマリカラーの設定をそのまま出力します。
    glUniform3i(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".srcRgb"), 
        GL_PRIMARY_COLOR, GL_PREVIOUS, GL_PREVIOUS);
    glUniform3i(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".srcAlpha"), 
        GL_PRIMARY_COLOR, GL_PREVIOUS, GL_PREVIOUS);
    glUniform1i(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".combineRgb"), 
        GL_REPLACE);
    glUniform1i(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".combineAlpha"), 
        GL_REPLACE);
    glUniform3i(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".operandRgb"), 
        GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
    glUniform3i(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".operandAlpha"), 
        GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
    glUniform1f(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".scaleRgb"), 
        1.0f);
    glUniform1f(
        glGetUniformLocation( m_ShapePgID, TEXENV_LAST_STAGE ".scaleAlpha"), 
        1.0f);
#undef TEXENV_LAST_STAGE
}

/*---------------------------------------------------------------------------
  Desc: プロジェクション、モデルビュー行列をUniformへ設定します
---------------------------------------------------------------------------*/
void GraphicsDrawing::SendMatrix()
{
    glUseProgram(m_ShapePgID);
    glBindAttribLocation(m_ShapePgID, 0, "aPosition");
    glBindAttribLocation(m_ShapePgID, 1, "aColor");
   
    
    // 射影行列を正射影に設定
    {
        // CTR 実機では、液晶が 90 度回転しているので、x, y を逆に設定する。
        const GLsizei lcdWidth  = static_cast<GLsizei>(m_ScreenSize.y); // 240
        const GLsizei lcdHeight = static_cast<GLsizei>(m_ScreenSize.x);

        glViewport(0, 0, lcdWidth, lcdHeight);

        // 左上原点とし、Y軸とZ軸の向きが逆になるように設定します。
        nn::math::MTX44 proj;
#ifdef USE_GL_LOOKAT        
        f32 znear   =  -100.0f;
        f32 zfar    =   100.0f;
#else
        f32 znear   =  0.0f;
        f32 zfar    =  1.0f;
#endif           
        f32 t       =  0;
        f32 b       =  lcdWidth;
        f32 l       =  0;
        f32 r       =  lcdHeight;        

        // PIVOT_UPSIDE_TO_TOPを引数とする場合は別途-90度の回転の必要はない
        nn::math::MTX44OrthoPivot(&proj, l, r, b, t, znear, zfar, nn::math::PIVOT_UPSIDE_TO_TOP);                            

        glUniformMatrix4fv(glGetUniformLocation(m_ShapePgID, "uProjection"), 1, GL_TRUE, (f32*)(proj));        
        // glUniform4fv(glGetUniformLocation(m_ShapePgID, "uProjection"), 4, proj.a);
    } 
        
    // モデルビュー行列を単位行列に設定    
    {
        nn::math::Matrix34 viewMtx, rot;
        const nn::math::VEC3 camPos(0.0f, 0.0f, 1.0f);
        const nn::math::VEC3 aimPos(0.0f, 0.0f, 0.0f);
        const nn::math::VEC3 camUp(0.0f, 1.0f, 0.0f);

        MTX34Identity(&viewMtx);
#ifdef USE_GL_LOOKAT
        MTX34LookAt(&viewMtx, &camPos, &camUp, &aimPos);
#endif
        nn::math::Matrix44 mv(viewMtx);
        glUniformMatrix4fv(glGetUniformLocation(m_ShapePgID, "uModelView"), 1, GL_TRUE, (f32*)(mv));
        // glUniform4fv(glGetUniformLocation(m_ShapePgID, "uModelView"), 3, mv.a);
    }
}

/*---------------------------------------------------------------------------
 Desc:  長方形の枠を描画
 
 Args:  pos     左上の座標
        size    サイズ
---------------------------------------------------------------------------*/
void GraphicsDrawing::DrawRectangle(const nn::math::VEC2& pos, const nn::math::VEC2& size)
{
    const GLushort vertex_indexs[VERTEX_STREAM_SIZE] =
    {
        1,  0,  2,  3,
        5,  4,  6,  7,
        9,  8,  10, 11,
        13, 12, 14, 15
    };
    
    s32 count = 0;            
                        
    count += this->BuildLine(
        nn::math::VEC4(pos.x,        pos.y, m_Depth2d, 1.0f), 
        nn::math::VEC4(pos.x+size.x, pos.y, m_Depth2d, 1.0f), 
        m_PositionStream);
        
    count += this->BuildLine(
        nn::math::VEC4(pos.x+size.x, pos.y,        m_Depth2d, 1.0f), 
        nn::math::VEC4(pos.x+size.x, pos.y+size.y, m_Depth2d, 1.0f), 
        &m_PositionStream[count]);

    count += this->BuildLine(
        nn::math::VEC4(pos.x+size.x, pos.y+size.y, m_Depth2d, 1.0f), 
        nn::math::VEC4(pos.x,        pos.y+size.y, m_Depth2d, 1.0f), 
        &m_PositionStream[count]);
    
    count += this->BuildLine(
        nn::math::VEC4(pos.x, pos.y+size.y, m_Depth2d, 1.0f), 
        nn::math::VEC4(pos.x, pos.y,        m_Depth2d, 1.0f), 
        &m_PositionStream[count]);

    glEnableVertexAttribArray(VTXATTR_POSITION);
    glEnableVertexAttribArray(VTXATTR_COLOR);
    glVertexAttribPointer(VTXATTR_POSITION, 4, GL_FLOAT, GL_FALSE, 0, &m_PositionStream[0]);
    glVertexAttribPointer(VTXATTR_COLOR, 4, GL_FLOAT, GL_FALSE, 0, &m_ColorStream[0]);

    const GLint vertexNum = sizeof(vertex_indexs) / sizeof(vertex_indexs[0]);    
    glDrawElements(GL_TRIANGLE_STRIP, vertexNum, GL_UNSIGNED_SHORT, vertex_indexs);
    
    NN_ASSERT (!glGetError());    
}

/*---------------------------------------------------------------------------
  Desc: 幅1ドットの長方形の枠を描画
        ※DrawRectangleで奇数幅の線を描画した際に正しく表示されないため

  Args: pos     左上端の座標
        size    サイズ
---------------------------------------------------------------------------*/
void GraphicsDrawing::DrawWidth1Rectangle(const nn::math::VEC2& pos, const nn::math::VEC2& size)
{
    this->FillRectangle(pos,                                 nn::math::VEC2(size.x, 1.0f));
    this->FillRectangle(nn::math::VEC2(pos.x+size.x, pos.y), nn::math::VEC2(1.0f, size.y));
    this->FillRectangle(nn::math::VEC2(pos.x, pos.y+size.y), nn::math::VEC2(size.x, 1.0f));
    this->FillRectangle(pos,                                 nn::math::VEC2(1.0f, size.y));
}

/*---------------------------------------------------------------------------
  Desc: 四角形を描画

  Args: pos     左上端の座標
        size    サイズ
---------------------------------------------------------------------------*/
void GraphicsDrawing::FillRectangle(const nn::math::VEC2& pos, const nn::math::VEC2& size)
{   
    this->BuildSquare( 
        nn::math::VEC4(pos.x+size.x, pos.y,          m_Depth2d, 1.0f),         
        nn::math::VEC4(pos.x,        pos.y,          m_Depth2d, 1.0f), 
        nn::math::VEC4(pos.x,        pos.y + size.y, m_Depth2d, 1.0f), 
        nn::math::VEC4(pos.x+size.x, pos.y + size.y, m_Depth2d, 1.0f),         
        m_PositionStream );
            
    glEnableVertexAttribArray(VTXATTR_POSITION);
    glEnableVertexAttribArray(VTXATTR_COLOR);
    glVertexAttribPointer(VTXATTR_POSITION, 4, GL_FLOAT, GL_FALSE, 0, &m_PositionStream[0]) ;
    glVertexAttribPointer(VTXATTR_COLOR, 4, GL_FLOAT, GL_FALSE, 0, &m_ColorStream[0]);

    const GLint vertexNum = sizeof(s_SquarVertexIndexs) / sizeof(s_SquarVertexIndexs[0]);    
    glDrawElements(GL_TRIANGLE_STRIP, vertexNum, GL_UNSIGNED_SHORT, s_SquarVertexIndexs);
    
    NN_ASSERT (!glGetError());    
}

/*---------------------------------------------------------------------------
  Desc: ラインを描画

  Args: p1  ラインの開始点
        p2  ラインの終了点
---------------------------------------------------------------------------*/
void GraphicsDrawing::DrawLine(const nn::math::VEC2& p1, const nn::math::VEC2& p2)
{ 
    this->BuildLine( 
        nn::math::VEC4(p1.x, p1.y, m_Depth2d, 1.0f),         
        nn::math::VEC4(p2.x, p2.y, m_Depth2d, 1.0f),
        m_PositionStream );
            
    glEnableVertexAttribArray(VTXATTR_POSITION);
    glEnableVertexAttribArray(VTXATTR_COLOR);
    glVertexAttribPointer(VTXATTR_POSITION, 4, GL_FLOAT, GL_FALSE, 0, &m_PositionStream[0]);
    glVertexAttribPointer(VTXATTR_COLOR, 4, GL_FLOAT, GL_FALSE, 0, &m_ColorStream[0]);

    const GLint vertexNum = sizeof(s_SquarVertexIndexs) / sizeof(s_SquarVertexIndexs[0]);    
    glDrawElements(GL_TRIANGLE_STRIP, vertexNum, GL_UNSIGNED_SHORT, s_SquarVertexIndexs);
    
    NN_ASSERT (!glGetError());
}

/*---------------------------------------------------------------------------
  Desc: プリミティブのカラーを設定
        ４つの頂点カラーを指定し、頂点バッファサイズ分繰り返しで同色を設定します

  Args: pt1         頂点座標１
        pt2         頂点座標２
        pt3         頂点座標３
        pt4         頂点座標４                        
        *pVecArray  頂点を生成するバッファの先頭アドレス
---------------------------------------------------------------------------*/
void GraphicsDrawing::SetColor(
    const nw::ut::Color8& color1, 
    const nw::ut::Color8& color2,
    const nw::ut::Color8& color3,
    const nw::ut::Color8& color4)
{
    for(int i=0; i<VERTEX_STREAM_SIZE/internal_uji::COLOR_NUM; i++)
    {    
        m_ColorStream[ 0+i*internal_uji::COLOR_NUM ] = color1;
        m_ColorStream[ 1+i*internal_uji::COLOR_NUM ] = color2;
        m_ColorStream[ 2+i*internal_uji::COLOR_NUM ] = color3;
        m_ColorStream[ 3+i*internal_uji::COLOR_NUM ] = color4;
    }    
}    

/*---------------------------------------------------------------------------
  Desc: プリミティブの深度値を設定します
---------------------------------------------------------------------------*/
void GraphicsDrawing::SetDepth2d(const f32& depth2d)
{
    m_Depth2d = depth2d;
}

/*---------------------------------------------------------------------------
  Desc: ライン描画用の頂点列を生成

  Args: pt1         ライン描画開始座標
        pt2         ライン描画終端座標
        *pVecArray  頂点を生成するバッファの先頭アドレス
---------------------------------------------------------------------------*/
s32 GraphicsDrawing::BuildLine(const nn::math::VEC4& pt1, const nn::math::VEC4& pt2, nn::math::VEC4 *pVecArray)
{
    NW_ASSERT( m_LineWidth >= 0.f );
    NW_NULL_ASSERT( pVecArray );

    static const f32 EVA_MATH_ERROR = 0.00001f;   
  
    f32 s_LineHalfWidth = 0.5f * m_LineWidth;

    f32 x01 = pt2.x - pt1.x;
    f32 y01 = pt2.y - pt1.y;
    
    f32 gradientX = 0.0f;
    f32 gradientY = 0.0f;
    
    if (nn::math::FAbs(x01) < EVA_MATH_ERROR)
    {
        gradientX = 1.0f;
    }
    else if (nn::math::FAbs(y01) < EVA_MATH_ERROR)
    {
        gradientY = 1.0f;
    }
    else
    {
        gradientX = y01;
        gradientY = (- x01);

        f32 gradient = gradientX * gradientX + gradientY * gradientY;
        gradient = nn::math::FSqrt(gradient);

        gradientX /= gradient;
        gradientY /= gradient;
    }
    
    gradientX *= s_LineHalfWidth;
    gradientY *= s_LineHalfWidth;
    
    nn::math::VEC4 vtx[4];
        
    vtx[0] = vtx[1] = pt1;
    vtx[2] = vtx[3] = pt2;

    vtx[0].x += gradientX; vtx[0].y += gradientY;
    vtx[1].x -= gradientX; vtx[1].y -= gradientY;
    vtx[2].x -= gradientX; vtx[2].y -= gradientY;        
    vtx[3].x += gradientX; vtx[3].y += gradientY;

#if 0  
    for(int i=0; i<4; i++)
        NN_LOG("X[%d]=%.2f Y[%d]=%.2f\n", i, vtx[i].x, i, vtx[i].y);
#endif   

    return this->BuildSquare(vtx[0], vtx[1], vtx[2], vtx[3], pVecArray);
}

/*---------------------------------------------------------------------------
  Desc: 三角形描画用の頂点列を生成

  Args: pt1         頂点座標１
        pt2         頂点座標２
        pt2         頂点座標３        
        *pVecArray  頂点を生成するバッファの先頭アドレス

  Rtns: 頂点数
---------------------------------------------------------------------------*/
s32 GraphicsDrawing::BuildTriangle(
    const nn::math::VEC4& pt1, 
    const nn::math::VEC4& pt2, 
    const nn::math::VEC4& pt3, 
    nn::math::VEC4 *pVecArray)
{
    NW_NULL_ASSERT( pVecArray );
    
    pVecArray[ 0 ] = pt1;
    pVecArray[ 1 ] = pt2;
    pVecArray[ 2 ] = pt3;
    
    return 3;
}

/*---------------------------------------------------------------------------
  Desc: 四角形描画用の頂点列を生成

  Args: pt1         頂点座標１
        pt2         頂点座標２
        pt2         頂点座標３
        pt2         頂点座標４                        
        *pVecArray  頂点を生成するバッファの先頭アドレス

  Rtns: 頂点数
---------------------------------------------------------------------------*/
s32 GraphicsDrawing::BuildSquare( 
    const nn::math::VEC4& pt1, 
    const nn::math::VEC4& pt2, 
    const nn::math::VEC4& pt3, 
    const nn::math::VEC4& pt4, 
    nn::math::VEC4 *pVecArray)
{
    NW_NULL_ASSERT( pVecArray );

    pVecArray[ 0 ] = pt1;
    pVecArray[ 1 ] = pt2;
    pVecArray[ 2 ] = pt3;
    pVecArray[ 3 ] = pt4;
    
    return 4;
}

//============================================================================
//============================================================================
//                              文字描画
//============================================================================
//============================================================================

/*
  Desc: シェーダの初期化を行います。
*/
void* GraphicsDrawing::InitFontShaders()
{
    // アロケータを引数とした後削除予定
    uji::sys::SysApp* sysApp = uji::sys::SysApp::GetInstance();    
    
    const wchar_t* shaders = NW_DEMO_FILE_PATH(L"sys/nwfont_RectDrawerShader.shbin");
    nn::fs::FileReader shaderReader(shaders);

    const u32 fileSize = (u32)shaderReader.GetSize();

    void* shaderBinary = sysApp->AllocateDeviceMemory(fileSize);
    NN_NULL_ASSERT(shaderBinary);

    s32 read = shaderReader.Read(shaderBinary, fileSize);
    NN_ASSERT(read == fileSize);

    const u32 vtxBufCmdBufSize = 
        nn::font::RectDrawer::GetVertexBufferCommandBufferSize(shaderBinary, fileSize);
    void *const vtxBufCmdBuf = sysApp->AllocateDeviceMemory(vtxBufCmdBufSize);
    NN_NULL_ASSERT(vtxBufCmdBuf);
    m_RectDrawer.Initialize(vtxBufCmdBuf, shaderBinary, fileSize);

    sysApp->Free(shaderBinary);

    return vtxBufCmdBuf;
}

/*
  Desc: 描画の初期設定を行います。

  Args: なし
  
  Rtns: なし
*/
void GraphicsDrawing::InitDrawingString(int width, int height)
{
    // カラーバッファ情報
    const nn::font::ColorBufferInfo colBufInfo = {height, width, PICA_DATA_DEPTH24_STENCIL8_EXT };

    const u32 screenSettingCommands[] =
    {

        // ビューポートの設定
        NN_FONT_CMD_SET_VIEWPORT( 0, 0, colBufInfo.width, colBufInfo.height ),

        // シザー処理を無効
        NN_FONT_CMD_SET_DISABLE_SCISSOR( colBufInfo ),

        // wバッファの無効化
        // デプスレンジの設定
        // ポリゴンオフセットの無効化
        NN_FONT_CMD_SET_WBUFFER_DEPTHRANGE_POLYGONOFFSET(
            0.0f,           // wScale : 0.0 でWバッファが無効
            0.0f,           // depth range near
            1.0f,           // depth range far
            0,              // polygon offset units : 0.0 で ポリゴンオフセットが無効
            colBufInfo),
    };

    nngxAdd3DCommand(screenSettingCommands, sizeof(screenSettingCommands), true);

    static const u32 s_InitCommands[] =
    {
        // カリングを無効
        NN_FONT_CMD_SET_CULL_FACE( NN_FONT_CMD_CULL_FACE_DISABLE ),

        // ステンシルテストを無効
        NN_FONT_CMD_SET_DISABLE_STENCIL_TEST(),

        // デプステストを無効
        // カラーバッファの全ての成分を書き込み可
        NN_FONT_CMD_SET_DEPTH_FUNC_COLOR_MASK(
            false,  // isDepthTestEnabled
            0,      // depthFunc
            true,   // depthMask
            true,   // red
            true,   // green
            true,   // blue
            true),  // alpha

        // アーリーデプステストを無効
        NN_FONT_CMD_SET_ENABLE_EARLY_DEPTH_TEST( false ),

        // フレームバッファアクセス制御
        NN_FONT_CMD_SET_FBACCESS(
            true,   // colorRead
            true,   // colorWrite
            false,  // depthRead
            false,  // depthWrite
            false,  // stencilRead
            false), // stencilWrite
    };

    nngxAdd3DCommand(s_InitCommands, sizeof(s_InitCommands), true);
}

/*
  Desc: ResFontを構築します。

  Args: font        - 構築するフォントへのポインタ
        filePath    - ロードするフォントリソースファイル名

  Rtns: ResFont構築の成否
*/
bool GraphicsDrawing::InitFont(const wchar_t* filePath)
{
    // アロケータを引数とした後削除予定
    uji::sys::SysApp* sysApp = uji::sys::SysApp::GetInstance();    
    
    // フォントリソースをロードします
    nn::fs::FileReader fontReader(filePath);

    s32 fileSize = (s32)fontReader.GetSize();
    if (fileSize <= 0)
    {
        return false;
    }

    void* buffer = sysApp->AllocateDeviceMemory(fileSize, nn::font::GlyphDataAlignment);
    if (buffer == NULL)
    {
        return false;
    }

    s32 readSize = fontReader.Read(buffer, fileSize);
    if (readSize != fileSize)
    {
        sysApp->Free(buffer);
        return false;
    }

    // フォントリソースをセットします
    bool bSuccess = m_Font.SetResource(buffer);
    NN_ASSERT(bSuccess);

    //--- 既にリソースをセット済みであるか，ロード済みであるか、リソースが不正な場合に失敗します。
    if (! bSuccess)
    {
        sysApp->Free(buffer);
    }

    // 描画用バッファを設定します。
    const u32 drawBufferSize = nn::font::ResFont::GetDrawBufferSize(buffer);
    void* drawBuffer = sysApp->AllocateDeviceMemory(drawBufferSize, 4);
    NN_NULL_ASSERT(drawBuffer);    
    m_Font.SetDrawBuffer(drawBuffer);

    return bSuccess;
}

/*
  Desc: ResFontを破棄します。

  Args: font - 破棄するフォントへのポインタ
*/
void GraphicsDrawing::CleanupFont(nn::font::ResFont* font)
{
    // アロケータを引数とした後削除予定
    uji::sys::SysApp* sysApp = uji::sys::SysApp::GetInstance();    
    
    // 描画用バッファがセットされているなら SetDrawBuffer 時に渡したバッファへの
    // ポインタが返ってきます。        
    void *const drawBuffer = font->GetDrawBuffer();
    if (drawBuffer != NULL)
    {
        sysApp->Free(drawBuffer);
    }

    // フォントがセットされているなら SetResource 時に渡したリソースへの
    // ポインタが返ってきます。
    void *const resource = font->RemoveResource();
    if (resource != NULL)
    {
        sysApp->Free(resource);
    }

    // RemoveResource 後は再度 SetResource するまでフォントとして使用できません。
}

/*
  Desc: 表示文字列用バッファを確保
*/
nn::font::DispStringBuffer* GraphicsDrawing::AllocDispStringBuffer(int charMax)
{
    // アロケータを引数とした後削除予定
    uji::sys::SysApp* sysApp = uji::sys::SysApp::GetInstance();    
    
    const u32 DrawBufferSize = nn::font::CharWriter::GetDispStringBufferSize(charMax);
    void *const bufMem = sysApp->AllocateDeviceMemory(DrawBufferSize);
    NN_NULL_ASSERT(bufMem);

    return nn::font::CharWriter::InitDispStringBuffer(bufMem, charMax);
}

/*
  Desc: モデルビュー行列と射影行列を設定
*/
void GraphicsDrawing::SendFontMatrix()
{
    // CTR 実機では、液晶が 90 度回転しているので、x, y を逆に設定する。
    const GLsizei lcdWidth  = static_cast<GLsizei>(m_ScreenSize.y); // 240
    const GLsizei lcdHeight = static_cast<GLsizei>(m_ScreenSize.x);

    // 射影行列を正射影に設定
    {
        // 左上原点とし、Y軸とZ軸の向きが逆になるように設定します。
        nn::math::MTX44 proj;
        f32 znear   =  0.0f;
        f32 zfar    = -1.0f;
        f32 t       =  0;
        f32 b       =  static_cast<f32>(m_ScreenSize.y);
        f32 l       =  0;
        f32 r       =  static_cast<f32>(m_ScreenSize.x);
        nn::math::MTX44Ortho(&proj, l, r, b, t, znear, zfar);

        // 実機の場合、画面を270度回転します。
        nn::math::MTX34 rot;
        nn::math::VEC3 zAxis(0.f, 0.f, 1.f);

        f32 displayRotate = 270.f;

        nn::math::MTX34RotAxisDeg(&rot, &zAxis, displayRotate);
        nn::math::MTX44 wk(rot);
        nn::math::MTX44Mult(&wk, &wk, &proj);
        m_RectDrawer.SetProjectionMtx(wk);
    }

    // モデルビュー行列を単位行列に設定
    {
        nn::math::MTX34 mv;
        nn::math::MTX34Identity(&mv);
        m_RectDrawer.SetViewMtxForText(mv);
    }
}

/*
  Desc: 文字列描画前処理
*/
void GraphicsDrawing::BeginDrawingString(nn::font::DispStringBuffer* pDrawStringBuf)
{
    this->InitDrawingString(m_ScreenSize.x, m_ScreenSize.y);
    m_TextWriter.SetDispStringBuffer(pDrawStringBuf);    
    
    m_TextWriter.StartPrint();     
}

/*
  Desc: 文字列描画後処理
*/
void GraphicsDrawing::EndDrawingString()
{
    m_TextWriter.EndPrint();
    m_RectDrawer.BuildTextCommand(&m_TextWriter);    
    
    m_RectDrawer.DrawBegin();
    {
        this->SendFontMatrix();
        m_TextWriter.UseCommandBuffer();
    }
    m_RectDrawer.DrawEnd();

    nngxUpdateState(NN_GX_STATE_ALL);
}

/*
  Desc: 等幅描画の為の設定を行います
*/
void GraphicsDrawing::SetFixedWidthFont(f32 height)
{
    m_TextWriter.SetFontSize(height);
    m_TextWriter.EnableFixedWidth(true);
    m_TextWriter.SetFixedWidth(height/2);    
}


//============================================================================
//============================================================================
//                              MISC
//============================================================================
//============================================================================

/*
  Desc: コマンドバッファのサイズをデバッグ出力
*/
void GraphicsDrawing::NN_LOG_CmdBufUsedSize()
{
    GLint param;
    GLint req_count;

    nngxGetCmdlistParameteri(NN_GX_CMDLIST_USED_BUFSIZE, &param);
    nngxGetCmdlistParameteri(NN_GX_CMDLIST_USED_REQCOUNT, &req_count);
    NN_LOG("CMD_USED_BUF_SIZE= %X REQ_COUNT= %d\n", param, req_count);
}


} // namespace sys
} // namespace uji
