/*---------------------------------------------------------------------------*
  Project: CTR_TRIAL
  File:    ProductionLog.cpp

  Copyright 2010 Nintendo.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

 *---------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/fnd.h>
#include <nn/fs.h>
#include <nn/crypto/crypto_Sha1.h>
#include <nn/Result.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <string>
#include <sstream>
#include <stack>

#include <nn/fs/CTR/MPCore/fs_ApiForHwCheck.h>
#include <nn/fs/CTR/MPCore/fs_FileSystemBase.h>
#include <nn/fs/CTR/fs_ArchiveTypesForSystem.h>
#include <nn/uds.h>

#include "ProductionLog.h"
#include "sys_Memory.h"

#include "../seq/TestResult.h"
#include "../seq/Config.h"


using namespace uji::seq;
using namespace uji::seq::pl;
using namespace std;

//
#pragma diag_suppress 2819

// ֎~̃`FbNs
bool ProductionLog::Check_ProhibitedString(string str) const
{

    // ',' ̃`FbN
    string::size_type index = str.find(",");
    if ( index != string::npos)
    {
        NN_LOG("%s ProhibitedString Found! [%s]\n", NN_FUNCTION, str.c_str());
        return false;
    }

    // '\n'̃`FbN
    index = str.find("\n");
    if ( index != string::npos)
    {
        NN_LOG("%s ProhibitedString Found! [%s]\n", NN_FUNCTION, str.c_str());
        return false;
    }

    return true;
}

// (1Ԗ)Ŏw肵 LogEntryɁAc̈̒lݒ肵AOGg쐬
LogResult ProductionLog::Create_LogEntry(LogEntry & le,
                                         int        testmode,
                                         string     testerName,
                                         string     state,
                                         string     version,
                                         string     date,
                                         string     time,
                                         string     testerId,
                                         string     msg) const
{
    //`FbN
    /// I[o[̃`FbN
    if ( testmode          >= Testmode(TESTMODE_MAX) ) { NN_LOG("Create_LogEntry testmode value over."); goto ARG_ERROR; }
    if ( testerName.size() > LOG_TESTERNAME_LEN )      { NN_LOG("Create_LogEntry testerName size over.");goto ARG_ERROR; }
    if ( state.size()      > LOG_STATE_LEN )           { NN_LOG("Create_LogEntry state size over.");     goto ARG_ERROR; }
    if ( version.size()    > LOG_VERSION_LEN )         { NN_LOG("Create_LogEntry version size over.");   goto ARG_ERROR; }
    if ( date.size()       > LOG_DATE_LEN )            { NN_LOG("Create_LogEntry date size over.");      goto ARG_ERROR; }
    if ( time.size()       > LOG_TIME_LEN )            { NN_LOG("Create_LogEntry time size over.");      goto ARG_ERROR; }
    if ( testerId.size()   > LOG_TESTERID_LEN )        { NN_LOG("Create_LogEntry testerId size over.");  goto ARG_ERROR; }
    if ( msg.size()        > LOG_MSG_LEN )             { NN_LOG("Create_LogEntry msg size over.");       goto ARG_ERROR; }

    /// ̃`FbNi󕶎łȂǂj
    if ( testerName.empty() ) { NN_LOG("Create_LogEntry testerName empty.");goto ARG_ERROR; }
    if ( state.empty() )      { NN_LOG("Create_LogEntry state empty.");     goto ARG_ERROR; }
    if ( version.empty() )    { NN_LOG("Create_LogEntry version empty.");   goto ARG_ERROR; }

    /// gp֎~̃`FbN
    if (
        (false == Check_ProhibitedString(testerName)) ||
        (false == Check_ProhibitedString(state))      ||
        (false == Check_ProhibitedString(version))    ||
        (false == Check_ProhibitedString(date))       ||
        (false == Check_ProhibitedString(time))       ||
        (false == Check_ProhibitedString(testerId))   ||
        (false == Check_ProhibitedString(msg))
       )
    {
         goto ARG_ERROR;
    }

    //f[^o^
    le.testmode = testmode;
    sprintf(le.testerName, "%s", testerName.c_str());
    sprintf(le.state,      "%s", state.c_str());
    sprintf(le.version,    "%s", version.c_str());
    sprintf(le.date,       "%s", date.c_str());
    sprintf(le.time,       "%s", time.c_str());
    sprintf(le.testerId,   "%s", testerId.c_str());
    sprintf(le.message,    "%s", msg.c_str());

    return  LogResultCode::SUCCESS;

ARG_ERROR:
    NN_LOG("ARGUMENT_ERROR!\n");
    return LogResultCode::ARGUMENT_ERROR;
}


LogResult ProductionLog::LogTable_Shrink(int id_end)
{
    LogResult  lr;

    //1/2 ŏ̗vfł邩ǂmFA
    //ł΁AENTRY_FULL_BOARD1STԂB
    if ( LogTable_IsCpuBoard1st() == true ) return LogResultCode::ENTRY_FULL_BOARD1ST;

    //2/2 łȂ΁Ae[u̗vf炵Ãf[^󂯂
    lr = LogTable_Rotate(id_end);
    if ( lr < 0 ) return LogResultCode::FUNC_CALL_ERROR;

    return LogResultCode::SUCCESS;
}

// OPsǉ
//  msgsĕ\ꍇ́A\nł͂ȂA$gƁB
//  msg','͋֎~B
LogResult ProductionLog::Add_1Line(int    testmode,
                                   string testerName,
                                   string state,
                                   string version,
                                   string date, string time,
                                   string testerId,
                                   string msg)
{
    LogResult  lr;
    LogEntry   le;

    // ^ꂽ,LogEntry쐬
    lr = Create_LogEntry(le, testmode,
                             testerName,
                             state,
                             version,
                             date,time,
                             testerId,
                             msg);

    if ( LogResultCode::SUCCESS != lr )
    {
        NN_LOG("%s ERROR(%s).\n",NN_FUNCTION, LogResultCode::ToString(lr).c_str());
        return lr;
    }

    lr = ProductionLog::Add_1Line( le );
    return lr;
}

LogResult ProductionLog::Add_1Line( LogEntry& le )
{
    LogResult  lr;
    // Ɋi[ׂAID擾
    int  id;
    id = Get_NextId();
    if ( LogResultCode::ENTRY_FULL == id )
    {
        lr = LogTable_Shrink(LOG_ENTRY_NUM_MAX - 1);
        if ( lr < 0 )
        {
            NN_LOG("shrink failed.\n");
            return lr;
        }
        else
        {
            id = LOG_ENTRY_NUM_MAX -1;
        }
    }

    // t@C̍őTCY < ̃Oe[ũTCY + VKGgTCY
    while ( m_fileSizeMax < (LogTable_GetWriteFileSize() + Get_LogEntryByteSize(le)))
    {
        lr = LogTable_Shrink(id-1);
        if ( lr < 0)
        {
            NN_LOG("!! shrink failed (%d)\n",lr);
            return lr;
        }
        else
        {
            NN_LOG("shrink success (%s) id=%d\n",le.testerName,id);
            id = id - 1;
        }

        if ( id < 0 ) return LogResultCode::UNKNOWN_ERROR;
    }

    //Oe[uւ̓o^
    lr = LogTable_Entry(id, le);
    if ( LogResultCode::SUCCESS != lr) goto ERROR;

    //nbV̌vZ
    Calc_DataHash();

    //t@Cւ̏
    lr = WriteProc();
    if ( LogResultCode::SUCCESS != lr) goto ERROR;

    return lr;

ERROR:
    NN_LOG("%s ERROR(%s).\n",NN_FUNCTION, LogResultCode::ToString(lr).c_str());
    return lr;
}


/*
    ŐṼOɏ㏑ǉ
    SCpG[WOł̂ݎgpĂAHvOł̎gp֎~
*/
LogResult ProductionLog::Overwrite_1Line(int    testmode,
                                         string testerName,
                                         string state,
                                         string version,
                                         string date, string time,
                                         string testerId,
                                         string msg)
{
    LogResult  lr;
    LogEntry   le, lastLe;

    // ^ꂽ,LogEntry쐬
    lr = Create_LogEntry(le, testmode,
                             testerName,
                             state,
                             version,
                             date,time,
                             testerId,
                             msg);

    if ( LogResultCode::SUCCESS != lr ) goto ERROR;

    // Ɋi[ׂIDŐṼOIDZo
    int  id;
    id = Get_NextId();
    if (id > 0) id--;
    lastLe = Get_LogEntry(id);
    
    // t@C̍őTCY < ̃Oe[ũTCY + VKGgTCY - ŐVOGgTCY
    while ( m_fileSizeMax < 
            (LogTable_GetWriteFileSize() + Get_LogEntryByteSize(le) - Get_LogEntryByteSize(lastLe)) )
    {
        lr = LogTable_Shrink(id-1);
        if ( lr < 0)
        {
            NN_LOG("!! shrink failed (%d)\n",lr);
            return lr;
        }
        else
        {
            NN_LOG("shrink success (%s) id=%d\n",le.testerName,id);
            id = id - 1;
        }

        if ( id < 0 ) return LogResultCode::UNKNOWN_ERROR;
    }

    //Oe[uւ̓o^
    lr = LogTable_Entry(id, le);
    if ( LogResultCode::SUCCESS != lr) goto ERROR;

    //nbV̌vZ
    Calc_DataHash();

    //t@Cւ̏
    lr = WriteProc();
    if ( LogResultCode::SUCCESS != lr) goto ERROR;

    return lr;

ERROR:
    NN_LOG("%s ERROR(%s).\n",NN_FUNCTION, LogResultCode::ToString(lr).c_str());
    return lr;
}


// w肵̍ŐVOGg̃CfbNXԂB
// ߂l: : B
//         : 炸
int ProductionLog::Get_LatestLogEntryIndex(const string testerName) const
{
    int foundId = -1;
    LogEntry le;

    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        le = Get_LogEntry(i);
        if (testerName == le.testerName)
        {
            foundId = i;
        }
    }
    return foundId;
}

// w肵̍ŐVOGg̃CfbNXԂB
// ߂l: : B
//         : 炸
int ProductionLog::Get_LatestLogEntryIndex(const string name, const int testmode) const
{
    int foundId = -1;
    LogEntry le;

    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        le = Get_LogEntry(i);
        if ((name == le.testerName) && (testmode == le.testmode))
        {
            foundId = i;
        }
    }

    return foundId;
}

// w肵̍ŐVOGg̃CfbNXԂB
// ߂l: : B
//         : 炸
int ProductionLog::Get_LatestLogEntryIndex(const string name,
                                           const int testmode,
                                           const string state) const
{
    int foundId = -1;
    LogEntry le;

    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        le = Get_LogEntry(i);
        if ((name == le.testerName) && (testmode == le.testmode) && (state == le.state))
        {
            foundId = i;
        }
    }

    return foundId;
}

// w肵̍ŐVOGg̃CfbNXԂB
// ߂l: : B
//         : 炸
int ProductionLog::Get_LatestLogEntryIndex( const int testmode,
                                            const string state) const
{
    int foundId = -1;
    LogEntry le;

    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        le = Get_LogEntry(i);
        if ((testmode == le.testmode) && (state == le.state))
        {
            foundId = i;
        }
    }

    return foundId;
}

// w肵̍ŐVOGg̃CfbNXԂB
// ߂l: : B
//         : 炸
int ProductionLog::Get_LatestLogEntryIndex( const int testmode ) const
{
    int foundId = -1;
    LogEntry le;

    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        le = Get_LogEntry(i);
        if ((testmode == le.testmode) && ( uji::seq::LogEntry::ENTRY_USED == le.used))
        {
            foundId = i;
        }
    }

    return foundId;
}


//
//  OGg̎gpςݗvf擾
//
int ProductionLog::Get_LogEntryNums(void) const
{
    DEBUG_WRITE("%s called.\n",NN_FUNCTION);
    int      count = 0;
    LogEntry le;

    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        le = Get_LogEntry(i);
        if ( LogEntry::ENTRY_USED == le.used) // used bitB
        {
            count++;
        }
    }
    return count;
}

/*
    OGg̍ővf擾
*/
int ProductionLog::Get_LogEntryNumsMax(void) const
{
    return LOG_ENTRY_NUM_MAX;
}

/*
    OGg̍őoCgTCY擾
*/
int ProductionLog::Get_LogEntryByteSizeMax(void) const
{
    int size = 0;

    size += 1;//testmode
    size += 1;//comma
    size += LOG_TESTERNAME_LEN;
    size += 1;
    size += LOG_STATE_LEN;
    size += 1;
    size += LOG_VERSION_LEN;
    size += 1;
    size += LOG_DATE_LEN;
    size += 1;
    size += LOG_TIME_LEN;
    size += 1;
    size += LOG_TESTERID_LEN;
    size += 1;
    size += LOG_MSG_LEN;
    size += 1;
    size += 1;//s

    return size;
}

/*
@OGgɊi[ID擾B

  ߂l: 0(LOG_ENTRY_NUM_MAX -1) : Ɋi[ꏊ
          LogResultCode::ENTRY_FULL   : Si[Ă(Ot)
*/
int ProductionLog::Get_NextId(void) const
{
    int already_used = -1;
    LogEntry  le;
    LogResult lr;

    //zS`FbN
    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        le = Get_LogEntry(i);
        if (LogEntry::ENTRY_USED == le.used) // used bitāAɎgĂAalready_usedɂ̒lB
        {
            already_used = i;
        }
    }

    if ( already_used == (LOG_ENTRY_NUM_MAX -1)) //  "LOG_ENTRY_NUM_MAX -1"͍Ō̃OGg̃CfbNX
    {
        NN_LOG("%s reached to LOG_ENTRY_NUM_MAX\n", NN_FUNCTION);
        lr = LogResultCode::ENTRY_FULL;
        return  lr; // ENTRY_FULL(vft)Ԃ
    }
    else
    {
        // S`FbŃAused bitĂŌ̃CfbNXĂ̂ŁA+1B
        // NN_LOG("return %d\n", already_used + 1);
        return (already_used + 1);
    }
}

// Oe[ũTCY擾
int  ProductionLog::LogTable_GetWriteFileSize(void)
{
    DEBUG_WRITE("%s called.\n",NN_FUNCTION);

    int      sizeTotal = 0;

    // Owb_
    sizeTotal = 42;

    // O{
    LogEntry le;
    for(int i=0; i < LOG_ENTRY_NUM_MAX; i++)
    {
        int size = 0;
        le = Get_LogEntry(i);
        if ( LogEntry::ENTRY_USED == le.used) // used bitB
        {
            size += 1;//testmode
            size += 1;//comma
            size += strlen(le.testerName);
            size += 1;
            size += strlen(le.state);
            size += 1;
            size += strlen(le.version);
            size += 1;
            size += strlen(le.date);
            size += 1;
            size += strlen(le.time);
            size += 1;
            size += strlen(le.testerId);
            size += 1;
            size += strlen(le.message);
            size += 1;
            size += 1;//s
        }
        sizeTotal = sizeTotal + size;
    }

//    NN_LOG("LogTable_GetWriteFileSize =%d\n",sizeTotal);
    return sizeTotal;
}


// ŏ̃GgOł邩ǂ
bool  ProductionLog::LogTable_IsCpuBoard1st(void)
{
    LogEntry le;
    le = Get_LogEntry(0);

    if ( 0 == strncmp(le.testerName, "CpuBoard",8))
    {
        return true;
    }
    else
    {
        return false;
    }
}

//ƂŐV̎wtesterName,wtestmodeOKł邩
bool ProductionLog::LogTable_LatestLogEntryIsOK(const std::string testerName, const int testmode)
{
    NN_LOG("%s", testerName.c_str());
    NN_LOG("%d", testmode);

    int index =  -1;

    index = Get_LatestLogEntryIndex(testerName, testmode);
    if ( index < 0 ) return false;

    LogEntry le;
    le = Get_LogEntry(index);

    if ( 0 == strncmp(le.state, "OK",2))
    {
        return true;
    }
    else
    {
        return false;
    }
}

// OGgo^
LogResult ProductionLog::LogTable_Entry(int id, LogEntry & le)
{
//    NN_LOG("%s id=(%d)",NN_FUNCTION, id);

    if ( id > (LOG_ENTRY_NUM_MAX -1))
    {
        NN_LOG("ARGUMENT ERROR\n");
        return LogResultCode::ARGUMENT_ERROR;
    }

    m_logTable[id]      = le;
    m_logTable[id].used = LogEntry::ENTRY_USED; //gprbg𗧂Ă

    DEBUG_WRITE("succeded.\n");
    return LogResultCode::SUCCESS;
}


// ̌OACpuBoardOő50oA
// t@CɕۑB
// TODO : ֐ύX(Oe[ueĂȂ)
LogResult ProductionLog::LogTable_Extract_CpuBoard(void)
{
    nn::Result nnr;
    nn::fs::FileOutputStream fos;

//@
// 1.stack CpuBoardO擪OASē˂
// 2.stack Aő50oî͎Ă)

    std::stack<LogEntry> stack ;    //LogEntry^̃X^bN
    LogEntry le;
    int i;
    for( i = 0; i < LOG_ENTRY_NUM_MAX ; i++)
    {
        le = Get_LogEntry(i);
        if ( 0 == strncmp(le.testerName, "CpuBoard", 8))
        {
            DEBUG_WRITE("push id=%d\n", i);
            stack.push(le);
        }
    }

    int index;
    index = stack.size() -1;
    if (index >= SALVAGE_CPUBOARD_MAX) index = SALVAGE_CPUBOARD_MAX -1;//50܂łȂ

    // Oe[uNA
    LogTable_Clear();

    // Oe[u֓o^
    LogResult lr;
    for( i = index; i >= 0 ; i--)
    {
        le = stack.top();
        stack.pop();
        DEBUG_WRITE("pop %d\n",i);
        lr = LogTable_Entry(i,le);
        if ( LogResultCode::SUCCESS != lr ) goto ERROR;
    }

    //nbV̌vZ
    Calc_DataHash();

    //t@C폜AVK쐬
    DeleteFile_Master("");

    //t@CȂ̂ŁAVKt@C̍쐬s
    nnr = fos.TryInitialize(m_pathMaster.c_str(), true);
    if (nnr.IsFailure())
    {
        NN_LOG("file create failed.\n");
        nn::dbg::PrintResult(nnr);
        goto ERROR;
    }

    fos.Write(BLANK_DATA_HEADER.c_str(),40+2);//swb_
    fos.Finalize();//t@C

    //t@Cւ̏
    lr = WriteProc_WithNoMasterBackup();
    if ( LogResultCode::SUCCESS != lr) goto ERROR;

    return lr;

ERROR:

    NN_LOG("%s ERROR(%s).\n",NN_FUNCTION,LogResultCode::ToString(lr).c_str());
    return lr;
}

// ēpɁAAPP_IEmsLabelOcԂCPUBoardȊÕO
LogResult ProductionLog::LogTable_ReconstructForReentry(void)
{
    int emsLabelIndex  = Get_LatestLogEntryIndex( "EmsLabel" );
    int appiStartIndex = Get_LatestLogEntryIndex( "App_I", uji::seq::pl::LINE, "START" );
    int appiOkIndex    = Get_LatestLogEntryIndex( "App_I", uji::seq::pl::LINE, "OK" );
    
    LogEntry emsLabel  = Get_LogEntry( emsLabelIndex );
    LogEntry appiStart = Get_LogEntry( appiStartIndex );
    LogEntry appiOk    = Get_LogEntry( appiOkIndex );
    
    LogResult lr = LogTable_Extract_CpuBoard();
    if( LogResultCode::SUCCESS != lr )
    {
        return lr;
    }
    
    // Index̏ꍇ͖ړĨOĂȂ߁A߂sȂ
    if( 0 <= emsLabelIndex )
    {
        if(( lr = Add_1Line( emsLabel )) != LogResultCode::SUCCESS )
        {
            return lr;
        }
    }
    if( 0 <= appiStartIndex )
    {
        if(( lr = Add_1Line( appiStart )) != LogResultCode::SUCCESS )
        {
            return lr;
        }
    }
    if( 0 <= appiOkIndex )
    {
        if(( lr = Add_1Line( appiOk )) != LogResultCode::SUCCESS )
        {
            return lr;
        }
    }
    
    return lr;
    
}

//
// Oe[u[e[gāA̗vf󂯂
// : id_end
// ߂l:
int ProductionLog::LogTable_Rotate(int id_end)
{
    if ( id_end <= 0 ) return LogResultCode::FUNC_CALL_ERROR;

// 擪vf(Y0)폜A 0  <- 1
//                           1  <- 2
//                           ...
//                          254 <- 255
//                          255 <- Vvf

    for(int i = 0; i < id_end; i++)
    {
      m_logTable[i] = m_logTable[i+1];
    }

    NN_LOG("LogTable_Rotate done (id_end =%d).\n", id_end);

    // ̗vfunusedrbg𗧂ĂāAgp
    LogEntry le;
    memset(&le, 0x00, sizeof(LogEntry));
    m_logTable[id_end] = le;
    m_logTable[id_end].used = LogEntry::ENTRY_UNUSED;

    // ̗vfȍ~Ô߃NA
    for(int i = (id_end+1) ; i < Get_LogEntryNumsMax() ; i++)
    {
      m_logTable[i] = le;
    }

    return id_end;
}

//
// w肵ID̃OGg擾B
//
const LogEntry ProductionLog::Get_LogEntry(const int index) const
{
    if ( index > (LOG_ENTRY_NUM_MAX -1))
    {
        //index ővf𒴂ĂAŌ̗vfԂ
        NN_LOG("%s argument is invalid (%d). greater %d.\n", NN_FUNCTION, index, (LOG_ENTRY_NUM_MAX - 1));
        return m_logTable[(LOG_ENTRY_NUM_MAX -1)];
    }
    else if ( index < 0)
    {
        //index 0菬΁Aŏiindex=0)̗vfԂB
        NN_LOG("%s argument is invalid (%d). lesser %d.\n", NN_FUNCTION, index, index);
        return m_logTable[0];
    }

    return m_logTable[index];
}

//
// Oe[ũnbVlvZAm_DataHash̒lXVB
//
LogResult ProductionLog::Calc_DataHash(void)
{
#if 0
    Dump_LogTableAll();
#endif

    // Oe[ũf[^Rs[pobt@m
    void * dataBuf = uji::sys::Alloc(m_fileSizeMax);//f[^obt@
    if ( NULL == dataBuf)
    {
        DEBUG_WRITE("%s dataBuf is NULL\n", NN_FUNCTION);
        return LogResultCode::ALLOCATE_ERROR;
    }
    memset(dataBuf, 0x00, m_fileSizeMax);

    char * dataPtr = NULL;       //f[^obt@ւ̃|C^
    dataPtr = static_cast<char *>(dataBuf);

    char line[LOG_LINEBUF_LEN]; //sobt@
    int  outputTotal = 0;       //o̓f[^TCY(oCg)
    LogEntry  le;
    for(int i = 0; i < LOG_ENTRY_NUM_MAX ;i++)
    {
        le = Get_LogEntry(i);
        if (le.used != LogEntry::ENTRY_UNUSED )       //gpGgȊO
        {
            memset(line, 0x00, sizeof(line));
            sprintf(line,"%d,%s,%s,%s,%s,%s,%s,%s,\n",le.testmode,
                                                      le.testerName,
                                                      le.state,
                                                      le.version,
                                                      le.date,
                                                      le.time,
                                                      le.testerId,
                                                      le.message);

            sprintf(dataPtr, line);                  //f[^ɏo
            dataPtr = dataPtr + strlen(line);
            outputTotal = outputTotal + strlen(line);//o̓TCYZ
        }

#if 0
        // Ot@C̃TCY𒴂ĂȂ`FbNAȂ΃G[
        if ( (m_fileSizeMax - LOG_LINEBUF_LEN) < outputTotal )
        {
            uji::sys::Free(dataBuf);
            NN_LOG("%s outputTotal is invalid.\n",NN_FUNCTION);
            NN_LOG(" outputTotal %d\n",outputTotal);
//            NN_LOG(" m_fileSizeMax %d\n",m_fileSizeMax);
//            NN_LOG(" LOG_LINEBUF_LEN %d\n",LOG_LINEBUF_LEN);
            return LogResultCode::FILESIZE_OVER;
        }
#endif
    }

    // nbVvZ
    nn::crypto::Sha1Context sha1;
    sha1.Initialize();
    // nbVvZpm
    int hashLength = 0;
    hashLength = sha1.GetHashLength();
    NN_LOG("hashLength:%d\n",hashLength);
    void* hashBuf = uji::sys::Alloc(hashLength);
    if ( NULL == hashBuf )
    {
        uji::sys::Free(dataBuf);
        return LogResultCode::ALLOCATE_ERROR;
    }
    memset(hashBuf, 0x00, sizeof(hashBuf));

    // nbVvZ
    NN_LOG("outputTotal:%d\n",outputTotal);

    /// outputTotal0 ̏ꍇAsha1.Update()֐̑20ƁAassertion(SDK2.4j
    /// assertion̂߁Am_DataHash̏lɋKl(HASH_NO_DATA)ݒB
    if (0 == outputTotal )
    {
        memset(m_DataHash, 0x00, sizeof(m_DataHash));
        sprintf(m_DataHash, "%s", HASH_NO_DATA.c_str());
        goto fin;
    }

    sha1.Update(dataBuf, outputTotal);
    NN_LOG("%s outputTotal=%d\n",NN_FUNCTION,outputTotal);
    sha1.GetHash(hashBuf);

    // nbVl̕\
    u8 * hashPtr;
    hashPtr = static_cast<u8 *>(hashBuf);
    memset(m_DataHash, 0x00, sizeof(m_DataHash));
    DEBUG_WRITE(" sha1 DataHash:[");
    for(int i = 0; i < hashLength; i++)
    {
        DEBUG_WRITE("%02x",*hashPtr);
        sprintf(&m_DataHash[i*2], "%02X",*hashPtr);
        hashPtr++;
    }
    DEBUG_WRITE("]\n");

fin:
    DEBUG_WRITE(" check m_DataHash[%s]\n",&m_DataHash[0]);

    // I
    sha1.Finalize();
    sha1.~Sha1Context();

    uji::sys::Free(dataBuf);
    uji::sys::Free(hashBuf);

    return LogResultCode::SUCCESS;
}

// siOf[^)͂AOe[uɓo^
void ProductionLog::Analyze_LogLine(u8 * line, int line_len)
{
    u8 * ptr = NULL;
    u8 * ptr2 = NULL;
    int charCount =0;
    int commaCount = 0;
    u8 mode     [LOG_LINEBUF_LEN];
    u8 name     [LOG_LINEBUF_LEN];
    u8 status   [LOG_LINEBUF_LEN];
    u8 ver      [LOG_LINEBUF_LEN];
    u8 date     [LOG_LINEBUF_LEN];
    u8 time     [LOG_LINEBUF_LEN];
    u8 testerId [LOG_LINEBUF_LEN];
    u8 msg      [LOG_LINEBUF_LEN];

    memset(mode,     0x00, sizeof(mode));
    memset(name,     0x00, sizeof(name));
    memset(status,   0x00, sizeof(status));
    memset(ver,      0x00, sizeof(ver));
    memset(date,     0x00, sizeof(date));
    memset(time,     0x00, sizeof(time));
    memset(testerId, 0x00, sizeof(testerId));
    memset(msg,      0x00, sizeof(msg));

// , āA͂Ă
    ptr = line;
    ptr2 = ptr;
    for(int i = 0; i < line_len; i++)
    {
        charCount++;
        if ( ',' == *ptr )
        {
            commaCount++;
            if (commaCount == 1)      { memcpy(mode,ptr2,(charCount-1));}
            else if (commaCount == 2) { memcpy(name,ptr2,(charCount-1));}
            else if (commaCount == 3) { memcpy(status,ptr2,(charCount-1));}
            else if (commaCount == 4) { memcpy(ver,ptr2,(charCount-1));}
            else if (commaCount == 5) { memcpy(date,ptr2,(charCount-1));}
            else if (commaCount == 6) { memcpy(time,ptr2,(charCount-1));}
            else if (commaCount == 7) { memcpy(testerId,ptr2,(charCount-1));}
            else if (commaCount == 8) { memcpy(msg,ptr2,(charCount-1));}
            ptr2 = ptr;
            ptr2++;
            charCount = 0;
        }
        ptr++;
    }

    // J}̐𒲂ׂB
    if ( commaCount > LOG_COMMA_MAX )
    {
        NN_LOG("%s invalid commaCount(%d)! skipped \n",NN_FUNCTION, commaCount);
        return;
    }
    else if (( 0 == commaCount ) && ( line_len != 0)) // commaCount =0 ŁAlineLength0łȂꍇ
    {
        // 1sڂ̃nbVl m_HashEntryValueɃRs[
        memset(m_HashEntryValue, 0x00, sizeof(m_HashEntryValue));
        memcpy(m_HashEntryValue, line, line_len);
        return;
    }
    else if ( 0 == commaCount)
    {
        // s
        //NN_LOG(" commaCount(%d) skip brank line.\n",commaCount);
        return;
    }
    else
    {
        int index  = -1;
        index = Get_NextId();
        DEBUG_WRITE("index:%d\n",index);
        DEBUG_WRITE("index(%d) mode(%s) name(%s) status(%s) ver(%s) date(%s) time(%s) testerId(%s) msg(%s)\n",
                           index, mode,name,status,ver,date,time,testerId,msg);
        if ( index < 0 )
        {
            NN_LOG("%s index(%d) is invalid!\n",NN_FUNCTION,index);
            return;
        }
        else
        {
            // Oe[uɓo^
            LogEntry le;
            Create_LogEntry(le,  atoi(reinterpret_cast<char *>(mode)),
                                     reinterpret_cast<char *>(name),
                                     reinterpret_cast<char *>(status),
                                     reinterpret_cast<char *>(ver),
                                     reinterpret_cast<char *>(date),
                                     reinterpret_cast<char *>(time),
                                     reinterpret_cast<char *>(testerId),
                                     reinterpret_cast<char *>(msg));

            LogTable_Entry(index, le);
        }
    }
}

//
// f[^[h
//
LogResult ProductionLog::LoadData(void * buf)
{

///TODO
///  bufs̈悾ANGƂB
///
    u8 * ptr = reinterpret_cast<u8 *>(buf);
    u8 * ptr2 = ptr;
    int i = 0;
    int line = 0;
    int charCount = 0;
    u8  lineBuf[LOG_LINEBUF_LEN];

    // LogTableNA
    LogTable_Clear();

    // obt@f[^ǂ݁ALogTableɓo^
    for(i =0; i < m_fileSize; i++)
    {
        charCount++;
        if ( 0x0a == *ptr ) //sR[h
        {
            memset(lineBuf,0x00,sizeof(lineBuf));
            memcpy(lineBuf,ptr2, (charCount-1));
            DEBUG_WRITE("(%d)[%s]\n",line,lineBuf);
            Analyze_LogLine(lineBuf, (charCount-1)); //œo^

            line++; //s̃JEg
            ptr2 = ptr;
            ptr2++;
            charCount = 0;
        }
        //NN_LOG("%c",*ptr);
        ptr++;
    }//for

    DEBUG_WRITE("line:%d\n",line);
    return LogResultCode::SUCCESS;
}

//
// Ot@C
//
// ̊֐̓Ot@C݂ȂԂłAgpłȂB
// ɃOt@Cꍇ́AFS_WRITE_ERRORIɕԂB
LogResult ProductionLog::InitFile(const wstring& path)
{
    LogResult lr = LogResultCode::SUCCESS;
    char line[LOG_LINEBUF_LEN];
    s32 writeBytes=0;     //݃TCY
    s32 writeBytesTotal=0;//݃TCYv

    NN_LOG("%s (%s) called.\n", NN_FUNCTION, path.c_str());
    nn::fs::FileOutputStream fos(path.c_str(), true);

    if ( fos.GetSize() != 0 )
    {
        //OԈĂ邽߁A
        //ɂAFS_WRITE_ERRORIɕԂ
        lr = LogResultCode::FS_WRITE_ERROR;
        NN_LOG(" fileSize(%d)\n", path.c_str(),fos.GetSize());
        return lr;
    }

    //wb_̏o
    LogTable_Clear();
    Calc_DataHash();

    memset(line, 0x00,sizeof(line));
    sprintf(line,"%s\n\n",m_DataHash);

    writeBytes = fos.Write(line,strlen(line));
    writeBytesTotal = writeBytes;

    fos.Finalize();
    NN_LOG(" writeBytesTotal(%d)\n", writeBytesTotal);
    return lr;
}

//
// Oe[ut@Cɏ
//
LogResult ProductionLog::WriteFile(const wstring& path, const string & message)
{
    nn::Result nnr;
    LogEntry  le;
    char line[LOG_LINEBUF_LEN];
    s32 writeBytes=0;      //݃TCY
    s32 writeBytesHeader=0;    //݃TCYwb_
    s32 writeBytesLinesTotal=0;//݃TCY O

    nn::os::Tick tick_start;//vJn
    nn::os::Tick tick_end;  //vI
    tick_start=  nn::os::Tick::GetSystemCurrent();

    nn::fs::FileOutputStream fos(path.c_str(), true);
    DEBUG_WRITE("%s [%ls] fileSize(%d)\n", NN_FUNCTION, path.c_str(),fos.GetSize());

    //t@CTCYMaxɁAShrinksAOGg̒ZȂƃS~邽߁A
    //t@C폜
    nnr = fos.TrySetSize(0);
    if (nnr.IsFailure())
    {
        NN_LOG("fos.TrySetSize failed.\n");
        nn::dbg::PrintResult(nnr);
        return LogResultCode::FS_WRITE_ERROR;
    }

    //wb_̏o
    //(⑫)nbVľvŹAe[uɃGgo^ƂƁA[ĥݍŝŁA
    //      ł͌vZȂB
    memset(line, 0x00,sizeof(line));
    sprintf(line,"%s\n\n",m_DataHash);
    writeBytes = fos.Write(line,strlen(line));
    writeBytesHeader = writeBytes;

    // f[^sobt@
    void* LINES_BUF = uji::sys::Alloc(m_fileSizeMax);
    memset(LINES_BUF, 0x00, m_fileSizeMax);

    //f[^̏o
    int writeLines = 0;//s
    int i = 0;
    int j = 0;
    char * writePtr = (char *)LINES_BUF;
    for(;i < LOG_ENTRY_NUM_MAX ;i++)
    {
        writeBytes = 0;
        le = Get_LogEntry(i);
        if (le.used != LogEntry::ENTRY_UNUSED )       //gpGgȊO
        {
            memset (line, 0x00, sizeof(line));
//NN_LOG("sizeof(line):%d\n",sizeof(line)); -> sizeof(line)256
            sprintf(line,"%d,%s,%s,%s,%s,%s,%s,%s,",le.testmode,
                                                    le.testerName,
                                                    le.state,
                                                    le.version,
                                                    le.date,
                                                    le.time,
                                                    le.testerId,
                                                    le.message);
#if 0
            NN_LOG(" strlen(line):%d line:[%s]\n",strlen(line), line);
#endif

            // Ot@C̍őTCY𒴂ĂȂ`FbNAȂΏIB
            if ( writeBytesLinesTotal + (strlen(line)+1) >= m_fileSizeMax  )
            {
                NN_LOG("%s i=%d (%s) FILESIZE_OVER !!\n", NN_FUNCTION, i, message.c_str());
                NN_LOG(" m_fileSizeMax(%d) < next writeBytesTotal(%d)\n", m_fileSizeMax, (writeBytesLinesTotal + strlen(line)+1));
                goto LINES_TOTAL_WRITE;
            }
            writeBytes = sprintf(writePtr, "%s\n",line);
            writeLines++;
            for(j = 0; j < strlen(line)+1; j++)
            {
                writePtr++;
            }
        }

        writeBytesLinesTotal = writeBytes + writeBytesLinesTotal;
    }

LINES_TOTAL_WRITE:
    fos.Write(LINES_BUF, writeBytesLinesTotal);
//    NN_LOG("writeLines %d writeBytesLinesTotal %d\n", writeLines, writeBytesLinesTotal);
    uji::sys::Free(LINES_BUF);

    fos.Finalize();

    tick_end =  nn::os::Tick::GetSystemCurrent();//vI
    int elapsed = (int)(tick_end - tick_start).ToTimeSpan().GetMilliSeconds();
    NN_LOG("%s            %s lines(%d) writeBytesTotal(%d) elapsedMsec(%d)\n",__func__, message.c_str(), writeLines,writeBytesHeader+writeBytesLinesTotal,elapsed);

    return LogResultCode::SUCCESS;
}

int  ProductionLog::Get_LogEntryByteSize(LogEntry & le) const
{

    char line[LOG_LINEBUF_LEN];

    memset (line, 0x00, sizeof(line));
    sprintf(line,"%d,%s,%s,%s,%s,%s,%s,%s,",le.testmode,
                                              le.testerName,
                                              le.state,
                                              le.version,
                                              le.date,
                                              le.time,
                                              le.testerId,
                                              le.message);

    return (strlen(line) + 1);
}


// Oe[u
void ProductionLog::LogTable_Clear(void)
{
    DEBUG_WRITE("%s sizeof(m_logTable)=%d\n",NN_FUNCTION, sizeof(m_logTable));
    memset(m_logTable, 0x00, sizeof(m_logTable));
    DEBUG_WRITE(" clear m_logTable size:(%d)\n", sizeof(m_logTable));
}

// fobOp֐: LogTable_v
void ProductionLog::LogTable_Dump(void) const
{
    char line[LOG_LINEBUF_LEN];

    int i = 0;
    for(i=0; i < LOG_ENTRY_NUM_MAX ; i++)
    {
        memset(line, 0x00, sizeof(line));

        // usedtOĂ݂̂̂dump
        if ( LogEntry::ENTRY_USED == m_logTable[i].used )
        {
            sprintf(line, "(%d)(%d,%s,%s,%s,%s,%s,%s,%s)",i
                                                         ,m_logTable[i].testmode
                                                         ,m_logTable[i].testerName
                                                         ,m_logTable[i].state
                                                         ,m_logTable[i].version
                                                         ,m_logTable[i].date
                                                         ,m_logTable[i].time
                                                         ,m_logTable[i].testerId
                                                         ,m_logTable[i].message);
            NN_LOG("%s\n",line);
        }
    }//for
}

// fobOp֐: LogTableS _v
void ProductionLog::LogTable_DumpAll(void) const
{
    char line[LOG_LINEBUF_LEN];
    int i = 0;

    for(i=0; i < LOG_ENTRY_NUM_MAX ; i++)
    {
        memset(line, 0x00, sizeof(line));
        // Gg̍ő吔ASdump
        sprintf(line, "(%d)(%d,%d,%s,%s,%s,%s,%s,%s,%s)",i
                                                        ,m_logTable[i].testmode
                                                        ,static_cast<int>(m_logTable[i].used)
                                                        ,m_logTable[i].testerName
                                                        ,m_logTable[i].state
                                                        ,m_logTable[i].version
                                                        ,m_logTable[i].date
                                                        ,m_logTable[i].time
                                                        ,m_logTable[i].testerId
                                                        ,m_logTable[i].message);
        NN_LOG("%s\n",line);
    }//for
}

// ޔEp
LogResult ProductionLog::Identify_RestoreTarget(void)
{
    return LogResultCode::SUCCESS;
}

/*---------------------------------------------------------------------------*
  End of file
 *---------------------------------------------------------------------------*/
