﻿/*--------------------------------------------------------------------------*
 Project:
 File: NWSound.cpp

 NintendoWare SimpleApp.cppを参考に作成しました

*--------------------------------------------------------------------------*/
#include <nn/snd.h>
#include <nw/os.h>
#include <nw/snd.h>

#include "NWSound.h"
#include "simple.csid"
#include "SoundInitializer.h"

namespace uji {
namespace eva {

namespace 
{

const s32 SOUND_HEAP_SIZE = 1 * 1024 * 1024;
const char SOUND_ARC_PATH[] = "rom:/snd/simple.bcsar";
const char DEMO_TITLE[] = "NWSoundApp";

} // anonymous namespace


//============================================================================
//      初期化・終了処理
//============================================================================
/*
    Desc: NWSoundを使用する時だけ実行する初期処理
*/
void NWSoundApp::OnInitialize()
{
    InitializeSoundSystem();

    // サウンドデータのロード
    if ( ! m_DataManager.LoadData( SEQ_MARIOKART, &m_Heap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
    }
    
    if ( ! m_DataManager.LoadData( SE_YOSHI, &m_Heap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
    }
}

/*
    Desc: SoundSystemの初期化
*/
void NWSoundApp::InitializeSoundSystem()
{
    // サウンドシステムの初期化
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        m_pMemoryForSoundSystem = sys::AllocDeviceMemory( workMemSize );

        nw::snd::SoundSystem::Initialize(
                param,
                reinterpret_cast<uptr>( m_pMemoryForSoundSystem ),
                workMemSize );
    }

    // サウンドアーカイブの初期化
    if ( ! m_Archive.Open( SOUND_ARC_PATH ) )
    {
        NW_ASSERTMSG( 0, "cannot open bcsar(%s)\n", SOUND_ARC_PATH );
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = m_Archive.GetHeaderSize();
        m_pMemoryForInfoBlock = sys::AllocDeviceMemory( infoBlockSize );
        if ( ! m_Archive.LoadHeader( m_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = m_DataManager.GetRequiredMemSize( &m_Archive );
        m_pMemoryForSoundDataManager = sys::AllocDeviceMemory( setupSize );
        m_DataManager.Initialize( &m_Archive, m_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = m_ArchivePlayer.GetRequiredMemSize( &m_Archive );
        m_pMemoryForSoundArchivePlayer = sys::AllocDeviceMemory( setupSize, 32 );
        size_t setupStrmBufferSize =
            m_ArchivePlayer.GetRequiredStreamBufferSize( &m_Archive );
        m_pMemoryForStreamBuffer = sys::AllocDeviceMemory( setupStrmBufferSize, 32 );
        bool result = m_ArchivePlayer.Initialize(
                &m_Archive,
                &m_DataManager,
                m_pMemoryForSoundArchivePlayer, setupSize,
                m_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        m_pMemoryForSoundHeap = sys::AllocDeviceMemory( SOUND_HEAP_SIZE );
        bool result = m_Heap.Create( m_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }
}

/*
    Desc: NWSoundの終了処理
          ※OnInitialize の終了処理です
*/
void NWSoundApp::OnFinalize()
{
    m_Heap.Destroy();    
    m_ArchivePlayer.Finalize();
    m_DataManager.Finalize();
    m_Archive.Close();
    
    nw::snd::SoundSystem::Finalize();

    sys::Free( m_pMemoryForSoundSystem );
    sys::Free( m_pMemoryForInfoBlock );
    sys::Free( m_pMemoryForSoundDataManager );
    sys::Free( m_pMemoryForSoundArchivePlayer );
    sys::Free( m_pMemoryForSoundHeap );
    sys::Free( m_pMemoryForStreamBuffer );
}


//============================================================================
//      EVAメニュー関連
//============================================================================

/*
    Desc: 操作説明表示
*/
void NWSoundApp::OnDrawUpLCD()
{
    sys::Menu::m_SubWindow->Printf ("\f");
    sys::Menu::m_SubWindow->Printf (" DEMO nw::snd %s\n\n", DEMO_TITLE);

    sys::Menu::m_SubWindow->Printf ("    -- usage --\n\n");
    sys::Menu::m_SubWindow->Printf ("    [A] Play Sequence Sound\n");
    sys::Menu::m_SubWindow->Printf ("    [X] Play Wave Sound\n");
#ifdef EVA_NWSOUND_USE_STREAM    
    sys::Menu::m_SubWindow->Printf ("    [Y] Play Stream Sound\n");
#endif
    sys::Menu::m_SubWindow->Printf ("    [B] Stop Sound\n");
    sys::Menu::m_SubWindow->Printf ("    [S] Exit");
}

/*
    Desc: パッド操作
*/
void NWSoundApp::OnUpdatePad(const sys::Pad& pad)
{
    if ( pad.IsButtonDown( sys::Pad::BUTTON_A ) )
    {
        m_Handle.Stop( 0 );
        bool result = m_ArchivePlayer.StartSound( &m_Handle, SEQ_MARIOKART ).IsSuccess();
        NN_LOG("[SEQ] SEQ_MARIOKART ... (%d)\n", result);
    }

    if ( pad.IsButtonDown( sys::Pad::BUTTON_X ) )
    {
        m_Handle.Stop( 0 );
        bool result = m_ArchivePlayer.StartSound( &m_Handle, SE_YOSHI ).IsSuccess();
        NN_LOG("[WSD] SE_YOSHI ... (%d)\n", result);
    }
#ifdef EVA_NWSOUND_USE_STREAM    
    if ( pad.IsButtonDown( sys::Pad::BUTTON_Y ) )
    {
        m_Handle.Stop( 0 );
        bool result = m_ArchivePlayer.StartSound( &m_Handle, STRM_MARIOKART ).IsSuccess();
        NN_LOG("[STRM] STRM_MARIOKART ... (%d)\n", result );
    }
#endif
    if ( pad.IsButtonDown( sys::Pad::BUTTON_B ) )
    {
        m_Handle.Stop( 3 );
    }
}

/*
    Desc: 毎フレーム実施する処理
          ※エージング音楽再生時は途中でパラメータの変更を行わないので・・
*/
void NWSoundApp::OnUpdate()
{
    m_ArchivePlayer.Update();
}

/*
    Desc: EVAメニューの本体
*/
void NWSoundApp::EvaRun()
{
    sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();
    
    SoundInitializer().InitSDK();    
    OnInitialize();
    
    do
    {          
        sys::Pad().UpdatePad();
        
        OnUpdatePad(sys::Pad());
        OnUpdate();
        
        OnDrawUpLCD();        

        // 画面更新
	    sys::Menu::m_WindowManager.Update();
	    sys::Menu::m_WindowManager.UpdatePad(sys::Pad());
	    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
	    gfx->m_DrawFramework->Clear();        
	    sys::Menu::m_WindowManager.DrawDisplay1();
	    gfx->m_DrawFramework->SwapBuffers();
        gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
    }
    while (!sys::Pad().IsButtonDown(sys::Pad::BUTTON_START));    
    
    OnFinalize();   
}


//============================================================================
//      TestList 呼び出し用
//============================================================================

/*
    Desc: エージングサウンド再生
*/
bool NWSoundApp::PlayAgingSound(seq::TestResult &result)
{
    SoundInitializer().InitSDK();    
    OnInitialize();

    m_Handle.Stop( 0 );
    result.m_Result = m_ArchivePlayer.StartSound(&m_Handle, SEQ_MARIOKART).IsSuccess();
    m_ArchivePlayer.Update();
    
    return result.m_Result;
}

/*
    Desc: エージングサウンド停止
*/
bool NWSoundApp::StopAgingSound(seq::TestResult &result)
{
    m_Handle.Stop( 3 );
    m_ArchivePlayer.Update();
    
    OnFinalize();
    SoundInitializer().FinalizeSDK();        
    
    return result.m_Result = true;    
}

namespace agingsnd{

NWSoundApp* pApp = 0;

bool StartAgingSound(seq::TestResult &result)
{
    pApp = new NWSoundApp;
    
    pApp->PlayAgingSound(result);
    
    return result.m_Result;
}

void UpdatePlayer()
{
    if (pApp)
        pApp->OnUpdate();
}

bool StopAgingSound(seq::TestResult &result)
{    
    pApp->StopAgingSound(result);
    
    delete pApp;
    pApp = 0;
    
    return result.m_Result;
}    
    
}   // namespace agingsnd


}   // namespace
}
