﻿/*---------------------------------------------------------------------------*
  Project:  
  File: TestKey.cpp

  キー入力テスト
 *---------------------------------------------------------------------------*/

//------------------------------------------------------------------
// アニメーションについて
//
//  animShare.clan ファイルでは、ペイン「Btn_A」にのみアニメーションが
//  設定されており、このアニメーションをグループ「Group_Key」に登録されている
//  各ペインで共有するようにアニメーション共有設定が行われています。
//  グループ「Group_Key」には、十字キーのセンターを除く全てのPictureペインが
//  登録されています。
//------------------------------------------------------------------

#include <GLES2/gl2.h>
#include <nn/hid/CTR/hid_Misc.h>        
#include <nw/demo.h>
    
#include "TestKey.h"
#include "../seq/Reception_board_interface.h"
#include "../mcu/TestMcuNotification.h"    

using namespace uji;
using namespace uji::sys;
using namespace uji::seq;

namespace uji {
namespace eva {
    

#define TEST_KEY_FILE_PATH(name) (L"rom:/" name)    

	// InputCheck時のキーカウント用変数
	int count_A = 0;
	int count_B = 0;
	int count_SELECT = 0;
	int count_START = 0;
	int count_RIGHT = 0;
	int count_LEFT = 0;
	int count_UP = 0;
	int count_DOWN = 0;
	int count_L = 0;
	int count_R = 0;
	int count_X = 0;
	int count_Y = 0;
	int count_HOME = 0;
	bool trigger_SEL = true;
	bool trigger_START = true;

/*---------------------------------------------------------------------------
  Desc: ファイルをロードします

  Args: allocator - アロケータです
        filePath  - ファイルパス名です
        align     - アライメントです
        
  Rtns: ロードしたファイルのバッファを返します
---------------------------------------------------------------------------*/  
nw::ut::MoveArray<u8> TestKey::LoadFile(
    nw::os::IAllocator* allocator,
    const wchar_t* filePath,
    u32 align /* = 32 */
)
{
    nn::fs::FileReader fileReader;

    nn::Result result = fileReader.TryInitialize(filePath);
    if (result.IsFailure())
    {
        return nw::ut::MoveArray<u8>();
    }

    s32 fileSize = static_cast<s32>(fileReader.GetSize());

    void* memory = allocator->Alloc(fileSize, static_cast<u8>(align));
    nw::ut::MoveArray<u8> buffer(memory, fileSize, allocator);
    buffer.resize(fileSize);

    fileReader.Read(&buffer.front(), fileSize);

    fileReader.Finalize();

    // コンテナを返してますが、バッファはコピーされません。
    // 所有権が移動します。
    return buffer;
}

/*---------------------------------------------------------------------------
  Desc: レイアウトリソースの初期化

  Args: なし 

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::InitializeLayoutResource()
{
    nw::lyt::Initialize(&memAllocator, &devMemAllocator);
    
    m_pLayout = new nw::lyt::Layout();

    // レイアウトのバイナリリソース(アーカイブ)を読み込み。
    if( uji::sys::PLATFORM_FTR == uji::sys::GetPlatformType())
    {
        m_fileLayout = LoadFile(
            &devMemAllocator, TEST_KEY_FILE_PATH(L"key/layout_ftr.arc"), 128);
    }else
    {
        m_fileLayout = LoadFile(
            &devMemAllocator, TEST_KEY_FILE_PATH(L"key/layout.arc"), 128);
    }
    if (m_fileLayout.empty())
    {
        NW_FATAL_ERROR("can not open layout archive.\n");
    }
    
    // バイナリリソースのルートディレクトリを指定してリソースアクセサを生成。
    m_pResAccessor = new nw::lyt::ArcResourceAccessor;
    if (!m_pResAccessor->Attach(m_fileLayout.begin(), "."))
    {
        NW_FATAL_ERROR("can not attach layout archive.\n");
    }

    // レイアウトリソースの読み込み
    {
        const void* lytRes = m_pResAccessor->GetResource(0, "animShare.bclyt");
        NW_NULL_ASSERT(lytRes);
        m_pLayout->Build(lytRes, m_pResAccessor);
    }

    // アニメーションリソースの読み込み
    {
        const void* lpaRes = m_pResAccessor->GetResource(0, "animShare_focus.bclan");
        NW_NULL_ASSERT(lpaRes);
        m_AnimRes.Set(lpaRes);
    }

    // アニメーション共有設定、アニメーション区間タグの関連グループを
    // 考慮したアニメーションのバインドを行います。
    m_pLayout->BindAnimationAuto(m_AnimRes, m_pResAccessor);

    // Layout::BindAnimationAuto() でバインドしたアニメーションは無効になっています。
    // 全体のアニメーションを有効にします。
    m_pLayout->GetRootPane()->SetAnimationEnable(m_AnimRes, true/*bEnable*/, true/*bRecursive*/);

    // グローバルなリソースファイルを読み込みます。
    {
        m_GraphicsResource.StartSetup();
        const wchar_t* resourcePath = 0;
        for (int i = 0;
             (resourcePath = m_GraphicsResource.GetResourcePath(i)) != NULL;
             ++i)
        {
            File file = LoadFile(&memAllocator, resourcePath);

            if (file.empty())
            {
                NW_FATAL_ERROR("can not open lyt resource file.");
            }

            m_GraphicsResource.SetResource(i, file.begin(), file.size(), false);
        }
        m_GraphicsResource.FinishSetup();
    }

    m_DrawInfo.SetGraphicsResource(&m_GraphicsResource);
}

/*---------------------------------------------------------------------------
  Desc: ボタン配列情報の初期化

  Args: なし

  Rtns: なし
---------------------------------------------------------------------------*/
void TestKey::InitializeButtonArray()
{
    m_ButtonArray.PushBack(new Button("Btn_A",      nn::hid::CTR::BUTTON_A));
    m_ButtonArray.PushBack(new Button("Btn_B",      nn::hid::CTR::BUTTON_B));
    m_ButtonArray.PushBack(new Button("Btn_X",      nn::hid::CTR::BUTTON_X));
    m_ButtonArray.PushBack(new Button("Btn_Y",      nn::hid::CTR::BUTTON_Y));    
    m_ButtonArray.PushBack(new Button("Btn_ST",     nn::hid::CTR::BUTTON_START));
    m_ButtonArray.PushBack(new Button("Key_R",      nn::hid::CTR::BUTTON_RIGHT));
    m_ButtonArray.PushBack(new Button("Key_L",      nn::hid::CTR::BUTTON_LEFT));
    m_ButtonArray.PushBack(new Button("Key_U",      nn::hid::CTR::BUTTON_UP));
    m_ButtonArray.PushBack(new Button("Key_D",      nn::hid::CTR::BUTTON_DOWN));
    m_ButtonArray.PushBack(new Button("Btn_R",      nn::hid::CTR::BUTTON_R));
    m_ButtonArray.PushBack(new Button("Btn_L",      nn::hid::CTR::BUTTON_L));
    m_ButtonArray.PushBack(new Button("Btn_SEL",    nn::hid::CTR::PSEUDO_BUTTON_SELECT));    
}

/*---------------------------------------------------------------------------
  Desc: 描画初期処理

  Args: なし

  Rtns: なし
---------------------------------------------------------------------------*/
void TestKey::InitDraw(int width, int height)
{
    // カラーバッファ情報
    // LCDの向きに合わせて、幅と高さを入れ替えています。
    const nw::font::ColorBufferInfo colBufInfo =
    {
        height, width, PICA_DATA_DEPTH24_STENCIL8_EXT
    };

    const u32 commands[] =
    {
        // ビューポートの設定
        NW_FONT_CMD_SET_VIEWPORT(0, 0, colBufInfo.width, colBufInfo.height),

        // シザー処理を無効
        NW_FONT_CMD_SET_DISABLE_SCISSOR(colBufInfo),

        // wバッファの無効化
        // デプスレンジの設定
        // ポリゴンオフセットの無効化
        NW_FONT_CMD_SET_WBUFFER_DEPTHRANGE_POLYGONOFFSET(
            0.0f,           // wScale : 0.0 でWバッファが無効
            0.0f,           // depth range near
            1.0f,           // depth range far
            0,              // polygon offset units : 0.0 で ポリゴンオフセットが無効
            colBufInfo),
    };

    nngxAdd3DCommand(commands, sizeof(commands), true);

    static const u32 constCommands[] =
    {
        // カリングを無効
        NW_FONT_CMD_SET_CULL_FACE(NW_FONT_CMD_CULL_FACE_DISABLE),

        // ステンシルテストを無効
        NW_FONT_CMD_SET_DISABLE_STENCIL_TEST(),

        // デプステストを無効
        // カラーバッファの全ての成分を書き込み可
        NW_FONT_CMD_SET_DEPTH_FUNC_COLOR_MASK(
            false,  // isDepthTestEnabled
            0,      // depthFunc
            true,   // depthMask
            true,   // red
            true,   // green
            true,   // blue
            true),  // alpha

        // アーリーデプステストを無効
        NW_FONT_CMD_SET_ENABLE_EARLY_DEPTH_TEST(false),

        // フレームバッファアクセス制御
        NW_FONT_CMD_SET_FBACCESS(
            true,   // colorRead
            true,   // colorWrite
            false,  // depthRead
            false,  // depthWrite
            false,  // stencilRead
            false), // stencilWrite
    };

    nngxAdd3DCommand(constCommands, sizeof(constCommands), true);
}

/*---------------------------------------------------------------------------
  Desc: 共通の初期化

  Args: なし

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::InitializeCommon()
{
	// InputCheck時のキーカウント用変数の初期化
	count_A = 0;
	count_B = 0;
	count_SELECT = 0;
	count_START = 0;
	count_RIGHT = 0;
	count_LEFT = 0;
	count_UP = 0;
	count_DOWN = 0;
	count_L = 0;
	count_R = 0;
	count_X = 0;
	count_Y = 0;
	count_HOME = 0;

	uji::sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();
    
    InitializeLayoutResource();
    InitializeButtonArray();
    
    gfx->m_DrawFramework->SetClearColor(NN_GX_DISPLAY0, 0.65f, 0.65f, 0.65f, 1.0f);
    gfx->m_DrawFramework->SetClearColor(NN_GX_DISPLAY1, 0.65f, 0.65f, 0.65f, 1.0f);

    // テキストウインドウ生成
    m_PutTriggerCountWindow = new sys::TextWindow(15, MAX_BUTTON_NUM, 14);
    (void)m_WindowManager.CreateWindow(m_PutTriggerCountWindow, NN_GX_DISPLAY1, 0, 0);    
    m_PutTriggerCountWindow->SetTitle("TRIGGER");
    // アクティブ指定を解除
    m_PutTriggerCountWindow->SetActiveFlag(false);
    
    // ウインドウマネージャのキー操作を無効に
    m_WindowManager.SetManagerPadControlFlag(false);    
}

/*---------------------------------------------------------------------------
  Desc: 共通の終了処理

  Args: なし

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::FinalizeCommon()
{
    m_pResAccessor->Detach();
    
    m_WindowManager.DestroyWindow(m_PutTriggerCountWindow);
    delete m_PutTriggerCountWindow;
    
    delete m_pLayout;
    delete m_pResAccessor;

    uji::sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();
    gfx->m_DrawFramework->SetClearColor(NN_GX_DISPLAY0, 
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_RED, 
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_GREEN, 
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_BLUE, 
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_ALPHA);
    gfx->m_DrawFramework->SetClearColor(NN_GX_DISPLAY1,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_RED, 
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_GREEN, 
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_BLUE, 
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_ALPHA);
}

/*---------------------------------------------------------------------------
  Desc: モデルビュー行列と射影行列を設定します

  Args: drawInfo    - 描画情報
        layout      - レイアウト

  Rtns: なし
---------------------------------------------------------------------------*/
void SetupCamera(nw::lyt::DrawInfo& drawInfo, const nw::lyt::Layout& layout)
{
    nw::ut::Rect layoutRect = layout.GetLayoutRect();

    f32 znear = 0.f;
    f32 zfar = 500.f;

    // 射影行列を正射影に設定
    // (Layoutデータは横向きなので向きを変換する)
    nw::math::MTX44 projMtx;
    nw::math::MTX44Ortho(
        &projMtx,
        layoutRect.bottom,  // left
        layoutRect.top,     // right
        -layoutRect.right,  // bottom
        -layoutRect.left,   // top
        znear,
        zfar);
    drawInfo.SetProjectionMtx(projMtx);

    // モデルビュー行列を設定
    // (Layoutデータは横向きなので画面の上方向はレイアウトの-X方向)
    nw::math::VEC3 pos(0, 0, 1);
    nw::math::VEC3 up(-1, 0, 0);
    nw::math::VEC3 target(0, 0, 0);

    nw::math::MTX34 viewMtx;
    nw::math::MTX34LookAt(&viewMtx, &pos, &up, &target);
    drawInfo.SetViewMtx(viewMtx);
}

/*---------------------------------------------------------------------------
  Desc: AnimationLinkを検索します
        指定したアニメーションリソースをペインやマテリアルにバインドしている
        nw::lyt::AnimationLink を検索し、最初に見つかったものを返します。
        指定されたペインの子孫を含めて検索します。

  Args: pPane   - 検索対象のルートとなるペインです
        animRes - アニメーションリソース

  Rtns: 見つかった場合は、nw::lyt::AnimationLink のポインタを返します。
        見つからなかった場合は、NULL を返します。
---------------------------------------------------------------------------*/ 
nw::lyt::AnimationLink* FindLayoutAnimationLink(
    nw::lyt::Pane*                pPane,
    const nw::lyt::AnimResource&  animRes
)
{
    // ペインアニメーションの検索
    if (nw::lyt::AnimationLink *const pAnimLink = pPane->FindAnimationLinkSelf(animRes))
    {
        return pAnimLink;
    }

    // マテリアルアニメーションの設定
    const u8 materialNum = pPane->GetMaterialNum();
    for (u8 i = 0; i < materialNum; ++i)
    {
        if (nw::lyt::AnimationLink *const pAnimLink = pPane->GetMaterial(i)->FindAnimationLink(animRes))
        {
            return pAnimLink;
        }
    }

    // 再帰的に検索します。
    for (nw::lyt::PaneList::Iterator it = pPane->GetChildList().GetBeginIter(); it != pPane->GetChildList().GetEndIter(); ++it)
    {
        if (nw::lyt::AnimationLink* pAnimLink = FindLayoutAnimationLink(&(*it), animRes))
        {
            return pAnimLink;
        }
    }

    return NULL;
}

/*---------------------------------------------------------------------------
  Desc: アニメーションリソースのフレーム値を設定します
        指定したアニメーションリソースの現在のフレーム値を全て同じに設定します。
        指定されたペインの子孫を含めて行います。

  Args: pPane   - 対象となるペインのルートとなるペイン
        animRes - アニメーションリソース
        frame   - 設定するフレーム値

  Rtns: なし
---------------------------------------------------------------------------*/
void SetAnimationFrame(
    nw::lyt::Pane* pPane, 
    const nw::lyt::AnimResource& animRes, 
    f32 frame)
{
    // ペインアニメーションの設定
    if (nw::lyt::AnimationLink *const pAnimLink = pPane->FindAnimationLinkSelf(animRes))
    {
        pAnimLink->GetAnimTransform()->SetFrame(frame);
    }

    // マテリアルアニメーションの設定
    const u8 materialNum = pPane->GetMaterialNum();
    for (u8 i = 0; i < materialNum; ++i)
    {
        if (nw::lyt::AnimationLink *const pAnimLink = pPane->GetMaterial(i)->FindAnimationLink(animRes))
        {
            pAnimLink->GetAnimTransform()->SetFrame(frame);
        }
    }

    // 再帰的に行います。
    for (nw::lyt::PaneList::Iterator it = pPane->GetChildList().GetBeginIter(); it != pPane->GetChildList().GetEndIter(); ++it)
    {
        SetAnimationFrame(&(*it), animRes, frame);
    }
}

/*---------------------------------------------------------------------------
  Desc: ペインのキーアニメーション番号を取得

  Args: pPane   - 対象となるペイン

  Rtns: アニメーション番号
---------------------------------------------------------------------------*/
KeyAnimation TestKey::GetKeyAnimation(nw::lyt::Pane* pPane)
{
    for (ButtonArray::iterator itr = m_ButtonArray.begin(); itr != m_ButtonArray.end(); itr++)
    {    
        if (std::strcmp((*itr)->GetPaneName(), pPane->GetName()) == 0)
        {
             return (*itr)->GetKeyAnimation();   
        }
    }
    
    NN_ASSERTMSG(false, "No PushBack pane to m_ButtonArray!\n");   
    return KEY_ANIM_RELEASE; // ここには来ません
}

/*---------------------------------------------------------------------------
  Desc: ペインのキーアニメーション番号を設定する

  Args: pPane   - 対象となるペイン
        keyAnim - アニメーション番号

  Rtns: なし
---------------------------------------------------------------------------*/
void TestKey::SetKeyAnimation(nw::lyt::Pane* pPane, KeyAnimation keyAnim)
{
    for (ButtonArray::iterator itr = m_ButtonArray.begin(); itr != m_ButtonArray.end(); itr++)
    {    
        if (std::strcmp((*itr)->GetPaneName(), pPane->GetName()) == 0)
        {
            (*itr)->SetKeyAnimation(keyAnim);   
        }
    }    
}

/*---------------------------------------------------------------------------
  Desc: キー入力情報から、ペインのキーアニメーション番号を設定する

  Args: hold   - キーの状態

  Rtns: なし
---------------------------------------------------------------------------*/
void TestKey::SetKeyInputAnimation(bit32 hold)
{
    for (ButtonArray::iterator itr = m_ButtonArray.begin(); itr != m_ButtonArray.end(); itr++)
    {    
        if (hold & (*itr)->GetBitButton())
        {
            (*itr)->SetKeyAnimation(KEY_ANIM_INPUT);
        }       
        // TODO: m_pCurrentTestButton を直接参照しないように変更する
        else if ((m_pCurrentTestButton) && (*m_pCurrentTestButton & (*itr)->GetBitButton()))
        {
            (*itr)->SetKeyAnimation(KEY_ANIM_FOCUS);
        }        
        else
        {
            (*itr)->SetKeyAnimation(KEY_ANIM_RELEASE);
        }
    }        
}
    
/*---------------------------------------------------------------------------
  Desc: アニメーションの状態を更新

  Args: pPane   - アニメーションを更新するペイン階層のルートとなるペイン
        animRes - アニメーションリソースオブジェクト

  Rtns: なし
---------------------------------------------------------------------------*/ 
void TestKey::UpdateAnimationFrame(nw::lyt::Pane* pPane, const nw::lyt::AnimResource& animRes)
{
    // 指定した アニメーションリソースをリンクする nw::lyt::AnimationLink を探します。
    // 最初に見つかったものが返ります。
    nw::lyt::AnimationLink *const pAnimationLink = FindLayoutAnimationLink(pPane, animRes);
    if (! pAnimationLink)
    {
        return;
    }

    // nw::lyt::AnimationLink から nw::lyt::AnimTransform 取得して、
    // 現在のフレーム値と最大フレームを取得します。 
    nw::lyt::AnimTransform *const pAnimTrans = pAnimationLink->GetAnimTransform();
    f32 animFrame = pAnimTrans->GetFrame();
    const f32 animFrameMax = static_cast<f32>(pAnimTrans->GetFrameSize() - 1);
    const f32 step = 1.f; 

    switch (GetKeyAnimation(pPane))
    {
    case KEY_ANIM_INPUT:
        // アニメーションがレイアウトエディタ（.bclan）で設定されている場合は、
        // ソースコード上で変更しても反映されないため、アニメーションを無効にする。           
        pPane->SetAnimationEnable(animRes, false, true);
        for (int i = 0; i < nw::lyt::VERTEXCOLOR_MAX; i++)
        {
            pPane->SetVtxColor(i, nw::ut::Color8(48, 48, 255, 255));
        }
        break;
    
    case KEY_ANIM_RELEASE:
        pPane->SetAnimationEnable(animRes, false, true);

        // ペインによって頂点カラーを変更
        if ((std::strcmp("Btn_R", pPane->GetName()) == 0) ||
            (std::strcmp("Btn_L", pPane->GetName()) == 0))
        {
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_RB, nw::ut::Color8(160, 160, 160, 255));
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_LB, nw::ut::Color8(160, 160, 160, 255));
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_RT, nw::ut::Color8(255, 255, 255, 255));
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_LT, nw::ut::Color8(255, 255, 255, 255));            
        }
        else
        {
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_RB, nw::ut::Color8(196, 196, 196, 255));
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_LB, nw::ut::Color8(255, 255, 255, 255));
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_RT, nw::ut::Color8(255, 255, 255, 255));
            pPane->SetVtxColor(nw::lyt::VERTEXCOLOR_LT, nw::ut::Color8(196, 196, 196, 255));            
        }
        break;
        
    case KEY_ANIM_FOCUS:
        pPane->SetAnimationEnable(animRes, true, true);                
        for (int i = 0; i < nw::lyt::VERTEXCOLOR_MAX; i++)
        {
            pPane->SetVtxColor(i, nw::ut::Color8(255, 48, 48, 255));            
        }
        // pPane以下の全てのペイン、マテリアルのアニメーションフレームを同じ値に設定。        
        SetAnimationFrame(pPane, animRes, m_focusAnimFrame);        
        break;
    
    default:
        break;
    }
}

/*---------------------------------------------------------------------------
  Desc: 各ペインの状態を更新

  Args: pGroup  - 更新する対象のペインが登録されているグループ
        animRes - アニメーションリソースオブジェクト

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::UpdatePane(nw::lyt::Group* pGroup, const nw::lyt::AnimResource& animRes)
{
    // キー情報の取得
    nn::hid::CTR::PadReader padReader;
    nn::hid::CTR::PadStatus padStatus;
    uji::sys::Pad pad;
    
    padReader.ReadLatest(&padStatus);
    
    // グループに登録されているペイン毎の処理
    nw::lyt::PaneLinkList& paneList = pGroup->GetPaneList();
    for (nw::lyt::PaneLinkList::Iterator it = paneList.GetBeginIter(); it != paneList.GetEndIter(); ++it)
    {
        // アニメーションを更新
        nw::lyt::Pane *const pPane = it->target;
        UpdateAnimationFrame(pPane, animRes);
    }
}

/*---------------------------------------------------------------------------
  Desc: 評価メニューから抜ける為のボタン情報を表示

  Args: gfx - 

  Rtns: なし
---------------------------------------------------------------------------*/  
void PutExitMessage(uji::sys::GraphicsDrawing* gfx)
{ 
#if 0 // RenderSystem 使用時は関数呼び出し元の設定で描画
    gfx->SetScreenSize(gfx->DISPLAY0_WIDTH, gfx->DISPLAY0_HEIGHT);
    gfx->m_DrawFramework->SetRenderingTarget(NN_GX_DISPLAY0);
#endif        
    gfx->SetFixedWidthFont(12);
    gfx->m_TextWriter.SetCursor(10, 225, 0);
    gfx->m_TextWriter.SetGradationMode(nn::font::CharWriter::GRADMODE_V);
    gfx->m_TextWriter.SetTextColor(nn::util::Color8::BLACK, nn::util::Color8::BLACK);
    gfx->BeginDrawingString(); 
        (void)gfx->m_TextWriter.Printf("EXIT-> START+L+R");
    gfx->EndDrawingString();    
    gfx->m_TextWriter.SetGradationMode(nn::font::CharWriter::GRADMODE_NONE);
}

/*---------------------------------------------------------------------------
  Desc: キーの入力情報を表示

  Args:  

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::PutInputCount(bit32 bitTrigger)
{
    int pos_y = 0;
    m_PutTriggerCountWindow->Printf("\f");
    m_PutTriggerCountWindow->SetTextColor(ATTR_COLOR_WHITE);
    
    for (ButtonArray::iterator itr = m_ButtonArray.begin(); itr != m_ButtonArray.end(); itr++)
    {    
        if (static_cast<bit32>(bitTrigger) & (*itr)->GetBitButton())
        {
            (*itr)->IncrementTriggerCount();
			// ここの(*itr)->GetBitButton()をセレクトも分かる関数に変える?
        }
        m_PutTriggerCountWindow->Gotoxy(0, pos_y);        
        m_PutTriggerCountWindow->Printf("%s", (*itr)->GetPaneName());
		// nw::lyt::Pane* pPane の pPane->GetName()で名前取れそう
        m_PutTriggerCountWindow->Gotoxy(8, pos_y++);        
        m_PutTriggerCountWindow->Printf(":%d", (*itr)->GetTriggerCount());

    }
}

/*---------------------------------------------------------------------------
  Desc: HOMEキーの入力情報を表示

  Args: なし

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::PutInputCount_HOME()
{
	nn::applet::ClearHomeButtonState();
	count_HOME++;
}

/*---------------------------------------------------------------------------
  Desc: キーの入力情報（スタート・セレクト）を表示

  Args:  

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::PutInputCount_SEL(bit32 bitTrigger)
{
    for (ButtonArray::iterator itr = m_ButtonArray.begin(); itr != m_ButtonArray.end(); itr++)
    {    
        if (static_cast<bit32>(bitTrigger) & (*itr)->GetBitButton())
        {
			if( bitTrigger & nn::hid::PSEUDO_BUTTON_SELECT )
			{
				if( trigger_SEL )	count_SELECT++;
				trigger_SEL = false;
			}
			if( bitTrigger & nn::hid::BUTTON_START )
			{
				if( trigger_START )		count_START++;
				trigger_START = false;
			}
        }
    }
}

/*---------------------------------------------------------------------------
  Desc: キーの入力情報（スタート・セレクト以外）を表示

  Args:  

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::PutInputCount_Sub(bit32 bitTrigger)
{
    int pos_y = 0;
    m_PutTriggerCountWindow->Printf("\f");
    m_PutTriggerCountWindow->SetTextColor(ATTR_COLOR_WHITE);
	bit32 bitbutton;
    
    for (ButtonArray::iterator itr = m_ButtonArray.begin(); itr != m_ButtonArray.end(); itr++)
    {
		bitbutton = (*itr)->GetBitButton();
        if (static_cast<bit32>(bitTrigger) & bitbutton)
        {
			if( bitbutton & nn::hid::BUTTON_A )		count_A++;
			if( bitbutton & nn::hid::BUTTON_B )		count_B++;
			if( bitbutton & nn::hid::BUTTON_RIGHT)	count_RIGHT++;
			if( bitbutton & nn::hid::BUTTON_LEFT )	count_LEFT++;
			if( bitbutton & nn::hid::BUTTON_UP )	count_UP++;
			if( bitbutton & nn::hid::BUTTON_DOWN )	count_DOWN++;
			if( bitbutton & nn::hid::BUTTON_R )		count_R++;
			if( bitbutton & nn::hid::BUTTON_L )		count_L++;
			if( bitbutton & nn::hid::BUTTON_X )		count_X++;
			if( bitbutton & nn::hid::BUTTON_Y )		count_Y++;
        }
        m_PutTriggerCountWindow->Gotoxy(0, pos_y);        
        m_PutTriggerCountWindow->Printf("%s", (*itr)->GetPaneName());
        m_PutTriggerCountWindow->Gotoxy(8, pos_y++);        
		if( itr == m_ButtonArray.begin() )			m_PutTriggerCountWindow->Printf(":%d",count_A );
		else if( itr == m_ButtonArray.begin()+1 )	m_PutTriggerCountWindow->Printf(":%d",count_B );
		else if( itr == m_ButtonArray.begin()+2 )	m_PutTriggerCountWindow->Printf(":%d",count_X );
		else if( itr == m_ButtonArray.begin()+3 )	m_PutTriggerCountWindow->Printf(":%d",count_Y );
		else if( itr == m_ButtonArray.begin()+4 )	m_PutTriggerCountWindow->Printf(":%d",count_START );
		else if( itr == m_ButtonArray.begin()+5 )	m_PutTriggerCountWindow->Printf(":%d",count_RIGHT );
		else if( itr == m_ButtonArray.begin()+6 )	m_PutTriggerCountWindow->Printf(":%d",count_LEFT );
		else if( itr == m_ButtonArray.begin()+7 )	m_PutTriggerCountWindow->Printf(":%d",count_UP );
		else if( itr == m_ButtonArray.begin()+8 )	m_PutTriggerCountWindow->Printf(":%d",count_DOWN );
		else if( itr == m_ButtonArray.begin()+9 )	m_PutTriggerCountWindow->Printf(":%d",count_R );
		else if( itr == m_ButtonArray.begin()+10 )	m_PutTriggerCountWindow->Printf(":%d",count_L );
		else if( itr == m_ButtonArray.begin()+11 )	m_PutTriggerCountWindow->Printf(":%d",count_SELECT );
		else										m_PutTriggerCountWindow->Printf(":error" );
    }
	m_PutTriggerCountWindow->Gotoxy(0, pos_y++);
    m_PutTriggerCountWindow->Printf("Btn_HOME");
    m_PutTriggerCountWindow->Gotoxy(8, pos_y);        
	m_PutTriggerCountWindow->Printf(":%d",count_HOME );
}

/*---------------------------------------------------------------------------
  Desc: 指定ボタンの入力チェック
        
  Args: bitButtonHold - 現在押されているボタン
        bitTestButton - テストするボタン

  Rtns: 入力チェックシーケンス番号
---------------------------------------------------------------------------*/  
INPUT_CHECK_SEQUENCE TestKey::InputCheck(bit32 bitButtonHold, bit32 bitTestButton)
{        
    // 禁則入力
    if (((bitButtonHold & PROHIBITED_UD_MASK) == PROHIBITED_UD_MASK) || 
        ((bitButtonHold & PROHIBITED_LR_MASK) == PROHIBITED_LR_MASK))
    {
        return m_InputCheckSequence = CHK_SEQ_PROHIBITED;
    }    
    
    switch (m_InputCheckSequence)
    {        
    case CHK_SEQ_PUSH:
        // テストボタンのみ押されていることを確認
        if (bitButtonHold == bitTestButton)
        {
            m_InputCheckSequence = CHK_SEQ_RELEASE;
        }
        break;

    case CHK_SEQ_RELEASE:
        // 全てのボタンが離されていることを確認
        if (!bitButtonHold)
        {
            m_InputCheckSequence = CHK_SEQ_END;            
        }
        break;        
        
    default:
        break;        
    }
    
    return m_InputCheckSequence;
}

/*---------------------------------------------------------------------------
  Desc: 指定ボタンの入力チェック
        
  Args: bitButtonHold - 現在押されているボタン
        bitTestButton - テストするボタン

  Rtns: 入力チェックシーケンス番号
---------------------------------------------------------------------------*/  
int TestKey::InputCheck_Button(bit32 bitButtonHold, bit32 bitTestButton)
{
    int ret = CHK_INPUT_ERROR;
    
    // 禁則入力
    if (((bitButtonHold & PROHIBITED_UD_MASK) == PROHIBITED_UD_MASK) || 
        ((bitButtonHold & PROHIBITED_LR_MASK) == PROHIBITED_LR_MASK))
    {
        m_InputCheckSequence = CHK_SEQ_PROHIBITED;
        return ret;
    }    
    
    switch (m_InputCheckSequence)
    {        
    case CHK_SEQ_PUSH:
        // テストボタンのみ押されていることを確認
        if (bitButtonHold == bitTestButton)
        {
            m_InputCheckSequence = CHK_SEQ_RELEASE;
            ret = CHK_INPUT_PUSH_BUTTON;
        }
        // テストボタン以外が押されている場合
        else if(bitButtonHold & (~bitTestButton))
        {
            ret = CHK_INPUT_ERROR;
        }
        else
        {
            ret = CHK_INPUT_INCOMPLETE;
        }
        break;

    case CHK_SEQ_RELEASE:
        // 全てのボタンが離されていることを確認
        if (!bitButtonHold)
        {
            m_InputCheckSequence = CHK_SEQ_END;
            ret = CHK_INPUT_RELEASE;
        }
        else
        {
            ret = CHK_INPUT_INCOMPLETE;
        }
        
        break;        
        
    default:
        break;        
    }
    
    return ret;
}

bool TestKey::InputCheck_NEC(u32 button, INPUT_CHECK_SEQUENCE seq)
{
    int ret;
    bool result = false;
    
    m_InputCheckSequence = seq;
    
    sys::Pad().UpdatePad();

    // 入力チェック
    while(1)
    {
        ret = InputCheck_Button(nn::hid::GetRawButtons(), button);
        
        if(ret == CHK_INPUT_PUSH_BUTTON && seq == CHK_SEQ_PUSH ||
           ret == CHK_INPUT_RELEASE && seq == CHK_SEQ_RELEASE)
        {
            result = true;
            break;
        }
        else if(ret == CHK_SEQ_PROHIBITED || ret == CHK_INPUT_ERROR)
        {
            break;
        }
    }
    
    return result;
}

/*---------------------------------------------------------------------------
  Desc: キーテスト実行

  Args: なし 

  Rtns: なし
---------------------------------------------------------------------------*/  
bool TestKey::ExecuteTest(void)
{  
    InitializeCommon();
    uji::sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();
    
    // ピクチャペインが登録されているグループのポインタを取得します。
    nw::lyt::Group *const pGroup_Key = m_pLayout->GetGroupContainer()->FindGroupByName("Group_Key");
            
    // キーテストの順番と組合せを作成
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_UP                                );
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_LEFT   | nn::hid::CTR::BUTTON_UP  ); 
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_LEFT                              );
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_LEFT   | nn::hid::CTR::BUTTON_DOWN);    
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_DOWN                              );    
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_RIGHT  | nn::hid::CTR::BUTTON_DOWN);    
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_RIGHT                             );
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_RIGHT  | nn::hid::CTR::BUTTON_UP  );
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::PSEUDO_BUTTON_SELECT);        
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_START);    
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_X);
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_Y);
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_B);
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_A);
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_R);
    m_TestButtonOrderArray.PushBack(nn::hid::CTR::BUTTON_L);

    m_pCurrentTestButton = m_TestButtonOrderArray.begin();

    nw::lyt::Drawer drawer;
    drawer.Initialize(m_GraphicsResource);
    nngxUpdateState(NN_GX_STATE_ALL);

    bool bResult = true;

    while (bResult == true)
    {
        sys::Pad().UpdatePad();

        if ((m_focusAnimFrame + 1) > 30.f)
        {
            m_focusAnimFrame = 0.f;
        }
        m_focusAnimFrame++;

        // 入力チェック
        INPUT_CHECK_SEQUENCE seq = InputCheck(nn::hid::GetRawButtons(), *m_pCurrentTestButton);   
        if (seq == CHK_SEQ_END)
        {
            m_InputCheckSequence = CHK_SEQ_PUSH;
            SetAnimationFrame(m_pLayout->GetRootPane(), m_AnimRes, 0);        

            m_pCurrentTestButton++;

            if (m_pCurrentTestButton == m_TestButtonOrderArray.end())
            {
                // 正常終了
                break;
            }
        }
        else if (seq == CHK_SEQ_PROHIBITED)
        {
            bResult = false;
            
            NN_LOG("GetRawButtons= %X\n", nn::hid::GetRawButtons());   
            // アニメーション情報を先行更新        
            SetKeyInputAnimation(nn::hid::GetRawButtons());    
            UpdatePane(pGroup_Key, m_AnimRes);
            m_pLayout->Animate();            
        }

        //ウインドウ処理
        m_WindowManager.Update();
        m_WindowManager.UpdatePad(sys::Pad());        

        /* ------------------------------------------------------------------------
                上画面描画
        ------------------------------------------------------------------------ */
        gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);  
        gfx->m_DrawFramework->Clear();
        /*
            レイアウトライブラリ描画
        */
        gfx->SetScreenSize(gfx->DISPLAY0_WIDTH, gfx->DISPLAY0_HEIGHT);
        {
            InitDraw(gfx->DISPLAY0_WIDTH, gfx->DISPLAY0_HEIGHT);
            
            SetupCamera(m_DrawInfo, *m_pLayout);

            m_pLayout->CalculateMtx(m_DrawInfo);

            // NW_LYT_DRAWER_VALIDATE(NN_GX_STATE_ALL);
            drawer.DrawBegin(m_DrawInfo);
            drawer.Draw(m_pLayout, m_DrawInfo);
            drawer.DrawEnd(m_DrawInfo);
            // nngxUpdateState(NN_GX_STATE_ALL);

            SetKeyInputAnimation(nn::hid::GetRawButtons());    
            UpdatePane(pGroup_Key, m_AnimRes);
            m_pLayout->Animate();
        }
        
        m_WindowManager.DrawDisplay0();
        gfx->m_DrawFramework->SwapBuffers();
        
        /* ------------------------------------------------------------------------
                下画面描画
        ------------------------------------------------------------------------ */
        gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
        gfx->m_DrawFramework->Clear();
        PutInputCount(sys::Pad().GetTrigger());        
        m_WindowManager.DrawDisplay1();
        gfx->m_DrawFramework->SwapBuffers();
        
        gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
    }

    FinalizeCommon();
    
    return bResult;
}

/*---------------------------------------------------------------------------
  Desc: 入力チェックの実行

  Args: なし 

  Rtns: なし
---------------------------------------------------------------------------*/  
void TestKey::ExecuteInputCheck(void)
{
    InitializeCommon();
    uji::sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();

    // ピクチャペインが登録されているグループのポインタを取得します。
    nw::lyt::Group *const pGroup_Key = m_pLayout->GetGroupContainer()->FindGroupByName("Group_Key");

    nw::lyt::Drawer drawer;
    drawer.Initialize(m_GraphicsResource);
    nngxUpdateState(NN_GX_STATE_ALL);

    do
    {
        sys::Pad().UpdatePad();
        
        //ウインドウ処理
        m_WindowManager.Update();
        m_WindowManager.UpdatePad(sys::Pad());        

        /* ------------------------------------------------------------------------
                上画面描画
        ------------------------------------------------------------------------ */
        gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);  
        gfx->m_DrawFramework->Clear();
        /*
            レイアウトライブラリ描画
        */        
        gfx->SetScreenSize(gfx->DISPLAY0_WIDTH, gfx->DISPLAY0_HEIGHT);
        {
            InitDraw(gfx->DISPLAY0_WIDTH, gfx->DISPLAY0_HEIGHT);
            
            SetupCamera(m_DrawInfo, *m_pLayout);

            m_pLayout->CalculateMtx(m_DrawInfo);

            drawer.DrawBegin(m_DrawInfo);
            drawer.Draw(m_pLayout, m_DrawInfo);
            drawer.DrawEnd(m_DrawInfo);

            SetKeyInputAnimation(nn::hid::GetRawButtons());    
            UpdatePane(pGroup_Key, m_AnimRes);
            m_pLayout->Animate();
        }
        
        PutExitMessage(gfx);
        m_WindowManager.DrawDisplay0();
        gfx->m_DrawFramework->SwapBuffers();
        
        /* ------------------------------------------------------------------------
                下画面描画
        ------------------------------------------------------------------------ */
        gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
        gfx->m_DrawFramework->Clear();
		// スタート・セレクトカウント用フラグの初期処理
		if( nn::hid::GetRawButtons() & nn::hid::PSEUDO_BUTTON_SELECT )	{}
		else	trigger_SEL = true;
		if( nn::hid::GetRawButtons() & nn::hid::BUTTON_START )	{}
		else	trigger_START = true;
        if ( nn::applet::IsExpectedToProcessHomeButton() )	PutInputCount_HOME();
		PutInputCount_SEL(nn::hid::GetRawButtons());
        PutInputCount_Sub(sys::Pad().GetTrigger());
        m_WindowManager.DrawDisplay1();
        gfx->m_DrawFramework->SwapBuffers();
        
        gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);    
    } while (!(sys::Pad().IsButtonPress(Pad::BUTTON_START) 
            && sys::Pad().IsButtonPress(Pad::BUTTON_L)                
            && sys::Pad().IsButtonPress(Pad::BUTTON_R)));

    sys::Pad().WaitReleaseKey();

    FinalizeCommon();
}

static void Eva_KeyTest()
{        
    seq::TestResult result;
    TestDigitalKey(result);
    
    sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();
    
    sys::Menu::m_SubWindow->Printf("\f");
    sys::Menu::m_SubWindow->Printf("<KEY INPUT TEST>\n");    
    sys::Menu::m_SubWindow->Printf("Result(%s)\n%s\n", result.m_Result ? "PASS" : "FAIL", result.m_String);
    sys::Menu::m_SubWindow->Printf("\n - Push B Button To Exit -");            
    
    // 画面更新
	sys::Menu::m_WindowManager.Update();
	sys::Menu::m_WindowManager.UpdatePad(sys::Pad());
	gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
	gfx->m_DrawFramework->Clear();        
	sys::Menu::m_WindowManager.DrawDisplay1();
	gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);

    do
    {
        sys::Pad().UpdatePad();
        nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(10));        
    }
    while (!sys::Pad().IsButtonDown(sys::Pad::BUTTON_B));   
    
}

static void Eva_InputCheck()
{
    TestKey* p = new TestKey;
    
    p->ExecuteInputCheck();
    
    delete p;
}

static void Eva_RemoteKeyTest()
{
    u8 remote;
    
    uji::sys::Pad pad;
    
    GraphicsDrawing* gfx = GraphicsDrawing::GetInstance();
    gfx->BeginDrawingShape();
    
    Menu::m_SubWindow->Gotoxy(0,0);
    Menu::m_SubWindow->SetTextColor(ATTR_COLOR_WHITE);
    
    do
    {
	    // パッド更新
	    pad.UpdatePadWithRemote();
	    remote = ( pad.IsButtonDown(sys::Pad::BUTTON_HEADPHONE) ) ? 0x01 : 0x00;
	    nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(10));
	    
	    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
	    gfx->m_DrawFramework->Clear();   
	    Menu::m_SubWindow->Printf("remote=%d\n", remote);     
		Menu::m_WindowManager.Update();
	    Menu::m_WindowManager.UpdatePad(pad);        
		Menu::m_WindowManager.DrawDisplay1();
	    gfx->m_DrawFramework->SwapBuffers();
	    
	    if( pad.IsButtonDown(Pad::BUTTON_B) )
	    {
	        break;
	    }
    }
    while(remote==0);
    
	nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(1000));
}

void OpenKeyMenu(void)
{
    ClassMenu *m = new ClassMenu( "Key Evaluation Menu" );

    m->Add( " Key Test   ", Eva_KeyTest );
    m->Add( " Input Check", Eva_InputCheck );
    m->Add( " Wifi Button Test", mcu::EvaWifiButtonTest);    
    m->Add( " HOME Button Test", mcu::EvaHomeButtonTest);        
    m->Add( " Remote Key Test", Eva_RemoteKeyTest );
    m->Open();

    delete m;
}

void KeyMenuForService(void)
{
    ClassMenu *m = new ClassMenu( "Key Evaluation Menu" );

    m->Add( " Input Check", Eva_InputCheck );
    m->Add( " Wifi Button Test", mcu::EvaWifiButtonTest);    
    m->Add( " HOME Button Test", mcu::EvaHomeButtonTest);        
    m->Open();

    delete m;
}

/*
  Desc: キー入力テスト
*/
bool TestDigitalKey(seq::TestResult &result)
{    
    TestKey* p = new TestKey;
    
    result.m_Result = p->ExecuteTest();
    
    delete p;
      
    return result.m_Result;
}

bool TestDigitalKey_NEC(seq::TestResult &result)
{    
    result.m_Result = true;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Up(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_UP, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Left_Up(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_LEFT | nn::hid::CTR::BUTTON_UP, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Left(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_LEFT, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Left_Down(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::BUTTON_LEFT | nn::hid::CTR::BUTTON_DOWN, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Down(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_DOWN, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Down_Right(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_RIGHT | nn::hid::CTR::BUTTON_DOWN, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Right(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_RIGHT, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Right_Up(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_RIGHT | nn::hid::CTR::BUTTON_UP, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Select(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::PSEUDO_BUTTON_SELECT, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Start(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_START, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_X(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_X, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_Y(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_Y, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_B(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_B, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_A(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_A, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_R(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_R, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Push_L(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_L, CHK_SEQ_PUSH );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Up(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_UP, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Left_up(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_LEFT | nn::hid::CTR::BUTTON_UP, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Left(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_LEFT, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Left_Down(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::BUTTON_LEFT | nn::hid::CTR::BUTTON_DOWN, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Down(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_DOWN, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Down_Right(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_RIGHT | nn::hid::CTR::BUTTON_DOWN, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Right(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_RIGHT, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Right_Up(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_RIGHT | nn::hid::CTR::BUTTON_UP, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Select(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::PSEUDO_BUTTON_SELECT, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Start(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_START, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_X(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_X, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_Y(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_Y, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_B(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_B, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_A(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_A, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_R(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_R, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

bool TestDigitalKey_Release_L(seq::TestResult &result)
{
    TestKey* p = new TestKey;
    
    result.m_Result = p->InputCheck_NEC( nn::hid::CTR::BUTTON_L, CHK_SEQ_RELEASE );
    delete p;
    
    return result.m_Result;
}

namespace key {
/*---------------------------------------------------------------------------
  Desc: リモートキー検査
---------------------------------------------------------------------------*/
bool TestRemoteKey(uji::seq::TestResult &result)
{
    u8 remote;
    
    uji::sys::Pad pad;
    nn::hid::CTR::PadReader padReader;
    
    // パッド更新
    pad.UpdatePadWithRemote();
    
    remote = ( pad.IsButtonDown(sys::Pad::BUTTON_HEADPHONE) ) ? 0x01 : 0x00;
    
    seq::g_CommC2T->WriteDpramBinary(&remote, sizeof(&remote));
    
    result.m_Result = true;
    return result.m_Result;
}
}	// namespace key



}   // namespace
}
