/*---------------------------------------------------------------------------*
  Project:  CTR_UJI
  File:     TestImport.cpp

  Copyright (C)2010 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

 *---------------------------------------------------------------------------*/
#include "TestImport.h"
#include "../seq/TestListManager.h"
#include "../seq/TestSequencer.h"
#include <nn/gx/CTR/gx_LcdBacklight.h>
#include <nn/am/am_Result.h>
#include <nn/nwm.h>
#include <algorithm>
#include <list>

#ifdef EVA_COMPOSITE
extern       char VERSION_STRING[];
#else
static const char VERSION_STRING[]= UJI_APPVER;
#endif
static const char VERSION_DATE[]  =	__DATE__ " " __TIME__ ;


using namespace uji;
using namespace uji::sys;

namespace uji {
namespace eva {
namespace import {

TestImport* TestImport::m_pInstance = 0;

#define DEBUG_MODE 0           // fobOɂ 1 

// LɂƃC|[gXbhƖ{Xbhœi݁XV݁XV..j
// ĉʂXV܂B{Xbh𖈃t[XVꍇ͒`𖳌ɁB
#define BLOCK_MAIN_THREAD

/*---------------------------------------------------------------------------
  Desc: fobOp
---------------------------------------------------------------------------*/
static void WaitInputA()
{
    // 
    // ʊmFp
    uji::sys::Pad pad;
    do {
        pad.UpdatePad();
        nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(16));
    } while(!pad.IsButtonDown(Pad::BUTTON_A));
    
}
static void DEBUG_NN_LOG( const char* format, ... )
{
    if(DEBUG_MODE==1)
    {
        va_list list;
        va_start(list, format);
        
        NN_LOG(format, list);
        
        va_end(list);
    }
}

/*---------------------------------------------------------------------------
  Desc: nn::am::ProgramInfo  id ŕёւ邽߂̔r֐
---------------------------------------------------------------------------*/
bool compare_id( const nn::am::ProgramInfo& left, const nn::am::ProgramInfo& right )
{
    return left.id < right.id;
}

/*---------------------------------------------------------------------------
  Desc: nn::ProgramId  id ŕёւ邽߂̔r֐
---------------------------------------------------------------------------*/
bool compare_id2( const nn::ProgramId& left, const nn::ProgramId& right )
{
    return left < right;
}

//===========================================================================
//===========================================================================
//                              C֐
//===========================================================================
//===========================================================================
/*---------------------------------------------------------------------------
    Desc: t@CC|[gXbh̎
    
    Args: importeeFile  - C|[gt@C̏
          pQueue        - pubLOL[ւ̃|C^
    
    Rtns: Ȃ
---------------------------------------------------------------------------*/
void TestImport::ImportThread(ImporteeFileInfo* importeeFile, nn::os::BlockingQueue* pQueue)
{
    ImportProgress progress;
    nn::os::Tick::Tick tick = nn::os::Tick::GetSystemCurrent();
    
    DEBUG_NN_LOG("  importing: %ls... ", importeeFile->filePath);
    
    // C|[gppCv擾
    nn::fs::FileOutputStream* stream;
    progress.retBegin = nn::am::BeginImportProgram(&stream, importeeFile->mediaType);
    pQueue->Enqueue(reinterpret_cast<uptr>(&progress));
    if( progress.retBegin.IsFailure() ) return;
    
    progress.readTotal =  0;
    progress.read      = -1;
    nn::fs::FileInputStream in(importeeFile->filePath);
    
    // t@CTCY擾
    progress.fileSize = in.GetSize();
    
    // 擾t@CTCY𑗐M
    pQueue->Enqueue(reinterpret_cast<uptr>(&progress));
    
    // C|[g
    s32    writeSize = 0;
    size_t readBufSize = sizeof(bit8) * 256 * 1024;
    bit8*  readBuf = reinterpret_cast<bit8*>(sys::Alloc(readBufSize, 32));
    while(s32 read = in.Read(readBuf, readBufSize))
    {
        progress.retWrite = stream->TryWrite(&writeSize, readBuf, progress.read = read);
        progress.readTotal += read;
        
        // C|[g󋵂𑗐M
        pQueue->Enqueue(reinterpret_cast<uptr>(&progress));
        if( progress.retWrite.IsFailure() ) return;
    }
    sys::Free(readBuf);
    
    // C|[gIiAŏI͎Ԃ|̂łł͍sȂj
    progress.retEnd = nn::am::EndImportProgramWithoutCommit(stream);
    pQueue->Enqueue(reinterpret_cast<uptr>(&progress));
    if( progress.retEnd.IsFailure() ) return;
    
    DEBUG_NN_LOG("done.\n");
    
    // oߎԂ𑗐M
    progress.msec = (nn::os::Tick::GetSystemCurrent()-tick).ToTimeSpan().GetMilliSeconds();
    pQueue->Enqueue(reinterpret_cast<uptr>(&progress));
}

/*---------------------------------------------------------------------------
  Desc: C|[gXbh̃bv֐
---------------------------------------------------------------------------*/
void TestImport::s_ImportThread(void* param)
{
    ImportThreadArgs* pImportThreadArgs = reinterpret_cast<ImportThreadArgs*>(param);
    TestImport* pTi = pImportThreadArgs->ti;
    pTi->ImportThread(&(pImportThreadArgs->importFile), pImportThreadArgs->queue);
}

/*---------------------------------------------------------------------------
  Desc: vOXo[̐Ɣj
---------------------------------------------------------------------------*/
void TestImport::CreateProgress( ImportInfo* info, int x, int y, int width, int height )
{
    info->pb = new uji::eva::ProgressBar( x, y, width, height, 0, info->fileSize );
}
void TestImport::DestroyProgress( ImportInfo* info )
{
    delete info->pb;
}

/*---------------------------------------------------------------------------
  Desc: OJEgtPrintf
---------------------------------------------------------------------------*/
void TestImport::LogPrintf(sys::TextWindow *wnd, sys::ATTR_TEXT_COLOR color, const char* format, ...)
{    
    sys::ATTR_TEXT_COLOR prev_color = wnd->GetTextColor();
    
    wnd->SetTextColor(color);
    va_list vlist;
    va_start(vlist, format);
    wnd->VPrintf(format, vlist);    
    va_end(vlist); 
    
    wnd->SetTextColor(prev_color);
}

/*---------------------------------------------------------------------------
  Desc: ʕ\̍XV
---------------------------------------------------------------------------*/
void TestImport::UpdateLCD(Current_Import_Mode mode)
{
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);
    gfx->m_DrawFramework->Clear();
    gfx->SetScreenSize(gfx->DISPLAY0_WIDTH, gfx->DISPLAY0_HEIGHT);
    
    // ʂɕ\eLq
    UpdateSubject_Main( mode );
    
    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
    
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();
    gfx->SetScreenSize(gfx->DISPLAY1_WIDTH, gfx->DISPLAY1_HEIGHT);
    
    // ʂɕ\eLq
    UpdateSubject_Sub( mode );
    
    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
    
}

/*---------------------------------------------------------------------------
  Desc: ʕ\̂ݍXV
---------------------------------------------------------------------------*/
void TestImport::UpdateLCD_Main(Current_Import_Mode mode)
{
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);
    gfx->m_DrawFramework->Clear();
    gfx->SetScreenSize(gfx->DISPLAY0_WIDTH, gfx->DISPLAY0_HEIGHT);
    
    // ʂɕ\eLq
    UpdateSubject_Main( mode );
    
    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
    
}

/*---------------------------------------------------------------------------
  Desc: ʕ\̂ݍXV
---------------------------------------------------------------------------*/
void TestImport::UpdateLCD_Sub(Current_Import_Mode mode)
{
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();
    gfx->SetScreenSize(gfx->DISPLAY1_WIDTH, gfx->DISPLAY1_HEIGHT);
    
    // ʂɕ\eLq
    UpdateSubject_Sub( mode );
    
    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
    
}

/*---------------------------------------------------------------------------
  Desc: Ct̍XV
---------------------------------------------------------------------------*/
void TestImport::UpdateSubject_Main(Current_Import_Mode mode)
{
    gfx->BeginDrawingString();
    gfx->SetFixedWidthFont(FONT_SIZE_L);
    
    // drcʂ̎擾
    GetCharTypeOfRemainingBatteryLv();
    
    if( mode==CHECK_WIRELESS )
    {
        gfx->SetFixedWidthFont(20);
        gfx->m_TextWriter.SetTextColor(nw::ut::Color8::CYAN);
        gfx->m_TextWriter.SetCursor(40, 110);
        gfx->m_TextWriter.Printf("Please turn on wireless switch!\n");
    }
    else if( mode==START_MENU )
    {
        gfx->m_TextWriter.SetTextColor(nw::ut::Color8::WHITE);
        gfx->m_TextWriter.SetCursor(20, 0);
        gfx->m_TextWriter.Printf("<< Data Import >>\n");
        gfx->m_TextWriter.Printf("                             [Battery %s(%%)]\n", m_RemainingBatteryLv);
        gfx->m_TextWriter.Printf("     ========================================\n");
        gfx->m_TextWriter.Printf("      - Program Version : Ver.%s\n", VERSION_STRING);
        gfx->m_TextWriter.Printf("      - Region          : %s\n",     UJI_REGION);
        gfx->m_TextWriter.Printf("      - Bonding Option  : %02d\n",   m_BondingOption);
        gfx->m_TextWriter.Printf("      - Wap Ssid        : %s\n",     m_ssid);
        gfx->m_TextWriter.Printf("      - RSP Version     : %s\n",     m_SystemPackageVersion.c_str());
        gfx->m_TextWriter.Printf("      - PreInstall Ver. : %s\n",     m_PreInstallPackageVersion.c_str());
        gfx->m_TextWriter.Printf("      - Mark            : %s\n",     m_Mark.c_str());
        gfx->m_TextWriter.Printf("     ========================================\n");
        gfx->m_TextWriter.Printf("\n");
        gfx->m_TextWriter.Printf("\n");
        gfx->m_TextWriter.SetTextColor(nw::ut::Color8::CYAN);
        gfx->m_TextWriter.Printf(" Please push (A) button when you confirm the message.\n");
    }
    else if( mode==IMPORT_CIA )
    {
        gfx->m_TextWriter.SetTextColor(nw::ut::Color8::WHITE);
        
        gfx->m_TextWriter.SetCursor(20, 0);
        gfx->m_TextWriter.Printf("<< Data Import >>\n");
        gfx->m_TextWriter.Printf("                             [Battery %s(%%)]\n", m_RemainingBatteryLv);
        gfx->m_TextWriter.Printf("     ========================================\n");
        gfx->m_TextWriter.Printf("      - Program Version : Ver.%s\n", VERSION_STRING);
        gfx->m_TextWriter.Printf("      - Region          : %s\n",     UJI_REGION);
        gfx->m_TextWriter.Printf("      - Bonding Option  : %02d\n",   m_BondingOption);
        gfx->m_TextWriter.Printf("      - Wap Ssid        : %s\n",     m_ssid);
        gfx->m_TextWriter.Printf("      - RSP Version     : %s\n",     m_SystemPackageVersion.c_str());
        gfx->m_TextWriter.Printf("      - PreInstall Ver. : %s\n",     m_PreInstallPackageVersion.c_str());
        gfx->m_TextWriter.Printf("      - Mark            : %s\n",     m_Mark.c_str());
        gfx->m_TextWriter.Printf("     ========================================\n");
        gfx->m_TextWriter.Printf("\n");
        gfx->m_TextWriter.Printf("\n");
        gfx->m_TextWriter.SetCursor(20, 153);
        gfx->m_TextWriter.Printf("Current: Importing %3d(%%) %10d/%10lld(byte)", // xxxxMB܂ŕ\\
                                 m_ImportInfoCurrent.percentage, 
                                 m_ImportInfoCurrent.importSize, 
                                 m_ImportInfoCurrent.fileSize);
        
        gfx->m_TextWriter.SetCursor(20, 193);
        gfx->m_TextWriter.Printf("Total  : Importing %3d(%%) %10d/%10lld(byte)", // xxxxMB܂ŕ\\
                                 m_ImportInfoTotal.percentage, 
                                 m_ImportInfoTotal.importSize, 
                                 m_ImportInfoTotal.fileSize);
        
        // vOXo[XV
        m_ImportInfoCurrent.pb->SetStep(m_ImportInfoCurrent.importSize);
        m_ImportInfoTotal.pb->SetStep(m_ImportInfoTotal.importSize);
        
        m_ImportInfoCurrent.pb->Draw(gfx);
        m_ImportInfoTotal.pb->Draw(gfx);
    }
    gfx->EndDrawingString();
}

/*---------------------------------------------------------------------------
  Desc: Tut̍XV
---------------------------------------------------------------------------*/
void TestImport::UpdateSubject_Sub(Current_Import_Mode mode)
{
    if( mode==CHECK_WIRELESS || mode==START_MENU || mode==CHECK_REMAINING_BATTERY_LV || mode==EXTRACT_AC_ADAPTER)
    {
        m_JpegDrawer->DrawPicture(NN_GX_DISPLAY1);
    }
    else
    {
        m_WindowManager.Update();
        m_WindowManager.DrawDisplay1();
    }
}

/*---------------------------------------------------------------------------
  Desc: wiftHg
---------------------------------------------------------------------------*/
void TestImport::SetDefaultBackGroundColor()
{
    gfx->m_DrawFramework->SetClearColor(NN_GX_DISPLAY0,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_RED,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_GREEN,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_BLUE,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_ALPHA);
    gfx->m_DrawFramework->SetClearColor(NN_GX_DISPLAY1,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_RED,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_GREEN,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_BLUE,
        GraphicsDrawing::DEFAULT_CLEAR_COLOR_ALPHA);
}

/*---------------------------------------------------------------------------
  Desc: wi
---------------------------------------------------------------------------*/
void TestImport::SetBlackBackGroundColor()
{
    gfx->m_DrawFramework->SetClearColor( NN_GX_DISPLAY0, 0.0f, 0.0f, 0.0f, 1.0f );
    gfx->m_DrawFramework->SetClearColor( NN_GX_DISPLAY1, 0.0f, 0.0f, 0.0f, 1.0f );
}

/*---------------------------------------------------------------------------
  Desc: SD J[hɃeLXgo
---------------------------------------------------------------------------*/
void TestImport::CreateTextToSdmcByArray( char* buf, const wchar_t* pathName )
{
    nn::fs::FileOutputStream ofile;
    nn::fs::TryDeleteFile( pathName );
    
    ofile.Initialize( pathName, true );
    ofile.Seek( 0, nn::fs::POSITION_BASE_BEGIN );
    ofile.Write( buf, strlen(buf) );
    ofile.Finalize();
}

/*---------------------------------------------------------------------------
  Desc: SD J[hɃeLXgo
---------------------------------------------------------------------------*/
void TestImport::CreateTitleListTextToSdmcByDeque( deque<nn::am::ProgramInfo> deq, const wchar_t* pathName )
{
    nn::fs::FileOutputStream ofile;
    nn::fs::TryDeleteFile( pathName );
    
    char* output1 = new char[50];
    deque<nn::am::ProgramInfo>::iterator ite = deq.begin();
    
    while( ite != deq.end() )
    {
        sprintf( output1, "id: %016llx  version: %5d\n", (*ite).id, (*ite).version );
        
        ofile.Initialize( pathName, true );
        ofile.Seek( ofile.GetSize(), nn::fs::POSITION_BASE_BEGIN );
        ofile.Write( output1, strlen(output1) );
        ofile.Finalize();
        
        ++ite;
    }
    
    delete[] output1;
}

/*---------------------------------------------------------------------------
  Desc: obe[cʂ𕶎Ŏ擾
---------------------------------------------------------------------------*/
bool TestImport::GetCharTypeOfRemainingBatteryLv()
{
    nn::Result nnr;
    u8 btLevel = 0;
    
    nnr = uji::eva::mcu::GetRemainingBatteryLv(&btLevel);
    if( nnr.IsFailure() ) return false;
    
    memset( m_RemainingBatteryLv, '\0', 4*sizeof(char) );
    sprintf( m_RemainingBatteryLv, "%3d", btLevel );
    return true;
}

/*---------------------------------------------------------------------------
  Desc: NG ̃G[bZ[Wꗗ
---------------------------------------------------------------------------*/
char* TestImport::OutputErrorString( ErrSeq_DataImport errCode, nn::Result nnResult )
{
    switch( errCode )
    {
    case PASS                                    : return "PASS";
    case ERR1_SDCARD_IS_NOT_WRITABLE             : return "SD Card isn't writable.";
    case ERR1_CHECK_CAPACITY                     : return m_SdSizeString;
    case ERR1_DB_IS_NOT_AVAILABLE                : return "Database isn't available.";
    case ERR1_GET_SDMC_SIZE                      : return "nn::fs::GetSdmcSize(): ResultMediaNotFound";
    case ERR1_INITIALIZE_DB_1                    : return "nn::am::InitializeTitleDatabase(): ResultInvalidEnumValue";
    case ERR1_INITIALIZE_DB_2                    : return "nn::am::InitializeTitleDatabase(): ResultMediaNotSupported";
    case ERR1_INITIALIZE_DB_3                    : return "nn::am::InitializeTitleDatabase(): ResultInvalidImportState";
    case ERR1_QUERY_AVAILABLE_DB_1               : return "nn::am::QueryAvailableTitleDatabase(): ResultInvalidEnumValue";
    case ERR1_QUERY_AVAILABLE_DB_2               : return "nn::am::QueryAvailableTitleDatabase(): ResultMediaNotSupported";
    case ERR1_QUERY_AVAILABLE_DB_3               : return "nn::am::QueryAvailableTitleDatabase(): ResultInvalidImportState";
    case ERR1_MOUNT_SDMC_1                       : return "nn::fs::MountSdmc(): ResultMediaNotFound";
    case ERR1_MOUNT_SDMC_2                       : return "nn::fs::MountSdmc(): ResultAlreadyExists";
    case ERR1_MOUNT_SDMC_3                       : return "nn::fs::MountSdmc(): ResultArchiveInvalidated";
    case ERR1_MOUNT_SDMC_4                       : return "nn::fs::MountSdmc(): ResultMediaAccessError";
    case ERR1_MOUNT_SDMC_5                       : return "nn::fs::MountSdmc(): ResultBadFormat";
    case ERR1_GET_REGION                         : return "Fail to Get Region.";
    case ERR1_UNKNOWN_REGION                     : return "Unknown Region.";
    case ERR1_SET_CONFIG                         : return "Fail to Set Region.";
    case ERR1_FLUSH_CONFIG                       : return "Fail to Flush Region.";
    case ERR1_GET_CONFIG                         : return "Fail to Get Config.";
    case ERR2_GET_PROGRAM_INFOS_FROM_CIA         : return "nn::am::GetProgramInfoFromCia(): ResultInvalidCiaFormat";
    case ERR2_UNKNOWN_APPLI                      : return "Unknown Application.";
    case ERR2_GET_PROGRAM_INFOS_1                : return "nn::am::GetProgramInfos(): ResultInvalidEnumValue";
    case ERR2_GET_PROGRAM_INFOS_2                : return "nn::am::GetProgramInfos(): ResultMediaNotSupported";
    case ERR2_GET_PROGRAM_INFOS_3                : return "nn::am::GetProgramInfos(): ResultInvalidSize";
    case ERR2_GET_PROGRAM_INFOS_4                : return "nn::am::GetProgramInfos(): ResultInternalDataCorrupted";
    case ERR2_INTERNAL_DELETE_TICKET             : return "nn::am::DeleteTicket()";
    case ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_1    : return "nn::am::BeginImportProgram(): ResultInvalidEnumValue";
    case ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_2    : return "nn::am::BeginImportProgram(): ResultMediaNotSupported";
    case ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_3    : return "nn::am::BeginImportProgram(): ResultInvalidImportState";
    case ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_4    : return "nn::am::BeginImportProgram(): ResultImportPipeAlreadyOpen";
    case ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_5    : return "nn::am::BeginImportProgram(): ResultInternalDataCorrupted";
    case ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_6    : return "nn::am::BeginImportProgram(): ResultAlreadyExists";
    case ERR2_INTERNAL_TRY_WRITE                 : return "Fail to Write.";
    case ERR2_INTERNAL_END_IMPORT_PROGRAM_1      : return "nn::am::EndImportProgramWithoutCommit(): ResultInvalidImportState";
    case ERR2_INTERNAL_END_IMPORT_PROGRAM_2      : return "nn::am::EndImportProgramWithoutCommit(): ResultInvalidPipe";
    case ERR2_INTERNAL_END_IMPORT_PROGRAM_3      : return "nn::am::EndImportProgramWithoutCommit(): ResultVerificationFailed";
    case ERR2_INTERNAL_END_IMPORT_PROGRAM_4      : return "nn::am::EndImportProgramWithoutCommit(): ResultNoEnoughSpace";
    case ERR2_INTERNAL_END_IMPORT_PROGRAM_5      : return "nn::am::EndImportProgramWithoutCommit(): ResultInternalDataCorrupted";
    case ERR2_INTERNAL_END_IMPORT_PROGRAM_6      : return "nn::am::EndImportProgramWithoutCommit(): ResultEsError";
    case ERR2_COMMIT_IMPORT_PROGRAMS             : return "nn::am::CommitImportPrograms()";
    case ERR2_INTENDED_PREINSTALL                : return "Description of the OMakefile is Wrong.";
    case ERR3_MOUNT_EXT_SAVE_DATA_1              : return "nn::fs::MountExtSaveData(): ResultMediaNotFound";
    case ERR3_MOUNT_EXT_SAVE_DATA_2              : return "nn::fs::MountExtSaveData(): DeleteExtSaveData";
    case ERR3_MOUNT_EXT_SAVE_DATA_3              : return "nn::fs::MountExtSaveData(): ResultBadFormat";
    case ERR3_MOUNT_EXT_SAVE_DATA_4              : return "nn::fs::MountExtSaveData(): ResultWriteProtected";
    case ERR3_MOUNT_EXT_SAVE_DATA_5              : return "nn::fs::MountExtSaveData(): ResultMediaAccessError";
    case ERR3_MOUNT_EXT_SAVE_DATA_6              : return "There's no way to exist ExtSaveData.";
    case ERR3_CREATE_EXT_SAVE_DATA_1             : return "nn::fs::CreateExtSaveData(): ResultNotEnoughSpace";
    case ERR3_CREATE_EXT_SAVE_DATA_2             : return "nn::fs::CreateExtSaveData(): ResultNotFormatted";
    case ERR3_CREATE_EXT_SAVE_DATA_3             : return "nn::fs::CreateExtSaveData(): ResultWriteProtected";
    case ERR3_CREATE_EXT_SAVE_DATA_4             : return "nn::fs::CreateExtSaveData(): ResultMediaAccessError";
    case ERR3_REMOUNT_EXT_SAVE_DATA              : return "Fail to Mount ExtSaveData.";
    case ERR3_CREATE_EXT_SAVE_DATA               : return "Fail to Create ExtSaveData.";
    case ERR3_OPEN_EXT_SAVE_DATA                 : return "Fail to Open ExtSaveData.";
    case ERR3_WRITE_EXT_SAVE_DATA                : return "Fail to Write ExtSaveData.";
    case ERR3_UNMOUNT_EXT_SAVE_DATA              : return "Fail to Unmount ExtSaveData.";
    case ERR4_MOUNT_TWLN_ARCHIVE                 : return "Fail to Mount Twln Archive.";
    case ERR4_OPEN_TWL_FONT_TABLE                : return "Fail to Create TWLFontTable.";
    case ERR4_SET_SIZE_TWL_FONT_TABLE            : return "Fail to Open TWLFontTable.";
    case ERR4_WRITE_TWL_FONT_TABLE               : return "Fail to Write TWLFontTable.";
    case ERR4_FLUSH_TWL_FONT_TABLE               : return "Fail to Flush TWLFontTable.";
    case ERR4_COMPARE_HASH                       : return m_HashString;
    case ERR4_UNMOUNT_TWLN_ARCHIVE               : return "Fail to Unmount Twln Archive.";
    case ERR4_DELETE_PROGRAM_1                   : return "nn::am::DeleteProgram(): ResultInvalidEnumValue";
    case ERR4_DELETE_PROGRAM_2                   : return "nn::am::DeleteProgram(): ResultInternalDataCorrupted";
    case ERR4_DELETE_TICKET                      : return "nn::am::DeleteTicket(): ResultInternalDataCorrupted";
    case ERR5_GARBAGE_IS_LEFT                    : return "Garbage is left.";
    case ERR5_EXIST_ITC                          : return m_ImportTitleContextsString;
    case ERR5_NEEDS_CLEANUP_1                    : return "nn::am::NeedsCleanup(): ResultInvalidEnumValue";
    case ERR5_NEEDS_CLEANUP_2                    : return "nn::am::NeedsCleanup(): ResultMediaNotSupported";
    case ERR5_NEEDS_CLEANUP_3                    : return "nn::am::NeedsCleanup(): ResultInternalDataCorrupted";
    case ERR5_GET_NUM_ITC_1                      : return "nn::am::GetNumImportTitleContexts(): ResultInvalidEnumValue";
    case ERR5_GET_NUM_ITC_2                      : return "nn::am::GetNumImportTitleContexts(): ResultMediaNotSupported";
    case ERR5_GET_NUM_ITC_3                      : return "nn::am::GetNumImportTitleContexts(): ResultInternalDataCorrupted";
    case ERR5_GET_NUM_PROGRAMS_1                 : return "nn::am::GetNumPrograms(): ResultInvalidEnumValue";
    case ERR5_GET_NUM_PROGRAMS_2                 : return "nn::am::GetNumPrograms(): ResultMediaNotSupported";
    case ERR5_GET_NUM_PROGRAMS_3                 : return "nn::am::GetNumPrograms(): ResultInternalDataCorrupted";
    case ERR5_GET_PROGRAM_LIST_1                 : return "nn::am::GetProgramList(): ResultInvalidEnumValue";
    case ERR5_GET_PROGRAM_LIST_2                 : return "nn::am::GetProgramList(): ResultMediaNotSupported";
    case ERR5_GET_PROGRAM_LIST_3                 : return "nn::am::GetProgramList(): ResultInvalidSize";
    case ERR5_GET_PROGRAM_LIST_4                 : return "nn::am::GetProgramList(): ResultInvalidAddress";
    case ERR5_GET_PROGRAM_LIST_5                 : return "nn::am::GetProgramList(): ResultInternalDataCorrupted";
    case ERR5_GET_PROGRAM_INFOS_1                : return "nn::am::GetProgramInfos(): ResultInvalidEnumValue";
    case ERR5_GET_PROGRAM_INFOS_2                : return "nn::am::GetProgramInfos(): ResultMediaNotSupported";
    case ERR5_GET_PROGRAM_INFOS_3                : return "nn::am::GetProgramInfos(): ResultInvalidSize";
    case ERR5_GET_PROGRAM_INFOS_4                : return "nn::am::GetProgramInfos(): ResultNotFound";
    case ERR5_GET_PROGRAM_INFOS_5                : return "nn::am::GetProgramInfos(): ResultInternalDataCorrupted";
    case ERR6_COMPARE_PROGRAM_INFOS_NUM          : return m_CompareProgramInfosNumString;
    case ERR6_COMPARE_PROGRAM_INFOS_ID_AND_VER   : return m_CompareProgramInfosIdAndVerString;
    case ERR6_COMPARE_NUM_OF_TICKETS_AND_PROGRAMS: return m_CompareNumOfTicketsAndProgramsString;
    case ERR6_COMPARE_ID_OF_TICKETS_AND_PROGRAMS : return m_CompareIdOfTicketsAndProgramsString;
    case ERR6_GET_NUM_TICKETS_1                  : return "nn::am::GetNumTickets(): ResultInternalDataCorrupted";
    case ERR6_GET_TICKET_LIST_1                  : return "nn::am::GetTicketList(): ResultInvalidSize";
    case ERR6_GET_TICKET_LIST_2                  : return "nn::am::GetTicketList(): ResultOutOfRange";
    case ERR6_GET_TICKET_LIST_3                  : return "nn::am::GetTicketList(): ResultInternalDataCorrupted";
    case ERR7_OPEN_TITLE_LIST                    : return "TitleList is not exist.";
    case ERR7_GET_SIZE_TITLE_LIST                : return "Fail to Get Size.";
    case ERR7_READ_TITLE_LIST                    : return "Fail to Read.";
    case ERR7_COMPARE_TITLE_LIST                 : return "Not match.";
    case ERR7_OPEN_PREINSTALL_TITLE_LIST         : return "PreinstallTitleList is not exist.";
    case ERR7_GET_SIZE_PREINSTALL_TITLE_LIST     : return "Fail to Get Size.";
    case ERR7_READ_PREINSTALL_TITLE_LIST         : return "Fail to Read.";
    case ERR7_COMPARE_PREINSTALL_TITLE_LIST      : return "Not match.";
    case ERR8_OPEN_DEFAULT_TITLE_LIST            : return "DefaultTitleList is not exist.";
    case ERR8_GET_SIZE_DEFAULT_TITLE_LIST        : return "Fail to Get Size.";
    case ERR8_READ_DEFAULT_TITLE_LIST            : return "Fail to Read.";
    case ERR8_COMPARE_DEFAULT_TITLE_LIST         : return "Not match.";
    case ERR9_GET_REMAINING_BATTERY_LV           : return "Fail to Get Remaining Battery Level.";
    case ERR9_TIME_OUT                           : return "Time out.";
    case ERR_FATAL                               : sprintf( m_FatalErrorString, "Level:       %d\n"
                                                                                "Summary:     %d\n"
                                                                                "Module:      %d\n"
                                                                                "Description: %d\n", 
                                                                                nnResult.GetLevel(), 
                                                                                nnResult.GetSummary(), 
                                                                                nnResult.GetModule(), 
                                                                                nnResult.GetDescription());
                                                   return m_FatalErrorString;
    case ERR_DEFAULT                             : return "Unknown Error.";
    }
    
    return "";
}


/*---------------------------------------------------------------------------
  Desc: ݒ
---------------------------------------------------------------------------*/
bool TestImport::Initialize()
{
    nn::Result nnResult;
    
    // oϐ̏ݒ
    {
        m_SysRegion                    = UNDEFINED;
        m_BondingOption                = 0x0;
        m_SystemPackageVersion         = "0.0.0-0X";
        m_PreInstallPackageVersion     = "---";
        m_Mark                         = "-----";
        
        m_RemainingBatteryLv           = reinterpret_cast<char*>(sys::Alloc(4*sizeof(char), 32));
        
        m_ImportInfoCurrent.fileSize   = 0;
        m_ImportInfoCurrent.importSize = 0;
        m_ImportInfoCurrent.percentage = 0;
        
        m_ImportInfoTotal.fileSize     = 0;
        m_ImportInfoTotal.importSize   = 0;
        m_ImportInfoTotal.percentage   = 0;
        
        m_AlreadyImportedFileSize      = 0;
        
        m_PreInstall                   = PREINSTALL;
        
        // fobOp
        uji::seq::Config config;
        NON_DELETE_PROGRAM     = config.Get().NonDeleteProgram;
        NON_DELETE_TICKET      = config.Get().NonDeleteTicket;
        NON_IMPORT_CIA         = config.Get().NonImportCia;
        
        THRESHOLD_REMAINING_BATTERY_LV = config.GetAlternative().RemainingBatteryLv;
        
        m_ProgressMsec         = 0;
        
        // NGbZ[Wp
        size_t bufSize  = sizeof(char) * 256 * 1024;
        m_SdSizeString                          = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        m_HashString                            = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        m_ImportTitleContextsString             = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        m_CompareProgramInfosNumString          = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        m_CompareProgramInfosIdAndVerString     = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        m_CompareNumOfTicketsAndProgramsString  = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        m_CompareIdOfTicketsAndProgramsString   = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        m_FatalErrorString                      = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        memset( m_SdSizeString,                         '\0', bufSize );
        memset( m_HashString,                           '\0', bufSize );
        memset( m_ImportTitleContextsString,            '\0', bufSize );
        memset( m_CompareProgramInfosNumString,         '\0', bufSize );
        memset( m_CompareProgramInfosIdAndVerString,    '\0', bufSize );
        memset( m_CompareNumOfTicketsAndProgramsString, '\0', bufSize );
        memset( m_CompareIdOfTicketsAndProgramsString,  '\0', bufSize );
        memset( m_FatalErrorString,                     '\0', bufSize );
    }
    
    // {fBOIvV
    m_BondingOption = uji::sys::GetBondingOption();
    
    if( "UPDATE_ONLY" != UJI_REGION )
    {
        // PackageVersion.dat o[W𒊏o
        s64    fileSize  = 0;
        s32    readSize  = 0;
        size_t bufSize  = sizeof(char) * 256 * 1024;
        char*  packBuf  = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        char*  rspBuf   = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        char*  pipBuf   = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        char*  markBuf  = reinterpret_cast<char*>(sys::Alloc(bufSize, 32));
        memset( packBuf, '\0', bufSize );
        memset( rspBuf,  '\0', bufSize );
        memset( pipBuf,  '\0', bufSize );
        memset( markBuf, '\0', bufSize );
        
        nn::fs::FileInputStream infile;
        nnResult = infile.TryInitialize( L"rom:/misc/PackageVersion.dat" );
        if( nnResult.IsFailure() ) return false;
        nnResult = infile.TryGetSize( &fileSize );
        if( nnResult.IsFailure() ) return false;
        nnResult = infile.TryRead( &readSize, packBuf, fileSize );
        if( nnResult.IsFailure() ) return false;
        infile.Finalize();
        
        int indexHyphenLeft  = strstr ( packBuf, "-" ) - packBuf;
        int indexHyphenRight = strrchr( packBuf, '-' ) - packBuf;
        char* ptrLeftParen  = strstr ( packBuf, "(");
        char* ptrRightParen = strstr ( packBuf, ")");
        
             if( m_BondingOption==0x0 )  strcpy( markBuf, "P" );  // ʎY@̏ꍇ
        else if( m_BondingOption==0x1 )  strcpy( markBuf, "D" );  // J@̏ꍇ
        else                             strcpy( markBuf, "" );

        //PackageVersion.dat(@햼)܂܂ĂꍇAMark̏ύX
        if ( ptrLeftParen != NULL && ptrRightParen != NULL ) {//(@햼)܂ޏꍇ
            int indexLeftParen  = ptrLeftParen - packBuf;
            int indexRightParen = ptrRightParen - packBuf;
            //NN_ASSERT(indexRightParen > indexLeftParen);//ʂɗĂARSPo[W\
            if( indexHyphenLeft == indexHyphenRight )
            {
                // PackageVersion.dat ɃvCXg[o[W܂܂ĂȂꍇ
                strcpy ( rspBuf, packBuf );
                strcpy ( pipBuf, "---" );
                strncat( markBuf, rspBuf,  indexLeftParen - 1 );
                strcat ( markBuf, "-");
                strncat( markBuf, ptrLeftParen + 1, indexRightParen - indexLeftParen - 1);
            }
            else
            {
                // PackageVersion.dat ɃvCXg[o[W܂܂Ăꍇ
                strncpy( rspBuf, packBuf,  indexHyphenRight );
                strcpy ( pipBuf, packBuf+1+indexHyphenRight );
                strncat( markBuf, rspBuf, indexLeftParen - 1);
                strcat ( markBuf, "-");
                strncat( markBuf, ptrLeftParen + 1, indexRightParen - indexLeftParen - 1);
                strcat ( markBuf, "-" );
                strcat ( markBuf, pipBuf );
                
            }
        } else {//(@햼)܂܂Ȃꍇ
            if( indexHyphenLeft==indexHyphenRight )
            {
                // PackageVersion.dat ɃvCXg[o[W܂܂ĂȂꍇ
                strcpy ( rspBuf, packBuf );
                strcpy ( pipBuf, "---" );
                strncat( markBuf, rspBuf,  strlen(rspBuf)-1 );
            }
            else
            {
                // PackageVersion.dat ɃvCXg[o[W܂܂Ăꍇ
                strncpy( rspBuf, packBuf,  indexHyphenRight );
                strcpy ( pipBuf, packBuf+1+indexHyphenRight );
                strncat( markBuf, rspBuf,  strlen(rspBuf)-1 );
                strcat ( markBuf, "-" );
                strcat ( markBuf, pipBuf );
            }
                    
        }
        
        m_SystemPackageVersion     = rspBuf;
        m_PreInstallPackageVersion = pipBuf;
        m_Mark                     = markBuf;
        
        sys::Free(packBuf);
        sys::Free(rspBuf);
        sys::Free(pipBuf);
        sys::Free(markBuf);
    }
    
    //Configݒǂ
    uji::seq::Config  *c = new uji::seq::Config;
        std::memset( m_ssid, 0x00, sizeof(m_ssid));
        u16 ssidU16 = c->GetAlternative().WapSsid;
        ssidU16 = ssidU16 % 10000;                 //4
        sprintf(m_ssid, "N-WAP%04d", ssidU16);
    delete c;
    
    // VXe[W
    // C-ARM
         if( "JPN_DEV"==UJI_REGION && CARM==true )  m_SysRegion = JPN_ARM;
    else if( "USA_DEV"==UJI_REGION && CARM==true )  m_SysRegion = USA_ARM;
    else if( "EUR_DEV"==UJI_REGION && CARM==true )  m_SysRegion = EUR_ARM;
    // ʎY@
    else if( "JPN"          == UJI_REGION )     m_SysRegion = JPN;
    else if( "USA"          == UJI_REGION )     m_SysRegion = USA;
    else if( "EUR"          == UJI_REGION )     m_SysRegion = EUR;
    else if( "AUS"          == UJI_REGION )     m_SysRegion = AUS;
    else if( "KOR"          == UJI_REGION )     m_SysRegion = KOR;
    else if( "CHN"          == UJI_REGION )     m_SysRegion = CHN;
    else if( "TWN"          == UJI_REGION )     m_SysRegion = TWN;
    else if( "UKV"          == UJI_REGION )     m_SysRegion = UKV;
    // J@
    else if( "JPN_DEV"      == UJI_REGION )     m_SysRegion = JPN;
    else if( "USA_DEV"      == UJI_REGION )     m_SysRegion = USA;
    else if( "EUR_DEV"      == UJI_REGION )     m_SysRegion = EUR;
    else if( "AUS_DEV"      == UJI_REGION )     m_SysRegion = AUS;
    // Jc[
    else if( "KMC"          == UJI_REGION )     m_SysRegion = KMC;
    // lbg[NΉV
    else if( "JPN_CUD_NW"     == UJI_REGION )   m_SysRegion = JPN;
    else if( "JPN_DEV_CUD_NW" == UJI_REGION )   m_SysRegion = JPN;
    else if( "USA_CUD_NW"     == UJI_REGION )   m_SysRegion = USA;
    else if( "USA_DEV_CUD_NW" == UJI_REGION )   m_SysRegion = USA;
    // C_pAbvf[gJ[h
    else if( "UPDATE_ONLY"  == UJI_REGION )     m_SysRegion = UPD;
    //FTRp(VXej[RSP6.0.0_łCTR/SPRƈقȂ邽)
    else if( "FTR_USA"    == UJI_REGION )   m_SysRegion = USA;
    else if( "FTR_EUR"    == UJI_REGION )   m_SysRegion = EUR;
    else if( "FTR_KOR"    == UJI_REGION )   m_SysRegion = KOR;
    else if( "FTR_USA_DEV"    == UJI_REGION )   m_SysRegion = USA;
    //
    else if( "FR12_PP2"        == UJI_REGION )   m_SysRegion = USA;
    else if( "FR12_PP2_DEV"    == UJI_REGION )   m_SysRegion = USA;
    else if( "FTR_SP1_USA_DEV"    == UJI_REGION )   m_SysRegion = USA;
    else if( "FTR_SP2_USA"        == UJI_REGION )   m_SysRegion = USA;
    else if( "FTR_SP2_USA_DEV"    == UJI_REGION )   m_SysRegion = USA;
    else if( "FTR_SP3_USA"        == UJI_REGION )   m_SysRegion = USA;
    else if( "FTR_SP3_USA_DEV"    == UJI_REGION )   m_SysRegion = USA;
        
    // Window 쐬
    uji::sys::Size size;
    size.width  = gfx->DISPLAY1_WIDTH;
    size.height = gfx->DISPLAY1_HEIGHT;
    m_LogWindow = 
        new sys::TextWindow((int)((double)gfx->DISPLAY1_WIDTH * 2 / (double)FONT_SIZE_S) -2 , 
                            (int)((double)gfx->DISPLAY1_HEIGHT / (double)FONT_SIZE_S) -2, 
                            FONT_SIZE_S);
    (void)m_WindowManager.CreateWindow(m_LogWindow, NN_GX_DISPLAY1, 0, 0);
    m_LogWindow->SetTitle("Log");
    m_LogWindow->SetSize(size);
    
    // wi
    SetBlackBackGroundColor();
    
    // Wifi XCb` off ̏ꍇ͓͂𑣂
    if( !nn::nwm::IsWifiOn() )
    {
        m_JpegDrawer = new sys::JpegDrawer;
        m_JpegDrawer->OpenPicture( L"rom:/jpeg/WifiButtonForCheck.jpg" );
        UpdateLCD(CHECK_WIRELESS);
        
        do{
            nn::os::Thread::Sleep( nn::fnd::TimeSpan::FromMilliSeconds(200));
        }while( !nn::nwm::IsWifiOn());
        
        // 摜̔j
        m_JpegDrawer->ClosePicture();
        delete m_JpegDrawer;
    }
    
    return true;
}

/*---------------------------------------------------------------------------
  Desc: I
---------------------------------------------------------------------------*/
bool TestImport::Finalize()
{
    // wiftHgɖ߂
    SetDefaultBackGroundColor();
    
    // Window j
    m_WindowManager.DestroyWindow(m_LogWindow);
    delete m_LogWindow;
    
    return true;
}


/*---------------------------------------------------------------------------
  Desc: VAio[SecureInfoBodyRegionݒ
        (bΉ߁A݂͎gp)
---------------------------------------------------------------------------*/
bool TestImport::WriteRegionFromSerial()
{
    // serialNo擾
    char serNo[nn::cfg::CTR::detail::CFG_SECURE_INFO_SERIAL_NO_LEN + 1];
    std::memset(serNo, 0x00, nn::cfg::CTR::detail::CFG_SECURE_INFO_SERIAL_NO_LEN + 1);
    uji::sys::GetSerialNumber(serNo);

    /* serialNoRegionSecureInfoBodyɏ*/
    if(uji::sys::WriteSerialNumber(serNo) == false)
    {
        return false;
    }

    return true;
}

    
/*---------------------------------------------------------------------------
  Desc: X^[gʕ\
---------------------------------------------------------------------------*/
bool TestImport::OpenStartMenu()
{
    Pad pad;
    nn::hid::PadReader padReader;
    nn::hid::PadStatus padStatus;
    
    // 摜
    m_JpegDrawer = new sys::JpegDrawer;
    switch( m_SysRegion )
    {
        case JPN:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_jpn.jpg" );   break;
        case USA:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_usa.jpg" );   break;
        case EUR:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_eur.jpg" );   break;
        case AUS:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_aus.jpg" );   break;
        case KOR:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_kor.jpg" );   break;
        case CHN:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_chn.jpg" );   break;
        case TWN:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_twn.jpg" );   break;
        case UKV:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_ukv.jpg" );   break;
        case KMC:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_kmc.jpg" );   break;
        case UPD:       m_JpegDrawer->OpenPicture( L"rom:/jpeg/update.jpg" );       break;
        case JPN_ARM:   m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_jpn.jpg" );   break;
        case USA_ARM:   m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_usa.jpg" );   break;
        case EUR_ARM:   m_JpegDrawer->OpenPicture( L"rom:/jpeg/import_eur.jpg" );   break;
        default:        delete m_JpegDrawer;                                        return false;
    }
    
    do
    {
        pad.UpdatePad();
        UpdateLCD(START_MENU);
    }
    while(!pad.IsButtonDown(Pad::BUTTON_A));

    // 摜̔j
    m_JpegDrawer->ClosePicture();
    delete m_JpegDrawer;
    
    return true;
}

/*---------------------------------------------------------------------------
  Desc: CreateSeed()  IVS XV邽߂Ƀn[hEFAZbg
---------------------------------------------------------------------------*/
bool TestImport::HardwareReset()
{
    nn::Result nnResult;
    nnResult = nn::applet::CTR::detail::HardwareResetAsync();
    if( nnResult.IsFailure() )
    {
        return false;
    }
    else
    {
        return true;
    }
    
}

/*---------------------------------------------------------------------------
  Desc: ACA_v^҂
---------------------------------------------------------------------------*/
bool TestImport::ExtractAcAdapter()
{
    u8           data;
    u8           value;
    nn::os::Tick start;
    nn::os::Tick current;
    s64          time;
        
    m_JpegDrawer = new sys::JpegDrawer;
    m_JpegDrawer->OpenPicture( L"rom:/jpeg/extract.jpg" );
    UpdateLCD(EXTRACT_AC_ADAPTER);

    start = nn::os::Tick::GetSystemCurrent();

    do{
        data  = uji::eva::mcu::ReadMcu(0x0f);
        value = data & 0x08;

        if ( value != 0x08 )
        {
            m_JpegDrawer->ClosePicture();
            delete m_JpegDrawer;

            return true;
        }

        nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(100));
            
        current = nn::os::Tick::GetSystemCurrent();
        time    = (current-start).ToTimeSpan().GetSeconds();
    }while( time < 1800 );

    m_JpegDrawer->ClosePicture();
    delete m_JpegDrawer;
    
    sprintf( m_TestResult.m_String, OutputErrorString( ERR9_TIME_OUT ) );
    return false;
}

/*---------------------------------------------------------------------------
  Desc: drcʃ`FbN
---------------------------------------------------------------------------*/
bool TestImport::CheckRemainingBatteryLv()
{
    nn::Result        nnr;
    bool              ret = false;
    ErrSeq_DataImport micro  = ERR_DEFAULT;
    u8                btLevel = 0;
    
    nnr = uji::eva::mcu::GetRemainingBatteryLv(&btLevel);
    if( nnr.IsFailure() )
    {
        micro = ERR9_GET_REMAINING_BATTERY_LV;
        goto FINALIZE;
    }
    
    // drcʂݒlႢ͏[d𑣂
    if( THRESHOLD_REMAINING_BATTERY_LV > btLevel )
    {
        u8           data;
        u8           value;
        nn::os::Tick start;
        nn::os::Tick current;
        s64          time;
        
        m_JpegDrawer = new sys::JpegDrawer;
        m_JpegDrawer->OpenPicture( L"rom:/jpeg/charge.jpg" );
        UpdateLCD(CHECK_REMAINING_BATTERY_LV);
        
        data  = uji::eva::mcu::ReadMcu(0x0f);
        value = data & 0x08;
        
        start = nn::os::Tick::GetSystemCurrent();
        do{
            if ( 0x08 == value )
            {
                // 摜̔j
                m_JpegDrawer->ClosePicture();
                delete m_JpegDrawer;
                
                micro = PASS;
                goto FINALIZE;
            }
            nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(100));
            data  = uji::eva::mcu::ReadMcu(0x0f);
            value = data & 0x08;
            
            current = nn::os::Tick::GetSystemCurrent();
            time    = (current-start).ToTimeSpan().GetSeconds();
        }while( time < 1800 );
        
        // 摜̔j
        m_JpegDrawer->ClosePicture();
        delete m_JpegDrawer;
        
        micro = ERR9_TIME_OUT;
        goto FINALIZE;
    } else {
        micro = PASS;
    }
    
FINALIZE:
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnr ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: SD J[hɃf[^x[X쐬ď
---------------------------------------------------------------------------*/
bool TestImport::InitializeSdmc()
{
    // vCXg[̏ꍇ͖ true Ԃ
    // i{I TestList  false ݒj
    if( !m_PreInstall ) return true;
    
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    s64  sdTotal    = 0;
    s64  sdFree     = 0;
    int  intTotal   = 0;
    int  intFree    = 0;
    bool createForcibly = true;
    bool isAvailable = false;
    
    const unsigned int THRESHOLD_OF_SDHC  = 2048*1024*1024UL;
    const int CLUSTER_SIZE_FAT32 = 32;
    
    /* 
        SD J[h݉\iSD J[h}ĂA SD J[h
         LOCK XCb`ɂ菑݋֎~ĂȂԁjł邩
    */
    if( nn::fs::IsSdmcWritable()==false )                       { micro = ERR1_SDCARD_IS_NOT_WRITABLE; goto FINALIZE; } // SD J[hݕs
    
    /* 
        SD J[h blank `FbN
        ̃t@C^tH_ꍇ 32KB gp邽߁A
        eʁ󂫗eʂł邱ƂmF邱ƂŃ`FbN
    */
    {
        using namespace nn::fs;
        nnResult = nn::fs::MountSdmc();
        if( nnResult.IsFailure() )
        {
                 if( nnResult <= ResultMediaNotFound() )        micro = ERR1_MOUNT_SDMC_1;
            else if( nnResult <= ResultAlreadyExists() )        micro = ERR1_MOUNT_SDMC_2;
            else if( nnResult <= ResultArchiveInvalidated() )   micro = ERR1_MOUNT_SDMC_3;
            else if( nnResult <= ResultMediaAccessError() )     micro = ERR1_MOUNT_SDMC_4;
            else if( nnResult <= ResultBadFormat() )            micro = ERR1_MOUNT_SDMC_5;
            else                                                micro = ERR_FATAL;
            goto FINALIZE;
        }
        nnResult = GetSdmcSize( &sdTotal, &sdFree );
        if( nnResult.IsFailure() )
        {
                 if( nnResult <= ResultNotFound() )             micro = ERR1_GET_SDMC_SIZE;   // SD J[hȂAFłȂ
            else                                                micro = ERR_FATAL;            // FatalError
            goto FINALIZE;
        }
        // FAT32 ̏ꍇA1NX^gpς݂ɂȂ
        intTotal = ( sdTotal <= THRESHOLD_OF_SDHC ) ? (int)(sdTotal/1024) : (int)(sdTotal/1024)-CLUSTER_SIZE_FAT32;
        intFree  = (int)(sdFree /1024);
        
        // SD J[h̑eʂƋ󂫗eʂȂ
        if( intTotal!=intFree )
        {
            micro = ERR1_CHECK_CAPACITY;
            sprintf( m_SdSizeString, "Total=%d[KB]\nFree =%d[KB]\n", intTotal, intFree );
            goto FINALIZE;
        }
    }
    
    /* 
         SD J[hɃf[^x[X쐬
         f[^x[X̏ԂɊւ炸A폜E쐬E
    */
    {
        using namespace nn::am;
        nnResult = InitializeTitleDatabase( nn::fs::MEDIA_TYPE_SDMC, createForcibly );
        if( nnResult.IsFailure() )
        {
                 if( nnResult == ResultInvalidEnumValue() )     micro = ERR1_INITIALIZE_DB_1; // sȃfBAw
            else if( nnResult == ResultMediaNotSupported() )    micro = ERR1_INITIALIZE_DB_2; // T|[gÕfBAw
            else if( nnResult == ResultInvalidImportState())    micro = ERR1_INITIALIZE_DB_3; // C|[gԂs
            else                                                micro = ERR_FATAL;            // FatalError
            goto FINALIZE;
        }
    }
    
    /* 
         ΏۃfBAɃf[^x[X݂邩
    */
    {
        using namespace nn::am;
        nnResult = QueryAvailableTitleDatabase(&isAvailable,  nn::fs::MEDIA_TYPE_SDMC);
        if( nnResult.IsFailure() )
        {
                 if( nnResult == ResultInvalidEnumValue() )     micro = ERR1_QUERY_AVAILABLE_DB_1; // sȃfBAw
            else if( nnResult == ResultMediaNotSupported() )    micro = ERR1_QUERY_AVAILABLE_DB_2; // T|[gÕfBAw
            else if( nnResult == ResultInvalidImportState())    micro = ERR1_QUERY_AVAILABLE_DB_3; // C|[gԂs
            else                                                micro = ERR_FATAL;                 // FatalError
            goto FINALIZE;
        }
    }
    
    // SD J[h̃^Cgf[^x[Xgps
    if( isAvailable==false )
    {
        micro = ERR1_DB_IS_NOT_AVAILABLE;
        goto FINALIZE;
    }
    
    // PASS
    micro = PASS;
    
FINALIZE:
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    nn::fs::Unmount( "sdmc:" );
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: vO폜
        `Pbgf[^x[XΏۂ̃`Pbg폜
---------------------------------------------------------------------------*/
bool TestImport::DeleteNandApp( nn::fs::MediaType media, nn::ProgramId id )
{
    nn::Result nnResult;
    
    // f[^Ȃꍇ͎t[
    if( NON_DELETE_PROGRAM==true )
    {
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE,   "TitleID = %016llx\n", id);
        UpdateLCD_Sub(DELETE_NAND_APP);
        goto DELETE_TICKET;
    }
    
    //------------------------------------------------------------------------
    // Av̍폜
    //------------------------------------------------------------------------
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE,   "TitleID = %016llx\n", id);
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_MAGENTA, "deleting program...");
    UpdateLCD_Sub(DELETE_NAND_APP);
    
    nnResult = nn::am::DeleteProgram( media, id );
    DEBUG_NN_LOG("  deleting program: %08xh... ", nnResult.GetPrintableBits());
    
    if( nnResult.IsFailure() )
    {
        if( nnResult == nn::am::ResultNotFound() )
        {
            // w肳ꂽvOȂꍇ PASS
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_MAGENTA, "NotFound.\n");
            UpdateLCD_Sub(DELETE_NAND_APP);
            DEBUG_NN_LOG("ResultNotFound.\n");
        }
        else if( nnResult == nn::am::ResultInvalidEnumValue() )
        {
            // sȃfBAw
            m_TestResult.m_Micro = (int)(ERR4_DELETE_PROGRAM_1/100);
            return false;
        }
        else if( nnResult == nn::am::ResultInternalDataCorrupted() )
        {
            // f[^j
            m_TestResult.m_Micro = (int)(ERR4_DELETE_PROGRAM_2/100);
            return false;
        }
        else
        {
            // sȃG[
            m_TestResult.m_Micro = (int)(ERR_FATAL/100);
            return false;
        }
    }
    else
    {
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_MAGENTA, "done.\n");
            UpdateLCD_Sub(DELETE_NAND_APP);
            DEBUG_NN_LOG("done.\n");
    }
    
DELETE_TICKET: 
    // `Pbg폜Ȃꍇ͏I
    if( NON_DELETE_TICKET==true )  return true;
    
    //------------------------------------------------------------------------
    // `Pbg̍폜
    //------------------------------------------------------------------------
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "deleting ticket...");
    UpdateLCD_Sub(DELETE_NAND_APP);
    
    nnResult = nn::am::DeleteTicket( id );
    DEBUG_NN_LOG("  deleting ticket: %08xh... ", nnResult.GetPrintableBits());
    if( nnResult.IsFailure() )
    {
        if( nnResult == nn::am::ResultNotFound() )
        {
            // w肳ꂽvOȂꍇ PASS
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "NotFound.\n");
            UpdateLCD_Sub(DELETE_NAND_APP);
            DEBUG_NN_LOG("ResultNotFound.\n");
        }
        else if( nnResult == nn::am::ResultInternalDataCorrupted() )
        {
            // f[^j
            m_TestResult.m_Micro = (int)(ERR4_DELETE_TICKET/100);
            return false;
        }
        else
        {
            // sȃG[
            m_TestResult.m_Micro = (int)(ERR_FATAL/100);
            return false;
        }
    }
    else
    {
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "done.\n");
            UpdateLCD_Sub(DELETE_NAND_APP);
            DEBUG_NN_LOG("done.\n");
    }
    
    return true;
}

/*---------------------------------------------------------------------------
  Desc: svȌvOƑΏۂ̃`Pbgw肵č폜
---------------------------------------------------------------------------*/
bool TestImport::DeleteTestProgram()
{
    bool ret = false;
    const wchar_t* IMPORTEE_AGING_DIR = L"nand:/title/00040000/";
    
    if ( uji::sys::GetBondingOption() == 0 )//ʎY@̏ꍇ
    {
        // TestMenu ̍폜iJ@̏ꍇǍ̃C|[gōŐVłށj
        ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                            uji::eva::import::TestImport::IMPORTEE_TEST_MENU_ID);
        if( !ret )  goto FINALIZE;
    }
    
    // TWL Kernel ̍폜iʎY@̏ꍇǍ̃C|[gōŐVłށj
    ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                        uji::eva::import::TestImport::IMPORTEE_TWL_KERNEL_ID);
    if( !ret )  goto FINALIZE;
    
    // AGB Kernel ̍폜
    ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                        uji::eva::import::TestImport::IMPORTEE_AGB_KERNEL_ID);
    if( !ret )  goto FINALIZE;
    
    // NANDAviG[WOAAvP[Vj̍폜ifBNg폜j
    // IMPORTEE_AGING_ID  IMPORTEE_APPLICATION_ID ͓ ID
    ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                        uji::eva::AgingImporter::IMPORTEE_AGING_ID);
    if( !ret )  goto FINALIZE;
    ret = DeleteDirectory(IMPORTEE_AGING_DIR);
    if( !ret )  goto FINALIZE;
    
    
    // TWL ݊p^Cg ̍폜iC-ARM̂݁j
    // DEVXe[^XȂ̂UJI_REGIONgpĂ
    if(( "JPN_DEV"==UJI_REGION && CARM==true ) || 
       ( "EUR_DEV"==UJI_REGION && CARM==true ) || 
       ( "USA_DEV"==UJI_REGION && CARM==true ))
    {
        // zCgXg
        ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                            uji::eva::import::TestImport::IMPORTEE_TWL_WHITELIST_ID);
        if( !ret )  goto FINALIZE;
        // _E[hvC
        ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                            uji::eva::import::TestImport::IMPORTEE_TWL_DLPLAY_ID);
        if( !ret )  goto FINALIZE;
        // o[W
        ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                            uji::eva::import::TestImport::IMPORTEE_TWL_VERSION_ID);
        if( !ret )  goto FINALIZE;
        // DSC^[lbgݒ
        ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                            uji::eva::import::TestImport::IMPORTEE_TWL_INTERNET_ID);
        if( !ret )  goto FINALIZE;
    }
    
    // A؍dl̏ꍇ̂݃_E[hvCAṽ^CgIDقȂׁAō폜Ă
    // dn΂悢̂m_SysRegiongp
    if(( KOR == m_SysRegion ) ||
       ( CHN == m_SysRegion ))
    {
        // _E[hvC
        ret = DeleteNandApp(nn::fs::MEDIA_TYPE_NAND, 
                            uji::eva::import::TestImport::IMPORTEE_TWL_DLPLAY_ID);
        if( !ret )  goto FINALIZE;
    }
    
    // PASS
    ret = true;
    
FINALIZE:
    
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: vCXg[AvǉCpɃvCXg[̃`Pbg폜
        vCXg[Av̂́ASDtH[}bgō폜
---------------------------------------------------------------------------*/
bool TestImport::DeletePreinstallTicket()
{
    bool ret = false;    
    nn::Result nnResult;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    //`Pbg擾
    int ticketNum = 0;
    nnResult = nn::am::GetNumTickets(&ticketNum);
    if(nnResult.IsFailure())
    {
        if( nnResult == nn::am::ResultInternalDataCorrupted() )      micro = ERR6_GET_NUM_TICKETS_1;
        else                                                         micro = ERR_FATAL;
        goto FINALIZE;
    }

    {
        //`PbgProgramId擾
        nn::ProgramId programIdList[ticketNum];
        nnResult = nn::am::GetTicketList(&ticketNum, programIdList, ticketNum, 0);
        if(nnResult.IsFailure())
        {
            if( nnResult == nn::am::ResultInvalidSize() )                micro = ERR6_GET_TICKET_LIST_1;
            else if( nnResult == nn::am::ResultOutOfRange() )            micro = ERR6_GET_TICKET_LIST_2;
            else if( nnResult == nn::am::ResultInternalDataCorrupted() ) micro = ERR6_GET_TICKET_LIST_3;
            else                                                         micro = ERR_FATAL;
            goto FINALIZE;
        }
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "\n");
        
        for(int i = 0; i < ticketNum; i++)
        {
            nn::fs::MediaType media      = nn::fs::MEDIA_TYPE_NAND;

            //CTR System Av
            if( nn::CTR::IsCtr(programIdList[i]) && nn::CTR::IsSystem(programIdList[i]) ) { media = nn::fs::MEDIA_TYPE_NAND; }

            //CTR User AvŊ^Cg ID ̖ 02
            else if( nn::CTR::IsCtr(programIdList[i]) &&  nn::CTR::GetVariationOf(programIdList[i]) == 2 ) { media = nn::fs::MEDIA_TYPE_NAND; }

            //CTR User Av
            else if( nn::CTR::IsCtr(programIdList[i]) ) { media = nn::fs::MEDIA_TYPE_SDMC; }

            //TWL AvŊ TWL System Avł
            else if( nn::CTR::IsTwlApp(programIdList[i]) &&  nn::CTR::IsTwlSystemApp(programIdList[i]) ) { media = nn::fs::MEDIA_TYPE_NAND; }

            //LȊO̕sȃAv
            else { micro  = ERR2_UNKNOWN_APPLI; goto FINALIZE; }

        
            // vCXg[^Cg̃`Pbg̍폜
            if ( media == nn::fs::MEDIA_TYPE_SDMC )
            {
                LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE,   "TitleID = %016llx\n", programIdList[i]);
                LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "deleting ticket ...");
                UpdateLCD_Sub(DELETE_PREINSTALL_TICKET);
            
                nnResult = nn::am::DeleteTicket( programIdList[i] );
                DEBUG_NN_LOG("  deleting ticket: %08xh... ", nnResult.GetPrintableBits());
                if( nnResult.IsFailure() )
                {
                    if( nnResult == nn::am::ResultNotFound() )
                    {
                        // w肳ꂽvOȂꍇ PASS
                        LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "NotFound.\n");
                        UpdateLCD_Sub(DELETE_PREINSTALL_TICKET);
                        DEBUG_NN_LOG("ResultNotFound.\n");
                    }
                    else if( nnResult == nn::am::ResultInternalDataCorrupted() )
                    {
                        // f[^j
                        m_TestResult.m_Micro = (int)(ERR4_DELETE_TICKET/100);
                        return false;
                    }
                    else
                    {
                        // sȃG[
                        m_TestResult.m_Micro = (int)(ERR_FATAL/100);
                        return false;
                    }
                }
                else
                {
                    LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "done.\n");
                    UpdateLCD_Sub(DELETE_PREINSTALL_TICKET);
                    DEBUG_NN_LOG("done.\n");
                }
            }
        }
    }
    
    // PASS
    micro = PASS;
    
FINALIZE:
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: fBNg݂邩`FbNāAΏB
---------------------------------------------------------------------------*/
bool TestImport::DeleteDirectory( const wchar_t* dirPath )
{
    nn::fs::Directory dir;	// fBNgIuWFNg
    nn::Result nnResult;	// Ug
    
    // f[^폜Ȃꍇ͏I
    if( NON_DELETE_PROGRAM==true )  return true;
    
    // w肳ꂽfBNgJƂ݂āA݂ȂΏI
    nnResult = dir.TryInitialize( dirPath );
    if( nnResult.IsFailure() ) return true;
    
    dir.Finalize();
    
    // 폜
    nnResult = nn::fs::TryDeleteDirectory( dirPath );
    if( nnResult.IsFailure())
    {
        DEBUG_NN_LOG("Read directry Failed\n");
        nn::dbg::PrintResult( nnResult );
        return false;
    }
    
    return true;
}

/*---------------------------------------------------------------------------
  Desc: CAL  CFG ɃRs[
---------------------------------------------------------------------------*/
bool TestImport::ResetSetting()
{
    nn::Result  nnResult;
    nnResult = nn::cfg::CTR::init::ResetDatabase();
    if( nnResult.IsFailure() )
    {
        return false;
    }
    else
    {
        return true;
    }
}

/*---------------------------------------------------------------------------
  Desc: e[W Home j[ ID 
---------------------------------------------------------------------------*/
bool TestImport::WriteHomeMenuId()
{
    nn::Result  nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    using namespace nn::cfg::CTR;

    // ʃOo
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "writing HomeMenuId...");
    UpdateLCD_Sub(WRITE_HOME_MENU_ID);
    
    // [W擾
    CfgRegionCode region;
    nnResult = init::GetRegion(&region);
    if( nnResult.IsFailure() )
    {
        // [W擾siʏ͎sȂj
        micro = ERR1_GET_REGION;
        goto FINALIZE;
    }
    
    // j[ύX
    detail::MenuInfoCfgData menuInfo;
    switch(region)
    {
#if 1
    case CFG_REGION_JAPAN:      menuInfo = detail::MENU_INFO_CFG_DEFAULT;       break;
    case CFG_REGION_AMERICA:    menuInfo = detail::MENU_INFO_CFG_DEFAULT_USA;   break;
    case CFG_REGION_EUROPE:     menuInfo = detail::MENU_INFO_CFG_DEFAULT_EUR;   break;
    case CFG_REGION_CHINA:      menuInfo.programId = 0x000400300000A102ull;     break;
    case CFG_REGION_KOREA:      menuInfo.programId = 0x000400300000A902ull;     break;
    case CFG_REGION_TAIWAN:     menuInfo.programId = 0x000400300000B102ull;     break;
#else
    /*
       TODO: 0.16.67 ȍ~̃t@[ł͈ȉ̒ʂ`ύXȂ΂ȂI
    */
    case CFG_REGION_JAPAN:      menuInfo = detail::MENU_INFO_CFG_DEFAULT;       break;
    case CFG_REGION_AMERICA:    menuInfo = detail::MENU_INFO_CFG_DEFAULT_US;    break;
    case CFG_REGION_EUROPE:     menuInfo = detail::MENU_INFO_CFG_DEFAULT_EU;    break;
    case CFG_REGION_CHINA:      menuInfo.programId = nn::CTR::MakeProgramId(
                                    nn::CTR::PROGRAM_ID_CATEGORY_APPLET,
                                    nn::CTR::PROGRAM_ID_UNIQUE_ID_SYSMENU_CN,
                                    nn::CTR::PROGRAM_ID_VERSION_SYSMENU_CN );   break;
    case CFG_REGION_KOREA:      menuInfo.programId = nn::CTR::MakeProgramId(
                                    nn::CTR::PROGRAM_ID_CATEGORY_APPLET,
                                    nn::CTR::PROGRAM_ID_UNIQUE_ID_SYSMENU_KR,
                                    nn::CTR::PROGRAM_ID_VERSION_SYSMENU_KR );   break;
    case CFG_REGION_TAIWAN:     menuInfo.programId = nn::CTR::MakeProgramId(
                                    nn::CTR::PROGRAM_ID_CATEGORY_APPLET,
                                    nn::CTR::PROGRAM_ID_UNIQUE_ID_SYSMENU_TW,
                                    nn::CTR::PROGRAM_ID_VERSION_SYSMENU_TW );   break;
#endif
    // m̃[Wiʏ͂ւ͂ĂȂj
    default:
        micro = ERR1_UNKNOWN_REGION;
        goto FINALIZE;
    }
    
    // j[ݒ𔽉f
    nnResult = init::SetConfig(
        GET_CFG_KEY(detail::NN_CFG_MENU, detail::NN_CFG_MENU_MENU_INFO),
        &menuInfo,
        sizeof(detail::MenuInfoCfgData));
    if( nnResult.IsFailure() )
    {
        // ݒ蔽fsiʏ͎sȂj
        micro = ERR1_SET_CONFIG;
        goto FINALIZE;
    }
    
    // NAND ɕۑ
    nnResult = init::FlushConfig();
    if( nnResult.IsFailure() )
    {
        // ۑsiʏ͎sȂj
        micro = ERR1_FLUSH_CONFIG;
        goto FINALIZE;
    }
    
    // ̃j[擾
    nnResult = nn::cfg::CTR::init::GetConfig(
        &menuInfo,
        sizeof(nn::cfg::CTR::detail::MenuInfoCfgData),
        GET_CFG_KEY(nn::cfg::CTR::detail::NN_CFG_MENU, nn::cfg::CTR::detail::NN_CFG_MENU_MENU_INFO));
    if( nnResult.IsFailure() )
    {
        // 擾siʏ͎sȂj
        micro = ERR1_GET_CONFIG;
        goto FINALIZE;
    }
    
    // j[̃vO ID \
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "done.[%016llX]\n\n", menuInfo.programId);
    UpdateLCD_Sub(WRITE_HOME_MENU_ID);
    
    // PASS
    micro = PASS;
    
FINALIZE:
    
    if( micro!=PASS )
    {
        sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: CFG ̒lύX
---------------------------------------------------------------------------*/
bool TestImport::ChangeConfigParameters()
{
    // C-ARM ȊȌꍇ͖ true Ԃ
    if(( m_SysRegion != JPN_ARM ) &&
       ( m_SysRegion != USA_ARM ) &&
       ( m_SysRegion != EUR_ARM ))
    {
        return false;
    }
    
    using namespace nn::cfg::CTR::detail;
    using namespace nn::cfg::CTR::init;
    
    const u16 CARM_EULA_SETTING_AGREE_VERSION   = 0x0100;   // 01.00
    const u8  CARM_OTHER_SETTING_SOUND_MODE     = 0x01;     // stereo
    const u8  CARM_LCD_SETTING_LUMINANCE_LEVEL  = 0x03;     // 3
    
    // cfg 
    nn::cfg::CTR::init::Initialize();
    
    // Eula Setting  Agree Version
    {
        // ݒ肪i[\̂w
        EulaInfoCfgData     eulaInfoCfgData;
        size_t              size = sizeof(EulaInfoCfgData);
        bit32               key  = GET_CFG_KEY(NN_CFG_EULA, NN_CFG_EULA_INFO);
        
        // ̐ݒ擾AݒύXAύX̐ݒ𔽉f
        if( GetConfig( &eulaInfoCfgData, size, key ).IsFailure() ) return false;
        eulaInfoCfgData.agreeVersion.version = CARM_EULA_SETTING_AGREE_VERSION;
        if( SetConfig( key, &eulaInfoCfgData, size ).IsFailure() ) return false;
    }
    
    // Other Setting  Sound Mode
    {
        // ݒ肪i[\̂w
        SoundSettingCfgData soundSettingCfgData;
        size_t              size = sizeof(SoundSettingCfgData);
        bit32               key  = GET_CFG_KEY(NN_CFG_SOUND, NN_CFG_SOUND_SETTING);
        
        // ̐ݒ擾AݒύXAύX̐ݒ𔽉f
        if( GetConfig( &soundSettingCfgData, size, key ).IsFailure() ) return false;
        soundSettingCfgData.soundOutputMode = CARM_OTHER_SETTING_SOUND_MODE;
        if( SetConfig( key, &soundSettingCfgData, size ).IsFailure() ) return false;
    }
    
    // Lcd Setting  LUMINANCE LEVEL
    {
        // ݒ肪i[\̂w
        BackLightCfgData    backLightCfgData;
        size_t              size = sizeof(BackLightCfgData);
        bit32               key  = GET_CFG_KEY(NN_CFG_LCD, NN_CFG_LCD_BACKLIGHT);
        
        // ̐ݒ擾AݒύXAύX̐ݒ𔽉f
        if( GetConfig( &backLightCfgData, size, key ).IsFailure() ) return false;
        backLightCfgData.luminanceLevel = CARM_LCD_SETTING_LUMINANCE_LEVEL;
        if( SetConfig( key, &backLightCfgData, size ).IsFailure() ) return false;
    }
    
    // Z[uf[^ɕۑ
    if( nn::cfg::CTR::init::FlushConfig().IsFailure() ) return false;
    
    // cfg I
    nn::cfg::CTR::init::Finalize();
    
    return true;
}

/*---------------------------------------------------------------------------
  Desc: Ñj[ύX
---------------------------------------------------------------------------*/
bool TestImport::ChangeOpeningMenu()
{
    using namespace nn::cfg::CTR::detail;
    using namespace nn::cfg::CTR::init;
    
    const bit64 SNAKE_OTHER_SETTING_MENU = 0x0004003000008102; // test menu
    
    if( m_BondingOption==0x1 ) // J@̏ꍇ̂
    {
        // cfg 
        nn::cfg::CTR::init::Initialize();
        
        // Other Setting  Menu
        {
            // ݒ肪i[\̂w
            MenuInfoCfgData menuInfoCfgData;
            size_t          size = sizeof(MenuInfoCfgData);
            bit32           key  = GET_CFG_KEY(NN_CFG_MENU, NN_CFG_MENU_MENU_INFO);
            
            // ̐ݒ擾AݒύXAύX̐ݒ𔽉f
            if( GetConfig( &menuInfoCfgData, size, key ).IsFailure() ) return false;
            menuInfoCfgData.programId = SNAKE_OTHER_SETTING_MENU;
            if( SetConfig( key, &menuInfoCfgData, size ).IsFailure() ) return false;
        }
        
        // Z[uf[^ɕۑ
        if( nn::cfg::CTR::init::FlushConfig().IsFailure() ) return false;
        
        // cfg I
        nn::cfg::CTR::init::Finalize();
    }
    return true;
}

/*---------------------------------------------------------------------------
  Desc: fBNgGgǂݍ
---------------------------------------------------------------------------*/
bool TestImport::ReadDirectory( nn::fs::DirectoryEntry* dirEntry, s32* entryNum, const wchar_t* dirPath )
{
    
    nn::fs::Directory dir;	// fBNgIuWFNg
    nn::Result nnResult;	// Ug
    nnResult = dir.TryInitialize( dirPath );	// w肳ꂽfBNgJƂ݂
    if( nnResult.IsFailure())
    {
        DEBUG_NN_LOG("dirInitialize Failed\n");
        nn::dbg::PrintResult( nnResult );
        return false;
    }
    
    nnResult = dir.TryRead( entryNum, &dirEntry[0], 255 );
    if( nnResult.IsFailure())
    {
        DEBUG_NN_LOG("Read directry Failed\n");
        nn::dbg::PrintResult( nnResult );
        return false;
    }
    
    for( int i=0; i<*entryNum; i++ )
    {
        m_ImportInfoTotal.fileSize += dirEntry[i].entrySize;
    }
    DEBUG_NN_LOG("totalFileSize = %lld\n", m_ImportInfoTotal.fileSize);
    
    return true;
}

/*---------------------------------------------------------------------------
  Desc: cia t@C݁i֐j
---------------------------------------------------------------------------*/
ErrSeq_DataImport TestImport::InternalImportCia( nn::fs::MediaType media, const wchar_t* filePath, nn::ProgramId id )
{
    nn::Result nnResult;
    
    // ܂Ȃꍇ͏I
    if( NON_IMPORT_CIA==true ) return PASS;
    
    // ɑ݂^Cgꍇ
    // ^CgႤP[XɑΉ邽߂ɁA`PbgĂB
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "deleting ticket...");
    UpdateLCD_Sub(IMPORT_CIA);
    nnResult = nn::am::DeleteTicket(id);
    if( nnResult.IsFailure() )
    {
        return ERR2_INTERNAL_DELETE_TICKET;
    }
    else
    {
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_CYAN, "done.\n");
        UpdateLCD_Sub(IMPORT_CIA);
    }
    
    // 
    nn::os::BlockingQueue queue;
    ImportProgress buffer[10];
    nn::os::Thread importThread;
    
    // ubLOL[̏
    queue.Initialize(reinterpret_cast<uptr*>(buffer), sizeof(buffer)/sizeof(ImportProgress));
    
    
    // C|[gXbh
    const size_t STACK_SIZE = 4096;
    ImportThreadArgs threadArgs;
    threadArgs.ti = this;
    threadArgs.queue = &queue;
    threadArgs.importFile.mediaType = media;
    threadArgs.importFile.filePath = filePath;
    importThread.StartUsingAutoStack(s_ImportThread, &threadArgs, STACK_SIZE, nn::os::DEFAULT_THREAD_PRIORITY-6);
    
    // C|[gppCv擾
    ImportProgress progress = *reinterpret_cast<ImportProgress*>(queue.Dequeue());
    nnResult = progress.retBegin;
    if( nnResult.IsFailure() )
    {
        // Xbh̔j
        importThread.Join();
        importThread.Finalize();
        
        using namespace nn::am;
             if( nnResult==ResultInvalidEnumValue() )     { return ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_1; } // sȃfBAw
        else if( nnResult==ResultMediaNotSupported() )    { return ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_2; } // T|[gÕfBAw
        else if( nnResult==ResultInvalidImportState() )   { return ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_3; } // C|[gԂs
        else if( nnResult==ResultImportPipeAlreadyOpen() ){ return ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_4; } // ɕʂ̃pCvJĂ
        else if( nnResult==ResultInternalDataCorrupted() ){ return ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_5; } // f[^j
        else if( nnResult==ResultAlreadyExists() )        { return ERR2_INTERNAL_BEGIN_IMPORT_PROGRAM_6; } // Temporary ł^Cg
        else                                              { return ERR_FATAL; }
    }
    
    // eC|[gt@CTCY̎擾҂
    progress = *reinterpret_cast<ImportProgress*>(queue.Dequeue());
    
    // eC|[gi
    {
        m_ImportInfoCurrent.fileSize = progress.fileSize;
        
        // Progress Bar ̐
        CreateProgress( &m_ImportInfoCurrent, 20, 170, 360, 16 );
        m_ImportInfoCurrent.pb->SetColor
        (
            nw::ut::FloatColor(0.1f, 0.85f, 0.1f, 1.0f),
            nw::ut::FloatColor(0.1f, 0.70f, 0.1f, 1.0f),
            nw::ut::FloatColor(0.9f, 0.9f, 0.9f, 1.0f),
            nw::ut::FloatColor(0.3f, 0.3f, 0.3f, 1.0f)
        );
    }
    
    // C|[gi
    {
        // Progress Bar ̐
        CreateProgress( &m_ImportInfoTotal,  20, 210, 360, 16 );
        m_ImportInfoTotal.pb->SetColor
        (
            nw::ut::FloatColor(0.1f, 0.1f, 0.90f, 1.0f),
            nw::ut::FloatColor(0.1f, 0.1f, 0.70f, 1.0f),
            nw::ut::FloatColor(0.9f, 0.9f, 0.9f, 1.0f),
            nw::ut::FloatColor(0.3f, 0.3f, 0.3f, 1.0f)
        );
    }
    
    // ʃOo
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_YELLOW, "importing...");
    UpdateLCD_Sub(IMPORT_CIA);
    
    // 
    while( progress.readTotal != progress.fileSize )
    {
#ifdef BLOCK_MAIN_THREAD
        // C|[g󋵂̎擾҂
        progress = *reinterpret_cast<ImportProgress*>(queue.Dequeue());
        nnResult = progress.retWrite;
        if( nnResult.IsFailure() )
        {
            // Xbh̔j
            importThread.Join();
            importThread.Finalize();
            return ERR2_INTERNAL_TRY_WRITE;
        }
#else
        ImportProgress *tmp;
        if (queue.TryDequeue(reinterpret_cast<uptr*>(&tmp)))
        {
            progress = *tmp;
        }
#endif
        
        // C|[gςݗe
        m_ImportInfoCurrent.importSize = progress.readTotal;
        m_ImportInfoTotal.importSize  = 
            progress.readTotal + static_cast<s32>(m_AlreadyImportedFileSize);
        
        // p[Ze[W
        m_ImportInfoCurrent.percentage = 
            static_cast<s32>((static_cast<double>(m_ImportInfoCurrent.importSize) /
                              static_cast<double>(m_ImportInfoCurrent.fileSize)) * 100.0);
        m_ImportInfoTotal.percentage = 
            static_cast<s32>((static_cast<double>(m_ImportInfoTotal.importSize) /
                              static_cast<double>(m_ImportInfoTotal.fileSize)) * 100.0);
        
        UpdateLCD(IMPORT_CIA);
        
    }
    
    // C|[gI
    progress = *reinterpret_cast<ImportProgress*>(queue.Dequeue());
    nnResult = progress.retEnd;
    if( nnResult.IsFailure() )
    {
        // Xbh̔j
        importThread.Join();
        importThread.Finalize();
        
        using namespace nn::am;
             if( nnResult==ResultInvalidImportState() )   { return ERR2_INTERNAL_END_IMPORT_PROGRAM_1; }   // C|[gԂs
        else if( nnResult==ResultInvalidPipe() )          { return ERR2_INTERNAL_END_IMPORT_PROGRAM_2; }   // w肳ꂽpCvs
        else if( nnResult==ResultVerificationFailed() )   { return ERR2_INTERNAL_END_IMPORT_PROGRAM_3; }   // ݃t@C̐؂Ɏs
        else if( nnResult==ResultNoEnoughSpace() )        { return ERR2_INTERNAL_END_IMPORT_PROGRAM_4; }   // ݂ɕKvȗ̈悪mۂłȂ
        else if( nnResult==ResultInternalDataCorrupted() ){ return ERR2_INTERNAL_END_IMPORT_PROGRAM_5; }   // f[^j
        else if( nnResult==ResultEsError() )              { return ERR2_INTERNAL_END_IMPORT_PROGRAM_6; }   // sȃf[^󂯎
        else                                              { return ERR_FATAL; }
    }
    
    // ݂Ɋ|Ԃ̎擾҂
    progress = *reinterpret_cast<ImportProgress*>(queue.Dequeue());
    
    // ʃOo
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_YELLOW, "done.");
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_BLUE, " @%d\n", progress.msec);
    UpdateLCD_Sub(IMPORT_CIA);
    
    //
    m_ProgressMsec = progress.msec;
    
    // ݍς݂̑t@CTCY
    m_AlreadyImportedFileSize = m_ImportInfoTotal.importSize;
    
    // Progress Bar ̔j
    DestroyProgress( &m_ImportInfoCurrent );
    DestroyProgress( &m_ImportInfoTotal );
    
    // XbȟEj
    importThread.Join();
    importThread.Finalize();
    
    // ubLOL[̔j
    queue.Finalize();
    
    return PASS;
}

/*---------------------------------------------------------------------------
  Desc: C|[g̍ŏI
---------------------------------------------------------------------------*/
nn::Result TestImport::CommitImportPrograms( nn::fs::MediaType media, nn::ProgramId* id, int num )
{
    nn::Result nnResult;
    
    // ʕ\
    char* string = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    memset( string, '\0', sizeof(char) * 256 * 1024);
    
    switch( media )
    {
        case nn::fs::MEDIA_TYPE_NAND:   sprintf( string, "nand" );  break;
        case nn::fs::MEDIA_TYPE_SDMC:   sprintf( string, "sdmc" );  break;
        default:                        sprintf( string, "----" );  break;
    }
    
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "committing(%s)...", string);
    UpdateLCD_Sub(COMMIT_IMPORT);
    
    // f[^܂Ȃꍇ͏I
    if( NON_IMPORT_CIA==true ) return nn::ResultSuccess();
    
    // R~bg
    DEBUG_NN_LOG("  committing... " );
    
    nnResult = nn::am::CommitImportPrograms( media, id, num );
    if( nnResult.IsFailure() )
    {
             if( nnResult == nn::am::ResultInvalidEnumValue() )
        {
            // sȃfBAw
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED, "InvalidEnumValue.\n");
            UpdateLCD_Sub(COMMIT_IMPORT);
            DEBUG_NN_LOG("ResultInvalidEnumValue.\n");
        }
        else if( nnResult == nn::am::ResultMediaNotSupported() )
        {
            // T|[gÕfBAw
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED, "MediaNotSupported.\n");
            UpdateLCD_Sub(COMMIT_IMPORT);
            DEBUG_NN_LOG("ResultMediaNotSupported.\n");
        }
        else if( nnResult == nn::am::ResultInvalidSize() )
        {
            // w肳ꂽvOs
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED, "InvalidSize.\n");
            UpdateLCD_Sub(COMMIT_IMPORT);
            DEBUG_NN_LOG("ResultInvalidSize.\n");
        }
        else if( nnResult == nn::am::ResultImportIncompleted() )
        {
            // CIA ݂̏IĂȂvOw
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED, "ImportIncompleted.\n");
            UpdateLCD_Sub(COMMIT_IMPORT);
            DEBUG_NN_LOG("ResultImportIncompleted.\n");
        }
        else if( nnResult == nn::am::ResultInternalDataCorrupted() )
        {
            // f[^j
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED, "InternalDataCorrupted.\n");
            UpdateLCD_Sub(COMMIT_IMPORT);
            DEBUG_NN_LOG("ResultInternalDataCorrupted.\n");
        }
        else
        {
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED, "FatalError.\n");
            UpdateLCD_Sub(COMMIT_IMPORT);
            DEBUG_NN_LOG("ResultFatalError.\n");
        }
    }
    else
    {
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "done.\n");
        UpdateLCD_Sub(COMMIT_IMPORT);
        DEBUG_NN_LOG("done.\n");
    }
    return nnResult;
}

/*---------------------------------------------------------------------------
  Desc: cia t@C { rom:/import/ ̏擾
---------------------------------------------------------------------------*/
bool TestImport::ImportCia()
{
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    /*
        Rom 擾 nand Ƀf[^
    */
    nn::fs::DirectoryEntry* dirEntry = new nn::fs::DirectoryEntry[ 255 ];   // fBNgGg
    s32                     entryNum = 0;                                   // Gg
    wchar_t*                dirPath  = new wchar_t[ 255 + 1 ];              // fBNgpX
    wchar_t*                filePath = new wchar_t[ 255 + 1 ];              // t@CpX
    
    // C|[gS cia t@C root
    std::wcscpy( dirPath, L"rom:/import/" );
    
    // fBNgGgǂݍ
    ReadDirectory( dirEntry, &entryNum, dirPath );
    
    // vOInfo  vOID ̔z
    nn::am::ProgramInfo   infos[entryNum];
    nn::ProgramId         id[entryNum];
    
    // Av^Cvƕۑ
    Appli_Type        appType    = CTR_SYSTEM_APPLI;
    nn::fs::MediaType media      = nn::fs::MEDIA_TYPE_NAND;
    
    // ۑ斈 ProgramInfo  ProgramId
    int                   i_r2n=0;
    int                   i_r2s=0;
    nn::am::ProgramInfo   infos_r2n[entryNum];
    nn::ProgramId         id_r2n[entryNum];
    nn::am::ProgramInfo   infos_r2s[entryNum];
    nn::ProgramId         id_r2s[entryNum];
    
    // CommitImportPrograms Ώۂ ProgramId
    int                   i_r2n_commit=0;
    int                   i_r2s_commit=0;
    nn::ProgramId         id_r2n_commit[entryNum];
    nn::ProgramId         id_r2s_commit[entryNum];
    
    // fobOpobt@
    char* buf               = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    char* id_Buf            = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    char* id_r2n_Buf        = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    char* id_r2s_Buf        = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    char* progressMsec_Buf  = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 4096, 32));
    memset( buf,              '\0', sizeof(char) * 256 * 1024);
    memset( id_Buf,           '\0', sizeof(char) * 256 * 1024);
    memset( id_r2n_Buf,       '\0', sizeof(char) * 256 * 1024);
    memset( id_r2s_Buf,       '\0', sizeof(char) * 256 * 1024);
    memset( progressMsec_Buf, '\0', sizeof(char) * 256 * 4096);
    
    // ㏑tO
    bool enableOverwrite = false;
    
    // cia t@C݊֐̌
    ErrSeq_DataImport  errInternalImportCia = ERR_DEFAULT;
    
    for( int i=0; i<entryNum; i++)
    {
        std::wcscpy( filePath, dirPath );
        std::wcscat( filePath, dirEntry[i].entryName );
        
        // ʃOo
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "[%d]\n", i+1);
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "%ls\n", filePath);
        UpdateLCD_Sub(IMPORT_CIA);
        
        // cia t@CvO擾B
        nnResult = nn::am::GetProgramInfoFromCia(&infos[i], filePath);
        if( nnResult.IsFailure() )
        {
            micro  = ERR2_GET_PROGRAM_INFOS_FROM_CIA;
            goto FINALIZE;
        }
        
        /*
            Av敪
            type =  0: CTR System Av
            type =  1: CTR User AvŊ^Cg ID ̖ 02
            type =  2: CTR User Av
            type =  3: TWL AvŊ TWL System Avł
            type = 99: LȊO͕sȃAv
        */
        int type = 99;
             if( nn::CTR::IsCtr(infos[i].id)    &&  nn::CTR::IsSystem(infos[i].id)          ) { type = 0; }
        else if( nn::CTR::IsCtr(infos[i].id)    &&  nn::CTR::GetVariationOf(infos[i].id)==2 ) { type = 1; }
        else if( nn::CTR::IsCtr(infos[i].id)                                                ) { type = 2; }
        else if( nn::CTR::IsTwlApp(infos[i].id) &&  nn::CTR::IsTwlSystemApp(infos[i].id)    ) { type = 3; }
        
        switch( type )
        {
            case 0:  appType = CTR_SYSTEM_APPLI;  media = nn::fs::MEDIA_TYPE_NAND;  break;
            case 1:  appType = CTR_USER_N_APPLI;  media = nn::fs::MEDIA_TYPE_NAND;  break;
            case 2:  appType = CTR_USER_S_APPLI;  media = nn::fs::MEDIA_TYPE_SDMC;  break;
            case 3:  appType = TWL_SYSTEM_APPLI;  media = nn::fs::MEDIA_TYPE_NAND;  break;
            default:
                micro  = ERR2_UNKNOWN_APPLI;
                goto FINALIZE;
        }
        
        switch( media )
        {
            case nn::fs::MEDIA_TYPE_NAND:
                {
                    infos_r2n[i_r2n] = infos[i];
                    id_r2n   [i_r2n] = infos[i].id;
                    i_r2n++;
                }
                break;
            case nn::fs::MEDIA_TYPE_SDMC:
                {
                    infos_r2s[i_r2s] = infos[i];
                    id_r2s   [i_r2s] = infos[i].id;
                    i_r2s++;
                }
                break;
        }
        
        /*
          SDKł̏㏑\L̂悤ɂȂĂ邽߁Aݎs{Ȃ΍킹KvB
          J@ : contentIDقȂĂ
          ʎY@ : contentIDقȂA^Cgo[WV
          A{̂ciat@CcontentIDr邱Ƃ́ASDKł͕s̂߁AL̑ΉƂB
          uVXepbP[WŃ^Cgo[WXVĂꍇAcontentIDύXĒv(JꕔΉ)
        */
        id[i] = infos[i].id;
        nn::am::ProgramInfo  info_Buf;
        nnResult = nn::am::GetProgramInfos( &info_Buf, media, id+i, 1);
        
        if( nnResult.IsFailure() )
        {
            if( nnResult == nn::am::ResultNotFound() )
            {
                // w肳ꂽID̃vOꍇ́AUpdater1stŃC|[gĂȂ
                // ^Cgׁ̈A
                enableOverwrite = true;
            }
            else
            {
                // LȊOnn::am::GetProgramInfosG[ɂȂ邱Ƃ͒ʏ킠蓾Ȃ
                     if( nnResult == nn::am::ResultInvalidEnumValue() )      micro  = ERR2_GET_PROGRAM_INFOS_1;
                else if( nnResult == nn::am::ResultMediaNotSupported() )     micro  = ERR2_GET_PROGRAM_INFOS_2;
                else if( nnResult == nn::am::ResultInvalidSize() )           micro  = ERR2_GET_PROGRAM_INFOS_3;
                else if( nnResult == nn::am::ResultInternalDataCorrupted() ) micro  = ERR2_GET_PROGRAM_INFOS_4;
                else                                                         micro  = ERR_FATAL;
                goto FINALIZE;
            }
        }
        else
        {
            // ʃOo
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_MAGENTA, 
                      "Latest:0x%04X <- Import:0x%04X\n", info_Buf.version, infos[i].version);
            UpdateLCD_Sub(IMPORT_CIA);
            
            if( uji::sys::GetBondingOption() == 1 && info_Buf.version < infos[i].version )//J@̏ꍇ
            {
                enableOverwrite = true;
            }
            else if( uji::sys::GetBondingOption() == 0 && info_Buf.version < infos[i].version )//ʎY@̏ꍇ
            {
                enableOverwrite = true;
            }
            else
            {
                // ͌Âo[W̏ꍇ͏܂A
                // ʁiC|[gςݗeʁAp[Ze[WAvOXo[ĵݍXV
                
                // C|[gςݗe
                nn::fs::FileInputStream in(filePath);
                m_ImportInfoCurrent.fileSize   = in.GetSize();
                m_ImportInfoCurrent.importSize = in.GetSize();
                m_ImportInfoTotal.importSize  = 
                    in.GetSize() + static_cast<s32>(m_AlreadyImportedFileSize);
                m_AlreadyImportedFileSize = m_ImportInfoTotal.importSize;
                
                // p[Ze[W
                m_ImportInfoCurrent.percentage = 
                    static_cast<s32>((static_cast<double>(m_ImportInfoCurrent.importSize) /
                                      static_cast<double>(m_ImportInfoCurrent.fileSize)) * 100.0);
                m_ImportInfoTotal.percentage = 
                    static_cast<s32>((static_cast<double>(m_ImportInfoTotal.importSize) /
                                      static_cast<double>(m_ImportInfoTotal.fileSize)) * 100.0);
                
                // Progress Bar ̐
                CreateProgress( &m_ImportInfoCurrent, 20, 170, 360, 16 );
                m_ImportInfoCurrent.pb->SetColor
                (
                    nw::ut::FloatColor(0.1f, 0.85f, 0.1f, 1.0f),
                    nw::ut::FloatColor(0.1f, 0.70f, 0.1f, 1.0f),
                    nw::ut::FloatColor(0.9f, 0.9f, 0.9f, 1.0f),
                    nw::ut::FloatColor(0.3f, 0.3f, 0.3f, 1.0f)
                );
                CreateProgress( &m_ImportInfoTotal,  20, 210, 360, 16 );
                m_ImportInfoTotal.pb->SetColor
                (
                    nw::ut::FloatColor(0.1f, 0.1f, 0.90f, 1.0f),
                    nw::ut::FloatColor(0.1f, 0.1f, 0.70f, 1.0f),
                    nw::ut::FloatColor(0.9f, 0.9f, 0.9f, 1.0f),
                    nw::ut::FloatColor(0.3f, 0.3f, 0.3f, 1.0f)
                );
                
                // ʍXV
                UpdateLCD(IMPORT_CIA);
                
                // Progress Bar ̔j
                DestroyProgress( &m_ImportInfoCurrent );
                DestroyProgress( &m_ImportInfoTotal );
            }
        }
        
        // ㏑tO true ȂΏ㏑
        if( enableOverwrite )
        {
            // tOɖ߂
            enableOverwrite = false;
            
            // cia t@C̏
            errInternalImportCia = InternalImportCia( media, filePath, id[i] );
            if( PASS != errInternalImportCia )
            {
                micro  = errInternalImportCia;
                goto FINALIZE;
            }
            
            // CommitImportPrograms p
            switch( media )
            {
                case nn::fs::MEDIA_TYPE_NAND:
                {
                    id_r2n_commit[i_r2n_commit] = id[i];
                    i_r2n_commit++;
                }
                break;
                case nn::fs::MEDIA_TYPE_SDMC:
                {
                    id_r2s_commit[i_r2s_commit] = id[i];
                    i_r2s_commit++;
                }
                break;
            }
        }
        else
        {
            // ʃOo
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_YELLOW, "not imported.\n");
            UpdateLCD_Sub(IMPORT_CIA);
        }
        
        ///////////////////////////////////////////////////////////////////////////////////////////
        if( DEBUG_MODE==1 )
        {
            sprintf( buf, "id: %016llx  version: %5d", infos[i].id, infos[i].version );
            strcat( id_Buf, buf );
            strcat( id_Buf, "\n");
            
            switch( appType )
            {
                case CTR_SYSTEM_APPLI:    strcat( buf, " (1) CTR_SYSTEM_APPLI\n" );    break;
                case CTR_USER_N_APPLI:    strcat( buf, " (2) CTR_USER_N_APPLI\n" );    break;
                case CTR_USER_S_APPLI:    strcat( buf, " (3) CTR_USER_S_APPLI\n" );    break;
                case TWL_SYSTEM_APPLI:    strcat( buf, " (4) TWL_SYSTEM_APPLI\n" );    break;
                default:                                                               break;
            }
            switch( media )
            {
                case nn::fs::MEDIA_TYPE_NAND:    strcat( id_r2n_Buf, buf );    break;
                case nn::fs::MEDIA_TYPE_SDMC:    strcat( id_r2s_Buf, buf );    break;
                default:                                                       break;
            }
            sprintf( buf, "%016llx, %lld, %lld", infos[i].id, m_ProgressMsec, infos[i].size );
            strcat( progressMsec_Buf, buf );
            strcat( progressMsec_Buf, "\n");
        }
        ///////////////////////////////////////////////////////////////////////////////////////////
    }
    
    delete[] filePath;
    delete[] dirPath;
    delete[] dirEntry;
    
    // C|[g̍ŏI
    if( i_r2n!=0 )
    {
        nnResult = CommitImportPrograms( nn::fs::MEDIA_TYPE_NAND, id_r2n_commit, i_r2n_commit );
        if( nnResult.IsFailure() )
        {
            micro  = ERR2_COMMIT_IMPORT_PROGRAMS;
            goto FINALIZE;
        }
    }
    if( i_r2s!=0 )
    {
        nnResult = CommitImportPrograms( nn::fs::MEDIA_TYPE_SDMC, id_r2s_commit, i_r2s_commit );
        if( nnResult.IsFailure() )
        {
            micro  = ERR2_COMMIT_IMPORT_PROGRAMS;
            goto FINALIZE;
        }
    }
    
    // Rom oϐɊi[
    m_RomAppInfo.media = nn::fs::MEDIA_TYPE_CTRCARD;
    m_RomAppInfo.num   = entryNum;
    for( int i=0; i<m_RomAppInfo.num; i++ )
    {
        m_RomAppInfo.infos[i]= infos[i];
        m_RomAppInfo.programs[i]= id[i];
    }
    
    // Rom  nand ɃC|[gAv
    m_Rom2NandAppInfo.media = nn::fs::MEDIA_TYPE_CTRCARD;
    m_Rom2NandAppInfo.num   = i_r2n;
    for( int i=0; i<m_Rom2NandAppInfo.num; i++ )
    {
        m_Rom2NandAppInfo.infos[i]= infos_r2n[i];
        m_Rom2NandAppInfo.programs[i]= id_r2n[i];
    }
    
    // Rom  sdmc ɃC|[gAv
    m_Rom2SdmcAppInfo.media = nn::fs::MEDIA_TYPE_CTRCARD;
    m_Rom2SdmcAppInfo.num   = i_r2s;
    for( int i=0; i<m_Rom2SdmcAppInfo.num; i++ )
    {
        m_Rom2SdmcAppInfo.infos[i]= infos_r2s[i];
        m_Rom2SdmcAppInfo.programs[i]= id_r2s[i];
    }
    
    // vCXg[̌O
    if( m_Rom2SdmcAppInfo.num!=0 )
    {
        char* code_Buf = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
        char* log_Buf  = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
        memset( code_Buf, '\0', sizeof(char) * 256 * 1024);
        memset( log_Buf,  '\0', sizeof(char) * 256 * 1024);
        
        for( int i=0; i<m_Rom2SdmcAppInfo.num; i++ )
        {
            // ObZ[WɃvCXg[Av̐iR[hƃ^CgID(j[N5)L^B
            nn::ProductCode pc;
            nn::am::GetProductCode(&pc, nn::fs::MEDIA_TYPE_SDMC, m_Rom2SdmcAppInfo.programs[i]);
            sprintf( code_Buf, "%016llx", m_Rom2SdmcAppInfo.programs[i] );
            sprintf( log_Buf,  "%d.%s(%s)", i+1, 
                                            static_cast<std::string>(pc.code ).substr(6,4).c_str(), 
                                            static_cast<std::string>(code_Buf).substr(9,5).c_str() );
            strcat( m_TestResult.m_String, log_Buf );
            strcat( m_TestResult.m_String, ( ((i+1)==3 || (i+1)==8) ? "\n" : "  ") );
            
            // l 80Byte ܂łȂ̂ iR[h{TitleID5(v13Byte)  5 ŁA񃍃Oo͂
            // ͍Ō TitleID ɂȂ烍Oo͂
            // ő 5*2=10 ܂ŃC|[gĂvȂ悤 "USRAPP", "USRAPP2" pӂĂB
            // seq/TesterLog/ProductionLogViewer.cpp 172sڂ\FɂĂ̋Lq
            if( (i+1)%5==0 || i==(m_Rom2SdmcAppInfo.num-1) )
            {
                     if( (i+1)<= 5 ) WriteProductionLog("PreInstall", "USRAPP" );
                else if( (i+1)<=10 ) WriteProductionLog("PreInstall", "USRAPP2");
                else                 WriteProductionLog("PreInstall", "-------"); // zO
                sprintf( m_TestResult.m_String, "" );
            }
        }
        
        // vCXg[݂Ӑ}̂ł邩iOMakefile  PREINSTALL=true ƂĂ邩j
        if( m_PreInstall!=true ) 
        {
            micro  = ERR2_INTENDED_PREINSTALL;
            goto FINALIZE;
        }
    }
    
    ///////////////////////////////////////////////////////////////////////////////////////////
    if( DEBUG_MODE==1 )
    {
        // SD J[hɏo
        nn::fs::MountSdmc();
        nn::fs::TryCreateDirectory( L"sdmc:/uji" );
        
        CreateTextToSdmcByArray( id_Buf,           L"sdmc:/uji/TitleInfos_Rom.txt" );
        CreateTextToSdmcByArray( id_r2n_Buf,       L"sdmc:/uji/TitleInfos_Rom2Nand.txt" );
        CreateTextToSdmcByArray( id_r2s_Buf,       L"sdmc:/uji/TitleInfos_Rom2Sdmc.txt" );
        CreateTextToSdmcByArray( progressMsec_Buf, L"sdmc:/uji/ImportProgressMsec.csv" );
        
        nn::fs::Unmount( "sdmc:" );
    }
    ///////////////////////////////////////////////////////////////////////////////////////////
    
    // PASS
    micro = PASS;
    
FINALIZE:
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        if( (int)(micro/100)==(int)(ERR2_GET_PROGRAM_INFOS/100)       || 
            (int)(micro/100)==(int)(ERR2_INTERNAL/100)                || 
            (int)(micro/100)==(int)(ERR2_COMMIT_IMPORT_PROGRAMS/100)  ) 
        {
            switch( media )
            {
                case nn::fs::MEDIA_TYPE_NAND:   m_TestResult.m_Micro = 100 + (int)(micro/100);  break;
                case nn::fs::MEDIA_TYPE_SDMC:   m_TestResult.m_Micro = 200 + (int)(micro/100);  break;
                default:                        m_TestResult.m_Micro = 500 + (int)(micro/100);  break;
            }
        }
        else
        {
            m_TestResult.m_Micro = (int)(micro / 100);;
        }
        ret = false;
        nn::gx::CTR::Backlight::Restore(nn::gx::CTR::LCD_BOTH);
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: t@[EFAXV
---------------------------------------------------------------------------*/
bool TestImport::UpdateFirmware()
{
    nn::Result nnResult;
    nnResult = nn::am::UpdateFirmwareAuto();
    if( nnResult.IsFailure() )
    {
        return false;
    }
    else
    {
        return true;
    }
}

/*---------------------------------------------------------------------------
  Desc: gZ[uf[^̃C|[g
---------------------------------------------------------------------------*/
bool TestImport::ImportExtSaveData()
{
    // vCXg[̏ꍇ͖ true Ԃ
    // i{I TestList  false ݒj
    if( !m_PreInstall ) return true;
    
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    bool needCreation = false;
    
    // gZ[uf[^ID
    const char*         archiveNameExtSaveData  = "extdata:";
    const std::wstring  romPathExtSaveDataW     = L"rom:/misc/SaveData.dat";
    const std::wstring  extPathExtSaveDataW     = L"extdata:/SaveData.dat";
    const nn::fs::ExtSaveDataId  extSaveDataId  = EXT_SAVE_DATA_NUMBER;
    
    // ۑgZ[uf[^obt@ɓǂݍ݂܂
    nn::fs::FileInputStream instream( romPathExtSaveDataW.c_str() );
    s64        extSaveDataBufSize = sizeof(wchar_t) * 256 * 1024;
    wchar_t*   extSaveDataBuf     = reinterpret_cast<wchar_t*>(sys::Alloc(extSaveDataBufSize, 32));
    s32        extSaveDataSize    = instream.Read(extSaveDataBuf, extSaveDataBufSize);
    
    // gZ[uf[^̃}Eg
    // ł͊gZ[uf[^݂͑Ȃ
    {
        nnResult = nn::fs::MountExtSaveData(archiveNameExtSaveData, extSaveDataId);
        if(nnResult.IsFailure())
        {
            if(nnResult <= nn::fs::ResultNotFound())
            {
                if(nnResult <= nn::fs::ResultMediaNotFound())
                {
                    // SDJ[hFłȂ
                    micro=ERR3_MOUNT_EXT_SAVE_DATA_1;
                    goto FREE_BUFFER;
                }
                
                // gZ[uf[^݂ĂȂ̂ō쐬
                needCreation = true;
            }
            else if((nnResult <= nn::fs::ResultNotFormatted()) ||
                    (nnResult <= nn::fs::ResultVerificationFailed()))
            {
                // gZ[uf[^蒼Kv
                nnResult = nn::fs::DeleteExtSaveData(extSaveDataId);
                if(nnResult.IsFailure())
                {
                    micro=ERR3_MOUNT_EXT_SAVE_DATA_2;
                    goto FREE_BUFFER;
                }
                needCreation = true;
            }
            else if(nnResult <= nn::fs::ResultBadFormat())
            {
                // SD J[htH[}bgKv
                micro=ERR3_MOUNT_EXT_SAVE_DATA_3;
                goto FREE_BUFFER;
            }
            else if(nnResult <= nn::fs::ResultOperationDenied())
            {
                if(nnResult <= nn::fs::ResultWriteProtected())
                {
                    // SD J[h݋֎~ɂĂ
                    micro=ERR3_MOUNT_EXT_SAVE_DATA_4;
                    goto FREE_BUFFER;
                }
                else if(nnResult <= nn::fs::ResultMediaAccessError())
                {
                    // ڐGsǓɂn[hEFAIȗv
                    micro=ERR3_MOUNT_EXT_SAVE_DATA_5;
                    goto FREE_BUFFER;
                }
                else
                {
                    // zÕG[
                    // SD J[h̃t@CfBNgǂݎpɂȂĂ铙̉\l܂
                    micro=ERR_FATAL;
                    goto FREE_BUFFER;
                }
            }
            else
            {
                // zÕG[
                micro=ERR_FATAL;
                goto FREE_BUFFER;
            }
        }
        else
        {
            // ȂgZ[uf[^݂
            micro=ERR3_MOUNT_EXT_SAVE_DATA_6;
            goto FREE_BUFFER;
        }
    }
    
    // gZ[uf[^̍쐬
    if( needCreation )
    {
        u32             dummy = 0;
        const size_t    maxDirectories  = 0;
        const size_t    maxFiles        = 3;
        
        // gZ[uf[^̍쐬
        nnResult = nn::fs::CreateExtSaveData(extSaveDataId, &dummy, sizeof(dummy), maxDirectories, maxFiles);
        if(nnResult.IsFailure())
        {
            if(nnResult <= nn::fs::ResultNotEnoughSpace())
            {
                // SD J[hɕKvȋ󂫗eʂȂ
                micro=ERR3_CREATE_EXT_SAVE_DATA_1;
                goto FREE_BUFFER;
            }
            else if(nnResult <= nn::fs::ResultNotFormatted())
            {
                // 쐬ɃG[A쐬łȂ
                micro=ERR3_CREATE_EXT_SAVE_DATA_2;
                goto FREE_BUFFER;
            }
            else if(nnResult <= nn::fs::ResultOperationDenied())
            {
                if(nnResult <= nn::fs::ResultWriteProtected())
                {
                    // SD J[h݋֎~ɂĂ
                    micro=ERR3_CREATE_EXT_SAVE_DATA_3;
                    goto FREE_BUFFER;
                }
                else if(nnResult <= nn::fs::ResultMediaAccessError())
                {
                    // ڐGsǓɂn[hEFAIȗv
                    micro=ERR3_CREATE_EXT_SAVE_DATA_4;
                    goto FREE_BUFFER;
                }
                else
                {
                    // zÕG[
                    micro=ERR_FATAL;
                    goto FREE_BUFFER;
                }
            }
            else
            {
                // zÕG[
                micro=ERR_FATAL;
                goto FREE_BUFFER;
            }
        }
        
        // Ń}EgɎsꍇ́AgZ[uf[^̎gp͕s\
        nnResult = nn::fs::MountExtSaveData(archiveNameExtSaveData, extSaveDataId);
        if(nnResult.IsFailure())
        {
            micro=ERR3_REMOUNT_EXT_SAVE_DATA;
            goto FREE_BUFFER;
        }

    }
    
    // gZ[uf[^ SaveData.dat ۑ
    {
        nn::fs::FileOutputStream    ostream;
        s32 out;
        
        // t@C쐬
        // gZ[uf[^ւ̃t@C쐬ɂ nn::fs::TryCreateFile gp
        nnResult = nn::fs::TryCreateFile(extPathExtSaveDataW.c_str(), extSaveDataSize);
        if(nnResult.IsFailure() && (nnResult <= nn::fs::ResultAlreadyExists()))
        {
            micro=ERR3_CREATE_EXT_SAVE_DATA;
            goto FREE_BUFFER;
        }
        
        // t@CJ
        nnResult = ostream.TryInitialize(extPathExtSaveDataW.c_str(), false);
        if(nnResult.IsFailure())
        {
            micro=ERR3_OPEN_EXT_SAVE_DATA;
            goto FREE_BUFFER;
        }
        
        // t@Cւ̏
        nnResult = ostream.TryWrite(&out, extSaveDataBuf, extSaveDataSize);
        if(nnResult.IsFailure())
        {
            micro=ERR3_WRITE_EXT_SAVE_DATA;
            goto FREE_BUFFER;
        }
        
        // t@C
        ostream.Finalize();
        
        // gZ[uf[^̃A}Eg
        nnResult = nn::fs::Unmount(archiveNameExtSaveData);
        if(nnResult.IsFailure())
        {
            micro=ERR3_UNMOUNT_EXT_SAVE_DATA;
            goto FREE_BUFFER;
        }
    }
    {
        micro = PASS;
        ret = true;
    }
    
FREE_BUFFER:
    sys::Free(extSaveDataBuf);
    
    if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
    else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
    m_TestResult.m_Micro = (int)(micro/100);
    
    return ret;

}

/*---------------------------------------------------------------------------
  Desc: sha1nbVlԂ
---------------------------------------------------------------------------*/
void TestImport::Sha1ContextGetHash( const void* dataBuf, s32 dataSize, char* string )
{
    nn::crypto::Sha1Context sha1;
    sha1.Initialize();

    // nbVvZpm
    int     hashLength = sha1.GetHashLength();
    void*   hashBuf    = uji::sys::Alloc( hashLength );
    memset( hashBuf, 0x00, sizeof(hashBuf) );
    
    // nbVvZ
    sha1.Update( dataBuf, dataSize );
    sha1.GetHash( hashBuf );
    
    // nbVl̕\
    u8 * hashPtr;
    hashPtr = static_cast<u8 *>(hashBuf);
    memset(string, 0x00, sizeof(string));
    
    for(int i = 0; i < hashLength; i++)
    {
        sprintf(string+(i*2), "%02X",*hashPtr);
        hashPtr++;
    }
    
    sha1.Finalize();
}

/*---------------------------------------------------------------------------
  Desc: TWLFontTable.dat ̏㏑
---------------------------------------------------------------------------*/
bool TestImport::WriteTWLFontTable()
{
    nn::Result                  nnResult;
    bool                        ret = false;
    ErrSeq_DataImport           micro  = ERR_DEFAULT;
    
    nn::fs::FileInputStream     instream;
    nn::fs::FileOutputStream    ostream;
    
    // TWLFontTable iA[JCuAROM^NANDۑj
    const char*         archiveNameTwln          = "twln:";
    const std::wstring  romPathTwlFontTableW     = L"rom:/misc/TWLFontTable.dat";
    const std::wstring  twlnPathTwlFontTableW    = L"twln:/sys/TWLFontTable.dat";
    
    // obt@
    s32     romSize     = 0;
    s32     twlnSize    = 0;
    s64     romBufSize  = sizeof(wchar_t) * 1024 * 1024;
    void*   romBuf      = reinterpret_cast<wchar_t*>(sys::Alloc(romBufSize, 32));
    s64     twlnBufSize = sizeof(wchar_t) * 1024 * 1024;
    void*   twlnBuf     = reinterpret_cast<wchar_t*>(sys::Alloc(twlnBufSize, 32));
    
    // nbVl̕
    char    romHashStringTwlFontTable[128];
    char    twlnHashStringTwlFontTable[128];

    // ʃOo
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "writing TWLFontTable...");
    UpdateLCD_Sub(WRITE_TWL_FONT_TABLE);
    
    // ㏑ TWLFontTable obt@ɓǂݍ݂܂
    instream.Initialize( romPathTwlFontTableW.c_str() );
    instream.TryRead( &romSize, romBuf, romBufSize );
    instream.Finalize();
    
    // ݑOɃnbVlvZ
    Sha1ContextGetHash( romBuf, romSize, romHashStringTwlFontTable );
    
    // Twln A[JCu}Eg
    nnResult = nn::fs::MountSpecialArchive(archiveNameTwln, nn::fs::CTR::ARCHIVE_TYPE_TWL_NAND);
    if(nnResult.IsFailure())
    {
        micro = ERR4_MOUNT_TWLN_ARCHIVE;
        goto FREE_BUFFER;
    }
    
    // t@CJ
    nnResult = ostream.TryInitialize(twlnPathTwlFontTableW.c_str(), false);
    if(nnResult.IsFailure())
    {
        micro = ERR4_OPEN_TWL_FONT_TABLE;
        goto FREE_BUFFER;
    }
    
    // t@CTCYw
    nnResult = ostream.TrySetSize(romSize);
    if(nnResult.IsFailure())
    {
        micro = ERR4_SET_SIZE_TWL_FONT_TABLE;
        goto FREE_BUFFER;
    }
    
    // t@Cւ̏
    s32 out;
    nnResult = ostream.TryWrite(&out, romBuf, romSize);
    if(nnResult.IsFailure())
    {
        micro = ERR4_WRITE_TWL_FONT_TABLE;
        goto FREE_BUFFER;
    }
    // t@CLbVfoCXɏ߂
    nnResult = ostream.TryFlush();
    if(nnResult.IsFailure())
    {
        micro = ERR4_FLUSH_TWL_FONT_TABLE;
        goto FREE_BUFFER;
    }
    
    // t@C
    ostream.Finalize();
    
    // ʃOo
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "done.\n\n");
    UpdateLCD_Sub(WRITE_TWL_FONT_TABLE);
    
    // ㏑ TWLFontTable obt@ɓǂݍ݂܂
    instream.Initialize( twlnPathTwlFontTableW.c_str() );
    instream.TryRead( &twlnSize, twlnBuf, twlnBufSize );
    instream.Finalize();
    
    // ݌̃nbVlvZ
    Sha1ContextGetHash( twlnBuf, twlnSize, twlnHashStringTwlFontTable );
    
    // ʃOo
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "comparing Hash...\n");
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_YELLOW, "rom  : %s\n", romHashStringTwlFontTable);
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_YELLOW, "twln : %s\n", twlnHashStringTwlFontTable);
    UpdateLCD_Sub(WRITE_TWL_FONT_TABLE);
    
    // r
    if( std::strcmp( romHashStringTwlFontTable, twlnHashStringTwlFontTable )!=0 )
    {
        micro = ERR4_COMPARE_HASH;
        sprintf( m_HashString, "rom  : %s\ntwln : %s\n", 
            romHashStringTwlFontTable, twlnHashStringTwlFontTable );
        goto FREE_BUFFER;
    }
    
    // ʃOo
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "match.\n\n");
    UpdateLCD_Sub(WRITE_TWL_FONT_TABLE);
    
    // Twln A[JCũA}Eg
    nnResult = nn::fs::Unmount(archiveNameTwln);
    if(nnResult.IsFailure())
    {
        micro = ERR4_UNMOUNT_TWLN_ARCHIVE;
        goto FREE_BUFFER;
    }
    
    {
        micro = PASS;
        ret = true;
    }
    
FREE_BUFFER:
    sys::Free(romBuf);
    sys::Free(twlnBuf);
    
    sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
    m_TestResult.m_Micro = (int)(micro/100);
    
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: f[^݌̃`FbN
        PDfBAɃC|[gr̃S~Ȃ
        QDC|[gReLXg݂^CgȂ
---------------------------------------------------------------------------*/
bool TestImport::WasProperlyImported( nn::fs::MediaType media, bool executeCheckNand )
{
    // vCXg[̏ꍇ͖ true Ԃ
    if( media==nn::fs::MEDIA_TYPE_SDMC && !m_PreInstall ) return true;
    
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    s32 contexts = 0;
    
    // ʕ\
    char* string = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    memset( string, '\0', sizeof(char) * 256 * 1024);
    
    switch( media )
    {
        case nn::fs::MEDIA_TYPE_NAND:   sprintf( string, "nand" );  break;
        case nn::fs::MEDIA_TYPE_SDMC:   sprintf( string, "sdmc" );  break;
        default:                        sprintf( string, "----" );  break;
    }
    
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "Check Imported(%s)...\n", string);
    UpdateLCD_Sub(WAS_PROPERLY_IMPORTED);
    
    // fBAɃC|[gr̃S~ȂmF
    // Nand ͋ÑVXeAv㏑ۂɃS~ƂĎc邽ߏO
    if( media==nn::fs::MEDIA_TYPE_SDMC
        || (media==nn::fs::MEDIA_TYPE_NAND && executeCheckNand) )
    {
        bool trash = false;
        nnResult = nn::am::NeedsCleanup( &trash, media );
        if( nnResult.IsFailure() )
        {
                 if( nnResult == nn::am::ResultInvalidEnumValue() )     { micro = ERR5_NEEDS_CLEANUP_1; goto FINALIZE; }
            else if( nnResult == nn::am::ResultMediaNotSupported() )    { micro = ERR5_NEEDS_CLEANUP_2; goto FINALIZE; }
            else if( nnResult == nn::am::ResultInternalDataCorrupted() ){ micro = ERR5_NEEDS_CLEANUP_3; goto FINALIZE; }
            else                                                        { micro = ERR_FATAL;            goto FINALIZE; }
        }
        if( trash )                                                     { micro = ERR5_GARBAGE_IS_LEFT; goto FINALIZE; }
    }
    
    // C|[gReLXg݂^CgȂmF
    {
        nnResult = nn::am::GetNumImportTitleContexts( &contexts, media );
        if( nnResult.IsFailure() )
        {
                 if( nnResult == nn::am::ResultInvalidEnumValue() )     { micro = ERR5_GET_NUM_ITC_1; goto FINALIZE; }
            else if( nnResult == nn::am::ResultMediaNotSupported() )    { micro = ERR5_GET_NUM_ITC_2; goto FINALIZE; }
            else if( nnResult == nn::am::ResultInternalDataCorrupted() ){ micro = ERR5_GET_NUM_ITC_3; goto FINALIZE; }
            else                                                        { micro = ERR_FATAL;          goto FINALIZE; }
        }
        if( contexts!=0 )                                               
        {
            micro = ERR5_EXIST_ITC;
            sprintf( m_ImportTitleContextsString, "ImportTitleContexts=%d", contexts );
            goto FINALIZE;
        }
    }
    
    // PASS
    micro = PASS;
    
FINALIZE:
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        switch( media )
        {
            case nn::fs::MEDIA_TYPE_NAND:   m_TestResult.m_Micro = 100 + (int)(micro/100);  break;
            case nn::fs::MEDIA_TYPE_SDMC:   m_TestResult.m_Micro = 200 + (int)(micro/100);  break;
            default:                        m_TestResult.m_Micro = 500 + (int)(micro/100);  break;
        }
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: Av擾
---------------------------------------------------------------------------*/
bool TestImport::GetAppInfo( nn::fs::MediaType media )
{
    // vCXg[̏ꍇ͖ true Ԃ
    if( media==nn::fs::MEDIA_TYPE_SDMC && !m_PreInstall ) return true;
    
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    // ʕ\
    char* string = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    memset( string, '\0', sizeof(char) * 256 * 1024);
    
    switch( media )
    {
        case nn::fs::MEDIA_TYPE_NAND:   sprintf( string, "nand" );  break;
        case nn::fs::MEDIA_TYPE_SDMC:   sprintf( string, "sdmc" );  break;
        default:                        sprintf( string, "----" );  break;
    }
    
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "Get AppInfo(%s)...\n", string);
    UpdateLCD_Sub(GET_APPINFO);
    
    
    s32 num = 0;
    {
        // C|[gĂvO擾
        nnResult = nn::am::GetNumPrograms( &num, media );
        if( nnResult.IsFailure() )
        {
                 if( nnResult == nn::am::ResultInvalidEnumValue() )      micro = ERR5_GET_NUM_PROGRAMS_1;
            else if( nnResult == nn::am::ResultMediaNotSupported() )     micro = ERR5_GET_NUM_PROGRAMS_2;
            else if( nnResult == nn::am::ResultInternalDataCorrupted() ) micro = ERR5_GET_NUM_PROGRAMS_3;
            else                                                         micro = ERR_FATAL;
            goto FINALIZE;
        }
        
        s32                     getNum = 0; // 擾 ProgramId ̐
        nn::ProgramId           id[num];    // ProgramId ̃Xg
        nn::am::ProgramInfo     infos[num]; // ProgramInfo ̃obt@
        
        // C|[gĂ ProgramId ̃Xg擾
        nnResult = nn::am::GetProgramList( &getNum, id, num, media );
        if( nnResult.IsFailure() )
        {
                 if( nnResult == nn::am::ResultInvalidEnumValue() )      micro = ERR5_GET_PROGRAM_LIST_1;
            else if( nnResult == nn::am::ResultMediaNotSupported() )     micro = ERR5_GET_PROGRAM_LIST_2;
            else if( nnResult == nn::am::ResultInvalidSize() )           micro = ERR5_GET_PROGRAM_LIST_3;
            else if( nnResult == nn::am::ResultInvalidAddress() )        micro = ERR5_GET_PROGRAM_LIST_4;
            else if( nnResult == nn::am::ResultInternalDataCorrupted() ) micro = ERR5_GET_PROGRAM_LIST_5;
            else                                                         micro = ERR_FATAL;
            goto FINALIZE;
        }
        
        // w肳ꂽ ProgramId ɏ]ăvOԂ
        nnResult = nn::am::GetProgramInfos( infos, media, id, num );
        if( nnResult.IsFailure() )
        {
                 if( nnResult == nn::am::ResultInvalidEnumValue() )      micro = ERR5_GET_PROGRAM_INFOS_1;
            else if( nnResult == nn::am::ResultMediaNotSupported() )     micro = ERR5_GET_PROGRAM_INFOS_2;
            else if( nnResult == nn::am::ResultInvalidSize() )           micro = ERR5_GET_PROGRAM_INFOS_3;
            else if( nnResult == nn::am::ResultNotFound() )              micro = ERR5_GET_PROGRAM_INFOS_4;
            else if( nnResult == nn::am::ResultInternalDataCorrupted() ) micro = ERR5_GET_PROGRAM_INFOS_5;
            else                                                         micro = ERR_FATAL;
            goto FINALIZE;
        }
        
        switch( media )
        {
            case nn::fs::MEDIA_TYPE_NAND: // Nand oϐɊi[
                m_NandAppInfo.media = media;
                m_NandAppInfo.num   = num;
                for( int i=0; i<m_NandAppInfo.num; i++ )
                {
                    m_NandAppInfo.infos[i]= infos[i];
                    m_NandAppInfo.programs[i]= id[i];
                }
                break;
            case nn::fs::MEDIA_TYPE_SDMC: // Sdmc oϐɊi[
                m_SdmcAppInfo.media = media;
                m_SdmcAppInfo.num   = num;
                for( int i=0; i<m_SdmcAppInfo.num; i++ )
                {
                    m_SdmcAppInfo.infos[i]= infos[i];
                    m_SdmcAppInfo.programs[i]= id[i];
                }
                break;
            default:
                break;
        }
    }
    
    // PASS
    micro = PASS;
    
FINALIZE:
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        switch( media )
        {
            case nn::fs::MEDIA_TYPE_NAND:   m_TestResult.m_Micro = 100 + (int)(micro/100);      break;
            case nn::fs::MEDIA_TYPE_SDMC:   m_TestResult.m_Micro = 200 + (int)(micro/100);      break;
            default:                        m_TestResult.m_Micro = 500 + (int)(micro/100);      break;
        }
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: f[^x[X ProgramInfo mF
---------------------------------------------------------------------------*/
bool TestImport::CompareProgramInfo( nn::fs::MediaType media )
{
    // vCXg[̏ꍇ͖ true Ԃ
    if( media==nn::fs::MEDIA_TYPE_SDMC && !m_PreInstall ) return true;
    
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    // ʕ\
    char* string = reinterpret_cast<char*>(sys::Alloc(sizeof(char) * 256 * 1024, 32));
    memset( string, '\0', sizeof(char) * 256 * 1024);
    
    switch( media )
    {
        case nn::fs::MEDIA_TYPE_NAND:   sprintf( string, "nand" );  break;
        case nn::fs::MEDIA_TYPE_SDMC:   sprintf( string, "sdmc" );  break;
        default:                        sprintf( string, "----" );  break;
    }
    
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "Compare ProgramInfo(%s)...\n", string);
    UpdateLCD_Sub(COMPARE_PROGRAM_INFO);
    
    // r
    char* cmp_to    = new char[50];
    char* cmp_from  = new char[50];
    
    // \[gΏۂƂReip
    deque<nn::am::ProgramInfo>  deq_to;
    deque<nn::am::ProgramInfo>  deq_from;
    
    // Ce[^
    deque<nn::am::ProgramInfo>::iterator it_to;
    deque<nn::am::ProgramInfo>::iterator it_from;
    
    
    // ݐ̃fBAɔrΏۂؑ
    // AvvȂꍇ͏I
    s32 num_a = 0;
    s32 num_b = 0;
    switch( media )
    {
        case nn::fs::MEDIA_TYPE_NAND:   num_a = m_NandAppInfo.num;  num_b = m_Rom2NandAppInfo.num;  break;
        case nn::fs::MEDIA_TYPE_SDMC:   num_a = m_SdmcAppInfo.num;  num_b = m_Rom2SdmcAppInfo.num;  break;
    }
    
    // r
    if( num_a != num_b )
    {
        micro = ERR6_COMPARE_PROGRAM_INFOS_NUM;
        sprintf( m_CompareProgramInfosNumString, "%s:%d != rom:%d", string, num_a, num_b );
        goto FINALIZE;
    }
    
    switch( media )
    {
        case nn::fs::MEDIA_TYPE_NAND: 
            for( int i=0; i<m_NandAppInfo.num; i++ )
            {
                deq_to.push_back( m_NandAppInfo.infos[i] );
                deq_from.push_back( m_Rom2NandAppInfo.infos[i] );
            }
            break;
        case nn::fs::MEDIA_TYPE_SDMC: 
            for( int i=0; i<m_SdmcAppInfo.num; i++ )
            {
                deq_to.push_back( m_SdmcAppInfo.infos[i] );
                deq_from.push_back( m_Rom2SdmcAppInfo.infos[i] );
            }
            break;
    }
    
    // \[g
    std::stable_sort( deq_to.begin(),    deq_to.end(),    compare_id );
    std::stable_sort( deq_from.begin(),  deq_from.end(),  compare_id );
    
    ///////////////////////////////////////////////////////////////////////////////////////////
    if( DEBUG_MODE==1 )
    {
        const wchar_t* pathName_to;
        const wchar_t* pathName_from;
        
        switch( media )
        {
            case nn::fs::MEDIA_TYPE_NAND: 
                pathName_to   = L"sdmc:/uji/SortTitleInfos_Nand.txt";
                pathName_from = L"sdmc:/uji/SortTitleInfos_Rom2Nand.txt";
                break;
            case nn::fs::MEDIA_TYPE_SDMC: 
                pathName_to   = L"sdmc:/uji/SortTitleInfos_Sdmc.txt";
                pathName_from = L"sdmc:/uji/SortTitleInfos_Rom2Sdmc.txt";
                break;
        }
        
        nn::fs::MountSdmc();
        nn::fs::TryCreateDirectory( L"sdmc:/uji" );
        
        CreateTitleListTextToSdmcByDeque( deq_to,   pathName_to   );
        CreateTitleListTextToSdmcByDeque( deq_from, pathName_from );
        
        nn::fs::Unmount( "sdmc:" );
    }
    ///////////////////////////////////////////////////////////////////////////////////////////
    
    //
    // ʂo
    it_to    = deq_to.begin();
    it_from  = deq_from.begin();
    
    while( it_to != deq_to.end() )
    {
        sprintf( cmp_to,    "%016llx %5d", (*it_to).id,    (*it_to).version );
        sprintf( cmp_from,  "%016llx %5d", (*it_from).id,  (*it_from).version  );
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "%s - %s", cmp_to, cmp_from);
        UpdateLCD_Sub(COMPARE_PROGRAM_INFO);
        DEBUG_NN_LOG("%s - %s", cmp_to, cmp_from);
        
        // r
        if( std::strcmp(cmp_to, cmp_from)!=0 )
        {
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED,   " X\n");
            UpdateLCD_Sub(COMPARE_PROGRAM_INFO);
            DEBUG_NN_LOG(" X\n");
            
            ///////////////////////////////////////////////////////////////////////////////////////////
            if( DEBUG_MODE==1 )
            {
                LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "---- WAIT INPUT (A) ----\n");
                UpdateLCD_Sub(COMPARE_PROGRAM_INFO);
                WaitInputA();
            }
            else
            {
                // vȂꍇ͏I
                micro   = ERR6_COMPARE_PROGRAM_INFOS_ID_AND_VER;
                sprintf( m_CompareProgramInfosIdAndVerString, "%s != %s", cmp_to, cmp_from );
                goto FINALIZE;
            }
            ///////////////////////////////////////////////////////////////////////////////////////////
        }
        else
        {
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, " O\n");
            UpdateLCD_Sub(COMPARE_PROGRAM_INFO);
            DEBUG_NN_LOG(" O\n");
        }
        ++it_to;
        ++it_from;
    }
    
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "done.\n");
    UpdateLCD_Sub(COMPARE_PROGRAM_INFO);
    
    // PASS
    micro = PASS;
    
FINALIZE:
    delete[] cmp_to;
    delete[] cmp_from;
    
    if( micro!=PASS )
    {
        sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        switch( media )
        {
            case nn::fs::MEDIA_TYPE_NAND:   m_TestResult.m_Micro = 100 + (int)(micro/100);     break;
            case nn::fs::MEDIA_TYPE_SDMC:   m_TestResult.m_Micro = 200 + (int)(micro/100);     break;
            default:                        m_TestResult.m_Micro = 500 + (int)(micro/100);     break;
        }
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: `Pbgf[^x[X̊mF
        EGetNumTickets  GetNumPrograms ԂvOlł
        EGetTicketList  GetProgramList ԂvOXglł
---------------------------------------------------------------------------*/
bool TestImport::CompareTicketDatabase()
{
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    s32 num_a = 0;
    s32 num_b = ((m_PreInstall) ? m_NandAppInfo.num + m_SdmcAppInfo.num : m_NandAppInfo.num);
    
    char* cmp_a  = new char[50];    // _a: GetProgramList ԂvOIDXg
    char* cmp_b  = new char[50];    // _b: m_NandAppInfo.programs + m_SdmcAppInfo.programs
    
    // ʕ\
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "Compare TicketDatabase...\n");
    UpdateLCD_Sub(COMPARE_TICKET_DB);
    
    // 
    // GetNumTickets Ԃ  H
    // GetNumPrograms Ԃ  m_NandAppInfo.num + m_SdmcAppInfo.num
    {
        nnResult = nn::am::GetNumTickets(&num_a);
        if(nnResult.IsFailure())
        {
            if( nnResult == nn::am::ResultInternalDataCorrupted() ) micro = ERR6_GET_NUM_TICKETS_1;
            else                                                    micro = ERR_FATAL;
            goto FINALIZE;
        }
        
        if( num_a != num_b )
        {
            DEBUG_NN_LOG("GetNumTickets != GetNumPrograms");
            micro = ERR6_COMPARE_NUM_OF_TICKETS_AND_PROGRAMS;
            sprintf( m_CompareNumOfTicketsAndProgramsString, "%d != %d", num_a, num_b );
            goto FINALIZE;
        }
    }
    // 
    // GetTicketList ԂvOIDXg  H
    // GetProgramList ԂvOIDXg  m_NandAppInfo.programs + m_SdmcAppInfo.programs
    {
        s32 getNum = 0;
        s32 index  = 0;
        nn::ProgramId id[num_a];
        
        deque<nn::ProgramId>  deq_a;
        deque<nn::ProgramId>  deq_b;
        deque<nn::ProgramId>::iterator it_a;
        deque<nn::ProgramId>::iterator it_b;
        
        nnResult = nn::am::GetTicketList(&getNum, id, num_a, index);
        if(nnResult.IsFailure())
        {
                 if( nnResult == nn::am::ResultInvalidSize() )           micro = ERR6_GET_TICKET_LIST_1;
            else if( nnResult == nn::am::ResultOutOfRange() )            micro = ERR6_GET_TICKET_LIST_2;
            else if( nnResult == nn::am::ResultInternalDataCorrupted() ) micro = ERR6_GET_TICKET_LIST_3;
            else                                                         micro = ERR_FATAL;
            goto FINALIZE;
        }
        
        for( int i=0; i<num_a; i++ )            { deq_a.push_back( id[i] ); }
        for( int i=0; i<m_NandAppInfo.num; i++ ){ deq_b.push_back( m_NandAppInfo.programs[i] ); }
        for( int i=0; i<m_SdmcAppInfo.num; i++ ){ deq_b.push_back( m_SdmcAppInfo.programs[i] ); }
        
        std::stable_sort( deq_a.begin(),  deq_a.end(),  compare_id2 );
        std::stable_sort( deq_b.begin(),  deq_b.end(),  compare_id2 );
        
        it_a = deq_a.begin();
        it_b = deq_b.begin();
        
        
        int j = 0;
        while( it_a != deq_a.end() )
        {
            sprintf( cmp_a, "%016llx", (*it_a) );
            sprintf( cmp_b, "%016llx", (*it_b) );
            LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "%s - %s", cmp_a, cmp_b);
            UpdateLCD_Sub(COMPARE_TICKET_DB);
            DEBUG_NN_LOG("%03d:  id = %s  mergeId = %s", j, cmp_a, cmp_b);
            
            // r
            if( std::strcmp(cmp_a, cmp_b)!=0 )
            {
                LogPrintf(m_LogWindow, sys::ATTR_COLOR_RED, " X\n");
                UpdateLCD_Sub(COMPARE_TICKET_DB);
                DEBUG_NN_LOG(" X\n");
                
                micro = ERR6_COMPARE_ID_OF_TICKETS_AND_PROGRAMS;
                sprintf( m_CompareIdOfTicketsAndProgramsString, "%s != %s", cmp_a, cmp_b );
                goto FINALIZE;
            }
            else
            {
                LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, " O\n");
                UpdateLCD_Sub(COMPARE_TICKET_DB);
                DEBUG_NN_LOG(" O\n");
            }
            ++it_a;
            ++it_b;
            j++;
        }
    }
    
    // PASS
    micro = PASS;
    
FINALIZE:
    delete[] cmp_a;
    delete[] cmp_b;
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: rom:/misc/TitleList.txt  nand:/title/ ̃t@Cr
---------------------------------------------------------------------------*/
bool TestImport::CompareTitleList()
{
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    const wchar_t* romPath  = L"rom:/misc/TitleList.txt";
    const wchar_t* nandPath = L"nand:/title/";
    
    s64    fileSize = 0;
    s32    readSize = 0;
    
    size_t romBufSize = sizeof(char) * 256 * 1024;
    char*  romBuf = reinterpret_cast<char*>(sys::Alloc(romBufSize, 32));
    memset( romBuf, '\0', romBufSize);
    
    size_t nandBufSize = sizeof(char) * 256 * 1024;
    char*  nandBuf = reinterpret_cast<char*>(sys::Alloc(nandBufSize, 32));
    memset( nandBuf, '\0', nandBufSize);
    
    // ʕ\
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "Compare TitleList...\n");
    UpdateLCD_Sub(COMPARE_TITLE_LIST);
    
    /*
        rom:/misc/TitleList.txt
    */
    {
        nn::fs::FileInputStream infile;
        nnResult = infile.TryInitialize( romPath );
        if(nnResult.IsFailure())
        {
            // TitleList.txt A
            // NeBJȃ^C~OŃJ[hƐڐGsǂNꍇɔ
            micro =  ERR7_OPEN_TITLE_LIST;
            goto FINALIZE;
        }
        
        nnResult = infile.TryGetSize( &fileSize );
        if(nnResult.IsFailure())
        {
            micro =  ERR7_GET_SIZE_TITLE_LIST;
            goto FINALIZE;
        }
        
        nnResult = infile.TryRead( &readSize, romBuf, fileSize );
        if(nnResult.IsFailure())
        {
            micro =  ERR7_READ_TITLE_LIST;
            goto FINALIZE;
        }
        
        infile.Finalize();
    }
    
    /*
        nand:/title/
    */
    {
        ReadNandAndOutputBuffer( nandBuf, nandPath );
    }
    
    ///////////////////////////////////////////////////////////////////////////////////////////
    if( DEBUG_MODE==1 )
    {
        int i = 0;
        while(i < strlen(romBuf))
        {
            DEBUG_NN_LOG("%s", romBuf+i);
            i = i + (NN_DBG_PRINTF_BUFFER_LENGTH - 1);
        }
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "File    size=%lld\n", fileSize);
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "RomBuf  size=%d\n", strlen(romBuf));
        UpdateLCD_Sub(COMPARE_TITLE_LIST);
        
        i = 0;
        while(i < strlen(nandBuf))
        {
            DEBUG_NN_LOG("%s", nandBuf+i);
            i = i + (NN_DBG_PRINTF_BUFFER_LENGTH - 1);
        }
        
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "NandBuf size=%d\n\n", strlen(nandBuf));
        UpdateLCD_Sub(COMPARE_TITLE_LIST);
        
        // SD J[hɏo
        nn::fs::MountSdmc();
        nn::fs::TryCreateDirectory( L"sdmc:/uji" );
        CreateTextToSdmcByArray( romBuf,   L"sdmc:/uji/TitleList_Rom.txt" );
        CreateTextToSdmcByArray( nandBuf,  L"sdmc:/uji/TitleList_Nand.txt" );
        nn::fs::Unmount( "sdmc:" );
        
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "---- WAIT INPUT (A) ----\n");
        UpdateLCD_Sub(COMPARE_TITLE_LIST);
        WaitInputA();
    }
    ///////////////////////////////////////////////////////////////////////////////////////////
    
    // r
    if( std::strcmp( nandBuf, romBuf )!=0 )
    {
        micro =  ERR7_COMPARE_TITLE_LIST;
        goto FINALIZE;
    }
    
    sys::Free(nandBuf);
    nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(16));
    
    // PASS
    micro = PASS;
    
FINALIZE:
    sys::Free(romBuf);
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: rom:/misc/PreInstallTitleList.txt 
        sdmc:/Nintendo 3DS/xxx/xxx/title/ ̃t@Cr
---------------------------------------------------------------------------*/
bool TestImport::ComparePreInstallTitleList()
{
    // vCXg[̏ꍇ͖ true Ԃ
    // i{I TestList  false ݒj
    if( !m_PreInstall ) return true;
    
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    const wchar_t* romPath  = L"rom:/misc/PreInstallTitleList.txt";
    const wchar_t* sdmcPath = L"sdmc:/Nintendo 3DS/";
    
    s64    fileSize = 0;
    s32    readSize = 0;
    
    size_t romBufSize = sizeof(char) * 256 * 1024;
    char*  romBuf = reinterpret_cast<char*>(sys::Alloc(romBufSize, 32));
    memset( romBuf, '\0', romBufSize);
    
    size_t sdmcBufSize = sizeof(char) * 256 * 1024;
    char*  sdmcBuf = reinterpret_cast<char*>(sys::Alloc(sdmcBufSize, 32));
    memset( sdmcBuf, '\0', sdmcBufSize);
    
    // ʕ\
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "Compare PreInstallTitleList...\n");
    UpdateLCD_Sub(COMPARE_TITLE_LIST);
    
    /*
        rom:/misc/PreInstallTitleList.txt
    */
    {
        nn::fs::FileInputStream infile;
        nnResult = infile.TryInitialize( romPath );
        if(nnResult.IsFailure())
        {
            // TitleList.txt A
            // NeBJȃ^C~OŃJ[hƐڐGsǂNꍇɔ
            micro =  ERR7_OPEN_PREINSTALL_TITLE_LIST;
            goto FINALIZE;
        }
        
        nnResult = infile.TryGetSize( &fileSize );
        if(nnResult.IsFailure())
        {
            micro =  ERR7_GET_SIZE_PREINSTALL_TITLE_LIST;
            goto FINALIZE;
        }
        
        nnResult = infile.TryRead( &readSize, romBuf, fileSize );
        if(nnResult.IsFailure())
        {
            micro =  ERR7_READ_PREINSTALL_TITLE_LIST;
            goto FINALIZE;
        }
        
        infile.Finalize();
    }
    
    /*
        sdmc:/Nintendo 3DS/xxx/xxx/title/
    */
    {
        nn::fs::MountSdmc();
        ReadSdmcAndOutputBufferEx( sdmcBuf, sdmcPath );
        nn::fs::Unmount( "sdmc:" );
    }
    
    ///////////////////////////////////////////////////////////////////////////////////////////
    if( DEBUG_MODE==1 )
    {
        int i = 0;
        while(i < strlen(romBuf))
        {
            DEBUG_NN_LOG("%s", romBuf+i);
            i = i + (NN_DBG_PRINTF_BUFFER_LENGTH - 1);
        }
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "File    size=%lld\n", fileSize);
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "RomBuf  size=%d\n", strlen(romBuf));
        UpdateLCD_Sub(COMPARE_TITLE_LIST);
        
        i = 0;
        while(i < strlen(sdmcBuf))
        {
            DEBUG_NN_LOG("%s", sdmcBuf+i);
            i = i + (NN_DBG_PRINTF_BUFFER_LENGTH - 1);
        }
        
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_WHITE, "SdmcBuf size=%d\n\n", strlen(sdmcBuf));
        UpdateLCD_Sub(COMPARE_TITLE_LIST);
        
        // SD J[hɏo
        nn::fs::MountSdmc();
        nn::fs::TryCreateDirectory( L"sdmc:/uji" );
        CreateTextToSdmcByArray( romBuf,   L"sdmc:/uji/PreInstallTitleList_Rom.txt" );
        CreateTextToSdmcByArray( sdmcBuf,  L"sdmc:/uji/PreInstallTitleList_Sdmc.txt" );
        nn::fs::Unmount( "sdmc:" );
        
        LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "---- WAIT INPUT (A) ----\n");
        UpdateLCD_Sub(COMPARE_TITLE_LIST);
        WaitInputA();
    }
    ///////////////////////////////////////////////////////////////////////////////////////////
    
    // r
    if( std::strcmp( sdmcBuf, romBuf )!=0 )
    {
        micro =  ERR7_COMPARE_PREINSTALL_TITLE_LIST;
        goto FINALIZE;
    }
    
    sys::Free(sdmcBuf);
    nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(16));
    
    // PASS
    micro = PASS;
    
FINALIZE:
    sys::Free(romBuf);
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: rom:/misc/DefaultTitleList.txt  nand:/title/ ̃t@Cr
---------------------------------------------------------------------------*/
bool TestImport::CompareDefaultTitleList()
{
    nn::Result nnResult;
    bool ret = false;
    ErrSeq_DataImport  micro  = ERR_DEFAULT;
    
    const wchar_t* romPath  = L"rom:/misc/DefaultTitleList.txt";
    const wchar_t* nandPath = L"nand:/title/";
    
    s64    fileSize = 0;
    s32    readSize = 0;
    
    size_t romBufSize = sizeof(char) * 256 * 1024;
    char*  romBuf = reinterpret_cast<char*>(sys::Alloc(romBufSize, 32));
    memset( romBuf, '\0', romBufSize);
    
    size_t nandBufSize = sizeof(char) * 256 * 1024;
    char*  nandBuf = reinterpret_cast<char*>(sys::Alloc(nandBufSize, 32));
    memset( nandBuf, '\0', nandBufSize);
    
    // ʕ\
    LogPrintf(m_LogWindow, sys::ATTR_COLOR_GREEN, "Compare DefaultTitleList...\n");
    UpdateLCD_Sub(COMPARE_TITLE_LIST);
    
    /*
        rom:/misc/DefaultTitleList.txt
    */
    {
        nn::fs::FileInputStream infile;
        nnResult = infile.TryInitialize( romPath );
        if(nnResult.IsFailure())
        {
            // TitleList.txt A
            // NeBJȃ^C~OŃJ[hƐڐGsǂNꍇɔ
            micro =  ERR8_OPEN_DEFAULT_TITLE_LIST;
            goto FINALIZE;
        }
        
        nnResult = infile.TryGetSize( &fileSize );
        if(nnResult.IsFailure())
        {
            micro =  ERR8_GET_SIZE_DEFAULT_TITLE_LIST;
            goto FINALIZE;
        }
        
        nnResult = infile.TryRead( &readSize, romBuf, fileSize );
        if(nnResult.IsFailure())
        {
            micro =  ERR8_READ_DEFAULT_TITLE_LIST;
            goto FINALIZE;
        }
        
        infile.Finalize();
    }
    
    /*
        nand:/title/
    */
    {
        ReadNandAndOutputBuffer( nandBuf, nandPath );
    }
    
    // r
    if( std::strcmp( nandBuf, romBuf )!=0 )
    {
        micro =  ERR8_COMPARE_DEFAULT_TITLE_LIST;
        goto FINALIZE;
    }
    
    sys::Free(nandBuf);
    nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(16));
    
    // PASS
    micro = PASS;
    
FINALIZE:
    sys::Free(romBuf);
    
    if( micro!=PASS )
    {
        if(micro != ERR_FATAL)  sprintf( m_TestResult.m_String, OutputErrorString( micro ) );
        else                    sprintf( m_TestResult.m_String, OutputErrorString( micro, nnResult ) );
        m_TestResult.m_Micro = (int)(micro/100);
        ret = false;
    }
    else
    {
        ret = true;
    }
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: O
---------------------------------------------------------------------------*/
bool TestImport::WriteProductionLog( const char* result )
{
    uji::seq::LogResult  lr;
    uji::seq::Config  *c = new uji::seq::Config;
    uji::seq::ProductionLog* pdl = new uji::seq::ProductionLog();
    
    pdl->Initialize();
    lr = pdl->Add_1Line( c->Get().TestMode,
                    g_TestListSet[c->GetAlternative().TestList].m_Name,
                    result,
                    VERSION_STRING,
                    "",
                    "",
                    "",
                    m_TestResult.m_String);
    if (lr == uji::seq::LogResultCode::SUCCESS)
    {
        m_TestResult.m_Result = true;
    }
    else
    {
        m_TestResult.m_Result = false;
    }
    pdl->Finalize();
    
    delete pdl;
    delete c;
    
    return m_TestResult.m_Result;
}
bool TestImport::WriteProductionLog( const char* name, const char* state )
{
    bool ret = false;
    
    uji::seq::LogResult  lr;
    uji::seq::Config  *c = new uji::seq::Config;
    uji::seq::ProductionLog* pdl = new uji::seq::ProductionLog();
    
    char resultMessage[80];
    FormToLogMessage( resultMessage, m_TestResult.m_String );
    
    pdl->Initialize();
    lr = pdl->Add_1Line( c->Get().TestMode, name, state, VERSION_STRING, "", "", "", resultMessage);
    if (lr == uji::seq::LogResultCode::SUCCESS) ret = true;
    else                                        ret = false;
    pdl->Finalize();
    
    delete pdl;
    delete c;
    
    return ret;
}
bool TestImport::WriteProductionLog( const char* name, const char* state, const char* msg )
{
    bool ret = false;
    
    uji::seq::LogResult  lr;
    uji::seq::Config  *c = new uji::seq::Config;
    uji::seq::ProductionLog* pdl = new uji::seq::ProductionLog();
    
    pdl->Initialize();
    lr = pdl->Add_1Line( c->Get().TestMode, name, state, VERSION_STRING, "", "", "", msg);
    if (lr == uji::seq::LogResultCode::SUCCESS) ret = true;
    else                                        ret = false;
    pdl->Finalize();
    
    delete pdl;
    delete c;
    
    return ret;
}

/*---------------------------------------------------------------------------
  Desc: bZ[Wv_NVOpɐ
---------------------------------------------------------------------------*/
void TestImport::FormToLogMessage( char *dst, const char *src )
{
    const int MAX_LENGTH = 80;

    //ő咷Ƀg~O
    strncpy( dst, src, MAX_LENGTH );
    if( strlen(src) > MAX_LENGTH ) dst[MAX_LENGTH]='\0';

    //\n$ɕϊA,Xy[Xɕϊ
    for( int i=0 ; dst[i]!='\0' ; i++ )
    {
        if( dst[i]=='\n' ) dst[i]='$';
        if( dst[i]==','  ) dst[i]=' ';
    }
}

/*---------------------------------------------------------------------------
  Desc: KHSMT[o[ƒʐM̃O`FbN(V[PXON/OFFύXs)
---------------------------------------------------------------------------*/
bool TestImport::CheckHsmLog()
{
    bool ret = true;
    
    /*ŐVƍŐV2Ԗڂ̃O擾*/
    uji::seq::ProductionLog *pl = new uji::seq::ProductionLog;
    pl->Initialize();
    int latestId = pl->Get_LogEntryNums() - 1;
    int secondLatestId = latestId - 1;
    if( latestId < 0 || secondLatestId < 0)
    {
        ret = false;
    }

    if (ret)
    {
        /*ŐVDataList,Hsm OKO΁AHSMʐMɍsĂ
          SendDataListsȂꍇlAŐVHsm OKłƂ*/
        uji::seq::LogEntry latest = pl->Get_LogEntry(latestId);
        uji::seq::LogEntry secondLatest = pl->Get_LogEntry(secondLatestId);
        if( strcmp(latest.testerName, "Hsm") == 0 && strcmp(latest.state, "OK") == 0 )
        {
            ret = true;
        }
        else if( strcmp(secondLatest.testerName, "Hsm") == 0 && strcmp(secondLatest.state, "OK") == 0
                 && strcmp(latest.testerName, "DataList") == 0 && strcmp(latest.state, "OK") == 0 )
        {
            ret = true;
        }
        else
        {
            ret = false;
        }
    }
    
    pl->Finalize();
    delete(pl);
    return ret;
}
    
//===========================================================================
//===========================================================================
//                             rhƗp
//===========================================================================
//===========================================================================
/*---------------------------------------------------------------------------
  Desc: ]pʍXV
---------------------------------------------------------------------------*/
void TestImport::UpdateDisplay( const char* Msg )
{
    gfx->SetFixedWidthFont(10);
    Menu::m_SubWindow->Printf( Msg );
    
    sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();
    sys::Menu::m_WindowManager.Update();
    sys::Menu::m_WindowManager.UpdatePad(sys::Pad());
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();        
    sys::Menu::m_WindowManager.DrawDisplay1();
    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
}

/*---------------------------------------------------------------------------
  Desc: TitleList.txt 쐬܂i SD J[h}EgKv܂j

  Args: str      ރf[^

  Rtns: Ȃ
---------------------------------------------------------------------------*/
void TestImport::CreateTitleListTextToSdmc( const char *str )
{
    nn::fs::FileOutputStream ofile;
    nn::fs::TryCreateDirectory( L"sdmc:/uji" );
    
    ofile.Initialize( L"sdmc:/uji/TitleList.txt", true );
    ofile.Seek( ofile.GetSize(), nn::fs::POSITION_BASE_BEGIN );
    ofile.Write( str, strlen(str) );
    ofile.Finalize();
}

static s32  s_totalNum  = 0;
/*---------------------------------------------------------------------------
  Desc: w肵NANDǂݍ sdmc:/uji/TitleList.txt ɏo͂֐

  Args: path      ΏۂNANDt@C̃pX

  Rtns: Ȃ
 ---------------------------------------------------------------------------*/
void TestImport::ReadNandAndOutputCheckTxt( const wchar_t* path )
{
    char output[255] = "";
    
    // fBNgGg
    nn::fs::DirectoryEntry* dirEntry = new nn::fs::DirectoryEntry[ 255 ];
    // fBNgGg
    s32  entryNum  = 0;
    
    if( ReadDirectory( dirEntry, &entryNum, path ) )
    {
        if( entryNum!=0 )
        {
            // fBNg̒gꍇ
            for( int i=0; i<entryNum; i++)
            {
                wchar_t* currentPath = new wchar_t[ 255 + 1 ];
                std::wcscpy( currentPath, path );
                std::wcscat( currentPath, dirEntry[i].entryName );
                if( dirEntry[i].attributes.isDirectory )
                {
                    // ΏۂfBNg̏ꍇ͍ċA
                    std::wcscat( currentPath, L"/" );
                    ReadNandAndOutputCheckTxt( currentPath );
                }
                else
                {
                    // Ώۂt@C̏ꍇ̓Oo
                    sprintf( output, "%ls, %lld,\n", currentPath, dirEntry[i].entrySize);
                    DEBUG_NN_LOG( " -%03d: %s", s_totalNum, output );
                    UpdateDisplay( output );
                    CreateTitleListTextToSdmc( output );
                    ++s_totalNum;
                }
                delete[] currentPath;
            }
        }
        else
        {
                    // fBNg̏ꍇ
                    sprintf( output, "%ls, %d,\n", path, 0);
                    DEBUG_NN_LOG( " -%03d: %s", s_totalNum, output );
                    UpdateDisplay( output );
                    CreateTitleListTextToSdmc( output );
                    ++s_totalNum;
        }
    }
    delete[] dirEntry;
}

/*---------------------------------------------------------------------------
  Desc: w肵NANDǂݍŃpXƃTCYobt@ɏo͂֐

  Args: buf       o͂obt@
        path      ΏۂNANDt@C̃pX

  Rtns: Ȃ
 ---------------------------------------------------------------------------*/
void TestImport::ReadNandAndOutputBuffer( char* buf, const wchar_t* path )
{
    char output[255] = "";
    
    // fBNgGg
    nn::fs::DirectoryEntry* dirEntry = new nn::fs::DirectoryEntry[ 255 ];
    // fBNgGg
    s32  entryNum  = 0;
    
    if( ReadDirectory( dirEntry, &entryNum, path ) )
    {
        if( entryNum!=0 )
        {
            // fBNg̒gꍇ
            for( int i=0; i<entryNum; i++)
            {
                wchar_t* currentPath = new wchar_t[ 255 + 1 ];
                std::wcscpy( currentPath, path );
                std::wcscat( currentPath, dirEntry[i].entryName );
                if( dirEntry[i].attributes.isDirectory )
                {
                    // ΏۂfBNg̏ꍇ͍ċA
                    std::wcscat( currentPath, L"/" );
                    ReadNandAndOutputBuffer( buf, currentPath );
                }
                else
                {
                    // Ώۂt@C̏ꍇ̓Oo
                    sprintf( output, "%ls, %lld,\n", currentPath, dirEntry[i].entrySize);
                    strcat( buf, output );
                    DEBUG_NN_LOG( " -%s", output );
                }
                delete[] currentPath;
            }
        }
        else
        {
                    // fBNg̏ꍇ
                    sprintf( output, "%ls, %d,\n", path, 0);
                    strcat( buf, output );
                    DEBUG_NN_LOG( " -%s", output );
        }
    }
    delete[] dirEntry;
}

/*---------------------------------------------------------------------------
  Desc: PreInstallTitleList.txt 쐬܂
        i SD J[h}EgKv܂j

  Args: str      ރf[^

  Rtns: Ȃ
---------------------------------------------------------------------------*/
void TestImport::CreatePreInstallTitleListTextToSdmc( const char *str )
{
    nn::fs::FileOutputStream ofile;
    nn::fs::TryCreateDirectory( L"sdmc:/uji" );
    
    ofile.Initialize( L"sdmc:/uji/PreInstallTitleList.txt", true );
    ofile.Seek( ofile.GetSize(), nn::fs::POSITION_BASE_BEGIN );
    ofile.Write( str, strlen(str) );
    ofile.Finalize();
}

/*---------------------------------------------------------------------------
  Desc: w肵SDMCǂݍŃfBNgQ
        /title tH_T

  Args: path      ΏۂSDMCt@C̃pX

  Rtns: Ȃ
 ---------------------------------------------------------------------------*/
void TestImport::ReadSdmcAndOutputCheckTxtEx( const wchar_t* path )
{
    char output[255] = "";
    
    // fBNgGg
    nn::fs::DirectoryEntry* dirEntry1 = new nn::fs::DirectoryEntry[ 255 ];
    nn::fs::DirectoryEntry* dirEntry2 = new nn::fs::DirectoryEntry[ 255 ];
    // fBNgGg
    s32  entryNum  = 0;
    
    if( ReadDirectory( dirEntry1, &entryNum, path ) )
    {
        wchar_t* currentPath = new wchar_t[ 255 + 1 ];
        
        std::wcscpy( currentPath, path );
        std::wcscat( currentPath, dirEntry1[0].entryName );
        std::wcscat( currentPath, L"/" );
        ReadDirectory( dirEntry2, &entryNum, currentPath );
        std::wcscat( currentPath, dirEntry2[0].entryName );
        std::wcscat( currentPath, L"/title/" );
        ReadSdmcAndOutputCheckTxt( currentPath );
        
        delete[] currentPath;
    }
    else
    {
        // w肵fBNg݂Ȃꍇ
        sprintf( output, "[%ls] not Exist.\n", path);
        UpdateDisplay( output );
    }
    delete[] dirEntry2;
    delete[] dirEntry1;
}

/*---------------------------------------------------------------------------
  Desc: w肵SDMCǂݍ /title ȉ̃AhX
        sdmc:/uji/PreInstallTitleList.txt ɏo͂֐

  Args: path      ΏۂSDMCt@C̃pX

  Rtns: Ȃ
 ---------------------------------------------------------------------------*/
void TestImport::ReadSdmcAndOutputCheckTxt( const wchar_t* path )
{
    char output[255] = "";
    
    // fBNgGg
    nn::fs::DirectoryEntry* dirEntry = new nn::fs::DirectoryEntry[ 255 ];
    // fBNgGg
    s32  entryNum  = 0;
    
    if( ReadDirectory( dirEntry, &entryNum, path ) )
    {
        if( entryNum!=0 )
        {
            // fBNg̒gꍇ
            for( int i=0; i<entryNum; i++)
            {
                wchar_t* currentPath = new wchar_t[ 255 + 1 ];
                std::wcscpy( currentPath, path );
                std::wcscat( currentPath, dirEntry[i].entryName );
                if( dirEntry[i].attributes.isDirectory )
                {
                    // ΏۂfBNg̏ꍇ͍ċA
                    std::wcscat( currentPath, L"/" );
                    ReadSdmcAndOutputCheckTxt( currentPath );
                }
                else
                {
                    // Ώۂt@C̏ꍇ̓Oo
                    sprintf( output, "%ls, %lld,\n", currentPath, dirEntry[i].entrySize);
                    strcpy( output, strstr( output, "/title" ) );
                    UpdateDisplay( output );
                    CreatePreInstallTitleListTextToSdmc( output );
                }
                delete[] currentPath;
            }
        }
        else
        {
                    // fBNg̏ꍇ
                    sprintf( output, "%ls, %d,\n", path, 0);
                    strcpy( output, strstr( output, "/title" ) );
                    UpdateDisplay( output );
                    CreatePreInstallTitleListTextToSdmc( output );
        }
    }
    delete[] dirEntry;
}

/*---------------------------------------------------------------------------
  Desc: w肵SDMCǂݍŃfBNgQ
        /title tH_T

  Args: path      ΏۂSDMCt@C̃pX

  Rtns: Ȃ
 ---------------------------------------------------------------------------*/
void TestImport::ReadSdmcAndOutputBufferEx( char* buf, const wchar_t* path )
{
    s32  entryNum  = 0;
    
    nn::fs::DirectoryEntry* dirEntry1 = new nn::fs::DirectoryEntry[ 255 ];
    nn::fs::DirectoryEntry* dirEntry2 = new nn::fs::DirectoryEntry[ 255 ];
    wchar_t* currentPath = new wchar_t[ 255 + 1 ];
    
    ReadDirectory( dirEntry1, &entryNum, path );
    std::wcscpy( currentPath, path );
    std::wcscat( currentPath, dirEntry1[0].entryName );
    std::wcscat( currentPath, L"/" );
    ReadDirectory( dirEntry2, &entryNum, currentPath );
    std::wcscat( currentPath, dirEntry2[0].entryName );
    std::wcscat( currentPath, L"/title/" );
    ReadSdmcAndOutputBuffer( buf, currentPath );
    
    delete[] currentPath;
    delete[] dirEntry2;
    delete[] dirEntry1;
}

/*---------------------------------------------------------------------------
  Desc: w肵SDMCǂݍ /title ȉ
        AhXƃTCYobt@ɏo͂֐

  Args: buf       o͂obt@
        path      ΏۂSDMCt@C̃pX

  Rtns: Ȃ
 ---------------------------------------------------------------------------*/
void TestImport::ReadSdmcAndOutputBuffer( char* buf, const wchar_t* path )
{
    char output[255] = "";
    
    // fBNgGg
    nn::fs::DirectoryEntry* dirEntry = new nn::fs::DirectoryEntry[ 255 ];
    // fBNgGg
    s32  entryNum  = 0;
    
    if( ReadDirectory( dirEntry, &entryNum, path ) )
    {
        if( entryNum!=0 )
        {
            // fBNg̒gꍇ
            for( int i=0; i<entryNum; i++)
            {
                wchar_t* currentPath = new wchar_t[ 255 + 1 ];
                std::wcscpy( currentPath, path );
                std::wcscat( currentPath, dirEntry[i].entryName );
                if( dirEntry[i].attributes.isDirectory )
                {
                    // ΏۂfBNg̏ꍇ͍ċA
                    std::wcscat( currentPath, L"/" );
                    ReadSdmcAndOutputBuffer( buf, currentPath );
                }
                else
                {
                    // Ώۂt@C̏ꍇ̓Oo
                    sprintf( output, "%ls, %lld,\n", currentPath, dirEntry[i].entrySize);
                    strcpy( output, strstr( output, "/title" ) );
                    strcat( buf, output );
                    DEBUG_NN_LOG( " -%s", output );
                }
                delete[] currentPath;
            }
        }
        else
        {
                    // fBNg̏ꍇ
                    sprintf( output, "%ls, %d,\n", path, 0);
                    strcpy( output, strstr( output, "/title" ) );
                    strcat( buf, output );
                    DEBUG_NN_LOG( " -%s", output );
        }
    }
    delete[] dirEntry;
}

} // namespace import
} // namespace eva
} // namespac

