/*--------------------------------------------------------------------------*
 Project:
 File: gyro_ShockTester.cpp


*--------------------------------------------------------------------------*/
#include "gyro_ShockTester.h"
#include "gyro_Logger.h"
#include "../mcu/McuInitializer.h"
#include "../sys/sys_SoundPlayer.h"
#include "../Sound/SoundInitializer.h"
#include "../seq/Config.h"
#include "../seq/TestIdDefine.h"
#include <nn/fs/CTR/MPCore/fs_ApiForHwCheck.h>
#include <nn/srv.h>
#include <nn/mcu.h>
#include <nn/snd.h>

#ifdef EVA_COMPOSITE
extern       char VERSION_STRING[];
#else
static const char VERSION_STRING[]= UJI_APPVER;
#endif
static const char VERSION_DATE[]  =	__DATE__ " " __TIME__ ;



namespace uji {
namespace eva {


// 静的メンバ変数の実体
char MicDemo::m_MicBuffer[MIC_BUFFER_SIZE] NN_ATTRIBUTE_ALIGN(nn::mic::BUFFER_ALIGNMENT_ADDRESS);


/*
    Desc: BCWAVファイルの再生関連
*/
namespace
{
    struct MyWaveBuffer : public nn::snd::WaveBuffer
    {
    public:
        void Dump()
        {
            NN_LOG("[WaveBuffer::Dump] bufAddr(%p) len(%6d|%08X) loop(%d)",
                   bufferAddress, sampleLength, sampleLength, loopFlag);

            if (pAdpcmContext == NULL)
            {
                NN_LOG("\n");
            }
            else
            {
                NN_LOG(" ps(%04X) yn1(%04X) yn2(%04X)\n",
                       pAdpcmContext->pred_scale,
                       pAdpcmContext->yn1, pAdpcmContext->yn2);
            }
        }
    };

    nn::fnd::ExpHeap expHeap;
    const int nDeviceMemorySize = (32 * 1024 * 1024);    // 使用するデバイスメモリのサイズ
    const int nFiles = 3;
    FILE_INFO apFile[nFiles] = {
        {L"rom:/bcwav/annotation.pcm16.bcwav",  ANNOTATION_PCM16_BEGIN, ANNOTATION_PCM16_END},
        {L"rom:/bcwav/ok.pcm16.bcwav",          OK_PCM16_BEGIN,         OK_PCM16_END},
        {L"rom:/bcwav/ng.pcm16.bcwav",          NG_PCM16_BEGIN,         NG_PCM16_END}
    };

    void* s_pBcwavBuffer[nFiles];
    MyWaveBuffer s_WaveBuffer[nFiles][2];

    // サウンドスレッド関連
    const int SOUND_THREAD_PRIORITY = 2;
    const int SOUND_THREAD_STACK_SIZE = 4096;
    nn::os::Thread threadSound;
    bool threadSoundFlag;
    void SoundThreadFunc(uptr arg)
    {
        NN_UNUSED_VAR(arg);
        threadSoundFlag = true;
        while (threadSoundFlag)
        {
            nn::snd::WaitForDspSync();      // DSP からのデータ受信を待つ
            nn::snd::SendParameterToDsp();  // パラメータを DSP に送信
        }
    }

    nn::snd::Voice* StartVoice(s32 i)
    {
        // Voice の設定
        nn::snd::Voice* pVoice = nn::snd::AllocVoice(128, NULL, NULL);
        NN_TASSERT_(pVoice);

        pVoice->SetupBcwav(reinterpret_cast<uptr>(s_pBcwavBuffer[i]), &s_WaveBuffer[i][0], &s_WaveBuffer[i][1]);
        s_WaveBuffer[i][0].Dump();
        s_WaveBuffer[i][1].Dump();

        nn::snd::MixParam mix;
        mix.mainBus[nn::snd::CHANNEL_INDEX_FRONT_LEFT ] = 2.0f;
        mix.mainBus[nn::snd::CHANNEL_INDEX_FRONT_RIGHT] = 2.0f;
        pVoice->SetMixParam(mix);
        pVoice->SetState(nn::snd::Voice::STATE_PLAY);

        return pVoice;
    }

    void StopVoice(nn::snd::Voice* pVoice)
    {
        pVoice->SetState(nn::snd::Voice::STATE_STOP);
        nn::snd::FreeVoice(pVoice);
    }
}

/*
    Desc: ジャイロ張り付きテスト
*/
bool ShockTester::TestGyroStiction(int sampling_num, u8* ng_code)
{
    using namespace gyro;

    const int Config_JudgeMax = 1000;
    const int Config_JudgeMin = -1000;

    // 閾値範囲外の値が継続していることを示すフラグ
    bool isFail[gyro::AXIS_RAW_NUM];
    for (int i=0; i<gyro::AXIS_RAW_NUM; i++)
    {
        isFail[i] = true;
    }
    LoopBuffer<nn::hid::GyroscopeLowStatus>* pLoopBuffer = m_ExtReader->GetExtdevCalib().loopBuffer;

    // 一定期間閾値範囲外の値が継続している場合に張り付きと判定する
    for (int i=0; i<sampling_num; i++)
    {
        s64 index = pLoopBuffer->LatestDataIndex() - i;
        nn::hid::GyroscopeLowStatus tmp = pLoopBuffer->GetData(index);

        if (isFail[AXIS_RAW_X] && (Config_JudgeMin < tmp.x && tmp.x < Config_JudgeMax)) { isFail[AXIS_RAW_X] = false; }
        if (isFail[AXIS_RAW_Y] && (Config_JudgeMin < tmp.y && tmp.y < Config_JudgeMax)) { isFail[AXIS_RAW_Y] = false; }
        if (isFail[AXIS_RAW_Z] && (Config_JudgeMin < tmp.z && tmp.z < Config_JudgeMax)) { isFail[AXIS_RAW_Z] = false; }

        if (!isFail[AXIS_RAW_X] && !isFail[AXIS_RAW_Y] && !isFail[AXIS_RAW_Z])
        {
            // 全ての軸で正常値を測定できたのでPASS
            *ng_code = 0;
            return true;
        }
    }

    // FAIL
    for (int i=0; i<gyro::AXIS_RAW_NUM; i++)
    {
        if (isFail[i])
        {
            *ng_code |= (0x1<<i);
        }
    }
    return false;
}

/*
    Desc: 加速度張り付きテスト
*/
bool ShockTester::TestAccelerometerStiction(int sampling_num, u8* ng_code)
{
    using namespace gyro;

    const int Config_JudgeMax = 2048;
    const int Config_JudgeMin = -2048;

    // 閾値範囲外の値が継続していることを示すフラグ
    bool isFail[gyro::AXIS_RAW_NUM];
    for (int i=0; i<gyro::AXIS_RAW_NUM; i++)
    {
        isFail[i] = true;
    }
    LoopBuffer<nn::hid::AccelerometerStatus>* pLoopBuffer = m_ExtReader->GetAccCalib().loopBuffer;

    // 一定期間閾値範囲外の値が継続している場合に張り付きと判定する
    for (int i=0; i<sampling_num; i++)
    {
        s64 index = pLoopBuffer->LatestDataIndex() - i;
        nn::hid::AccelerometerStatus tmp = pLoopBuffer->GetData(index);

        if (isFail[AXIS_RAW_X] && (Config_JudgeMin < tmp.x && tmp.x < Config_JudgeMax)) { isFail[AXIS_RAW_X] = false; }
        if (isFail[AXIS_RAW_Y] && (Config_JudgeMin < tmp.y && tmp.y < Config_JudgeMax)) { isFail[AXIS_RAW_Y] = false; }
        if (isFail[AXIS_RAW_Z] && (Config_JudgeMin < tmp.z && tmp.z < Config_JudgeMax)) { isFail[AXIS_RAW_Z] = false; }

        if (!isFail[AXIS_RAW_X] && !isFail[AXIS_RAW_Y] && !isFail[AXIS_RAW_Z])
        {
            // 全ての軸で正常値を測定できたのでPASS
            *ng_code = 0;
            return true;
        }
    }

    // FAIL
    for (int i=0; i<gyro::AXIS_RAW_NUM; i++)
    {
        if (isFail[i])
        {
            *ng_code |= (0x1<<i);
        }
    }
    return false;
}

/*
    Desc: マイク張り付きテスト
*/
bool ShockTester::TestMicStiction(int sampling_num)
{
    using namespace gyro;

    const int Config_JudgeMax = 1024;
    const int Config_JudgeMin = -1024;

    // 一定期間閾値範囲外の値が継続している場合に張り付きと判定する
    for (int i=0; i<sampling_num; i++)
    {
        s64 index = m_MicLoopBuffer->LatestDataIndex() - i;
        s16 tmp = m_MicLoopBuffer->GetData(index);

        if (Config_JudgeMin < tmp && tmp < Config_JudgeMax)
        {
            return true;
        }
    }
    return false;
}

/*
    Desc: 本体NANDへ検査ログを書きこみます
*/
seq::LogResult ShockTester::WriteProductionLog(bool result, int minor, u32 micro, char* string)
{
    seq::LogResult lr;

    uji::seq::Config *c = new uji::seq::Config;
    uji::seq::ProductionLog* pdl = new uji::seq::ProductionLog();

    pdl->Initialize();
    {
        char strNgCode[seq::pl::LOG_MSG_LEN];

        if (result == true) std::sprintf(strNgCode, "OK");
        else                std::sprintf(strNgCode, "%02d-%02d-%02d", G_CTR_ACCEL, minor, static_cast<int>(micro));

        lr = pdl->Add_1Line(c->Get().TestMode, "Shock", strNgCode, VERSION_STRING, "", "", "", string);
    }
    delete pdl;
    delete c;

    return lr;
}

/*
    Desc: グラフ更新
*/
void ShockTester::UpdateGraph()
{
    // ジャイロ・加速度生値グラフを作成
    {
        s64 latestIndex = m_ExtReader->GetExtdevCalib().loopBuffer->LatestDataIndex();
        s64 startIndex = (m_GyroRawLastAddedIndex == 0) ? 0 : m_GyroRawLastAddedIndex + 1;

        // グラフデータ更新
        for (s64 idx = startIndex; idx <= latestIndex; idx++)
        {
            nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);
            m_GyroRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpGyro.x);
            m_GyroRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpGyro.y);
            m_GyroRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpGyro.z);

            nn::hid::AccelerometerStatus tmpAcc = m_ExtReader->GetAccCalib().loopBuffer->GetData(idx);
            m_AccRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpAcc.x);
            m_AccRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpAcc.y);
            m_AccRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpAcc.z);
        }
        m_GyroRawLastAddedIndex = latestIndex;
    }

    // マイクグラフ作成
    {
        s16 micData[m_ScreenWidth];

        micDemo.GetMicData(micData, m_ScreenWidth);
        for (int i=0; i<m_ScreenWidth; i++)
        {
            m_MicLine->GetLoopBuffer()->AddNewData(micData[i]);
            // はりつき判定用データも更新
            // 毎フレーム120データの取得（1/8180*120≒14.6ms なので重複データは無い）
            m_MicLoopBuffer->AddNewData(micData[i]);
        }
        m_MicLatestData = micData[m_ScreenWidth-1];
    }
}

/*
    Desc: 画面更新
*/
void ShockTester::UpdateScreen(sys::GraphicsDrawing* gfx)
{
    /* ------------------------------------------------------------------------
                上画面描画
    ------------------------------------------------------------------------ */
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);
    gfx->m_DrawFramework->Clear();

    const f32 BASE_GYRO_DATA_X = 10.0f;
    const f32 BASE_ACC_DATA_X = 140.0f;
    const f32 BASE_MIC_DATA_X = 270.0f;
    const f32 BASE_DATA_Y = 200.0f;

    gfx->m_DrawFramework->SetFontSize(8.0f);

    // ジャイロ生値
    gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X: %6d\n", m_ExtReader->GetExtdevCalib().latestStatus.x);
    gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y: %6d\n", m_ExtReader->GetExtdevCalib().latestStatus.y);
    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
    gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z: %6d\n", m_ExtReader->GetExtdevCalib().latestStatus.z);
    m_GyroRawGraphDrawing->Draw(gfx);

    // 加速度生値
    gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y,        "X: %6d\n", m_ExtReader->GetAccCalib().latestStatus.x);
    gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+10.0f,  "Y: %6d\n", m_ExtReader->GetAccCalib().latestStatus.y);
    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+20.0f,  "Z: %6d\n", m_ExtReader->GetAccCalib().latestStatus.z);
    m_AccRawGraphDrawing->Draw(gfx);

    // マイク
    gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 1.0f);
    gfx->m_DrawFramework->DrawText(BASE_MIC_DATA_X, BASE_DATA_Y+20.0f,  "%6d\n", m_MicLatestData);
    m_MicGraphDrawing->Draw(gfx);

    gfx->m_DrawFramework->SwapBuffers();

    /* ------------------------------------------------------------------------
                下画面描画
    ------------------------------------------------------------------------ */
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();

    // 画像表示
    if (m_TextureNo != TEXTURE_NONE)
    {
        jpegDrawer->DrawPicture(NN_GX_DISPLAY1);
    }
    // テキスト
    gfx->m_DrawFramework->SetColor(0.3f, 0.30f, 7.0f);
    gfx->m_DrawFramework->DrawText(0,  0, "Shock Tester Ver.%s", VERSION_STRING);
    gfx->m_DrawFramework->DrawText(0, 10, "SEQUENCE No.(%d) Log(%d)", m_StictionTestCount, m_LogResult);
    gfx->m_DrawFramework->DrawText(0, 20, "GYRO(%s:%Xh) ACCL(%s:%Xh) MIC(%s)",
        m_TestSequence != TESTSEQ_RESULT    ?   "YET"    :
                            m_ResultGyro    ?   "PASS"   :  "FAIL", m_NgcodeGyro,
        m_TestSequence != TESTSEQ_RESULT    ?   "YET"    :
                            m_ResultAccl    ?   "PASS"   :  "FAIL", m_NgcodeAccl,
        m_TestSequence != TESTSEQ_RESULT    ?   "YET"    :
                             m_ResultMic    ?   "PASS"   :  "FAIL");
    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
}

/*
    Desc: ジャイロ・加速度の張り付き検査
          ※表示は速度優先の為、RenderSystemの文字・図形描画を使用します。
*/
void ShockTester::Run()
{
    const int Config_SamplingDataNum = 80;
    const int Config_MicSamplingDataNum = 1024; // リングバッファサイズまで検査可能
    char* strSound[] = {"rom:/SoundPlayer/ng.wav", "rom:/SoundPlayer/ok.wav"};

    int sub_seq = 0;
    sys::SoundPlayer *sp = new sys::SoundPlayer;
    jpegDrawer = new sys::JpegDrawer;
    sys::GraphicsDrawing* gfx = sys::GraphicsDrawing::GetInstance();

    m_StictionTestCount = 0;
    m_ResultAccl = true;
    m_ResultGyro = true;
    m_ResultMic = true;
    m_TestResult = true;
    m_NgcodeAccl = 0;
    m_NgcodeGyro = 0;
    m_LogResult = 0;

    /*
        拡張リーダ
    */
    nn::hid::GyroscopeLowReader extDevReader;
    nn::hid::GyroscopeLowCalibrator extDevCalibrator;
    nn::hid::AccelerometerCalibrator accCalibrator;

    m_ExtReader->Initialize();
    // サンプリング開始
    m_ExtReader->SamplingStart(
        extDevCalibrator,
        extDevReader,
        accCalibrator);

    /*
        マイク初期化
    */
    // CTRマイクの初期化
    micDemo.Initialize();
    micDemo.Start();

    /*
        グラフデータ初期化
    */
    f32 f_width  = static_cast<f32>(m_ScreenWidth);
    f32 f_height = static_cast<f32>(m_ScreenHeight);

    // ジャイロ（生値）
    m_GyroRawGraphDrawing = new GyroGraphDrawing(
        "GYRO(RAW)",
        f_height/65536.0f, 10.0f, 10.0f, f_width, f_height, nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_X]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Y]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Z]);

    // 加速度（生値）
    m_AccRawGraphDrawing  = new GyroGraphDrawing(
        "ACCL(RAW)",
        f_height/16384.0f, 140.0f, 10.0f, f_width, f_height, nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_X]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Y]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Z]);

    // マイク
    m_MicGraphDrawing  = new GyroGraphDrawing(
        "MIC(s16bit)",
        f_height/65536.0f, 270.0f, 10.0f, f_width, f_height, nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_MicGraphDrawing->AddLine(m_MicLine);

    /*
        MCUも使います
    */
    eva::mcu::McuInitializer().HwCheckInit();
    // MCU プロセスにリクエストを送るための Camera クラスを構築。このクラスは自動生成される
    nn::mcu::CTR::HwCheck mcu(
        eva::mcu::McuInitializer().GetHSession()    // 確立されたセッションのハンドルを
    );                                              // コンストラクタに指定

    /*
        サウンド関連
    */
    // dsp, snd の初期化
    uji::eva::SoundInitializer().InitSDK();
    
    if( uji::sys::PLATFORM_FTR == uji::sys::GetPlatformType())
    {
        // FTRの場合はサウンド出力をモノラルに
        nn::snd::SetSoundOutputMode(nn::snd::OUTPUT_MODE_MONO);
    }else
    {
        // サウンド出力をステレオに
        nn::snd::SetSoundOutputMode(nn::snd::OUTPUT_MODE_STEREO);
    }

    // サウンドスレッドを起動（DSP 割り込みイベント待ち）
    threadSound.StartUsingAutoStack(SoundThreadFunc, NULL, SOUND_THREAD_STACK_SIZE, SOUND_THREAD_PRIORITY);

    for (int i = 0; i < nFiles; i++)
    {
#ifdef ROM_FS_USE
        nn::fs::FileReader fileReader = nn::fs::FileReader(apFileNames[i]);
        size_t fileSize = fileReader.GetSize();
        s_pBcwavBuffer[i] = reinterpret_cast<u8*>(uji::sys::AllocDeviceMemory(fileSize, 32));
        s32 size = fileReader.Read(s_pBcwavBuffer[i], fileSize);
        NN_LOG("[%s] fileSize(%d) loadSize(%d) fileSize(%d)\n", apFileNames[i], fileSize, size, fileSize);
        fileReader.Finalize();
#else
        // オブジェクトファイルとしてリンクしたデータを使用
        s32  fileSize = static_cast<s32>(apFile[i].objEnd - apFile[i].objBegin);
        uptr begin = reinterpret_cast<uptr>(apFile[i].objBegin);
        s_pBcwavBuffer[i] = reinterpret_cast<u8*>(sys::AllocDeviceMemory(fileSize, 32));
        std::memcpy(s_pBcwavBuffer[i], reinterpret_cast<void*>(begin), fileSize);
        nn::snd::FlushDataCache(reinterpret_cast<uptr>(s_pBcwavBuffer[i]), fileSize);
#endif
    }

    nn::snd::Voice* apVoice[nFiles] = { NULL, };
    s32 sndFileIndex = 0;

    // 本体を閉じた状態で音を出せるようにします
    nn::snd::SetHeadphoneOutOnShellClose(false);

    do
    {
        // パッド情報更新
        sys::Pad().UpdatePad();

        switch (m_TestSequence)
        {
        // 張り付き検査
        case TESTSEQ_STICTION:
            // 判定中はLEDを消灯します
            mcu.SetPowerLEDPattern(static_cast<nn::mcu::CTR::PowerLedPattern>(3));

            // 一定サイズのデータがサンプリングされるのを待つ
            if ((m_ExtReader->GetExtdevCalib().loopBuffer->GetValidDataNum() > Config_SamplingDataNum) &&
                (m_MicLoopBuffer->GetValidDataNum() > Config_MicSamplingDataNum))
            {
                // 張り付きチェック
                m_ResultAccl = TestAccelerometerStiction(Config_SamplingDataNum, &m_NgcodeAccl);
                m_ResultGyro = TestGyroStiction(Config_SamplingDataNum, &m_NgcodeGyro);
                m_ResultMic  = TestMicStiction(Config_MicSamplingDataNum);
                m_StictionTestCount++;

                // PASS
                if (m_ResultAccl && m_ResultGyro && m_ResultMic)
                {
                    switch (m_StictionTestCount)
                    {
                    case 1:
                    case 2:
                    case 3:
                        m_TestSequence = TESTSEQ_SHOCK;
                        break;
                    case 4:
                        m_TestSequence = TESTSEQ_RESULT;
                        m_TestResult = true;
                        break;
                    default:
                        break;
                    }
                }
                // FAIL
                else
                {
                    m_TestResult = false;
                    m_TestSequence = TESTSEQ_RESULT;
                }
            }
            break;

        // ショック検査
        case TESTSEQ_SHOCK:
            switch (sub_seq)
            {
            case 0:
                // 画像更新
                {
                    FILE_INFO texinfo = GetTexture(m_TextureNo = TEXTURE_SHOCK_TARGET);
                    jpegDrawer->OpenPicture(texinfo.objBegin, static_cast<s32>(texinfo.objEnd-texinfo.objBegin));
                    UpdateScreen(gfx);
                }

                // ショック実施をLED青で指示
                mcu.SetPowerLEDPattern(static_cast<nn::mcu::CTR::PowerLedPattern>(5));
                // 注意音の再生
                if (apVoice[sndFileIndex] == NULL)
                {
                    apVoice[sndFileIndex] = StartVoice(sndFileIndex);
                    NN_LOG("--- start voice (file %d)\n", sndFileIndex);
                }
                nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(300));
                if (apVoice[sndFileIndex])
                {
                    StopVoice(apVoice[sndFileIndex]);
                    apVoice[sndFileIndex] = NULL;
                    NN_LOG("--- stop voice (file %d)\n", sndFileIndex);
                }
                sub_seq++;
                break;
            case 1:
                // Rボタンで次の検査に
                if (sys::Pad().IsButtonDown(sys::Pad::BUTTON_R))
                {
                    // 測定バッファのクリア
                    m_ExtReader->GetExtdevCalib().loopBuffer->Clear();
                    m_ExtReader->GetAccCalib().loopBuffer->Clear();
                    m_MicLoopBuffer->Clear();

                    jpegDrawer->ClosePicture();
                    m_TestSequence = TESTSEQ_STICTION;
                    sub_seq = 0;
                }
                break;
            default:
                break;
            }
            break;

        // 検査結果表示
        case TESTSEQ_RESULT:
            switch (sub_seq)
            {
            case 0:
                int led_pattern;

                // 検査ログ
                m_LogResult = WriteProductionLog(m_TestResult, m_StictionTestCount, 0, "");

                if (m_TestResult == true)
                {
                    led_pattern = 5;
                    m_PassLedStartTick = nn::os::Tick::GetSystemCurrent();
                    m_bPassLedToggle = true;
                    sndFileIndex = 1;
                    sub_seq++;
                }
                else
                {
                    led_pattern = 4;
                    sndFileIndex = 2;
                    sub_seq+=2;
                }
                {
                    FILE_INFO texinfo = GetTexture(m_TextureNo = m_TestResult ? TEXTURE_PASS : TEXTURE_FAIL);
                    jpegDrawer->OpenPicture(texinfo.objBegin, texinfo.objEnd-texinfo.objBegin);
                    UpdateScreen(gfx);
                }
                mcu.SetPowerLEDPattern(static_cast<nn::mcu::CTR::PowerLedPattern>(led_pattern));
                if (apVoice[sndFileIndex] == NULL)
                {
                    apVoice[sndFileIndex] = StartVoice(sndFileIndex);
                    NN_LOG("--- start voice (file %d)\n", sndFileIndex);
                }
                nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(1500));
                if (apVoice[sndFileIndex])
                {
                    StopVoice(apVoice[sndFileIndex]);
                    apVoice[sndFileIndex] = NULL;
                    NN_LOG("--- stop voice (file %d)\n", sndFileIndex);
                }
                break;
            case 1:
                // 青LEDを点滅表示
                if ((int)(nn::os::Tick::GetSystemCurrent() - m_PassLedStartTick).ToTimeSpan().GetMilliSeconds()%800 < 400)
                {
                    if (m_bPassLedToggle)
                    {
                        mcu.SetPowerLEDPattern(static_cast<nn::mcu::CTR::PowerLedPattern>(3));
                        m_bPassLedToggle = !m_bPassLedToggle;
                    }
                }
                else
                {
                    if (!m_bPassLedToggle)
                    {
                        mcu.SetPowerLEDPattern(static_cast<nn::mcu::CTR::PowerLedPattern>(5));
                        m_bPassLedToggle = !m_bPassLedToggle;
                    }
                }
                break;
            default:
                break;
            }
            break;

        default:
            break;
        }

        // グラフ更新
        UpdateGraph();

        // 画面更新
        UpdateScreen(gfx);
    }
    while(!sys::Pad().IsButtonDown(sys::Pad::BUTTON_B) || !(m_TestSequence == TESTSEQ_RESULT && sub_seq > 0 ));

    /*
        終了処理
    */
    m_ExtReader->SamplingStop();
    m_ExtReader->Finalize();

    micDemo.End();
    micDemo.Finalize();
    // グラフの破棄
    delete m_GyroRawGraphDrawing;
    delete m_AccRawGraphDrawing;
    // 画像描画の破棄
    jpegDrawer->ClosePicture();
    delete jpegDrawer;

    for (int i = 0; i < nFiles; i++)
    {
        if (apVoice[i])
        {
            StopVoice(apVoice[i]);
            apVoice[i] = NULL;
        }
    }
    // サウンドスレッドの破棄
    threadSoundFlag = false;
    threadSound.Join();
    threadSound.Finalize();
    eva::SoundInitializer().FinalizeSDK();
    
    for (int i = 0; i < nFiles; i++)
        uji::sys::Free(s_pBcwavBuffer[i]);

    delete sp;

    // restore gl state ※RenderSystemの文字描画実行の際は必ず戻すこと
    for (int attrindex = 0; attrindex < 12; attrindex++)
        glDisableVertexAttribArray(attrindex);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_SCISSOR_TEST);
    glDisable(GL_STENCIL_TEST);
    glDisable(GL_BLEND);
    glActiveTexture(GL_TEXTURE0);
    glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
    glDepthMask(GL_TRUE);
}


//============================================================================
//      マイク関連
//============================================================================

/*
    Desc: 初期処理
*/
void MicDemo::Initialize()
{
    // mic の初期化
    nn::Result result = nn::mic::Initialize();
    NN_UTIL_PANIC_IF_FAILED(result);

    // マイクライブラリにバッファを設定
    result = nn::mic::SetBuffer(m_MicBuffer, MIC_BUFFER_SIZE);
    NN_UTIL_PANIC_IF_FAILED(result);

    // サンプリングバッファサイズを取得
    size_t size;
    result = nn::mic::GetSamplingBufferSize(&size);
    NN_UTIL_PANIC_IF_FAILED(result);
    m_SamplingLength = size / sizeof(s16);

    // MICアンプの設定
    result = nn::mic::GetAmpGain(&m_MicGain);
    NN_UTIL_PANIC_IF_FAILED(result);

    result = nn::mic::SetAmpGain(MIC_GAIN);
    NN_UTIL_PANIC_IF_FAILED(result);

    // MICの電源ON
    result = nn::mic::SetAmp(true);
    NN_UTIL_PANIC_IF_FAILED(result);
}

/*
    Desc: 終了処理
*/
void MicDemo::Finalize()
{
    // MICの電源OFF
    nn::Result result = nn::mic::SetAmp(false);
    NN_UTIL_PANIC_IF_FAILED(result);

    // MICアンプを元に戻す
    result = nn::mic::SetAmpGain(m_MicGain);
    NN_UTIL_PANIC_IF_FAILED(result);

    result = nn::mic::StopSampling();
    NN_UTIL_PANIC_IF_FAILED(result);

    // マイクライブラリに設定したバッファを使わないようにする
    result = nn::mic::ResetBuffer();
    NN_UTIL_PANIC_IF_FAILED(result);

    result = nn::mic::Finalize();
    NN_UTIL_PANIC_IF_FAILED(result);
}

void MicDemo::Start()
{
    // マイクサンプリング開始
    nn::Result result;
    result = nn::mic::StartSampling(nn::mic::SAMPLING_TYPE_SIGNED_16BIT,
                                    nn::mic::SAMPLING_RATE_8180,
                                    0,                                   // オフセット
                                    m_SamplingLength * sizeof(s16),      // サイズ
                                    true);
    NN_UTIL_PANIC_IF_FAILED(result);
}

void MicDemo::End()
{
    // マイクサンプリング終了
    nn::Result result;
    result = nn::mic::StopSampling();
    NN_UTIL_PANIC_IF_FAILED(result);
}

void MicDemo::GetMicData(s16* pDest, size_t size)
{
    uptr src;
    nn::Result result = nn::mic::GetLastSamplingAddress(&src);
    NN_UTIL_PANIC_IF_FAILED(result);

    s16* data = reinterpret_cast<s16*>(m_MicBuffer);
    u32 offs = src - reinterpret_cast<uptr>(m_MicBuffer);

    for (int i = 0; i < size; i++)
    {
        *(pDest + i) = data[(offs - i) % m_SamplingLength];
    }
}


} // namespace
}
