/*--------------------------------------------------------------------------*
 Project:
 File: gyro_LoopBuffer.h
 

*--------------------------------------------------------------------------*/

#ifndef __LOOP_BUFFER_H__
#define __LOOP_BUFFER_H__

#include <nn/types.h>
#include "sys.h"

namespace uji {
namespace eva {

/*
    リングバッファ
*/
template <class T> class LoopBuffer
{
public:
    /*
        コンストラクタ
    */
    // デフォルトコンストラクタ
    LoopBuffer() : m_Buffer(0), m_BufferLength(0), m_BufferAddedNum(0)
    {   
    }    
    // buffer_length分のバッファを確保します
    LoopBuffer(s32 buffer_length) : m_BufferLength(buffer_length), m_BufferAddedNum(0)
    {
        m_Buffer = static_cast<T*>(sys::Alloc(sizeof(T) * m_BufferLength));
    }
    // 参照用のバッファを引数に与えます
    LoopBuffer(s32 buffer_length, T* buffer) : m_BufferLength(buffer_length), m_Buffer(buffer), m_BuffNum(0)
    {
    }
    
    /*
        デストラクタ
    */
    ~LoopBuffer()
    {
        if (m_Buffer != 0) 
        {
            sys::Free(m_Buffer);
        }
    }
    
    // 新しくデータを格納する
    // バッファ長以上格納している場合はループしてデータを上書きする
    void AddNewData(const T& data)
    {
        s32 index = m_BufferAddedNum % m_BufferLength;
        m_Buffer[index] = data;

        m_BufferAddedNum++;
    }

    // 有効なデータの総数を返す。上限はバッファ長となる。
    s32 GetValidDataNum()
    {
        return (m_BufferAddedNum < m_BufferLength) ? m_BufferAddedNum : m_BufferLength;
    }

    // idx番目のデータを取得
    T& GetData(s64 idx)
    {
        if (idx >= m_BufferAddedNum)
        {
            NN_LOG("Get Invalid Data idx: %lld >= num: %d \n", idx, m_BufferAddedNum);
        }
        return m_Buffer[idx % m_BufferLength];
    }

    // 指定インデックスのデータは有効か？
    bool IsValidIndex(s64 idx)
    {
        return (idx >= OldestDataIdx() && idx <= NewestDataIdx());
    }

    // 最古のデータのインデックスを取得
    s64 OldestDataIndex()
    {
        return (m_BufferLength >= m_BufferAddedNum) ? 0 : (m_BufferAddedNum - m_BufferLength);
    }

    // 最新データのインデックスを取得
    s64 LatestDataIndex() { return m_BufferAddedNum - 1; }

    // バッファ内部のすべてのデータをクリアする(indexを初期化する)
    void Clear() { m_BufferAddedNum = 0; }

    // バッファ長を返す
    s32 GetBufferLength() const { return m_BufferLength; }

private :
    T* m_Buffer;

    s32 m_BufferLength;
    s64 m_BufferAddedNum;
};

}} // namespace

#endif // __LOOP_BUFFER_H__
