/*--------------------------------------------------------------------------*
 Project:
 File: gyro_ExtReader.h
 

*--------------------------------------------------------------------------*/
#include "gyro_ExtReader.h"
//#include <nn/hid/CTR/hid_ApiForGyroscopeLow.h>
    
namespace uji {
namespace eva {


/*
  Desc: 生値サンプリング用スレッドの実体
  
  Args: なし
  
  rtns: なし
*/
void GyroExtReader::ThreadGyroSamplingRawData()
{    
    // インスタンス生成のタイミングで加速度がサンプリングされるSDK-0.12より
    nn::hid::AccelerometerReader accReader;    
    
    nn::hid::GyroscopeLowStatus gyroRawData;
    nn::hid::Gyroscope& gyroscopeLow = nn::hid::GetGyroscope();    
    nn::hid::AccelerometerStatus accRawData;    
    nn::fnd::TimeSpan timeout = nn::fnd::TimeSpan::FromMilliSeconds(18);            
    
    NN_LOG("Create thread(ThreadGyroSamplingRawData)\n");
        
    while (true)
    {        
        // デバイスがサンプリングするのを待ちます
        // 全データの取り逃しがないか確認するためにタイムアウトを設定（デバッグ用）
        if (gyroscopeLow.WaitSampling(timeout) != true)
        {
            NN_LOG("Warnning: Gyro Sampling Time Out!!\n");
        }
#if 0   
        // 加速度をサンプリングを待つ場合
        // ※加速度とジャイロは同じサンプリング周期であることと、
        //   ジャイロのデータを検査で優先で使用するためジャイロのサンプリング待ちで統一
        nn::hid::Accelerometer& accelerometer = nn::hid::GetAccelerometer();
        if (accelerometer.WaitSampling(timeout) != true)
#endif    
        // Gyro生値取得
        {
            GetExtdevCalib().reader->ReadRaw(&gyroRawData);
            GetExtdevCalib().loopBuffer->AddNewData(gyroRawData);
            GetExtdevCalib().latestStatus = gyroRawData;
            GetExtdevCalib().latestReadLen = 1;
        }
        
        // Accelerometer生値取得
        {
            GetAccCalib().reader->ReadRaw(&accRawData);
            GetAccCalib().loopBuffer->AddNewData(accRawData);
            GetAccCalib().latestStatus = accRawData;
            GetAccCalib().latestReadLen = 1;            
        }

        if (m_EventSamplingThreadBreak.TryWait())
        {
            NN_LOG("Exit thread(ThreadGyroSamplingRawData)\n");            
            break;   
        }        
    }
}

/*
  Desc: 生値サンプリング用スレッドラップ関数
  
  Args: param - メンバ関数実行時のオブジェクトのポインタ

  Rtns: なし  
*/
void GyroExtReader::s_ThreadGyroSamplingRawData(void* param)
{
    GyroExtReader* pGyroExtReader = reinterpret_cast<GyroExtReader*>(param);
    pGyroExtReader->ThreadGyroSamplingRawData();
} 

/*
  Desc: サンプリングデータを読み込み、各バッファに格納します
        キャリブレ済みデータの取得で使用。

  Args: pEvent      - キャンセル完了を通知するイベント
        cancelled   - キャンセルされたかどうかを表す変数
  
  Rtns: なし
*/
void GyroExtReader::SamplingAlarmHandler(bool cancelled)
{
    if (cancelled)
    {
        NN_LOG("Gyro Read Alarm Handler Cancelled\n");
        m_CancelEvent.Signal();
    }
    else
    {        
        nn::hid::GyroscopeLowStatus extDevBuffer[TMP_SAMPLING_BUFFER_SIZE]; 
        s32 readLen;
        
        // 未取得データの読み込み。バッファには新しいものから順番に読込まれる。
        GetExtdev().reader->Read(extDevBuffer, &readLen, TMP_SAMPLING_BUFFER_SIZE);
        NN_ASSERTMSG(readLen <= TMP_SAMPLING_BUFFER_SIZE, "Error: readLen(%d)\n", readLen);
        GetExtdev().latestReadLen = readLen;
        
        while (readLen)
        {
            // リングバッファに古いデータから追加
            GetExtdev().loopBuffer->AddNewData(extDevBuffer[readLen-1]);
            --readLen;
        }
        
        // 最新データの保持
        GetExtdev().latestStatus = extDevBuffer[0];
    }    
}

/*
  Desc: 周期アラームハンドラのラップ関数
  
  Args: param       - メンバ関数実行時のオブジェクトのポインタ
        cancelled   - キャンセル
  
  Rtns: なし
*/
void GyroExtReader::s_SamplingAlarmHandler(void* param, bool cancelled)
{
    GyroExtReader* pGyroExtReader = reinterpret_cast<GyroExtReader*>(param);
    pGyroExtReader->SamplingAlarmHandler(cancelled);
}

/*
  Desc: サンプリングを開始します
        ※サンプリング中（周期アラーム設定後）に呼ばれても何もしません
  
  Args: extCalibReader  - GyroscopeLowCalibratorのインスタンス
        extDevReader    - GyroscopeLowReaderのインスタンス
        accCalibReader  - AccelerometerCalibratorのインスタンス
  
  Rtns: 0(正常終了) / 1(サンプリング中) 
*/
s32 GyroExtReader::SamplingStart(
    nn::hid::GyroscopeLowCalibrator& extCalibReader,
    nn::hid::GyroscopeLowReader& extDevReader, 
    nn::hid::AccelerometerCalibrator& accCalibReader)
{
    if (IsSampling() == true)
    {
        // サンプリング中
        return 1;
    }

    // 生値サンプリング用スレッド生成
    GetExtdevCalib().reader = &extCalibReader;
    GetAccCalib().reader = &accCalibReader;

    const size_t STACK_SIZE = 4096;
    m_ThreadRawSampling.StartUsingAutoStack(s_ThreadGyroSamplingRawData, this, STACK_SIZE);

    // 周期アラームを設定（10ミリ秒周期）
    GetExtdev().reader = &extDevReader;
    
    nn::fnd::TimeSpan tSpan = nn::fnd::TimeSpan::FromMilliSeconds(ALARM_READ_INTERVAL_MSEC);
//    一時的にコメントアウト   
//    m_Alarm.SetPeriodic(tSpan, s_SamplingAlarmHandler, this);
    m_IsSamplingFlag = true;
    
    return 0;
}

/*
  Desc: サンプリングを停止します
  
  Args: なし
  
  Rtns: なし
*/
void GyroExtReader::SamplingStop()
{    
     if (IsSampling() == false)
     {
        return;
     }  
        
    // 周期アラームのキャンセル
    m_Alarm.Cancel();
    while (!m_Alarm.CanSet()){ nn::os::Thread::Yield(); }

    /*
        生値サンプリングスレッド終了
    */
    m_EventSamplingThreadBreak.Signal();    // スレッド終了を通知
    m_ThreadRawSampling.Join();             // スレッド終了待ち
    m_ThreadRawSampling.Finalize();         // スレッド破棄

    m_IsSamplingFlag = false;  
}

/*
  Desc: リングバッファのメモリを再確保します
        ※既にサンプリング済みのデータの保障はありません

  Args: bufferLength - リングバッファの長さ
  
  Rtns: なし
*/    
void GyroExtReader::ReallocateRingBuffer(s32 bufferLength)
{
    // サンプリングを一時停止
    SamplingStop();
    
    // 現在のバッファを削除
    delete GetExtdevCalib().loopBuffer; GetExtdevCalib().loopBuffer = 0;    
    delete GetExtdev().loopBuffer; GetExtdev().loopBuffer = 0;
    delete GetAccCalib().loopBuffer; GetAccCalib().loopBuffer = 0;
    
    // 新規に確保
    GetExtdevCalib().loopBuffer = new LoopBuffer<nn::hid::GyroscopeLowStatus>(bufferLength);
    GetExtdev().loopBuffer = new LoopBuffer<nn::hid::GyroscopeLowStatus>(bufferLength);
    GetAccCalib().loopBuffer = new LoopBuffer<nn::hid::AccelerometerStatus>(bufferLength); 
}

/*
  Desc: 初期処理
  
  Args: なし
  
  Rtns: なし
*/
void GyroExtReader::Initialize()
{    
    // アラームの初期化
    m_Alarm.Initialize();
    
    // イベントの初期化
    m_EventSamplingThreadBreak.Initialize(false);
 
#if 0    
    osPrintf("calib init");
    for(s32 i = MAKER_NONE;i<MAKER_NUM;i++)
    {
        DEFAULT_GYRO_CALIB_PARAM[i].x.motionless = TYPICAL_GYRO_PARAM[i][0];
        DEFAULT_GYRO_CALIB_PARAM[i].x.posi78 = TYPICAL_GYRO_PARAM[i][1];
        DEFAULT_GYRO_CALIB_PARAM[i].x.nega78 = TYPICAL_GYRO_PARAM[i][2];
        DEFAULT_GYRO_CALIB_PARAM[i].y.motionless = TYPICAL_GYRO_PARAM[i][0];
        DEFAULT_GYRO_CALIB_PARAM[i].y.posi78 = TYPICAL_GYRO_PARAM[i][1];
        DEFAULT_GYRO_CALIB_PARAM[i].y.nega78 = TYPICAL_GYRO_PARAM[i][2];
        DEFAULT_GYRO_CALIB_PARAM[i].z.motionless = TYPICAL_GYRO_PARAM[i][0];
        DEFAULT_GYRO_CALIB_PARAM[i].z.posi78 = TYPICAL_GYRO_PARAM[i][1];
        DEFAULT_GYRO_CALIB_PARAM[i].z.nega78 = TYPICAL_GYRO_PARAM[i][2];

        gyroCalibParam[i] = DEFAULT_GYRO_CALIB_PARAM[i];
    }
#endif    
}

/*
  Desc: 終了処理
  
  Args: なし
  
  Rtns: なし
*/
void GyroExtReader::Finalize()
{    
    // アラームをキャンセル
    m_Alarm.Cancel();

    // キャンセルが完了するまで待機
    while(!m_Alarm.CanSet()) {     
        NN_LOG("Finalize: Wait Cancel\n");
        m_CancelEvent.Wait();
    }
    m_Alarm.Finalize();
    
    // イベント終了処理
    m_EventSamplingThreadBreak.Finalize();
}





} // namespace
}
