/*--------------------------------------------------------------------------*
 Project:
 File: gyro_EvaSampling.cpp


*--------------------------------------------------------------------------*/

#include <nn/fs/CTR/MPCore/fs_ApiForHwCheck.h>
#include "../seq/Config.h"
#include "../seq/ConfigDefine.h"
#include "../seq/TestListManager.h"
#include "gyro_EvaSampling_forF.h"
#include "gyro_UdsClient.h"
#include "sys_GetSerialNumber.h"

#ifdef EVA_COMPOSITE
extern       char VERSION_STRING[];
#else
static const char VERSION_STRING[]= UJI_APPVER;
#endif
static const char VERSION_DATE[]  =	__DATE__ " " __TIME__ ;

using namespace uji;
using namespace uji::sys;
using namespace uji::seq;

#define SD_WRITE_NG_1     8001
#define SD_WRITE_NG_2     8002
#define SD_WRITE_NG_3     8003
#define SD_WRITE_NG_4     8004
#define SD_WRITE_NG_5     8005
#define RESULT_OK         0
#define RESULT_NG         1




namespace uji {
namespace eva {
namespace forF {

static s32 check_acc_x_flag = 0;   // 加速度センサx軸の貼り付きがないことが確認できると１になるフラグ
static s32 check_acc_y_flag = 0;   // 加速度センサy軸の貼り付きがないことが確認できると１になるフラグ
static s32 check_acc_z_flag = 0;   // 加速度センサz軸の貼り付きがないことが確認できると１になるフラグ
static s32 check_gyro_x_flag = 0;  // ジャイロセンサx軸の貼り付きがないことが確認できると１になるフラグ
static s32 check_gyro_y_flag = 0;  // ジャイロセンサy軸の貼り付きがないことが確認できると１になるフラグ
static s32 check_gyro_z_flag = 0;  // ジャイロセンサz軸の貼り付きがないことが確認できると１になるフラグ
static s32 acc_diffmax_x = 0;
static s32 acc_diffmax_y = 0;
static s32 acc_diffmax_z = 0;
static s32 gyro_diffmax_x = 0;
static s32 gyro_diffmax_y = 0;
static s32 gyro_diffmax_z = 0;

static s64 GetMilliSecond( nn::os::Tick start_time );
static u32 WriteData2SD(const wstring& path, string strBuf);
static s32 CreateDirectorySdmc(const wchar_t* path);


//============================================================================
//                         加速度センサ検査
//============================================================================
bool StickTest(seq::TestResult &result)
{

    uji::eva::forF::GyroEvaSampling *p = new uji::eva::forF::GyroEvaSampling;
    result = p->Run2();

    delete p;
    return result.m_Result;

}


/*
    Desc: 1sec間（100サンプル）のMax-Minを算出する
*/
void GyroEvaSampling::CalculateMaxMin()
{
    s64 latestIndex = m_ExtReader->GetExtdevCalib().loopBuffer->LatestDataIndex();

    // 105サンプルサンプリングされたら算出
    if ((latestIndex - m_LastCalcLatestDataIndex) >= 105)
    {
        NN_LOG("OK GetSample:%lld\n X:%d\n Y:%d\n Z:%d\n\n\n", (latestIndex - m_LastCalcLatestDataIndex), m_AccRawDiff.x, m_AccRawDiff.y, m_AccRawDiff.z );
        nn::hid::AccelerometerStatus acc_max = {-37268/2, -32768/2, -32768/2};
        nn::hid::AccelerometerStatus acc_min = { 37267/2,  32767/2,  32767/2};
        nn::hid::GyroscopeLowStatus gyro_max = {-37268/2, -32768/2, -32768/2};
        nn::hid::GyroscopeLowStatus gyro_min = { 37267/2,  32767/2,  32767/2};

        NN_LOG("\n\n latestIndex:%lld  m:%lld \n\n" , latestIndex, m_LastCalcLatestDataIndex );

        // Max-Minを算出
        for (s64 idx = m_LastCalcLatestDataIndex; idx <= latestIndex; idx++)
        {
            if( (idx-m_LastCalcLatestDataIndex) < 5 )
            {
                 NN_LOG ("\nidx:%lld m_LastCalcLatestDataIndex:%lld \n", idx, m_LastCalcLatestDataIndex );
            }
            else
            {
                nn::hid::AccelerometerStatus tmpAcc = m_ExtReader->GetAccCalib().loopBuffer->GetData(idx);
                nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);

                // 加速度生値 最大／最小値の更新
                if (acc_max.x < tmpAcc.x) { acc_max.x = tmpAcc.x; }
                if (acc_max.y < tmpAcc.y) { acc_max.y = tmpAcc.y; }
                if (acc_max.z < tmpAcc.z) { acc_max.z = tmpAcc.z; }
                if (acc_min.x > tmpAcc.x) { acc_min.x = tmpAcc.x; }
                if (acc_min.y > tmpAcc.y) { acc_min.y = tmpAcc.y; }
                if (acc_min.z > tmpAcc.z) { acc_min.z = tmpAcc.z; }

                // ジャイロ生値 最大／最小値の更新
                if (gyro_max.x < tmpGyro.x) { gyro_max.x = tmpGyro.x; }
                if (gyro_max.y < tmpGyro.y) { gyro_max.y = tmpGyro.y; }
                if (gyro_max.z < tmpGyro.z) { gyro_max.z = tmpGyro.z; }
                if (gyro_min.x > tmpGyro.x) { gyro_min.x = tmpGyro.x; }
                if (gyro_min.y > tmpGyro.y) { gyro_min.y = tmpGyro.y; }
                if (gyro_min.z > tmpGyro.z) { gyro_min.z = tmpGyro.z; }
            }
        }

        m_AccRawDiff.x = acc_max.x - acc_min.x;
        m_AccRawDiff.y = acc_max.y - acc_min.y;
        m_AccRawDiff.z = acc_max.z - acc_min.z;

        m_GyroRawDiff.x = gyro_max.x - gyro_min.x;
        m_GyroRawDiff.y = gyro_max.y - gyro_min.y;
        m_GyroRawDiff.z = gyro_max.z - gyro_min.z;

        if(m_AccRawDiff.x > acc_diffmax_x) acc_diffmax_x = m_AccRawDiff.x;
        if(m_AccRawDiff.y > acc_diffmax_y) acc_diffmax_y = m_AccRawDiff.y;
        if(m_AccRawDiff.z > acc_diffmax_z) acc_diffmax_z = m_AccRawDiff.z;

        if(m_GyroRawDiff.x > gyro_diffmax_x) gyro_diffmax_x = m_GyroRawDiff.x;
        if(m_GyroRawDiff.y > gyro_diffmax_y) gyro_diffmax_y = m_GyroRawDiff.y;
        if(m_GyroRawDiff.z > gyro_diffmax_z) gyro_diffmax_z = m_GyroRawDiff.z;

        m_LastCalcLatestDataIndex = latestIndex;
    }

}




/*
    Desc: グラフデータ更新
*/
void GyroEvaSampling::UpdateGraph()
{
    // ジャイロ・加速度生値グラフを作成
    {
        s64 latestIndex = m_ExtReader->GetExtdevCalib().loopBuffer->LatestDataIndex();
        s64 startIndex = (m_GyroRawLastAddedIndex == 0) ? 0 : m_GyroRawLastAddedIndex + 1;

        // グラフデータ更新
        for (s64 idx = startIndex; idx <= latestIndex; idx++)
        {
            nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);
            m_GyroRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpGyro.x);
            m_GyroRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpGyro.y);
            m_GyroRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpGyro.z);

            nn::hid::AccelerometerStatus tmpAcc = m_ExtReader->GetAccCalib().loopBuffer->GetData(idx);
            m_AccRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpAcc.x);
            m_AccRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpAcc.y);
            if (m_IsGraphRangeSmall){
                // 通常設置状態でグラフが表示範囲外になってしまうのでオフセットします
                m_AccRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpAcc.z + 1024);
            } else {
                m_AccRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpAcc.z);
            }
            // 生値をSDに保存
            if (m_IsSDCardDetected && m_IsLogging)
            {
                if (!m_GyroLogger.Printf(
                        "%d,%d,%d," "%d,%d,%d\n",
                        tmpGyro.x, tmpGyro.y, tmpGyro.z, tmpAcc.x, tmpAcc.y, tmpAcc.z)
                    )
                {
                    NN_LOG("Logger buffer is Over!\n");
                    // バッファが溢れたので現時点までのデータをSD保存
                    m_GyroLogger.Flush();
                }
            }

        }
        m_GyroRawLastAddedIndex = latestIndex;
    }

    // キャリブレ済みジャイロデータ
    {
        s64 latestIndex = m_ExtReader->GetExtdev().loopBuffer->LatestDataIndex();
        s64 startIndex = (m_GyroLastAddedIndex == 0) ? 0 : m_GyroLastAddedIndex + 1;

        // グラフデータ更新
        for (s64 idx = startIndex; idx <= latestIndex; idx++)
        {
            nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdev().loopBuffer->GetData(idx);
            m_GyroLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpGyro.x);
            m_GyroLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpGyro.y);
            m_GyroLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpGyro.z);
        }
        m_GyroLastAddedIndex = latestIndex;
    }

}




/*
    Desc: 画面更新
*/
s32 GyroEvaSampling::UpdateScreen(sys::GraphicsDrawing* gfx, s64 elapsed_time)
{

    enum
    {
        TEST_OK = 0,
        NG_X = 1,
        NG_Y = 2,
        NG_XY = 3,
        NG_Z = 4,
        NG_XZ = 5,
        NG_YZ = 6,
        NG_XYZ = 7
    };

    s32 sample_result = 0;
    s32 ng_code = 0;
    s32 gyro_result = 0;

    // グラフにジャイロの生値を表示するかキャリブレ値を表示するか切り替えるフラグ(キャリブレ値を表示する場合は0にしてビルドし直す。 FLOWERではキャリブレ値を見る予定が無いのでこの変数を削除してもいいかも)
    m_IsGyroGraphRaw = 1;

    // コンフィグ用
    Config::Member accMember;
    Config accConfig;
    accMember = accConfig.Get();

    /* ------------------------------------------------------------------------
                上画面描画
    ------------------------------------------------------------------------ */
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);
    gfx->m_DrawFramework->Clear();

    const f32 BASE_GYRO_DATA_X = 10.0f;
    const f32 BASE_ACC_DATA_X = 205.0f;
    const f32 BASE_DATA_Y = 200.0f;

    if(gyro_diffmax_x > accMember.Sampling_GyroX_Judge) check_gyro_x_flag = 1;
    if(gyro_diffmax_y > accMember.Sampling_GyroY_Judge) check_gyro_y_flag = 1;
    if(gyro_diffmax_z > accMember.Sampling_GyroZ_Judge) check_gyro_z_flag = 1;
    gyro_result = !check_gyro_x_flag + ((!check_gyro_y_flag) << 1) + ((!check_gyro_z_flag) << 2);

    gfx->m_DrawFramework->SetFontSize(8.0f);

    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y-30,     " S:%d  ", gyro_result );
    gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y-20,     " X:%4d Y:%4d Z:%4d\n", accMember.Sampling_GyroX_Judge, accMember.Sampling_GyroY_Judge, accMember.Sampling_GyroZ_Judge );

    if (m_IsGyroGraphRaw)
    {
        // ジャイロ生値
        ReaderImpl<nn::hid::GyroscopeLowCalibrator, nn::hid::GyroscopeLowStatus> extDevCalib = m_ExtReader->GetExtdevCalib();
        gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
        if(check_gyro_x_flag==1) gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X: %3d(p-p: %3d) %3d 0\n", extDevCalib.latestStatus.x, m_GyroRawDiff.x, gyro_diffmax_x);
        else                     gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X: %3d(p-p: %3d) %3d X\n", extDevCalib.latestStatus.x, m_GyroRawDiff.x, gyro_diffmax_x);

        gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
        if(check_gyro_y_flag==1) gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y: %3d(p-p: %3d) %3d 0\n", extDevCalib.latestStatus.y, m_GyroRawDiff.y, gyro_diffmax_y);
        else                     gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y: %3d(p-p: %3d) %3d X\n", extDevCalib.latestStatus.y, m_GyroRawDiff.y, gyro_diffmax_y);

        gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
        if(check_gyro_z_flag==1) gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z: %3d(p-p: %3d) %3d 0\n", extDevCalib.latestStatus.z, m_GyroRawDiff.z, gyro_diffmax_z);
        else                     gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z: %3d(p-p: %3d) %3d X\n", extDevCalib.latestStatus.z, m_GyroRawDiff.z, gyro_diffmax_z);

        m_GyroRawGraphDrawing->Draw(gfx);
    }
    else
    {
        // ジャイロキャリブレ値
        gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
        if(check_gyro_x_flag==1) gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X: %3d %3d 0\n", m_ExtReader->GetExtdev().latestStatus.x, gyro_diffmax_x);
        else                     gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X: %3d %3d X\n", m_ExtReader->GetExtdev().latestStatus.x, gyro_diffmax_x);

        gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
        if(check_gyro_y_flag==1) gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y: %3d %3d 0\n", m_ExtReader->GetExtdev().latestStatus.y, gyro_diffmax_y);
        else                     gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y: %3d %3d X\n", m_ExtReader->GetExtdev().latestStatus.y, gyro_diffmax_y);

        gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
        if(check_gyro_z_flag==1) gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z: %3d %3d 0\n", m_ExtReader->GetExtdev().latestStatus.z, gyro_diffmax_z);
        else                     gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z: %3d %3d X\n", m_ExtReader->GetExtdev().latestStatus.z, gyro_diffmax_z);
        m_GyroGraphDrawing->Draw(gfx);
    }

    if(acc_diffmax_x  > accMember.Sampling_AccX_Judge)  check_acc_x_flag = 1;
    if(acc_diffmax_y  > accMember.Sampling_AccY_Judge)  check_acc_y_flag = 1;
    if(acc_diffmax_z  > accMember.Sampling_AccZ_Judge)  check_acc_z_flag = 1;
    ng_code = !check_acc_x_flag + ((!check_acc_y_flag) << 1) + ((!check_acc_z_flag) << 2);

    // 加速度生値
    ReaderImpl<nn::hid::AccelerometerCalibrator, nn::hid::AccelerometerStatus> accCalib = m_ExtReader->GetAccCalib();
    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y-30,     " S:%d  T:%lld", ng_code, elapsed_time );
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y-20,     " X:%4d Y:%4d Z:%4d\n", accMember.Sampling_AccX_Judge, accMember.Sampling_AccY_Judge, accMember.Sampling_AccZ_Judge );

    gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
    if(check_acc_x_flag==1) gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y,        "X: %4d(p-p: %3d) %3d 0\n", accCalib.latestStatus.x, m_AccRawDiff.x, acc_diffmax_x);
    else                    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y,        "X: %4d(p-p: %3d) %3d X\n", accCalib.latestStatus.x, m_AccRawDiff.x, acc_diffmax_x);

    gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
    if(check_acc_y_flag==1) gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+10.0f,  "Y: %4d(p-p: %3d) %3d 0\n", accCalib.latestStatus.y, m_AccRawDiff.y, acc_diffmax_y);
    else                    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+10.0f,  "Y: %4d(p-p: %3d) %3d X\n", accCalib.latestStatus.y, m_AccRawDiff.y, acc_diffmax_y);

    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
    if(check_acc_z_flag==1) gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+20.0f,  "Z: %4d(p-p: %3d) %3d 0\n", accCalib.latestStatus.z, m_AccRawDiff.z, acc_diffmax_z);
    else                    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+20.0f,  "Z: %4d(p-p: %3d) %3d X\n", accCalib.latestStatus.z, m_AccRawDiff.z, acc_diffmax_z);
    m_AccRawGraphDrawing->Draw(gfx);

    gfx->m_DrawFramework->SwapBuffers();

    /* ------------------------------------------------------------------------
               下画面描画
    ------------------------------------------------------------------------ */
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();

    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(0,  0, "A: LOG Start/Stop <NOW %s>", m_IsLogging ? "Recording":"Standby");
    gfx->m_DrawFramework->DrawText(0, 10, "B: Exit To Menu(Save To SD-CARD)");
    gfx->m_DrawFramework->DrawText(0, 20, "X: Switch Graph Range(Wide/Narrow)");
    gfx->m_DrawFramework->DrawText(0, 30, "Y: Switch GYRO Graph(RAW/CALB)");
    gfx->m_DrawFramework->DrawText(0, 50, "SD-CARAD(%s)", m_IsSDCardDetected ? "Detected" : "NOT Detected");

    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);

    return ng_code + gyro_result * 100;
}




/* ------------------------------------------------------------------------
    Desc: ジャイロ・加速度の貼り付き検査
          全ての軸の最大値と最小値の差が一定以上開くまで
          サンプリンググラフ表示を表示します。
          ※表示は速度優先の為、RenderSystemの文字・図形描画を使用します。
   ------------------------------------------------------------------------*/
seq::TestResult GyroEvaSampling::Run2()
{

    NN_LOG("\n\n\nRun2 Wait X \n\n\n");

    seq::TestResult acc_result;

    // 時間計測用
    nn::os::Tick start_tick;
    s64 test_time = 0;

    s32 CheckButtonBFlag = 0;

    // コンフィグ用
    Config::Member accMember;
    Config accConfig;
    accMember = accConfig.Get();

    typedef struct {
        char title[0x20];
        f32 coefficient;
    } GraphMode;

    // 検査モードを画面に表示する際に使う
    const char TestMode[10][15]={
        "LINE",
        "QC",
        "REPAIR",
        "SC",
        "SUBSIDIARY",
        "SUBMIT",
        "RESERVED1",
        "RESERVED2",
        "RESERVED3",
        "DEBUG",
    };

    nn::Result nnr;
    s32 sample_result = 0;
    string line;
    const wstring SD_PATH_DP0_4 = L"sdmc:/uji/Stick/AccGyroMax.csv";
    u32 sd_ng_code = RESULT_OK;
    bit8 mac[nn::uds::MAC_ADDRESS_SIZE];

    char strTestMode[20] = "";              // LINE,QCなどのテストモード文字列  (項目名：TestMode)
    char str_config_crc[10] = "";           // コンフィグのCRCを取得後、文字列作成  (項目名：CONFIG_CRC)
    char str_mac[32];                       // MACアドレス文字列
    char m_serialNo[nn::cfg::CTR::detail::CFG_SECURE_INFO_SERIAL_NO_LEN +1];//シリアルNO
    char result_str[128];
    char strTime[32];
    char strMaxXYZ[128];
    char str_error[64] = "";

    nn::hid::GyroscopeLowReader extDevReader;
    nn::hid::GyroscopeLowCalibrator extDevCalibrator;
    nn::hid::AccelerometerCalibrator accCalibrator;
    // 現在はキャリブレ済みのデータではなく軸を変換しただけのデータの為
    // SDK0.13リリース後は差し替えの必要あり。
    // TODO:

    check_acc_x_flag = 0;   // 加速度センサx軸の貼り付きがないことが確認できると１になるフラグ
    check_acc_y_flag = 0;   // 加速度センサy軸の貼り付きがないことが確認できると１になるフラグ
    check_acc_z_flag = 0;   // 加速度センサz軸の貼り付きがないことが確認できると１になるフラグ
    check_gyro_x_flag = 0;  // ジャイロセンサx軸の貼り付きがないことが確認できると１になるフラグ
    check_gyro_y_flag = 0;  // ジャイロセンサy軸の貼り付きがないことが確認できると１になるフラグ
    check_gyro_z_flag = 0;  // ジャイロセンサz軸の貼り付きがないことが確認できると１になるフラグ

    acc_diffmax_x = 0;
    acc_diffmax_y = 0;
    acc_diffmax_z = 0;

    gyro_diffmax_x = 0;
    gyro_diffmax_y = 0;
    gyro_diffmax_z = 0;

    m_ExtReader->Initialize();

    // サンプリング開始
    m_ExtReader->SamplingStart(
        extDevCalibrator,
        extDevReader,
        accCalibrator);


    /*
        グラフデータ初期化
    */
    // ジャイロ（生値）
    GraphMode GraphGyroRaw[] = {
        {"GYRO(RAW)",               static_cast<f32>(GRAPH_HEIGHT)/65536.0f},
        {"GYRO(RAW) Range:+-575",   static_cast<f32>(GRAPH_HEIGHT)/(575.0f*2.0f)}
    };
    m_GyroRawGraphDrawing =
        new GyroGraphDrawing(
            GraphGyroRaw[m_IsGraphRangeSmall].title,
            GraphGyroRaw[m_IsGraphRangeSmall].coefficient,
            10.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_X]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Y]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Z]);

    // ジャイロ（キャリブレ済み）
    m_GyroGraphDrawing =
        new GyroGraphDrawing(
            "GYRO(CALIB)",
            static_cast<f32>(GRAPH_HEIGHT)/65536.0f,
            10.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_X]);
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_Y]);
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_Z]);

    // 加速度（生値）
    GraphMode GraphAccRaw[] = {
        {"ACCL(RAW)",               static_cast<f32>(GRAPH_HEIGHT)/16384.0f},
        {"ACCL(RAW) Range:+-100",   static_cast<f32>(GRAPH_HEIGHT)/200.0f}
    };
    m_AccRawGraphDrawing  =
        new GyroGraphDrawing(
            GraphAccRaw[m_IsGraphRangeSmall].title,
            GraphAccRaw[m_IsGraphRangeSmall].coefficient,
            205.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_X]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Y]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Z]);

    // 時間計測開始
    start_tick = nn::os::Tick::GetSystemCurrent();

    do
    {
        // グラフデータの更新
        UpdateGraph();

        // 1s間のMax-Minを算出
        CalculateMaxMin();

        test_time = GetMilliSecond(start_tick);

        // 画面更新
        sample_result = UpdateScreen(sys::GraphicsDrawing::GetInstance(), test_time);

        // 検査開始から一定時間が経過している場合
        if ( (test_time) >= accMember.SamplingTimeOut * 1000 )
        {
            break;
        }
    }
    while( (sample_result != 0) || (test_time/1000 < accMember.SamplingInvalidTime) );


    /*
        終了処理
    */
    if (m_IsSDCardDetected)
    {
        m_GyroLogger.Flush();
        m_GyroLogger.CloseFile();

        // SD カード アンマウント
        NN_UTIL_PANIC_IF_FAILED(nn::fs::Unmount("sdmc:"));
    }

    m_ExtReader->SamplingStop();
    m_ExtReader->Finalize();

    delete m_GyroRawGraphDrawing;
    delete m_GyroGraphDrawing;
    delete m_AccRawGraphDrawing;

    // restore gl state ※RenderSystemの文字描画実行の際は必ず戻すこと
    for (int index = 0; index < 12; index++)
        glDisableVertexAttribArray(index);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_SCISSOR_TEST);
    glDisable(GL_STENCIL_TEST);
    glDisable(GL_BLEND);
    glActiveTexture(GL_TEXTURE0);
    glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
    glDepthMask(GL_TRUE);

    acc_result.m_Micro = sample_result;

    if(acc_result.m_Micro==0) sprintf(result_str, ",OK,");
    else sprintf(result_str, ",%04u,", acc_result.m_Micro);

    // LINE,QCなどのテストモード文字列  (項目名：TestMode)
    sprintf(strTestMode, ",%s,", TestMode[accMember.TestMode]);

    // コンフィグのCRCを取得後、文字列作成  (項目名：ConfigCRC)
    uji::seq::Config *c = new uji::seq::Config;
    sprintf(str_config_crc, "%d,", c->GetCheckCode());
    delete c;

    //ＭＡＣアドレス取得（ＵＤＳを使用する）し、MacAddress文字列  (項目名：MacAddress)を作成
    sys::GetMacAddress(mac);
    sprintf(str_mac, "[%02X::%02X::%02X::%02X::%02X::%02X],", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

    // SerialNumberの取得
    uji::sys::GetSerialNumber( m_serialNo );

    sprintf( strTime, "%04lld,", test_time );
    sprintf( strMaxXYZ, "%d,%d,%d,%d,%d,%d\n", gyro_diffmax_x, gyro_diffmax_y, gyro_diffmax_z, acc_diffmax_x, acc_diffmax_y, acc_diffmax_z );


    line = static_cast<string>(VERSION_STRING)      // 検査プログラムのバージョン
        + static_cast<string>(strTestMode)      // テストモード
        + static_cast<string>(str_config_crc)   // コンフィグのCRC
        + static_cast<string>(str_mac)          // マックアドレス
        + static_cast<string>(m_serialNo)       // シリアルナンバー
        + static_cast<string>(result_str)       // エラーコード(項目名：Result)
        + static_cast<string>(strTime)          // サンプリングにかかった時間
        + static_cast<string>(strMaxXYZ);       // 加速度・ジャイロセンサの各軸サンプリング値の最大値-最小値(各軸の値が一定以上であれば検査OK)

    if( accMember.TestMode == uji::seq::pl::LINE )
    {

        // SDカードのマウント
        nnr = nn::fs::MountSdmc();
        if (nnr.IsFailure())
        {
            sample_result = SD_WRITE_NG_1;
            strcpy(str_error, "CHECK SD CARD");
        }
        else
        {
            // ujiディレクトリチェック（無ければ新規作成）
            sd_ng_code = CreateDirectorySdmc(L"sdmc:/uji");

            if( sd_ng_code != RESULT_OK )
            {
                sample_result = SD_WRITE_NG_2;
                strcpy(str_error, "MAKE uji DIRECTORY ERROR");
            }
            else
            {

                // /uji/Stickディレクトリチェック（無ければ新規作成）
                sd_ng_code = CreateDirectorySdmc(L"sdmc:/uji/Stick");

                if( sd_ng_code != RESULT_OK )
                {
                    // 1回目のディレクトリ作成NGと見分けるためにNGコードに+1しています。
                    sample_result = SD_WRITE_NG_3;
                    strcpy(str_error, "MAKE uji/Stick DIRECTORY ERROR");
                }
                else
                {
                    // SDカードに出力
                    sd_ng_code = WriteData2SD( SD_PATH_DP0_4, line );
                    if( sd_ng_code != RESULT_OK )
                    {
                        sample_result = SD_WRITE_NG_4;
                        strcpy(str_error, "WRITE SD DATA ERROR");
                    }
                }

            }

            // SDカードのアンマウント
            NN_LOG("sdmc archive unmount ");
            nnr = nn::fs::Unmount("sdmc:");
            if ( nnr.IsFailure())
            {
                sample_result = SD_WRITE_NG_5;
                strcpy(str_error, "UNMOUNT SD ERROR");
            }
        }
    }

    if(sample_result == 0)
    {
            sprintf( acc_result.m_String, "OK\n GYRO-X:%d Y:%d Z:%d  ACC-X:%d Y:%d Z:%d", gyro_diffmax_x, gyro_diffmax_y, gyro_diffmax_z, acc_diffmax_x, acc_diffmax_y, acc_diffmax_z );
            acc_result.m_Result = 1;    // OKの場合
    }
    else
    {
        sprintf( acc_result.m_String, "NG_CODE:%04d  %s\n GYRO-X:%d Y:%d Z:%d  ACC-X:%d Y:%d Z:%d ", sample_result, str_error, gyro_diffmax_x, gyro_diffmax_y, gyro_diffmax_z, acc_diffmax_x, acc_diffmax_y, acc_diffmax_z );
        acc_result.m_Result = 0;                // NGの場合
    }

    return acc_result;
}




/*!--------------------------------------------------------------------------*
  @brief        時間計測関数
  引数      計測開始時点
  戻り値    start_timeから経過した時間(mS)
 *---------------------------------------------------------------------------*/
static s64 GetMilliSecond( nn::os::Tick start_time )
{

    nn::os::Tick currentTick = nn::os::Tick::GetSystemCurrent();

    float millisecond = (float)((currentTick - start_time).ToTimeSpan().GetMicroSeconds() / 1000.0);

    return millisecond;

}


/*---------------------------------------------------------------------------
  Desc: SDカードにデータを書き込みます

  Args: path        対象ファイルのフルパス
        strBuf      書き込むデータ

  Rtns: なし
---------------------------------------------------------------------------*/
static u32 WriteData2SD(const wstring& path, string strBuf)
{

    const char TITLE[256] = "Ver,"              // バージョン情報
                            "TestMode,"         // LINE,QCなどのテストモード
                            "ConfigCRC,"        // コンフィグのCRC
                            "MacAddress,"       // ＭＡＣアドレス
                            "SerialNo,"         // シリアルナンバー
                            "Result,"           // 検査結果
                            "Time,"             // 検査にかかった時間
                            "GYRO-X,"           // ジャイロセンサX軸最大値
                            "GYRO-Y,"           // ジャイロセンサY軸最大値
                            "GYRO-Z,"           // ジャイロセンサZ軸最大値
                            "ACC-X,"            // 加速度センサX軸最大値
                            "ACC-Y,"            // 加速度センサY軸最大値
                            "ACC-Z,"            // 加速度センサZ軸最大値
                            "\n";

    // ファイルを書き込みためのオブジェクト作成
    nn::fs::FileStream fos;
    nn::Result result = fos.TryInitialize(path.c_str(), nn::fs::OPEN_MODE_READ   |
                                                        nn::fs::OPEN_MODE_WRITE  |
                                                        nn::fs::OPEN_MODE_CREATE );
    if(result.IsSuccess())
    {
        s64 size = fos.GetSize();
        NN_LOG("FILE_SIZE= %lld\n", size);

        if(size == 0)
        {
            // 初回のみ上記TITLE文字列をファイルに書き込む
            fos.Write(TITLE, strlen(TITLE), true);
        }
        else
        {
            // 二回目以降はファイルの終端に移動
            fos.Seek( -1, nn::fs::POSITION_BASE_END );
        }
        fos.Write(strBuf.c_str(), strlen(strBuf.c_str()), true);
        fos.Write("@", 1, true);
    }
    else return RESULT_NG;

    // ファイルを閉じる
    fos.Finalize();

    return RESULT_OK;
}


/*---------------------------------------------------------------------------
  Desc: SDカードに指定フォルダが存在するか確認します。(フォルダが無い場合はフォルダを作成します。)

  Args: path        対象フォルダのフルパス

  Rtns: OK/NG
---------------------------------------------------------------------------*/
s32 CreateDirectorySdmc(const wchar_t* path)
{
    nn::fs::Directory dir;
    nn::Result result = dir.TryInitialize( path );

    // ディレクトリがない場合は新規作成
    if(result.IsFailure())
    {
        NN_LOG("Can't Initialize! So Create Directory... \n");

        result = nn::fs::TryCreateDirectory( path );

        dir.Finalize();

        if(result.IsFailure())
        {
            NN_LOG("* Warning: No Create Directory! \n");
            return RESULT_NG;
        }
    }
    return RESULT_OK;
}



} // namespace forF
} // namespace
}

