/*--------------------------------------------------------------------------*
 Project:
 File: gyro_EvaSampling.cpp


*--------------------------------------------------------------------------*/
#include "gyro_EvaSampling.h"
#include <nn/fs/CTR/MPCore/fs_ApiForHwCheck.h>
#include "../sound/EvaSound.h"

using namespace uji::seq;


namespace uji {
namespace eva {

#define ALARM_FLAG_OFF   0
#define ALARM_FLAG_ON    1

/*
    Desc: 1sec間（100サンプル）のMax-Minを算出する
*/
void GyroEvaSampling::CalculateMaxMin(bool soundFlag)
{
    s64 latestIndex = m_ExtReader->GetExtdevCalib().loopBuffer->LatestDataIndex();

    Config::Member cnfMember;                   // コンフィグ用
    Config accConfig;                           // コンフィグ用
    cnfMember = accConfig.Get();

    static bool flag = 0;

    // 100サンプルサンプリングされたら算出
    if ((latestIndex - m_LastCalcLatestDataIndex) >= cnfMember.SampleStabilityNum)
    {
        nn::hid::AccelerometerStatus acc_max = {-37268/2, -32768/2, -32768/2};
        nn::hid::AccelerometerStatus acc_min = { 37267/2,  32767/2,  32767/2};
        nn::hid::GyroscopeLowStatus gyro_max = {-37268/2, -32768/2, -32768/2};
        nn::hid::GyroscopeLowStatus gyro_min = { 37267/2,  32767/2,  32767/2};

        // Max-Minを算出
        for (s64 idx = m_LastCalcLatestDataIndex; idx <= latestIndex; idx++)
        {
            {
                nn::hid::AccelerometerStatus tmpAcc = m_ExtReader->GetAccCalib().loopBuffer->GetData(idx);

                // 加速度生値 最大／最小値の更新
                if (acc_max.x < tmpAcc.x) { acc_max.x = tmpAcc.x; }
                if (acc_max.y < tmpAcc.y) { acc_max.y = tmpAcc.y; }
                if (acc_max.z < tmpAcc.z) { acc_max.z = tmpAcc.z; }
                if (acc_min.x > tmpAcc.x) { acc_min.x = tmpAcc.x; }
                if (acc_min.y > tmpAcc.y) { acc_min.y = tmpAcc.y; }
                if (acc_min.z > tmpAcc.z) { acc_min.z = tmpAcc.z; }
            }

            {
                nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);

                // ジャイロ生値 最大／最小値の更新
                if (gyro_max.x < tmpGyro.x) { gyro_max.x = tmpGyro.x; }
                if (gyro_max.y < tmpGyro.y) { gyro_max.y = tmpGyro.y; }
                if (gyro_max.z < tmpGyro.z) { gyro_max.z = tmpGyro.z; }
                if (gyro_min.x > tmpGyro.x) { gyro_min.x = tmpGyro.x; }
                if (gyro_min.y > tmpGyro.y) { gyro_min.y = tmpGyro.y; }
                if (gyro_min.z > tmpGyro.z) { gyro_min.z = tmpGyro.z; }
            }
        }

        m_AccRawDiff.x = acc_max.x - acc_min.x;
        m_AccRawDiff.y = acc_max.y - acc_min.y;
        m_AccRawDiff.z = acc_max.z - acc_min.z;

        m_GyroRawDiff.x = gyro_max.x - gyro_min.x;
        m_GyroRawDiff.y = gyro_max.y - gyro_min.y;
        m_GyroRawDiff.z = gyro_max.z - gyro_min.z;

        m_LastCalcLatestDataIndex = latestIndex;
    }

    if(soundFlag==ALARM_FLAG_ON)
    {
        if( ( m_AccRawDiff.x > cnfMember.AccSampleStblRange )   ||
            ( m_AccRawDiff.y > cnfMember.AccSampleStblRange )   ||
            ( m_AccRawDiff.z > cnfMember.AccSampleStblRange )   ||
            ( m_GyroRawDiff.x > cnfMember.GyroSampleStblRange ) ||
            ( m_GyroRawDiff.y > cnfMember.GyroSampleStblRange ) ||
            ( m_GyroRawDiff.z > cnfMember.GyroSampleStblRange )    )
        {
            EvaSoundStability(SAMPLING_SOUND_ON);
            oldAccRawDiff = m_AccRawDiff;
            oldGyroRawDiff = m_GyroRawDiff;
        }
        else
        {
            EvaSoundStability(SAMPLING_SOUND_OFF);
        }
    }
}

/*
    Desc: グラフデータ更新
*/
void GyroEvaSampling::UpdateGraph()
{
    // ジャイロ・加速度生値グラフを作成
    {
        s64 latestIndex = m_ExtReader->GetExtdevCalib().loopBuffer->LatestDataIndex();
        s64 startIndex = (m_GyroRawLastAddedIndex == 0) ? 0 : m_GyroRawLastAddedIndex + 1;

        // グラフデータ更新
        for (s64 idx = startIndex; idx <= latestIndex; idx++)
        {
            nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);
            m_GyroRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpGyro.x);
            m_GyroRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpGyro.y);
            m_GyroRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpGyro.z);

            nn::hid::AccelerometerStatus tmpAcc = m_ExtReader->GetAccCalib().loopBuffer->GetData(idx);
            m_AccRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpAcc.x);
            m_AccRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpAcc.y);
            if (m_IsGraphRangeSmall){
                // 通常設置状態でグラフが表示範囲外になってしまうのでオフセットします
                m_AccRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpAcc.z + 1024);
            } else {
                m_AccRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpAcc.z);
            }
            // 生値をSDに保存
            if (m_IsSDCardDetected && m_IsLogging)
            {
                if (!m_GyroLogger.Printf(
                        "%d,%d,%d," "%d,%d,%d\n",
                        tmpGyro.x, tmpGyro.y, tmpGyro.z, tmpAcc.x, tmpAcc.y, tmpAcc.z)
                    )
                {
                    NN_LOG("Logger buffer is Over!\n");
                    // バッファが溢れたので現時点までのデータをSD保存
                    m_GyroLogger.Flush();
                }
            }

        }
        m_GyroRawLastAddedIndex = latestIndex;
    }

    // キャリブレ済みジャイロデータ
    {
        s64 latestIndex = m_ExtReader->GetExtdev().loopBuffer->LatestDataIndex();
        s64 startIndex = (m_GyroLastAddedIndex == 0) ? 0 : m_GyroLastAddedIndex + 1;

        // グラフデータ更新
        for (s64 idx = startIndex; idx <= latestIndex; idx++)
        {
            nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdev().loopBuffer->GetData(idx);
            m_GyroLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpGyro.x);
            m_GyroLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpGyro.y);
            m_GyroLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpGyro.z);
        }
        m_GyroLastAddedIndex = latestIndex;
    }

#if 0
    /*
       加速度生値グラフを作成
       ※現状、ExtReaderではジャイロ生値のタイミングで加速度も取得している為
         上記のコード内に加速度取得もまとめているため未使用。
    */
    {
       s64 latestIndex = m_ExtReader->GetAccCalib().loopBuffer->LatestDataIndex();

        // グラフデータ更新
        for (s64 idx = m_AccLastAddedIndex; idx <= latestIndex; idx++)
        {
            nn::hid::AccelerometerStatus tmpAcc = m_ExtReader->GetAccCalib().loopBuffer->GetData(idx);
            m_AccRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpAcc.x);
            m_AccRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpAcc.y);
            m_AccRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpAcc.z);
        }
        m_AccLastAddedIndex = latestIndex;
    }
#endif

}

/*
    Desc: 画面更新
*/
void GyroEvaSampling::UpdateScreen(sys::GraphicsDrawing* gfx)
{
    /* ------------------------------------------------------------------------
                上画面描画
    ------------------------------------------------------------------------ */
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);
    gfx->m_DrawFramework->Clear();

    const f32 BASE_GYRO_DATA_X = 10.0f;
    const f32 BASE_ACC_DATA_X = 205.0f;
    const f32 BASE_DATA_Y = 200.0f;

    Config::Member cnfMember;                   // コンフィグ用
    Config accConfig;                           // コンフィグ用
    cnfMember = accConfig.Get();

    gfx->m_DrawFramework->SetFontSize(8.0f);

    if (m_IsGyroGraphRaw)
    {
        // ジャイロ生値
        ReaderImpl<nn::hid::GyroscopeLowCalibrator, nn::hid::GyroscopeLowStatus> extDevCalib = m_ExtReader->GetExtdevCalib();
        gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y-10.0f,  "LIMIT:%d\n", cnfMember.GyroSampleStblRange);
        gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X:%05d(p-p:%04d) %d\n", extDevCalib.latestStatus.x, m_GyroRawDiff.x, oldGyroRawDiff.x);
        gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y:%05d(p-p:%04d) %d\n", extDevCalib.latestStatus.y, m_GyroRawDiff.y, oldGyroRawDiff.y);
        gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z:%05d(p-p:%04d) %d\n", extDevCalib.latestStatus.z, m_GyroRawDiff.z, oldGyroRawDiff.z);
        m_GyroRawGraphDrawing->Draw(gfx);
    }
    else
    {
        // ジャイロキャリブレ値
        gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X:%6d\n", m_ExtReader->GetExtdev().latestStatus.x);
        gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y:%6d\n", m_ExtReader->GetExtdev().latestStatus.y);
        gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z:%6d\n", m_ExtReader->GetExtdev().latestStatus.z);
        m_GyroGraphDrawing->Draw(gfx);
    }
    // 加速度生値
    ReaderImpl<nn::hid::AccelerometerCalibrator, nn::hid::AccelerometerStatus> accCalib = m_ExtReader->GetAccCalib();

    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y-10.0f,  "LIMIT:%d\n", cnfMember.AccSampleStblRange);
    gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y,        "X:%05d(p-p:%04d) %d\n", accCalib.latestStatus.x, m_AccRawDiff.x, oldAccRawDiff.x);
    gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+10.0f,  "Y:%05d(p-p:%04d) %d\n", accCalib.latestStatus.y, m_AccRawDiff.y, oldAccRawDiff.y);
    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);
    gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+20.0f,  "Z:%05d(p-p:%04d) %d\n", accCalib.latestStatus.z, m_AccRawDiff.z, oldAccRawDiff.z);
    m_AccRawGraphDrawing->Draw(gfx);

    gfx->m_DrawFramework->SwapBuffers();

    /* ------------------------------------------------------------------------
               下画面描画
    ------------------------------------------------------------------------ */
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();

    gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 0.0f);
    gfx->m_DrawFramework->DrawText(0,  0, "A: LOG Start/Stop <NOW %s>", m_IsLogging ? "Recording":"Standby");
    gfx->m_DrawFramework->DrawText(0, 10, "B: Exit To Menu(Save To SD-CARD)");
    gfx->m_DrawFramework->DrawText(0, 20, "X: Switch Graph Range(Wide/Narrow)");
    gfx->m_DrawFramework->DrawText(0, 30, "Y: Switch GYRO Graph(RAW/CALB)");
    gfx->m_DrawFramework->DrawText(0, 50, "SD-CARAD(%s)", m_IsSDCardDetected ? "Detected" : "NOT Detected");

    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
}

/*
    Desc: ジャイロ・加速度のサンプリンググラフ表示
          ※表示は速度優先の為、RenderSystemの文字・図形描画を使用します。
          ※SDカードが挿入されている場合は、生値（ジャイロ・加速度）を保存します。
*/
void GyroEvaSampling::Run()
{
    typedef struct {
        char title[0x20];
        f32 coefficient;
    } GraphMode;

    /*
        拡張リーダ
    */
    nn::hid::GyroscopeLowCalibrator extDevCalibrator;
    nn::hid::AccelerometerCalibrator accCalibrator;
    // 現在はキャリブレ済みのデータではなく軸を変換しただけのデータの為
    // SDK0.13リリース後は差し替えの必要あり。
    // TODO:
    nn::hid::GyroscopeLowReader extDevReader;

    m_ExtReader->Initialize();
    // サンプリング開始
    m_ExtReader->SamplingStart(
        extDevCalibrator,
        extDevReader,
        accCalibrator);

    oldAccRawDiff.x = 0;
    oldAccRawDiff.y = 0;
    oldAccRawDiff.z = 0;
    oldGyroRawDiff.x = 0;
    oldGyroRawDiff.y = 0;
    oldGyroRawDiff.z = 0;

    /*
        グラフデータ初期化
    */
    // ジャイロ（生値）
    GraphMode GraphGyroRaw[] = {
        {"GYRO(RAW)",               static_cast<f32>(GRAPH_HEIGHT)/65536.0f},
        {"GYRO(RAW) Range:+-575",   static_cast<f32>(GRAPH_HEIGHT)/(575.0f*2.0f)}
    };
    m_GyroRawGraphDrawing =
        new GyroGraphDrawing(
            GraphGyroRaw[m_IsGraphRangeSmall].title,
            GraphGyroRaw[m_IsGraphRangeSmall].coefficient,
            10.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_X]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Y]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Z]);

    // ジャイロ（キャリブレ済み）
    m_GyroGraphDrawing =
        new GyroGraphDrawing(
            "GYRO(CALIB)",
            static_cast<f32>(GRAPH_HEIGHT)/65536.0f,
            10.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_X]);
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_Y]);
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_Z]);

    // 加速度（生値）
    GraphMode GraphAccRaw[] = {
        {"ACCL(RAW)",               static_cast<f32>(GRAPH_HEIGHT)/16384.0f},
        {"ACCL(RAW) Range:+-100",   static_cast<f32>(GRAPH_HEIGHT)/200.0f}
    };
    m_AccRawGraphDrawing  =
        new GyroGraphDrawing(
            GraphAccRaw[m_IsGraphRangeSmall].title,
            GraphAccRaw[m_IsGraphRangeSmall].coefficient,
            205.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_X]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Y]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Z]);

    /*
        SDカード検出
    */
    if (m_GyroLogger.IsSdmcInserted())
    {
        nn::Result nnResult = nn::fs::MountSdmc();
        if (nnResult.IsSuccess())
        {
            bool result = m_GyroLogger.CreateDirectory();
            NN_LOG("CreateDir(%d)\n", result);

            // ログファイル名はマックアドレスとします
            wchar_t fileName[0x30];
            bit8 mac[nn::uds::MAC_ADDRESS_SIZE];
            sys::GetMacAddress(mac);
            std::swprintf(fileName, sizeof(fileName), L"Graph_%X%X%X%X%X%X.csv", mac[0], mac[1],mac[2],mac[3],mac[4],mac[5]);
            m_GyroLogger.OpenFile(fileName, static_cast<u32>(1024*100), GyroLogger::SDLOG_WRITE_TYPE_CONTINUANCE);

            m_IsSDCardDetected = true;
            m_GyroLogger.Printf("Gyro_X, Gyro_Y, Gyro_Z, Accl_X, Accl_Y, Accl_Z\n");
        }
        else
        {
            m_IsSDCardDetected = false;
        }
    }

    do
    {
        // パッド情報更新
        sys::Pad().UpdatePad();

        if (sys::Pad().IsButtonDown(sys::Pad::BUTTON_A) && m_IsSDCardDetected)
        {
            // SDにログを保存
            m_IsLogging = !m_IsLogging;
        }

        if (sys::Pad().IsButtonDown(sys::Pad::BUTTON_X))
        {
            // グラフレンジの切替
            m_IsGraphRangeSmall = !m_IsGraphRangeSmall;
            // Gyro Raw
            m_GyroRawGraphDrawing->SetTitle(GraphGyroRaw[m_IsGraphRangeSmall].title);
            m_GyroRawGraphDrawing->SetCoefficient(GraphGyroRaw[m_IsGraphRangeSmall].coefficient);
            // Acc Raw
            m_AccRawGraphDrawing->SetTitle(GraphAccRaw[m_IsGraphRangeSmall].title);
            m_AccRawGraphDrawing->SetCoefficient(GraphAccRaw[m_IsGraphRangeSmall].coefficient);
        }

        if (sys::Pad().IsButtonDown(sys::Pad::BUTTON_Y))
        {
            // GYRO生値/キャリブレ済グラフの切替
            m_IsGyroGraphRaw = !m_IsGyroGraphRaw;
        }

        // グラフデータの更新
        UpdateGraph();

        // 1s間のMax-Minを算出
        CalculateMaxMin(ALARM_FLAG_OFF);

        // 画面更新
        UpdateScreen(sys::GraphicsDrawing::GetInstance());
    }
    while(!sys::Pad().IsButtonDown(sys::Pad::BUTTON_B));

    /*
        終了処理
    */
    if (m_IsSDCardDetected)
    {
        m_GyroLogger.Flush();
        m_GyroLogger.CloseFile();

        // SD カード アンマウント
        NN_UTIL_PANIC_IF_FAILED(nn::fs::Unmount("sdmc:"));
    }

    m_ExtReader->SamplingStop();
    m_ExtReader->Finalize();

    delete m_GyroRawGraphDrawing;
    delete m_GyroGraphDrawing;
    delete m_AccRawGraphDrawing;

    // restore gl state ※RenderSystemの文字描画実行の際は必ず戻すこと
    for (int index = 0; index < 12; index++)
        glDisableVertexAttribArray(index);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_SCISSOR_TEST);
    glDisable(GL_STENCIL_TEST);
    glDisable(GL_BLEND);
    glActiveTexture(GL_TEXTURE0);
    glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
    glDepthMask(GL_TRUE);
}


/*
    Desc: ジャイロ・加速度の安定性確認
          サンプリンググラフ表示＋不安定時に警告音出力
*/
void GyroEvaSampling::GyroCheckStability()
{
    typedef struct {
        char title[0x20];
        f32 coefficient;
    } GraphMode;

    /*
        拡張リーダ
    */
    nn::hid::GyroscopeLowCalibrator extDevCalibrator;
    nn::hid::AccelerometerCalibrator accCalibrator;
    // 現在はキャリブレ済みのデータではなく軸を変換しただけのデータの為
    // SDK0.13リリース後は差し替えの必要あり。
    // TODO:
    nn::hid::GyroscopeLowReader extDevReader;

    m_ExtReader->Initialize();
    // サンプリング開始
    m_ExtReader->SamplingStart(
        extDevCalibrator,
        extDevReader,
        accCalibrator);

    oldAccRawDiff.x = 0;
    oldAccRawDiff.y = 0;
    oldAccRawDiff.z = 0;
    oldGyroRawDiff.x = 0;
    oldGyroRawDiff.y = 0;
    oldGyroRawDiff.z = 0;

    /*
        グラフデータ初期化
    */
    // ジャイロ（生値）
    GraphMode GraphGyroRaw[] = {
        {"GYRO(RAW)",               static_cast<f32>(GRAPH_HEIGHT)/65536.0f},
        {"GYRO(RAW) Range:+-575",   static_cast<f32>(GRAPH_HEIGHT)/(575.0f*2.0f)}
    };
    m_GyroRawGraphDrawing =
        new GyroGraphDrawing(
            GraphGyroRaw[m_IsGraphRangeSmall].title,
            GraphGyroRaw[m_IsGraphRangeSmall].coefficient,
            10.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_X]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Y]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Z]);

    // ジャイロ（キャリブレ済み）
    m_GyroGraphDrawing =
        new GyroGraphDrawing(
            "GYRO(CALIB)",
            static_cast<f32>(GRAPH_HEIGHT)/65536.0f,
            10.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_X]);
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_Y]);
    m_GyroGraphDrawing->AddLine(m_GyroLine[gyro::AXIS_RAW_Z]);

    // 加速度（生値）
    GraphMode GraphAccRaw[] = {
        {"ACCL(RAW)",               static_cast<f32>(GRAPH_HEIGHT)/16384.0f},
        {"ACCL(RAW) Range:+-100",   static_cast<f32>(GRAPH_HEIGHT)/200.0f}
    };
    m_AccRawGraphDrawing  =
        new GyroGraphDrawing(
            GraphAccRaw[m_IsGraphRangeSmall].title,
            GraphAccRaw[m_IsGraphRangeSmall].coefficient,
            205.0f,
            10.0f,
            static_cast<f32>(GRAPH_WIDTH),
            static_cast<f32>(GRAPH_HEIGHT),
            nw::ut::FloatColor(1.0f, 1.0f, 1.0f, 1.0f)
    );
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_X]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Y]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Z]);

    /*
        SDカード検出
    */
    if (m_GyroLogger.IsSdmcInserted())
    {
        nn::Result nnResult = nn::fs::MountSdmc();
        if (nnResult.IsSuccess())
        {
            bool result = m_GyroLogger.CreateDirectory();
            NN_LOG("CreateDir(%d)\n", result);

            // ログファイル名はマックアドレスとします
            wchar_t fileName[0x30];
            bit8 mac[nn::uds::MAC_ADDRESS_SIZE];
            sys::GetMacAddress(mac);
            std::swprintf(fileName, sizeof(fileName), L"Graph_%X%X%X%X%X%X.csv", mac[0], mac[1],mac[2],mac[3],mac[4],mac[5]);
            m_GyroLogger.OpenFile(fileName, static_cast<u32>(1024*100), GyroLogger::SDLOG_WRITE_TYPE_CONTINUANCE);

            m_IsSDCardDetected = true;
            m_GyroLogger.Printf("Gyro_X, Gyro_Y, Gyro_Z, Accl_X, Accl_Y, Accl_Z\n");
        }
        else
        {
            m_IsSDCardDetected = false;
        }
    }

    EvaSoundStability(SAMPLING_SOUND_START);

    do
    {
        // パッド情報更新
        sys::Pad().UpdatePad();

        if (sys::Pad().IsButtonDown(sys::Pad::BUTTON_A) && m_IsSDCardDetected)
        {
            // SDにログを保存
            m_IsLogging = !m_IsLogging;
        }

        if (sys::Pad().IsButtonDown(sys::Pad::BUTTON_X))
        {
            // グラフレンジの切替
            m_IsGraphRangeSmall = !m_IsGraphRangeSmall;
            // Gyro Raw
            m_GyroRawGraphDrawing->SetTitle(GraphGyroRaw[m_IsGraphRangeSmall].title);
            m_GyroRawGraphDrawing->SetCoefficient(GraphGyroRaw[m_IsGraphRangeSmall].coefficient);
            // Acc Raw
            m_AccRawGraphDrawing->SetTitle(GraphAccRaw[m_IsGraphRangeSmall].title);
            m_AccRawGraphDrawing->SetCoefficient(GraphAccRaw[m_IsGraphRangeSmall].coefficient);
        }

        if (sys::Pad().IsButtonDown(sys::Pad::BUTTON_Y))
        {
            // GYRO生値/キャリブレ済グラフの切替
            m_IsGyroGraphRaw = !m_IsGyroGraphRaw;
        }

        // グラフデータの更新
        UpdateGraph();

        // 1s間のMax-Minを算出
        CalculateMaxMin(ALARM_FLAG_ON);

        // 画面更新
        UpdateScreen(sys::GraphicsDrawing::GetInstance());
    }
    while(!sys::Pad().IsButtonDown(sys::Pad::BUTTON_B));

    /*
        終了処理
    */
    if (m_IsSDCardDetected)
    {
        m_GyroLogger.Flush();
        m_GyroLogger.CloseFile();

        // SD カード アンマウント
        NN_UTIL_PANIC_IF_FAILED(nn::fs::Unmount("sdmc:"));
    }

    EvaSoundStability(SAMPLING_SOUND_END);

    m_ExtReader->SamplingStop();
    m_ExtReader->Finalize();

    delete m_GyroRawGraphDrawing;
    delete m_GyroGraphDrawing;
    delete m_AccRawGraphDrawing;

    // restore gl state ※RenderSystemの文字描画実行の際は必ず戻すこと
    for (int index = 0; index < 12; index++)
        glDisableVertexAttribArray(index);

    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_SCISSOR_TEST);
    glDisable(GL_STENCIL_TEST);
    glDisable(GL_BLEND);
    glActiveTexture(GL_TEXTURE0);
    glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
    glDepthMask(GL_TRUE);
}



} // namespace
}
