/*--------------------------------------------------------------------------*
 Project:
 File: gyro_AgingTester.cpp
 
 ジャイロのエージングテストです。
 加速度検査工程の先頭で実施します。
 
 久保さん修正あり版

*--------------------------------------------------------------------------*/
#include "gyro_AgingTester.h"
#include "../mcu/McuInitializer.h"
#include "../mcu/TestMcu.h"    
#include "../sys/sys_SoundPlayer.h"
#include "../sys/sys_GetSerialNumber.h"        
#include "../seq/TestIdDefine.h"    
#include "../backlight/BackLightTest.h"    
#include <nn/fs/CTR/MPCore/fs_ApiForHwCheck.h>
#include <nn/srv.h>
#include <nn/mcu.h>
#include <nn/cfg/CTR/cfg_ApiInit.h>
#include <nn/gx/CTR/gx_LcdPowerManager.h>

#ifdef EVA_COMPOSITE
extern       char VERSION_STRING[];
#else
static const char VERSION_STRING[]= UJI_APPVER;
#endif
static const char VERSION_DATE[]  =	__DATE__ " " __TIME__ ;
    

namespace uji {
namespace eva {
    
        
// #define LOGGING_LOW_DATA    
        
//============================================================================
//      MISC
//============================================================================

/*
    Desc: お知らせランプの赤色点灯
*/
void GyroAgingTester::SetInfoLedRed()
{    
    nn::mcu::CTR::InfoLedPattern infoLedPattern =
    {
        0,           // 1フレームの長さ(単位MCUチック/約2ms)
        0,           // 現在の色からフレーム指定色に到達するまでの時間
                     // 通常はframeTimeと同じ値を指定してください
        255,         // 最終フレームを繰り返す回数(255なら無限に繰り返す)
        0,           // 予約
        0            // R1,R2..R32, G1,G2..G32, B1,B2..B32
    };
    
    eva::mcu::McuInitializer().HwCheckInit();
    // MCU プロセスにリクエストを送るためのクラスを構築。このクラスは自動生成される
    nn::mcu::CTR::HwCheck mcu(
        eva::mcu::McuInitializer().GetHSession()    // 確立されたセッションのハンドルを
    );                                              // コンストラクタに指定
    // 赤点灯
    for(int i = 0; i < 32; i++)
    {
        infoLedPattern.frameColors[i] = 255;
    }
    for(int i = 32; i < 96; i++)
    {
        infoLedPattern.frameColors[i] = 0;
    }
    mcu.SetInfoLEDPattern(infoLedPattern);
}

/*
    Desc: ジャイロ・加速度生値グラフ更新
*/
void GyroAgingTester::UpdateGraph()
{
    // ジャイロ・加速度生値グラフを作成        
    s64 latestIndex = m_ExtReader->GetExtdevCalib().loopBuffer->LatestDataIndex();
    s64 startIndex = (m_GyroRawLastAddedIndex == 0) ? 0 : m_GyroRawLastAddedIndex + 1;  

    // グラフデータ更新
    for (s64 idx = startIndex; idx <= latestIndex; idx++)
    {
        nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);
        m_GyroRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpGyro.x);
        m_GyroRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpGyro.y);
        m_GyroRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpGyro.z);
                
        nn::hid::AccelerometerStatus tmpAcc = m_ExtReader->GetAccCalib().loopBuffer->GetData(idx);
        m_AccRawLine[gyro::AXIS_RAW_X]->GetLoopBuffer()->AddNewData(tmpAcc.x);
        m_AccRawLine[gyro::AXIS_RAW_Y]->GetLoopBuffer()->AddNewData(tmpAcc.y);
        m_AccRawLine[gyro::AXIS_RAW_Z]->GetLoopBuffer()->AddNewData(tmpAcc.z);
        
#ifdef LOGGING_LOW_DATA    
        // 生値をSDに保存
        if (m_IsSDCardDetected)
        {
            if (!m_GyroDataLogger.Printf(
                    "%d,%d,%d," "%d,%d,%d\n",
                    tmpGyro.x, tmpGyro.y, tmpGyro.z, tmpAcc.x, tmpAcc.y, tmpAcc.z)
                )
            {
                NN_LOG("Logger buffer is Over!\n");
                // バッファが溢れたので現時点までのデータをSD保存
                m_GyroDataLogger.Flush();
            }      
        }
#endif                
    }
    m_GyroRawLastAddedIndex = latestIndex;    
}

/*
    Desc: 画面更新
*/
void GyroAgingTester::UpdateScreen(sys::GraphicsDrawing* gfx)
{    
    const f32 BASE_GYRO_DATA_X = 10.0f;
    const f32 BASE_ACC_DATA_X = 205.0f;
    const f32 BASE_DATA_Y = 200.0f;
    
    if (m_Is3DMode)    
    {            
        // 3Dモード
        gfx->m_DrawFramework->SetLcdModeUji(NN_GX_DISPLAYMODE_STEREO);

        /* ------------------------------------------------------------------------
                上画面描画 左
        ------------------------------------------------------------------------ */
        gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);    
        gfx->m_DrawFramework->Clear();
        m_WndManager.DrawDisplay0();
    
        gfx->m_DrawFramework->Transfer();
    
        /* ------------------------------------------------------------------------
                上画面描画 右
        ------------------------------------------------------------------------ */
        gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0_EXT);
        gfx->m_DrawFramework->Clear();
        m_WndManager.DrawDisplay0();

        gfx->m_DrawFramework->Transfer();    
   
        gfx->m_DrawFramework->SwapBuffers_ParallaxBarrier();
    }
    else
    {
        // ノーマルモード
        gfx->m_DrawFramework->SetLcdModeUji(NN_GX_DISPLAYMODE_NORMAL);

        /* ------------------------------------------------------------------------
                上画面描画 左
        ------------------------------------------------------------------------ */
        gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY0);    
        gfx->m_DrawFramework->Clear();
        m_WndManager.DrawDisplay0();
    
        gfx->m_DrawFramework->SetFontSize(8.0f);
    
        // ジャイロ生値
        gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y,       "X: %6d\n", m_ExtReader->GetExtdevCalib().latestStatus.x);
        gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+10.0f, "Y: %6d\n", m_ExtReader->GetExtdevCalib().latestStatus.y);
        gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);  
        gfx->m_DrawFramework->DrawText(BASE_GYRO_DATA_X, BASE_DATA_Y+20.0f, "Z: %6d\n", m_ExtReader->GetExtdevCalib().latestStatus.z);
        m_GyroRawGraphDrawing->Draw(gfx);
    
        // 加速度生値
        gfx->m_DrawFramework->SetColor(1.0f, 0.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y,        "X: %6d\n", m_ExtReader->GetAccCalib().latestStatus.x);
        gfx->m_DrawFramework->SetColor(0.0f, 1.0f, 0.0f);
        gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+10.0f,  "Y: %6d\n", m_ExtReader->GetAccCalib().latestStatus.y);
        gfx->m_DrawFramework->SetColor(0.0f, 0.0f, 1.0f);  
        gfx->m_DrawFramework->DrawText(BASE_ACC_DATA_X, BASE_DATA_Y+20.0f,  "Z: %6d\n", m_ExtReader->GetAccCalib().latestStatus.z);
        m_AccRawGraphDrawing->Draw(gfx);        
 
        gfx->m_DrawFramework->SwapBuffers();        
    }
    
    /* ------------------------------------------------------------------------
                下画面描画
    ------------------------------------------------------------------------ */
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();

    jpegDrawer->DrawPicture(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->DrawText(0, 0, "SD-CARAD(%s)", m_IsSDCardDetected ? "Detected" : "Not Detected");            

    gfx->m_DrawFramework->Transfer();
    gfx->m_DrawFramework->SwapBuffers_ParallaxBarrier();
    
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);
        
    // restore gl state ※RenderSystemの文字描画実行の際は必ず戻すこと
    for (int index = 0; index < 12; index++)
        glDisableVertexAttribArray(index);        
}

/*
    Desc: 測定データのSD保存
          ※ExtReader で取得済みのデータを全て保存します
          ※Clearにてバッファの初期化を行った際は、Clear後に取得したデータのみ保存されます
    
    Args: gyroLogger    - GyroLogger へのポインタ
          isAllTestPass - テスト結果
          micro         - テスト結果のマイクロコード
          
    Rtns: なし          
*/
void GyroAgingTester::SaveTestDataToSD(GyroLogger* gyroLogger, bool isAllTestPass, u32 micro)
{    
    if (!m_IsSDCardDetected)
    {
        return;   
    }

    m_DebugWnd->Printf("\nNow Saving..");                    

    if (gyroLogger->GetFileSize() == 0)
    {
        // ラベル
        gyroLogger->Printf("TestVer.,""NG Code,""MacAddress,""SerialNo.,");
        gyroLogger->Printf("Stable[Pulse](ms),Stable[PulseEx](ms),Stable[Aging_1st](ms),Stable[Aging_2nd](ms),");
        gyroLogger->Printf("Max_X[Pulse],Max_Y[Pulse],Max_Z[Pulse],"
                           "Min_X[Pulse],Min_Y[Pulse],Min_Z[Pulse],");
        gyroLogger->Printf("Diff_X[Pulse],Diff_Y[Pulse],Diff_Z[Pulse],");
        gyroLogger->Printf("Ave_X[Pulse],Ave_Y[Pulse],Ave_Z[Pulse],"
                           "Ave_X[Aging_1st],Ave_Y[Aging_1st],Ave_Z[Aging_1st],"
                           "Ave_X[Aging_2nd],Ave_Y[Aging_2nd],Ave_Z[Aging_2nd],");
        gyroLogger->Printf("Ave_X[1st-2nd],Ave_Y[1st-2nd],Ave_Z[1st-2nd],");
        gyroLogger->Printf("CONFIG[Pulse],CONFIG[PulseEx],CONFIG[Aging],");
        // 改行
        gyroLogger->Printf("\n");
    }                       

    // プログラムバージョン
    gyroLogger->Printf("Ver.%s,", VERSION_STRING);
        
    // NGコード
    if (isAllTestPass)  gyroLogger->Printf("-,");
    else                gyroLogger->Printf("%d-%d-%d(%s),",
                            G_CTR_ACCEL, TID_GYRO_AGING_TEST, micro, GetTestSequenceName(m_TestErrorSeqence));        
    
    // MACアドレス
    bit8 mac[6]={0,0,0,0,0,0};
    uji::sys::GetMacAddress(mac);
    gyroLogger->Printf(
        "%02X-%02X-%02X-%02X-%02X-%02X,", 
        mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]
    ); 
    // シリアル番号
    char serialNo[nn::cfg::CTR::detail::CFG_SECURE_INFO_SERIAL_NO_LEN +1];
    sys::GetSerialNumber(serialNo);
    gyroLogger->Printf("%s,", serialNo);    
    // 安定時間
     gyroLogger->Printf(
        "%d,%d,%d,%d,", 
        m_PulseStableTime, m_PulseStableTimeEx, m_Aging1stStableTime, m_Aging2ndStableTime);    
    // Pulse
    gyroLogger->Printf(
        "%d,%d,%d,%d,%d,%d,""%d,%d,%d,",
        m_PulseMax.x, m_PulseMax.y, m_PulseMax.z, m_PulseMin.x, m_PulseMin.y, m_PulseMin.z,
        m_PulseMax.x-m_PulseMin.x, m_PulseMax.y-m_PulseMin.y, m_PulseMax.z-m_PulseMin.z        
    );
    // 平均値
    gyroLogger->Printf(
        "%d,%d,%d,%d,%d,%d,%d,%d,%d,",
        m_PulseAve.x, m_PulseAve.y, m_PulseAve.z,
        m_Aging1stAve.x, m_Aging1stAve.y, m_Aging1stAve.z, 
        m_Aging2ndAve.x, m_Aging2ndAve.y, m_Aging2ndAve.z
    );
    // Aging差分
    gyroLogger->Printf(
        "%d,%d,%d,",
        m_AbsDiff.x, m_AbsDiff.y, m_AbsDiff.z
    );    
    // 判定基準
    gyroLogger->Printf(
        "%d,%d,%d,",
        Config_PulseMax, Config_PulseMaxEx, Config_AgingAveDiffMax    
    );

    // 改行
    gyroLogger->Printf("\n");
    gyroLogger->Flush();
    
    m_DebugWnd->Printf(" Save Finished");   
}


//============================================================================
//      テスト関連
//============================================================================

/*
    Desc: 新規に測定されたデータの開始と終了インデックスを取得
    
    Rtns: 新規更新データ数
*/
int GyroAgingTester::GetNewIndex(s64& startIndex, s64& latestIndex)
{        
    // 最新データのインデックスを取得
    latestIndex = m_ExtReader->GetExtdevCalib().loopBuffer->LatestDataIndex();
 
    if ((latestIndex == m_GyroPrevLatestIndex) || (latestIndex == -1))
    // データ更新がないまたはサンプリング数0の場合
    {
        startIndex = latestIndex;
        return 0;
    }
    else
    {
        // サンプリング開始後(測定データあり)の初インデックス取得の際は0番目のデータを開始インデックスとする
        // 2番目の測定データからは前回の最新インデックス+1を開始インデックスとする
        startIndex = (m_GyroPrevLatestIndex == -1) ? 0 : m_GyroPrevLatestIndex + 1;
        m_GyroPrevLatestIndex = latestIndex;
        // 更新データ数
        return (latestIndex - startIndex + 1);
    }
}

/*
    Desc: 測定データ中にパルスが含まれているかチェックします
          対象はResetPulseTest() 呼び出し後のデータです。
    
    Args: 
    
    Rtns: true パルスなし / false パルス確認
*/
bool GyroAgingTester::CheckPulse(
    s64 startIndex,
    s64 latestIndex,
    s32 pulseMaxThreshold,    
    nn::hid::GyroscopeLowStatus& GStatMax,
    nn::hid::GyroscopeLowStatus& GStatMin
)
{   
    // 新たに取得したデータに対して算出
    for (s64 idx = startIndex; idx <= latestIndex; idx++)
    {
        nn::hid::GyroscopeLowStatus tmpGyro = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);
        
        // ジャイロ生値 最大／最小値の更新
        if (GStatMax.x < tmpGyro.x) { GStatMax.x = tmpGyro.x; }
        if (GStatMax.y < tmpGyro.y) { GStatMax.y = tmpGyro.y; }
        if (GStatMax.z < tmpGyro.z) { GStatMax.z = tmpGyro.z; }                
        if (GStatMin.x > tmpGyro.x) { GStatMin.x = tmpGyro.x; }
        if (GStatMin.y > tmpGyro.y) { GStatMin.y = tmpGyro.y; }
        if (GStatMin.z > tmpGyro.z) { GStatMin.z = tmpGyro.z; }        
    }
            
    // 差分を算出して判定
    if (((GStatMax.x - GStatMin.x) <= pulseMaxThreshold) &&
        ((GStatMax.y - GStatMin.y) <= pulseMaxThreshold) &&
        ((GStatMax.z - GStatMin.z) <= pulseMaxThreshold))
    {
        return true;   
    }

    return false;   
}

/*
    Desc: パルステスト用パラメータのリセット
          CheckPulse の対象データは本関数呼出し後のデータです。
    
    Args: passSamplingNum   - OK判定に必要な安定サンプリング数
          msTimeOut         - タイムアウト
          
    Rtns: 無し
*/
void GyroAgingTester::ResetPulseTestSequence(int passSamplingNum, int passSamplingNumEx, int msTimeOut)
{
    m_PulseCheckSeqID = PULSESEQ_START;
    m_PulseTestPassSamplingNum = passSamplingNum;
    m_PulseTestPassSamplingNumEx = passSamplingNumEx;    
    m_msPulseTestTimeOut = msTimeOut;
    m_IsPass = (passSamplingNum == 0) ? true : false;
    m_IsPassEx = (passSamplingNumEx == 0) ? true : false;
}    

/*
    Desc: パルステストシーケンス
    
    Args: GStatMax      - 算出したMax値の参照
          GStatMin      - 算出したMin値の参照
          refStableTime - 安定時間格納変数への参照
          
    Rtns: 
*/
GyroAgingTester::PULSETESET_SEQUENCE 
    GyroAgingTester::DoPulseTestSequence(
        nn::hid::GyroscopeLowStatus& GStatMax,
        nn::hid::GyroscopeLowStatus& GStatMin,
        int& refStableTime,
        int& refStableTimeEx,
        nn::hid::GyroscopeLowStatus* pStableData,
        nn::hid::GyroscopeLowStatus* pStableDataEx
    )
{    
    int dataNum;
    int stableTime;
    bool isEnd = false;
        
    static nn::hid::GyroscopeLowStatus lowMax;
    static nn::hid::GyroscopeLowStatus lowMaxEx;      
    static nn::hid::GyroscopeLowStatus lowMin;
    static nn::hid::GyroscopeLowStatus lowMinEx;
        
    // for Debug
    static s64 prevStartIndex;   
        
    switch(m_PulseCheckSeqID)
    {
    case PULSESEQ_START:
        m_PulseCheckSeqStart = nn::os::Tick::GetSystemCurrent();
        // 測定バッファのクリア
        m_ExtReader->GetExtdevCalib().loopBuffer->Clear();
        m_ExtReader->GetAccCalib().loopBuffer->Clear();
        // 測定用パラメータの初期化
        m_GyroPrevLatestIndex   = -1;
        m_GyroStableDataNum     = 0;
        m_GyroStableDataNumEx   = 0;
        lowMax.x = lowMax.y = lowMax.z = lowMaxEx.x = lowMaxEx.y = lowMaxEx.z = -37268/2;
        lowMin.x = lowMin.y = lowMin.z = lowMinEx.x = lowMinEx.y = lowMinEx.z = 37268/2;
        // Next
        m_PulseCheckSeqID = PULSESEQ_CHEK;
        
        // for Debug
        prevStartIndex = -1;
        break;
             
    case PULSESEQ_CHEK:
        s64 startIndex, latestIndex;

        dataNum = GetNewIndex(startIndex, latestIndex);
        stableTime = (nn::os::Tick::GetSystemCurrent() - m_PulseCheckSeqStart).ToTimeSpan().GetMilliSeconds();        
                
        if (dataNum > 0)
        {
            // for Debug 
            if (startIndex == prevStartIndex)
                SYS_PANIC("startIndex == prevStartIndex");
            prevStartIndex = startIndex;

            // 2.28dps
            if (m_PulseTestPassSamplingNum && !m_IsPass && pStableData)
            {
                refStableTime = stableTime;
            
                if (CheckPulse(startIndex, latestIndex, Config_PulseMax, lowMax, lowMin))
                {
                    for (s64 idx=startIndex; idx <= latestIndex; idx++)
                    {
                        pStableData[m_GyroStableDataNum] = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);
                        if ((++m_GyroStableDataNum) == m_PulseTestPassSamplingNum) 
                        {
                            // 安定データを必要数測定出来た時点で終了
                            m_IsPass = true;
                            break;
                        }                        
                    }
                }
                else
                // リトライ
                {
                    m_GyroStableDataNum = 0;
                    lowMax.x = lowMax.y = lowMax.z = -37268/2;
                    lowMin.x = lowMin.y = lowMin.z =  37268/2;                     
                }
            }
            // 1.80dps
            if (m_PulseTestPassSamplingNumEx && !m_IsPassEx && pStableDataEx)
            {
                refStableTimeEx = stableTime;
            
                if (CheckPulse(startIndex, latestIndex, Config_PulseMaxEx, lowMaxEx, lowMinEx))
                {
                    for (s64 idx = startIndex; idx <= latestIndex; idx++)
                    {
                        pStableDataEx[m_GyroStableDataNumEx] = m_ExtReader->GetExtdevCalib().loopBuffer->GetData(idx);
                        if ((++m_GyroStableDataNumEx) == m_PulseTestPassSamplingNumEx)
                        {
                            // 安定データを必要数測定出来た時点で終了
                            m_IsPassEx = true;
                            break;
                        }
                    }        
                }
                else
                // リトライ
                {
                    m_GyroStableDataNumEx = 0;
                    lowMaxEx.x = lowMaxEx.y = lowMaxEx.z = -37268/2;
                    lowMinEx.x = lowMinEx.y = lowMinEx.z =  37268/2;                    
                }
            }
        }                   
        if (stableTime > m_msPulseTestTimeOut)
        {
            m_PulseCheckSeqID = PULSESEQ_FAIL;
            isEnd = true;
        }                      
        else if (m_IsPass && m_IsPassEx)
        {
            m_PulseCheckSeqID = PULSESEQ_PASS;                           
            isEnd = true;
        }
        if (isEnd)
        {
            GStatMax.x = ::std::max(lowMax.x, lowMaxEx.x);
            GStatMax.y = ::std::max(lowMax.y, lowMaxEx.y);
            GStatMax.z = ::std::max(lowMax.z, lowMaxEx.z);            
            GStatMin.x = ::std::min(lowMin.x, lowMinEx.x);
            GStatMin.y = ::std::min(lowMin.y, lowMinEx.y);
            GStatMin.z = ::std::min(lowMin.z, lowMinEx.z);            
        }
        
        m_DebugWnd->Printf("\f");
        m_DebugWnd->Printf("- %s -\n", GetTestSequenceName(m_TestSequence));
        
        if (m_PulseTestPassSamplingNum)
        {         
            m_DebugWnd->Printf("2.24(dps)\n");   
            m_DebugWnd->Printf("-----------------------\n");                     
            m_DebugWnd->Printf("%04d(ms) %03d/%03d\n", refStableTime, m_GyroStableDataNum, m_PulseTestPassSamplingNum);
            m_DebugWnd->Printf("X max:%05d  min:%05d  max-min:%04d\n", lowMax.x , lowMin.x, lowMax.x - lowMin.x);
            m_DebugWnd->Printf("Y max:%05d  min:%05d  max-min:%04d\n", lowMax.y , lowMin.y, lowMax.y - lowMin.y);
            m_DebugWnd->Printf("Z max:%05d  min:%05d  max-min:%04d",   lowMax.z , lowMin.z, lowMax.z - lowMin.z);
            m_DebugWnd->Printf("\n\n");
        }
        if (m_PulseTestPassSamplingNumEx)
        {                
            m_DebugWnd->Printf("1.80(dps)\n");
            m_DebugWnd->Printf("-----------------------\n");                         
            m_DebugWnd->Printf("%04d(ms) %03d/%03d\n", refStableTimeEx, m_GyroStableDataNumEx, m_PulseTestPassSamplingNumEx);        
            m_DebugWnd->Printf("X max:%05d  min:%05d  max-min:%04d\n", lowMaxEx.x , lowMinEx.x, lowMaxEx.x - lowMinEx.x);
            m_DebugWnd->Printf("Y max:%05d  min:%05d  max-min:%04d\n", lowMaxEx.y , lowMinEx.y, lowMaxEx.y - lowMinEx.y);
            m_DebugWnd->Printf("Z max:%05d  min:%05d  max-min:%04d",   lowMaxEx.z , lowMinEx.z, lowMaxEx.z - lowMinEx.z);
        }
        break;
        
    default:
        break;         
    }
    
    return m_PulseCheckSeqID;
}

/*
    Desc: ジャイロ生値の平均値算出
    
    Args: pData     - バッファアドレス
          dataNum   - バッファサイズ
          
    Rtns: 算出した平均値
*/
nn::hid::GyroscopeLowStatus GyroAgingTester::GetAverage(nn::hid::GyroscopeLowStatus* pData, int dataNum)
{
    s32 sum_x, sum_y, sum_z;
    
    // 指定区間の平均を算出
    sum_x = sum_y = sum_z = 0;
    for (int i=0; i<dataNum; i++)
    {
        sum_x += pData[i].x;
        sum_y += pData[i].y;
        sum_z += pData[i].z;
    }
    // 平均値算出
    nn::hid::GyroscopeLowStatus ave = {
        static_cast<s16>(sum_x/dataNum), 
        static_cast<s16>(sum_y/dataNum), 
        static_cast<s16>(sum_z/dataNum)          
    };
    
    return ave;
}

/*
    Desc: SDカード初期化処理
*/
void GyroAgingTester::InitSDCard()
{
    if (m_GyroLogger.IsSdmcInserted())
    {
        nn::Result nnResult = nn::fs::MountSdmc();
        if (nnResult.IsSuccess())
        {
            bool dir_result = m_GyroLogger.CreateDirectory();
            NN_LOG("CreateDir(%d)\n", dir_result);    
                
            wchar_t fileName[0x30];
            std::swprintf(fileName, sizeof(fileName), L"Aging.csv");
            m_GyroLogger.OpenFile(fileName, static_cast<u32>(1024*100), GyroLogger::SDLOG_WRITE_TYPE_CONTINUANCE);

#ifdef LOGGING_LOW_DATA             
#if 1       // データログファイル名はシリアルナンバーとします
            char serialNo[nn::cfg::CTR::detail::CFG_SECURE_INFO_SERIAL_NO_LEN +1];
            sys::GetSerialNumber(serialNo);
            std::swprintf(fileName, sizeof(fileName), L"Graph_%s.csv", serialNo);            
#else
            // データログファイル名はマックアドレスとします
            bit8 mac[nn::uds::MAC_ADDRESS_SIZE];
            sys::GetMacAddress(mac);
            std::swprintf(fileName, sizeof(fileName), L"Graph_%X%X%X%X%X%X.csv", mac[0], mac[1],mac[2],mac[3],mac[4],mac[5]);
#endif            
            m_GyroDataLogger.OpenFile(fileName, static_cast<u32>(1024*100), GyroLogger::SDLOG_WRITE_TYPE_CONTINUANCE);        
            m_GyroDataLogger.Printf("Gyro_X, Gyro_Y, Gyro_Z, Accl_X, Accl_Y, Accl_Z\n");
#endif        
            m_IsSDCardDetected = true;
        }
        else
        {
            m_IsSDCardDetected = false;        
        }
    }    
}

/*
    Desc: ジャイロエージング検査
          ※SDカードが挿入されている場合は、生値（ジャイロ・加速度）を保存します。
*/
bool GyroAgingTester::Test(seq::TestResult &result)
{    
    nn::os::Tick current, start;
    sys::GraphicsDrawing* gfx = sys::GraphicsDrawing::GetInstance();
    jpegDrawer = new sys::JpegDrawer; 
    
    const int FONT_SIZE = 16;
    const int DEBUG_WND_WIDTH  = 400/(FONT_SIZE/2);
    const int DEBUG_WND_HEIGHT = 14;    

    // 処理状況出力用ウインドウの生成    
    m_DebugWnd = new sys::TextWindow(DEBUG_WND_WIDTH, DEBUG_WND_HEIGHT, FONT_SIZE);
    m_DebugWnd->SetTitle("GYRO AGING TEST");
    m_WndManager.CreateWindow(m_DebugWnd, NN_GX_DISPLAY0, 0, 0);
    // アクティブ指定を解除
    m_DebugWnd->SetActiveFlag(false);    
    
    /*
        拡張リーダ
    */
    nn::hid::GyroscopeLowReader extDevReader;
    nn::hid::GyroscopeLowCalibrator extDevCalibrator;    
    nn::hid::AccelerometerCalibrator accCalibrator;        
    // サンプリング開始    
    m_ExtReader->Initialize();
    m_ExtReader->SamplingStart(extDevCalibrator, extDevReader, accCalibrator);
        
    /*
        グラフデータ初期化  
    */
#if 0
    f32 f_width  = static_cast<f32>(m_ScreenWidth);
    f32 f_height = static_cast<f32>(m_ScreenHeight);    
    // ジャイロ（生値）
    m_GyroRawGraphDrawing = new GyroGraphDrawing(
        "GYRO(RAW)",
        f_height/(575.0f*2.0f), 10.0f, 60.0f, f_width, f_height, nw::ut::FloatColor(0.85f, 0.75f, 0.85f, 1.0f)
    );
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_X]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Y]);
    m_GyroRawGraphDrawing->AddLine(m_GyroRawLine[gyro::AXIS_RAW_Z]);
    // 加速度（生値）
    m_AccRawGraphDrawing  = new GyroGraphDrawing(
        "ACCL(RAW)",
        f_height/200.0f, 205.0f, 60.0f, f_width, f_height, nw::ut::FloatColor(0.85f, 0.75f, 0.85f, 1.0f)
    );    
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_X]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Y]);
    m_AccRawGraphDrawing->AddLine(m_AccRawLine[gyro::AXIS_RAW_Z]);
#endif             
        
    // SDカード検出  
    InitSDCard();
        
    // 拡張バッファのクリアカラーの設定
    gfx->m_DrawFramework->SetClearColor(NN_GX_DISPLAY0_EXT, 0.1f, 0.1f, 0.1f, 1.0f);
    
    // バックライトの設定
    nn::gx::CTR::LcdPowerManager::TurnOnAll();
    sys::WaitMilliSeconds(300);

    // バックライト輝度最大
    nn::gx::CTR::Backlight::SetLuminanceLevel(nn::gx::CTR::LCD_BOTH, 5);
    // アクティブバックライトOFF
    nn::gx::CTR::Backlight::DisableActiveBl(nn::gx::CTR::LCD_BOTH);
    sys::WaitMilliSeconds(300);
                                   
    // 画像更新
    jpegDrawer->OpenPicture(L"rom:/jpeg/Donttoutch.jpg");
    
#ifndef EVA_LCD_AND_KEY
    // お知らせランプの点灯
    SetInfoLedRed();
#endif
    // 測定開始時間 
    start = nn::os::Tick::GetSystemCurrent();
    
    result.m_Result = true;
                                                          
    int tTmp;                                                          
    do
    {
        // パッド情報更新
        sys::Pad().UpdatePad();
                
        switch (m_TestSequence)
        {
        // エージング（1ST）
        case TESTSEQ_MEASUREMENT_AGING_1ST:
            switch(m_TestSubSeqence)
            {
            case 0:
                ResetPulseTestSequence(100, 0, 20000);
                m_TestSubSeqence++;
            case 1:
                switch (DoPulseTestSequence(
                    m_Aging1stMax, m_Aging1stMin, m_Aging1stStableTime, tTmp, 
                    m_Aging1stStableData, NULL))
                {
                case PULSESEQ_PASS:
                    m_Aging1stAve       = GetAverage(m_Aging1stStableData, 100);
                    m_TestSequence      = TESTSEQ_PULSE_CHECK;
                    m_TestSubSeqence    = 0;
                    break;
                case PULSESEQ_FAIL:
                    result.m_Result = false;
                    result.m_Micro = ERROR_MICRO_PULSE;
                    sprintf(
                        result.m_String, 
                        "PULSE(AGING 1ST)\n"
                        "Max-> X:Y:Z(%d,%d,%d)\n" 
                        "Min-> X:Y:Z(%d,%d,%d)",
                        m_Aging1stMax.x, m_Aging1stMax.y, m_Aging1stMax.z, 
                        m_Aging1stMin.x, m_Aging1stMin.y, m_Aging1stMin.z 
                    );
                    m_TestErrorSeqence  = TESTSEQ_MEASUREMENT_AGING_1ST;
                    m_TestSequence      = TESTSEQ_SAVETOSD;
                    break;
                }
                break;
            }            
            break;
    
        // パルステスト
        case TESTSEQ_PULSE_CHECK:  
            switch(m_TestSubSeqence)
            {
            case 0:
                ResetPulseTestSequence(500, 100, 20000);
                m_TestSubSeqence++;
            case 1:
                switch (DoPulseTestSequence(
                    m_PulseMax, m_PulseMin, m_PulseStableTime, m_PulseStableTimeEx, 
                    m_PulseStableData, m_PulseExStableData))
                {
                case PULSESEQ_PASS:
                    m_PulseAve          = GetAverage(m_PulseStableData, 500);                                       
                    m_TestSequence      = TESTSEQ_MEASUREMENT_AGING_2ND;
                    m_TestSubSeqence    = 0;
                    break;
                case PULSESEQ_FAIL:
                    result.m_Result = false;
                    if (!m_IsPass && !m_IsPassEx) { result.m_Micro = ERROR_MICRO_PULSE_BOTH; }
                    else                          { result.m_Micro = (!m_IsPass) ? ERROR_MICRO_PULSE : ERROR_MICRO_PULSE_EX; }
                    sprintf(
                        result.m_String,
                        "PULSE\n"
                        "Max-> X:Y:Z(%d,%d,%d)\n" 
                        "Min-> X:Y:Z(%d,%d,%d)",   
                        m_PulseMax.x, m_PulseMax.y, m_PulseMax.z, 
                        m_PulseMin.x, m_PulseMin.y, m_PulseMin.z
                    );
                    m_TestErrorSeqence  = TESTSEQ_PULSE_CHECK;                                                                            
                    m_TestSequence      = TESTSEQ_SAVETOSD;
                    break;
                }
                break;
            }            
            break;
        
        // エージング（2ND）
        case TESTSEQ_MEASUREMENT_AGING_2ND:  
            switch(m_TestSubSeqence)
            {
            case 0:
                // エージングデータ測定に必要な待ち時間の算出
                {
                    int tPulseTest = (m_PulseStableTime > m_PulseStableTimeEx) ? m_PulseStableTime : m_PulseStableTimeEx;
                    int elapsed = m_Aging1stStableTime + tPulseTest;
                    if (elapsed < 9000)
                    {
                        s64 wait_time           = 9000 - elapsed;
                        s64 elapsed_time        = 0;
                        nn::os::Tick wait_start = nn::os::Tick::GetSystemCurrent();
                        
                        do
                        {
                            m_DebugWnd->Printf("\f");
                            m_DebugWnd->Printf("Please Wait %d(ms)", wait_time - elapsed_time);
                            
                            UpdateGraph();                            
                            UpdateScreen(gfx);
                             
                            elapsed_time = (nn::os::Tick::GetSystemCurrent() - wait_start).ToTimeSpan().GetMilliSeconds();
                        }
                        while (elapsed_time <= wait_time);
                    }
                }
                ResetPulseTestSequence(100, 0, 20000);
                m_TestSubSeqence++;
            case 1:
                switch (DoPulseTestSequence(
                    m_Aging2ndMax, m_Aging2ndMin, m_Aging2ndStableTime, tTmp, 
                    m_Aging2ndStableData, NULL))
                {
                case PULSESEQ_PASS:
                    m_Aging2ndAve       = GetAverage(m_Aging2ndStableData, 100);
                    m_TestSequence      = TESTSEQ_AGING_CHECK;
                    m_TestSubSeqence    = 0;
                    break;
                case PULSESEQ_FAIL:
                    result.m_Result = false;
                    result.m_Micro  = ERROR_MICRO_PULSE;
                    sprintf(
                        result.m_String,
                        "PULSE(AGING 2ND)\n"
                        "Max-> X:Y:Z(%d,%d,%d)\n" 
                        "Min-> X:Y:Z(%d,%d,%d)",
                        m_Aging2ndMax.x, m_Aging2ndMax.y, m_Aging2ndMax.z, 
                        m_Aging2ndMin.x, m_Aging2ndMin.y, m_Aging2ndMin.z 
                    );
                    m_TestErrorSeqence  = TESTSEQ_MEASUREMENT_AGING_2ND;                                                       
                    m_TestSequence      = TESTSEQ_SAVETOSD;
                    break;
                }
                break;
            }            
            break;        

        // エージング検査
        case TESTSEQ_AGING_CHECK:                     
            // 平均値の差分の絶対値を算出            
            m_AbsDiff.x = std::abs(m_Aging1stAve.x - m_Aging2ndAve.x);
            m_AbsDiff.y = std::abs(m_Aging1stAve.y - m_Aging2ndAve.y); 
            m_AbsDiff.z = std::abs(m_Aging1stAve.z - m_Aging2ndAve.z);
    
            // 判定
            if ((m_AbsDiff.x > Config_AgingAveDiffMax) || 
                (m_AbsDiff.y > Config_AgingAveDiffMax) || 
                (m_AbsDiff.z > Config_AgingAveDiffMax))
            {
                    result.m_Result = false;
                    result.m_Micro = ERROR_MICRO_AGING;
                    sprintf(
                        result.m_String, 
                        "AGING TEST\n"
                        "ave X:Y:Z(%d,%d,%d) (%d,%d,%d)\n"
                        "abs_diff X:Y:Z(%d,%d,%d)\n",             
                        m_Aging1stAve.x, m_Aging1stAve.y, m_Aging1stAve.z, 
                        m_Aging2ndAve.x, m_Aging2ndAve.y, m_Aging2ndAve.z,
                        m_AbsDiff.x, m_AbsDiff.y, m_AbsDiff.z
                    );
                    m_TestErrorSeqence = TESTSEQ_AGING_CHECK;                    
            }
#ifdef LOGGING_LOW_DATA                         
            m_TestSequence = TESTSEQ_COLLECT_DATA;
#else
            m_TestSequence = TESTSEQ_SAVETOSD;    
#endif    
            break;
            
        // 測定データ収集    
        case TESTSEQ_COLLECT_DATA:
#ifdef LOGGING_LOW_DATA                                     
            // バックライト輝度最小
            nn::gx::CTR::Backlight::SetLuminanceLevel(nn::gx::CTR::LCD_BOTH, 1);
            // アクティブバックライトON
            nn::gx::CTR::Backlight::EnableActiveBl(nn::gx::CTR::LCD_BOTH);
            sys::WaitMilliSeconds(300);
                    
            // 測定バッファのクリア
            m_ExtReader->GetExtdevCalib().loopBuffer->Clear();
            m_ExtReader->GetAccCalib().loopBuffer->Clear();
                    
            // 描画モードを戻します
            m_Is3DMode = false;
                
            if (m_IsSDCardDetected)
            {            
                m_GyroDataLogger.Printf("LuminanceLevel<1> 3D<Off>\n");
            }
            
            {
                s64 measuring_time = 1000;
                s64 elapsed_time = 0;
                nn::os::Tick wait_start = nn::os::Tick::GetSystemCurrent();

                do
                {
                    m_DebugWnd->Printf("\f");
                    m_DebugWnd->Printf("LuminanceLevel<1> 3D<Off>\n");
                    m_DebugWnd->Printf("Please Wait %d(ms)", measuring_time - elapsed_time);
                            
                    UpdateGraph();                            
                    UpdateScreen(gfx);
                             
                    elapsed_time = (nn::os::Tick::GetSystemCurrent() - wait_start).ToTimeSpan().GetMilliSeconds();
                }
                while (elapsed_time <= measuring_time);
                
                m_ExtReader->SamplingStop();
            }
#endif                        
            m_TestSequence = TESTSEQ_SAVETOSD;
            break;       

        // データをSDに保持
        case TESTSEQ_SAVETOSD:            
            SaveTestDataToSD(&m_GyroLogger, result.m_Result, result.m_Micro);            
            
            if (!result.m_Result)
            {
                // お知らせランプ赤点滅
                eva::mcu::NotifyLedRedBlinker();
            }
            m_TestSequence = TESTSEQ_RESULT;
            break;

        default:
            break;            
        }

        // グラフ更新
        UpdateGraph();
        
        // 画面更新
        UpdateScreen(gfx);                      
    }
    while(!(m_TestSequence == TESTSEQ_RESULT));
    
    /*
        終了処理
    */
    if (m_IsSDCardDetected)
    {
        m_GyroLogger.Flush();
        m_GyroLogger.CloseFile();

#ifdef LOGGING_LOW_DATA                         
        m_GyroDataLogger.Flush();
        m_GyroDataLogger.CloseFile();
#endif
        // SD カード アンマウント
        NN_UTIL_PANIC_IF_FAILED(nn::fs::Unmount("sdmc:"));        
    }        
    m_ExtReader->SamplingStop();
    m_ExtReader->Finalize();
    
#if 0
    delete m_GyroRawGraphDrawing;
    delete m_AccRawGraphDrawing;    
#endif
    jpegDrawer->ClosePicture();
    delete jpegDrawer; 
    
    m_WndManager.DestroyWindow(m_DebugWnd);    
    delete m_DebugWnd;

    // restore gl state ※RenderSystemの文字描画実行の際は必ず戻すこと
    for (int index = 0; index < 12; index++)
        glDisableVertexAttribArray(index);
    
    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_SCISSOR_TEST);
    glDisable(GL_STENCIL_TEST);
    glDisable(GL_BLEND);
    glActiveTexture(GL_TEXTURE0);
    glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
    glDepthMask(GL_TRUE);
    
    // 描画モードを戻します
    gfx->m_DrawFramework->SetLcdModeUji(NN_GX_DISPLAYMODE_NORMAL);
    gfx->ReInitRenderSystem();

    return result.m_Result;
}

/*
    Desc: Evaメニュー呼び出し用
*/
void GyroAgingTester::EvaGyroAging()
{
    seq::TestResult result;    
    GyroAgingTester gaTester;
        
    sys::GraphicsDrawing *gfx = uji::sys::GraphicsDrawing::GetInstance();

    gaTester.Test(result);
        
    sys::Menu::m_SubWindow->Printf("\f");
    sys::Menu::m_SubWindow->Printf("<AGING EVA>\n");    
    sys::Menu::m_SubWindow->Printf("Result(%s)\n%s\n", result.m_Result ? "PASS" : "FAIL", result.m_String);
    sys::Menu::m_SubWindow->Printf("\n - Push B Button To Exit -");            
    
    // 画面更新
    sys::Menu::m_WindowManager.Update();
    sys::Menu::m_WindowManager.UpdatePad(sys::Pad());
    gfx->m_DrawFramework->SetRenderTarget(NN_GX_DISPLAY1);
    gfx->m_DrawFramework->Clear();        
    sys::Menu::m_WindowManager.DrawDisplay1();
    gfx->m_DrawFramework->SwapBuffers();
    gfx->m_DrawFramework->WaitVsync(NN_GX_DISPLAY_BOTH);

    do
    {
        sys::Pad().UpdatePad();
        nn::os::Thread::Sleep(nn::fnd::TimeSpan::FromMilliSeconds(10));        
    }
    while (!sys::Pad().IsButtonDown(sys::Pad::BUTTON_B));   
}


namespace gyroC {

/*
    Desc: ジャイロエージングテスト（テストリスト呼び出し用）
*/
bool GyroAgingTest(seq::TestResult &result)
{
    GyroAgingTester gaTester;
    
    return gaTester.Test(result);
}

}


} // namespace
}
