/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"
#include "Tga.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "TextureAll"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace TextureAll{

/* buffer id */
static GLuint array_buffer_id = 0;
static GLuint element_array_buffer_id = 0;

/* program id */
static GLuint pgid = 0;

/* shader id */
static GLuint shid = 0;

/* Specific to the sample */
#define TEXID_NUM			3
#define PROC_TEXID_NUM		7
static GLuint texcollection = 0;
static GLuint texid[TEXID_NUM] = {0};
static GLuint texid_proc[PROC_TEXID_NUM] = {0};

static GLfloat coords[] = {
	-1.f, -1.f, 0.f, 1.f,
	1.f, -1.f, 0.f, 1.f,
	1.f, 1.f, 0.f, 1.f,
	-1.f, 1.f, 0.f, 1.f,
};

static GLfloat color[] = {
	1.f, 0.f, 0.f, 1.f,
	0.f, 1.f, 0.f, 1.f,
	0.f, 0.f, 1.f, 1.f,
	1.f, 1.f, 1.f, 1.f,
};

static GLfloat texture[] = {
	0.f,0.f,0.f,0.f,
	1.f,0.f,0.f,0.f,
	1.f,1.f,0.f,0.f,
	0.f,1.f,0.f,0.f,
};

static GLfloat proc_texture[] = {
	-1.f,-1.f,0.f,0.f,
	-1.f,1.f,0.f,0.f,
	1.f,1.f,0.f,0.f,
	1.f,-1.f,0.f,0.f,
};

static GLushort idxs[] = {
	0, 1, 2, 3
};



static int set_1D_tex(GLfloat* RR, GLfloat* GG, GLfloat* BB, GLfloat* AA, int N)
{
	memset(RR, 0, 512 * sizeof(GLfloat));
	memset(GG, 0, 512 * sizeof(GLfloat));
	memset(BB, 0, 512 * sizeof(GLfloat));
	memset(AA, 0, 512 * sizeof(GLfloat));

	RR[7*N]=0x00/255.0f;	GG[7*N]=0x00/255.0f;	BB[7*N]=0x00/255.0f;	AA[7*N]=0xFF/255.0f;
	RR[6*N]=0x00/255.0f;	GG[6*N]=0x00/255.0f;	BB[6*N]=0x20/255.0f;	AA[6*N]=0xFF/255.0f;
	RR[5*N]=0x20/255.0f;	GG[5*N]=0x00/255.0f;	BB[5*N]=0x00/255.0f;	AA[5*N]=0xFF/255.0f;
	RR[4*N]=0x20/255.0f;	GG[4*N]=0x00/255.0f;	BB[4*N]=0x20/255.0f;	AA[4*N]=0xFF/255.0f;
	RR[3*N]=0x00/255.0f;	GG[3*N]=0x20/255.0f;	BB[3*N]=0x00/255.0f;	AA[3*N]=0xFF/255.0f;
	RR[2*N]=0x00/255.0f;	GG[2*N]=0x20/255.0f;	BB[2*N]=0x20/255.0f;	AA[2*N]=0xFF/255.0f;
	RR[1*N]=0x20/255.0f;	GG[1*N]=0x20/255.0f;	BB[1*N]=0x00/255.0f;	AA[1*N]=0xFF/255.0f;
	RR[0*N]=0x20/255.0f;	GG[0*N]=0x20/255.0f;	BB[0*N]=0x20/255.0f;	AA[0*N]=0xFF/255.0f;

	for (int x = 0; x < 255; x++)
	{
		RR[256+x] = RR[x+1] - RR[x];
		GG[256+x] = GG[x+1] - GG[x];
		BB[256+x] = BB[x+1] - BB[x];
		AA[256+x] = AA[x+1] - AA[x];
	}
	RR[255] = 0.0f;
	GG[255] = 0.0f;
	BB[255] = 0.0f;
	AA[255] = 0.0f;

	return N * 8;
}

static void LoadSin(float map_1D2D_LUT[256], GLfloat f)
{
	for (int i = 0; i < 128; i++)
		map_1D2D_LUT[i] = sin(fmod(f * DMP_PI * i/128.f, DMP_PI));

	for (int i = 0; i < 127; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];

	map_1D2D_LUT[255] = sin(fmod(f * DMP_PI, DMP_PI)) - map_1D2D_LUT[127];
}

static void LoadFlat(float map_1D2D_LUT[256], float a)
{
	for (int i = 0; i < 128; i++)
	{
		map_1D2D_LUT[i] = a;
		map_1D2D_LUT[i + 128] = 0;
	}
}

static void load_ProceduralTexture(float* F_FunctionRGB)
{
	/* Declaration */
	float F_FunctionA[256], F_Noise[256];
	LoadFlat(F_FunctionA, 1.0f);
	memset(F_Noise, 0, 256 * sizeof(float));

	/* Texture collection */
	glBindTexture(GL_LUT_TEXTURE0_DMP, texid_proc[0]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionRGB);

	glBindTexture(GL_LUT_TEXTURE1_DMP, texid_proc[1]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionA);
	
	glBindTexture(GL_LUT_TEXTURE2_DMP, texid_proc[2]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_Noise);

	const int NN = 512;
	GLfloat RR[NN], GG[NN], BB[NN], AA[NN];
	int texSize = set_1D_tex(RR, GG, BB, AA, 1);

	glBindTexture(GL_LUT_TEXTURE3_DMP, texid_proc[3]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, RR);

	glBindTexture(GL_LUT_TEXTURE4_DMP, texid_proc[4]);
	glTexImage1D(GL_LUT_TEXTURE4_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, GG);

	glBindTexture(GL_LUT_TEXTURE5_DMP, texid_proc[5]);
	glTexImage1D(GL_LUT_TEXTURE5_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, BB);

	glBindTexture(GL_LUT_TEXTURE6_DMP, texid_proc[6]);
	glTexImage1D(GL_LUT_TEXTURE6_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, AA);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerRgbMap"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerAlphaMap"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerNoiseMap"), 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerR"), 3);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerG"), 4);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerB"), 5);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerA"), 6);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].samplerType"), GL_TEXTURE_PROCEDURAL_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].texcoord"), GL_TEXTURE2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptMinFilter"), GL_LINEAR);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexWidth"), texSize);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexOffset"), 0);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptRgbMap"), GL_PROCTEX_ADDSQRT2_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaMap"), GL_PROCTEX_ADDSQRT2_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaSeparate"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampU"), GL_SYMMETRICAL_REPEAT_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampV"), GL_SYMMETRICAL_REPEAT_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftU"), GL_NONE_DMP);
    glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftV"), GL_NONE_DMP);
}

static void tx_createMipmap(const GLvoid* _data, GLubyte* result, GLenum _format, GLenum _type, GLsizei _width, GLsizei _height)
{
	int i, j;
	GLsizei w, h;
	
	w = _width >> 1;
	h = _height >> 1;
	
	switch (_format)
	{
		case GL_RGBA: case GL_RGBA8_OES:
		case GL_SHADOW_DMP:
			switch (_type)
			{
				case GL_UNSIGNED_BYTE:
					for (i = 0; i < h; i++)
					{
						for (j = 0; j < w; j++)
						{
							GLuint pixel[4];
							GLuint r, g, b, a;
							
							pixel[0] = ((GLuint*)_data)[i * 2 * _width + j * 2];
							pixel[1] = ((GLuint*)_data)[i * 2 * _width + j * 2 + 1];
							pixel[2] = ((GLuint*)_data)[(i * 2 + 1) * _width + j * 2];
							pixel[3] = ((GLuint*)_data)[(i * 2 + 1) * _width + j * 2 + 1];
							r = ((pixel[0] & 0xff) + (pixel[1] & 0xff) + (pixel[2] & 0xff) + (pixel[3] & 0xff)) >> 2;
							g = (((pixel[0] >> 8) & 0xff) + ((pixel[1] >> 8) & 0xff) + ((pixel[2] >> 8) & 0xff) + ((pixel[3] >> 8) & 0xff)) >> 2;
							b = (((pixel[0] >> 16) & 0xff) + ((pixel[1] >> 16) & 0xff) + ((pixel[2] >> 16) & 0xff) + ((pixel[3] >> 16) & 0xff)) >> 2;
							a = (((pixel[0] >> 24) & 0xff) + ((pixel[1] >> 24) & 0xff) + ((pixel[2] >> 24) & 0xff) + ((pixel[3] >> 24) & 0xff)) >> 2;
							((GLuint*)result)[i * w + j] = ((a << 24) & 0xff000000) | ((b << 16) & 0xff0000) | ((g << 8) & 0xff00) | (r & 0xff);
						}
					}
					break ;
				case GL_UNSIGNED_SHORT_4_4_4_4:
					for (i = 0; i < h; i++)
					{
						for (j = 0; j < w; j++)
						{
							GLushort pixel[4];
							GLushort r, g, b, a;
							
							pixel[0] = ((GLushort*)_data)[i * 2 * _width + j * 2];
							pixel[1] = ((GLushort*)_data)[i * 2 * _width + j * 2 + 1];
							pixel[2] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2];
							pixel[3] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2 + 1];
							r = (((pixel[0] >> 12) & 0xf) + ((pixel[1] >> 12) & 0xf) + ((pixel[2] >> 12) & 0xf) + ((pixel[3] >> 12) & 0xf)) >> 2;
							g = (((pixel[0] >>  8) & 0xf) + ((pixel[1] >>  8) & 0xf) + ((pixel[2] >>  8) & 0xf) + ((pixel[3] >>  8) & 0xf)) >> 2;
							b = (((pixel[0] >>  4) & 0xf) + ((pixel[1] >>  4) & 0xf) + ((pixel[2] >>  4) & 0xf) + ((pixel[3] >>  4) & 0xf)) >> 2;
							a = ((pixel[0] & 0xf) + (pixel[1] & 0xf) + (pixel[2] & 0xf) + (pixel[3] & 0xf)) >> 2;
							((GLushort*)result)[i * w + j] = ((r << 12) & 0xf000) | ((g << 8) & 0xf00) | ((b << 4) & 0xf0) | (a & 0xf);
						}
					}
					break ;
				case GL_UNSIGNED_SHORT_5_5_5_1:
					for (i = 0; i < h; i++)
					{
						for (j = 0; j < w; j++)
						{
							GLushort pixel[4];
							GLushort r, g, b, a;
							
							pixel[0] = ((GLushort*)_data)[i * 2 * _width + j * 2];
							pixel[1] = ((GLushort*)_data)[i * 2 * _width + j * 2 + 1];
							pixel[2] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2];
							pixel[3] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2 + 1];
							r = (((pixel[0] >> 11) & 0x1f) + ((pixel[1] >> 11) & 0x1f) + ((pixel[2] >> 11) & 0x1f) + ((pixel[3] >> 11) & 0x1f)) >> 2;
							g = (((pixel[0] >>  6) & 0x1f) + ((pixel[1] >>  6) & 0x1f) + ((pixel[2] >>  6) & 0x1f) + ((pixel[3] >>  6) & 0x1f)) >> 2;
							b = (((pixel[0] >>  1) & 0x1f) + ((pixel[1] >>  1) & 0x1f) + ((pixel[2] >>  1) & 0x1f) + ((pixel[3] >>  1) & 0x1f)) >> 2;
							a = ((pixel[0] & 0x1) + (pixel[1] & 0x1) + (pixel[2] & 0x1) + (pixel[3] & 0x1)) >> 2;
							((GLushort*)result)[i * w + j] = ((r << 11) & 0xf800) | ((g << 6) & 0x7c0) | ((b << 1) & 0x3e) | (a & 0x1);
						}
					}
					break ;
			}
			break ;
		case GL_RGB: case GL_RGB8_OES:
			switch ( _type )
			{
				case GL_UNSIGNED_BYTE:
					for (i = 0; i < h; i++)
					{
						for (j = 0; j < w; j++)
						{
							GLushort r, g, b;
							r = ((GLubyte*)_data)[i * 6 * _width + j * 6] + ((GLubyte*)_data)[i * 6 * _width + j * 6 + 3]
								+ ((GLubyte*)_data)[(i * 6 + 3) * _width + j * 6] + ((GLubyte*)_data)[(i * 6 + 3) * _width + j * 6 + 3];
							g = ((GLubyte*)_data)[i * 6 * _width + j * 6 + 1] + ((GLubyte*)_data)[i * 6 * _width + j * 6 + 4]
								+ ((GLubyte*)_data)[(i * 6 + 3) * _width + j * 6 + 1] + ((GLubyte*)_data)[(i * 6 + 3) * _width + j * 6 + 4];
							b = ((GLubyte*)_data)[i * 6 * _width + j * 6 + 2] + ((GLubyte*)_data)[i * 6 * _width + j * 6 + 5]
								+ ((GLubyte*)_data)[(i * 6 + 3) * _width + j * 6 + 2] + ((GLubyte*)_data)[(i * 6 + 3) * _width + j * 6 + 5];
							((GLubyte*)result)[i * w * 3 + j * 3] = (GLubyte)(r >> 2);
							((GLubyte*)result)[i * w * 3 + j * 3 + 1] = (GLubyte)(g >> 2);
							((GLubyte*)result)[i * w * 3 + j * 3 + 2] = (GLubyte)(b >> 2);
						}
					}
					break;
				case GL_UNSIGNED_SHORT_5_6_5:
					for (i = 0; i < h; i++)
					{
						for (j = 0; j < w; j++)
						{
							GLushort pixel[4];
							GLushort r, g, b;
							
							pixel[0] = ((GLushort*)_data)[i * 2 * _width + j * 2];
							pixel[1] = ((GLushort*)_data)[i * 2 * _width + j * 2 + 1];
							pixel[2] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2];
							pixel[3] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2 + 1];
							r = (((pixel[0] >> 11) & 0x1f) + ((pixel[1] >> 11) & 0x1f) + ((pixel[2] >> 11) & 0x1f) + ((pixel[3] >> 11) & 0x1f)) >> 2;
							g = (((pixel[0] >>  5) & 0x3f) + ((pixel[1] >>  5) & 0x3f) + ((pixel[2] >>  5) & 0x3f) + ((pixel[3] >>  5) & 0x3f)) >> 2;
							b = ((pixel[0] & 0x1f) + (pixel[1] & 0x1f) + (pixel[2] & 0x1f) + (pixel[3] & 0x1f)) >> 2;
							((GLushort*)result)[i * w + j] = ((r << 11) & 0xf800) | ((g << 5) & 0x7e0) | (b & 0x1f);
						}
					}
					break ;
			}
			break ;
		case GL_ALPHA:
		case GL_LUMINANCE:
			for (i = 0; i < h; i++)
			{
				for (j = 0; j < w; j++)
				{
					GLushort pixel[4];
					
					pixel[0] = (GLushort)((GLubyte*)_data)[i * 2 * _width + j * 2];
					pixel[1] = (GLushort)((GLubyte*)_data)[i * 2 * _width + j * 2 + 1];
					pixel[2] = (GLushort)((GLubyte*)_data)[(i * 2 + 1) * _width + j * 2];
					pixel[3] = (GLushort)((GLubyte*)_data)[(i * 2 + 1) * _width + j * 2 + 1];
					((GLubyte*)result)[i * w + j] = (GLubyte)((pixel[0] + pixel[1] + pixel[2] + pixel[3]) >> 2);
				}
			}
			break;
		case GL_LUMINANCE_ALPHA:
		case GL_HILO8_DMP:
			for (i = 0; i < h; i++)
			{
				for (j = 0; j < w; j++)
				{
					GLushort pixel[4];
					GLushort r, g;
					
					pixel[0] = ((GLushort*)_data)[i * 2 * _width + j * 2];
					pixel[1] = ((GLushort*)_data)[i * 2 * _width + j * 2 + 1];
					pixel[2] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2];
					pixel[3] = ((GLushort*)_data)[(i * 2 + 1) * _width + j * 2 + 1];
					r = (((pixel[0] >> 8) & 0xff) + ((pixel[1] >> 8) & 0xff) + ((pixel[2] >> 8) & 0xff) + ((pixel[3] >> 8) & 0xff)) >> 2;
					g = ((pixel[0] & 0xff) + (pixel[1] & 0xff) + (pixel[2] & 0xff) + (pixel[3] & 0xff)) >> 2;
					((GLushort*)result)[i * w + j] = ((r << 8) & 0xff00) | (g & 0xff);
				}
			}
			break ;
	}
	
	return;
}	/* static void tx_createMipmap(const GLvoid* _data, GLenum _format, GLenum _type, GLsizei _width, GLsizei _height) */

static void loadMipMapTexture(const char * _name, unsigned _target, bool& _useAlpha, bool gen_mipmap, bool _is2D = true, int *_width=0, unsigned _forceformat=0);

static void loadMipMapTexture(const char * _name, unsigned _target, bool& _useAlpha, bool gen_mipmap, bool _is2D, int *_width, unsigned _forceformat)
{
	unsigned char* pixels;
	unsigned width, height, format, type, orientation;
	
	(void)_useAlpha;
	pixels = (unsigned char*)dmpLoadTGA(_name, &width, &height, &format, &type, &orientation);
	if (!pixels)
		return;
	
	switch (format)
	{
		case IMAGE_RGB:		format = GL_RGB;	break;
		case IMAGE_RGBA:	format = GL_RGBA;	break;
		default:
			if (pixels)
				free(pixels);
			return;
	}
	
	/* force to convert format */
	if (_forceformat != 0 && format != _forceformat)
	{
		unsigned i;
		unsigned char* newpixels = 0;
		
		switch (format)
		{
			case GL_RGB:
				switch (_forceformat)
				{
					case GL_RGBA:
						newpixels = (unsigned char*)malloc(width * height * 4);
						if (!newpixels)
						{
							free(pixels);
							return;
						}
						for (i = 0; i < width * height; i++)
						{
							newpixels[i * 4 + 0] = pixels[i * 3 + 0];
							newpixels[i * 4 + 1] = pixels[i * 3 + 1];
							newpixels[i * 4 + 2] = pixels[i * 3 + 2];
							newpixels[i * 4 + 3] = 0xff;
						}
						break;
				}
				break;
			case GL_RGBA:
				switch (_forceformat)
				{
					case GL_RGB:
						newpixels = (unsigned char*)malloc(width * height * 3);
						if (!newpixels)
						{
							free(pixels);
							return;
						}
						for (i = 0; i < width * height; i++)
						{
							newpixels[i * 3 + 0] = pixels[i * 4 + 0];
							newpixels[i * 3 + 1] = pixels[i * 4 + 1];
							newpixels[i * 3 + 2] = pixels[i * 4 + 2];
						}
						break;
				}
				break;
		}
		
		if (newpixels)
		{
			format = _forceformat;
			free(pixels);
			pixels = newpixels;
		}
	}

	int output_level = 1;
	if (gen_mipmap)
	{
		int pixelsize = 0;
		int converted_format = 0;
		switch (format)
		{
			case GL_RGBA:	pixelsize = 4;	converted_format = GL_RGBA; break;
			case GL_RGB:	pixelsize = 3;	converted_format = GL_RGB; break;
			default:
				if (pixels) free(pixels);
				return;
		}
		unsigned char* newpixels = (unsigned char*)malloc(width * height * pixelsize * 2);	// alloc enough size for all mip surface
		int prevoffset = 0;
		int offset = 0;
		memcpy(&newpixels[offset], pixels, width * height * pixelsize);
		
		int w = width;
		int h = height;
		for (; w > 8 &&  h > 8; w /= 2, h /= 2)
		{
			offset += w * h * pixelsize;
			tx_createMipmap((const GLvoid*)&newpixels[prevoffset], &newpixels[offset], converted_format, type, w, h);
			output_level++;
			prevoffset = offset;
		}
		
		free(pixels);
		pixels = newpixels;
	}
	
	switch (type)
	{
		case IMAGE_UNSIGNED_BYTE:	type = GL_UNSIGNED_BYTE;	break;
		default:
			if (pixels)
				free(pixels);
			return;
	}
	
	if (_is2D)
		glTexImage2D(_target, (output_level > 1) ? -output_level : 0, format, width, height, 0, format, type, pixels);
	else
		glTexImage1D(_target, (output_level > 1) ? -output_level : 0, format, width, 0, format, type, pixels);
	
	if (pixels)
		free(pixels);
	
	if (_width)
		(*_width) = width;
}


static void load_textures(void)
{
	bool bUseAlpha ;
	/* Binding of the texture as a texture collection */
	glGenTextures( 1, &texcollection );
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, texcollection);

	/* Texture 0 */
	glActiveTexture(GL_TEXTURE0);
	glGenTextures( TEXID_NUM, &texid[0] );
	glBindTexture( GL_TEXTURE_2D, texid[0] );

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST );
	GLuint err = glGetError();
	while(err);

	loadMipMapTexture(FILE_APP_ROOT "dmp0128_r.tga", GL_TEXTURE_2D, bUseAlpha, true);
	
	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[0].samplerType"), GL_TEXTURE_2D );

	/* Texture 1 */
	glActiveTexture(GL_TEXTURE1);
	glBindTexture( GL_TEXTURE_2D, texid[1] );

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST );

	loadMipMapTexture(FILE_APP_ROOT "dmp0128_g.tga", GL_TEXTURE_2D, bUseAlpha, true);
	
	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[1].samplerType"), GL_TEXTURE_2D );

	/* Texture 2 */
	glActiveTexture(GL_TEXTURE2);
	glBindTexture( GL_TEXTURE_2D, texid[2] );

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST );

	loadMipMapTexture(FILE_APP_ROOT "dmp0128_b.tga", GL_TEXTURE_2D, bUseAlpha, true);
	
	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[2].samplerType"), GL_TEXTURE_2D );
	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[2].texcoord"), GL_TEXTURE1 );

	/* procedural texture */
	GLfloat F_FunctionRGB[256];
	GLfloat t = 0 / 100.f;

	glGenTextures( PROC_TEXID_NUM, &texid_proc[0] );
	LoadSin(F_FunctionRGB, 3.f + sin(DMP_PI * t) * 1.5f);
	load_ProceduralTexture( F_FunctionRGB);

	/* Texture Combiner0 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE0, GL_TEXTURE1, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_TEXTURE0, GL_TEXTURE1, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[0].combineRgb"), GL_ADD );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[0].combineAlpha"), GL_ADD );
	/* Texture Combiner1 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcRgb"), GL_PREVIOUS, GL_TEXTURE2, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcAlpha"), GL_PREVIOUS, GL_TEXTURE2, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[1].combineRgb"), GL_ADD );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[1].combineAlpha"), GL_ADD );
	/* Texture Combiner2 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].srcRgb"), GL_PREVIOUS, GL_TEXTURE3, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].srcAlpha"), GL_PREVIOUS, GL_TEXTURE3, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[2].combineRgb"), GL_ADD );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[2].combineAlpha"), GL_ADD );
}


static unsigned frame_num = 0;

static int drawframe(void)
{	
	glClearColor(0.1f , 0.008f * (frame_num % 100), 0.8f, 1.f );
	glClearDepthf( 1.f );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	glBindTexture(GL_TEXTURE_COLLECTION_DMP, texcollection);

	glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, coords) ;
	glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, 0, color);
	glVertexAttribPointer(2, 4, GL_FLOAT, GL_FALSE, 0, texture);
	glVertexAttribPointer(3, 4, GL_FLOAT, GL_FALSE, 0, proc_texture);

	mat4_t m2;
	GLfloat mv[16];
	m2 = mat4_t::scale(1.0f - (GLfloat)(0.01 * (frame_num % 100)), 1.0f - (GLfloat)(0.01 * (frame_num % 100)), 1.0f - (GLfloat)(0.01 * (frame_num % 100)));
	m2 *= mat4_t::rotate((GLfloat)frame_num,0,0,1);
	m2.toFloatArr(mv);
	glUniformMatrix4fv( glGetUniformLocation( pgid, "uModelView" ), 1, GL_FALSE, mv );


	GLfloat F_FunctionRGB[256];
	GLfloat t = frame_num / 100.f;
	LoadSin(F_FunctionRGB, 3.f + sin(DMP_PI * t) * 1.5f);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionRGB);

	glDrawElements(GL_TRIANGLE_FAN, 4, GL_UNSIGNED_SHORT, idxs);

	glFinish();

	swap_buffer();

	glBindTexture(GL_TEXTURE_COLLECTION_DMP, 0);

	frame_num++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame_num = 0;

	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation( pgid, 0, "aPosition" );
	glBindAttribLocation( pgid, 1, "aColor" );
	glBindAttribLocation( pgid, 2, "aTexCoord" );
	glBindAttribLocation( pgid, 3, "aTexCoord_proc" );

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	GLfloat proj[16];
	mat4_t m = mat4_t::ortho(-1.f, 1.f, -1.f, 1.f, -1.f, 1.f);
	m.toFloatArr(proj);
	glUniformMatrix4fv( glGetUniformLocation( pgid, "uProjection" ), 1, GL_FALSE, proj );


	glViewport( 0, 0, WIDTH, HEIGHT );

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);
	glEnableVertexAttribArray(2);
	glEnableVertexAttribArray(3);

	load_textures();

	glEnable( GL_CULL_FACE );
	glFrontFace( GL_CCW );

	return 0;
}

/* finalization */
static void finalize()
{
	if (texid[0]) {
		glDeleteTextures( TEXID_NUM, texid );
		memset(texid, 0, sizeof(GLuint) * TEXID_NUM);
	}

	if (texid_proc[0]) {
		glDeleteTextures( PROC_TEXID_NUM, texid_proc );
		memset(texid_proc, 0, sizeof(GLuint) * PROC_TEXID_NUM);
	}

	if (texcollection) {
		glDeleteTextures( 1, &texcollection );
		texcollection = 0;
	}

	if (array_buffer_id) {
		glDeleteBuffers(1, &array_buffer_id);
		array_buffer_id = 0;
	}
	if (element_array_buffer_id) {
		glDeleteBuffers(1, &element_array_buffer_id);
		element_array_buffer_id = 0;
	}
	if (shid) {
		glDeleteShader(shid);
		shid = 0;
	}
	if (pgid) {
		glDeleteProgram(pgid);
		pgid = 0;
	}
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	/* finalization */
	finalize();
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

