/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

/*
 * Comments
 * --------
 *
 * This sample illustrates an adaptive tesselation of Catmull-Clark subdivision
 * surface described as a set of subdivision primitives.
 * Subdivision primitives are generated from original quadrangle mesh with at most one irregular
 * vertex per quadrangle face (irregular means the number of adjacent edges is not 4).
 * The difference between original and subdivided mesh is most noticeable at
 * silhouette. Different vertex normals WRT different adjacent faces are preserved
 * (not averaged), so normal (and other parameter) discontinuities are kept in subdivided mesh.
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "SubdivisionSimple"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace SubdivisionSimple{

/* program id */
GLuint pgid;

/* shader id */
GLuint shid[2];

template <typename TYPE> class simple_vector
{
public:
	simple_vector()
	{
		element_size = 512;
		current_size = 0;
		element = (TYPE*)malloc(element_size * sizeof(TYPE));
	}
	~simple_vector()
	{
		free(element);
	}
	int size(){ return current_size; }
	void clear(){ current_size = 0; }
	void push_back(TYPE d)
	{
		element[current_size++] = d;
		if (current_size == element_size)
		{
			TYPE* new_element = (TYPE*)malloc(2 * element_size * sizeof(TYPE));
			memcpy(new_element, element, sizeof(TYPE) * element_size);
			element_size *= 2;
			free(element);
			element = new_element;
		}
	}
	TYPE& operator[](int idx) { return element[idx]; }
	
	typedef TYPE*	iterator;
	iterator begin() { return element; }
	iterator end() { return &element[current_size]; }
	

protected:
	TYPE*	element;
	int		current_size;
	int		element_size;
};

GLuint array_buffer_id;
GLuint element_array_buffer_id;
GLsizei index_num;

/* generate simple subdivision object */
static void load_objects(void)
{
	struct object_info_t
	{
		simple_vector<unsigned short> idx;
		simple_vector<float> v;
		simple_vector<float> n;
		simple_vector<float> c;
		simple_vector<float> t;
	} simple_info;
	
	for (unsigned int i = 0; i < 7; i++)
	{
		for (unsigned int j = 0; j < 7; j++)
		{

			float x = -3.f + (float)i;
			float y = -3.f + (float)j;
			float z = 0.5f*(x*x + y*y);
			simple_info.v.push_back(x);
			simple_info.v.push_back(y);
			simple_info.v.push_back(z);
			/*
			simple_info.c.push_back(( (j*7+i)%3==0 )? 1.f: 0.f);
			simple_info.c.push_back(( (j*7+i)%3==1 )? 1.f: 0.f);
			simple_info.c.push_back(( (j*7+i)%3==2 )? 1.f: 0.f);
			*/
			simple_info.c.push_back(( (j*7+i)%2==0 )? 1.f: 0.f);
			simple_info.c.push_back(( (j*7+i)%2==0 )? 1.f: 0.f);
			simple_info.c.push_back(( (j*7+i)%2==0 )? 1.f: 0.f);
		}
	}

	for (unsigned int i = 0; i < 4; i++)
	{
		for (unsigned int j = 0; j < 4; j++)
		{
			simple_info.idx.push_back(16);
			simple_info.idx.push_back(( j+2 )*7 + ( i+2 ));
			simple_info.idx.push_back(( j+2 )*7 + ( i+1 ));
			simple_info.idx.push_back(( j+1 )*7 + ( i+1 ));
			simple_info.idx.push_back(( j+1 )*7 + ( i+2 ));
			simple_info.idx.push_back(( j+3 )*7 + ( i+2 ));
			simple_info.idx.push_back(( j+3 )*7 + ( i+1 ));
			simple_info.idx.push_back(( j+3 )*7 + ( i+0 ));
			simple_info.idx.push_back(( j+2 )*7 + ( i+0 ));
			simple_info.idx.push_back(( j+1 )*7 + ( i+0 ));
			simple_info.idx.push_back(( j+0 )*7 + ( i+0 ));
			simple_info.idx.push_back(( j+0 )*7 + ( i+1 ));
			simple_info.idx.push_back(( j+0 )*7 + ( i+2 ));
			simple_info.idx.push_back(( j+0 )*7 + ( i+3 ));
			simple_info.idx.push_back(( j+1 )*7 + ( i+3 ));
			simple_info.idx.push_back(( j+2 )*7 + ( i+3 ));
			simple_info.idx.push_back(( j+3 )*7 + ( i+3 ));
		}
	}

	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	
	glBufferData(GL_ARRAY_BUFFER, sizeof(float) * (simple_info.v.size() + simple_info.c.size()), 0, GL_STATIC_DRAW);
	glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(float) * simple_info.v.size(), &simple_info.v[0]);
	glBufferSubData(GL_ARRAY_BUFFER, sizeof(float) * simple_info.v.size(), sizeof(float) * simple_info.c.size(), &simple_info.c[0]);

	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);
	glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(sizeof(float) * simple_info.v.size()));
	
	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);
	
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(unsigned short) * simple_info.idx.size(), &simple_info.idx[0], GL_STATIC_DRAW);
	index_num = simple_info.idx.size();
}

static int frame = 0;

int drawframe(void)
{
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	mat4_t modelview;
	GLfloat m[16];
	
	/* projection setting */
	/* modelview setting */
	modelview = mat4_t::lookAt(0.f, 0.f, 15.f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	modelview = modelview * mat4_t::translate(0.f, 0.f, 10*sinf((float)frame / 36.f));
	modelview = modelview * mat4_t::rotate(-(float)frame, 0.f, 1.f, 0.f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	glDrawElements(GL_GEOMETRY_PRIMITIVE_DMP, (GLsizei)index_num, GL_UNSIGNED_SHORT, 0);
	glFinish();

	/* it is possible to save the content of the buffer */
	/*
	char fname[256];
	sprintf(fname, "frame-%04d.tga", f);
	outputImage(WIDTH, HEIGHT, fname);
	*/

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid[0] = glCreateShader(GL_VERTEX_SHADER);
	shid[1] = glCreateShader(GL_GEOMETRY_SHADER_DMP);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;

	glShaderBinary(2, shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid[0]);
	glAttachShader(pgid, shid[1]);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aColor");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glDisable(GL_CULL_FACE);

	load_objects();
	
	/* set subdivision level */
	glUniform1f(glGetUniformLocation(pgid, "dmp_Subdivision.level"), 1);
	
	/* set fragment lighting disabling */
	glUniform1i(glGetUniformLocation(pgid, "dmp_Subdivision.fragmentLightingEnabled"), GL_FALSE);
	
	mat4_t proj;
	GLfloat m[16];
	
	/* projection setting */
	proj = mat4_t::frustum(-0.07f, 0.07f, -0.07f * HEIGHT / WIDTH, 0.07f * HEIGHT / WIDTH, 0.2f, 200.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	glDeleteBuffers(1, &array_buffer_id);
	glDeleteBuffers(1, &element_array_buffer_id);
	glDeleteProgram(pgid);
	glDeleteShader(shid[0]);
	glDeleteShader(shid[1]);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

