/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "SubdivisionLoopSimple"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace SubdivisionLoopSimple{

/* program id */
GLuint pgid;

/* shader id */
GLuint shid[2];

GLuint array_buffer_id;
GLuint element_array_buffer_id;

GLuint g_num_index;
GLuint g_num_subd_index;

template <typename TYPE> class simple_set
{
public:
	simple_set()
	{
		element_size = 128;
		current_size = 0;
		element = (TYPE*)malloc(element_size * sizeof(TYPE));
	}
	~simple_set()
	{
		free(element);
	}
	int size(){ return current_size; }
	void clear(){ current_size = 0; }
	void insert(TYPE d)
	{
		int i;
		for (i = 0; i < current_size; i++)
		{
			if (element[i] == d)
				break;
		}
		if (i == current_size)
		{
			element[current_size++] = d;
			if (current_size == element_size)
			{
				TYPE* new_element = (TYPE*)malloc(2 * element_size * sizeof(TYPE));
				memcpy(new_element, element, sizeof(TYPE) * element_size);
				element_size *= 2;
				free(element);
				element = new_element;
			}
		}
	}
	
	typedef TYPE*	iterator;
	iterator begin() { return element; }
	iterator end() { return &element[current_size]; }

protected:
	TYPE*	element;
	int		current_size;
	int		element_size;
};

template <typename TYPE> class simple_vector
{
public:
	simple_vector()
	{
		element_size = 512;
		current_size = 0;
		element = (TYPE*)malloc(element_size * sizeof(TYPE));
	}
	~simple_vector()
	{
		free(element);
	}
	int size(){ return current_size; }
	void clear(){ current_size = 0; }
	void push_back(TYPE d)
	{
		element[current_size++] = d;
		if (current_size == element_size)
		{
			TYPE* new_element = (TYPE*)malloc(2 * element_size * sizeof(TYPE));
			memcpy(new_element, element, sizeof(TYPE) * element_size);
			element_size *= 2;
			free(element);
			element = new_element;
		}
	}
	TYPE& operator[](int idx) { return element[idx]; }
	
	typedef TYPE*	iterator;
	iterator begin() { return element; }
	iterator end() { return &element[current_size]; }
	

protected:
	TYPE*	element;
	int		current_size;
	int		element_size;
};

static GLushort* make_loopsubdivision_info(GLushort* tri_index, GLuint num_index, GLuint* num_subd_index, GLfloat* valence_array)
{
	simple_vector<GLushort>	loop_subd_index;
	simple_set<GLushort> e0s, e1s, e2s;
	
	for (unsigned i = 0; i < num_index; i += 3)
	{
		GLushort v0, v1, v2;
		GLushort e00, e01, e0p, e10, e11, e1p, e20, e21, e2p;
		
		e00 = e01 = e0p = e10 = e11 = e1p = e20 = e21 = e2p = 0;	/* avoid warning */
		e0s.clear();
		e1s.clear();
		e2s.clear();
		
		/* get main vertices */
		v0 = tri_index[i];
		v1 = tri_index[i + 1];
		v2 = tri_index[i + 2];
		
		if (v0 == v1 || v1 == v2 || v2 == v0)
			continue;	/* this module doesn't support degenerate triangle. */
		
		for (unsigned j = 0; j < num_index; j += 3)
		{
			unsigned match_mask = 0;
			bool find_v[3] = {false};
			
			if (tri_index[j] == tri_index[j + 1] || tri_index[j + 1] == tri_index[j + 2] || tri_index[j + 2] == tri_index[j])
				continue;	/* this module doesn't support degenerate triangle. */
			
			/* find vertex sharing edge with v0, v1, v2 */
			for (int k = 0; k < 3; k++)
			{
				if (tri_index[j + k] == v0)
				{
					e0s.insert(tri_index[j + (k + 1) % 3]);
					e0s.insert(tri_index[j + (k + 2) % 3]);
					find_v[0] = true;
					match_mask |= 1 << k;
				}
				if (tri_index[j + k] == v1)
				{
					e1s.insert(tri_index[j + (k + 1) % 3]);
					e1s.insert(tri_index[j + (k + 2) % 3]);
					find_v[1] = true;
					match_mask |= 1 << k;
				}
				if (tri_index[j + k] == v2)
				{
					e2s.insert(tri_index[j + (k + 1) % 3]);
					e2s.insert(tri_index[j + (k + 2) % 3]);
					find_v[2] = true;
					match_mask |= 1 << k;
				}
			}
			
			/* find ei0 */
			if (find_v[0] && !find_v[1] && find_v[2])
			{
				switch (match_mask)
				{
					case 0x6: e00 = tri_index[j + 0]; break;
					case 0x5: e00 = tri_index[j + 1]; break;
					case 0x3: e00 = tri_index[j + 2]; break;
				}
			}
			else if (find_v[0] && find_v[1] && !find_v[2])
			{
				switch (match_mask)
				{
					case 0x6: e10 = tri_index[j + 0]; break;
					case 0x5: e10 = tri_index[j + 1]; break;
					case 0x3: e10 = tri_index[j + 2]; break;
				}
			}
			else if (!find_v[0] && find_v[1] && find_v[2])
			{
				switch (match_mask)
				{
					case 0x6: e20 = tri_index[j + 0]; break;
					case 0x5: e20 = tri_index[j + 1]; break;
					case 0x3: e20 = tri_index[j + 2]; break;
				}
			}
		}

		for (unsigned j = 0; j < num_index; j += 3)
		{
			if (tri_index[j] == tri_index[j + 1] || tri_index[j + 1] == tri_index[j + 2] || tri_index[j + 2] == tri_index[j])
				continue;	/* this module doesn't support degenerate triangle. */
			
			for (unsigned m = 0; m < 3; m++)
			{
				unsigned match_mask = 0;
				bool find_v[3] = {false};
				
				GLushort vi0, vi1, ei1_0, ei0_p, ei1_1;
				vi0 = vi1 = ei1_0 = ei0_p = ei1_1 = 0;	/* avoid warning */
				
				switch (m)
				{
					case 0: vi0 = v0; vi1 = v1; ei1_0 = e10; break;
					case 1: vi0 = v1; vi1 = v2; ei1_0 = e20; break;
					case 2: vi0 = v2; vi1 = v0; ei1_0 = e00; break;
				}
				
				for (int k = 0; k < 3; k++)
				{
					if (tri_index[j + k] == vi0)
					{
						find_v[0] = true;
						match_mask |= 1 << k;
					}
					if (tri_index[j + k] == vi1)
					{
						find_v[1] = true;
						match_mask |= 1 << k;
					}
					if (tri_index[j + k] == ei1_0)
					{
						find_v[2] = true;
						match_mask |= 1 << k;
					}
				}
				
				/* find eip and ei1 */
				if (find_v[0] && !find_v[1] && find_v[2])
				{
					switch (match_mask)
					{
						case 0x6: ei0_p = tri_index[j + 0]; break;
						case 0x5: ei0_p = tri_index[j + 1]; break;
						case 0x3: ei0_p = tri_index[j + 2]; break;
					}
					switch (m)
					{
						case 0: e0p = ei0_p; break;
						case 1: e1p = ei0_p; break;
						case 2: e2p = ei0_p; break;
					}
				}
				else if (!find_v[0] && find_v[1] && find_v[2])
				{
					switch (match_mask)
					{
						case 0x6: ei1_1 = tri_index[j + 0]; break;
						case 0x5: ei1_1 = tri_index[j + 1]; break;
						case 0x3: ei1_1 = tri_index[j + 2]; break;
					}
					switch (m)
					{
						case 0: e11 = ei1_1; break;
						case 1: e21 = ei1_1; break;
						case 2: e01 = ei1_1; break;
					}
				}
			}
		}
		
		/* make index array */
		loop_subd_index.push_back((GLushort)(e0s.size() + e1s.size() + e2s.size() + 3));
		loop_subd_index.push_back(v0);
		loop_subd_index.push_back(v1);
		loop_subd_index.push_back(v2);
		for (simple_set<GLushort>::iterator iter = e0s.begin(); iter != e0s.end(); ++iter)
			loop_subd_index.push_back(*iter);
		for (simple_set<GLushort>::iterator iter = e1s.begin(); iter != e1s.end(); ++iter)
			loop_subd_index.push_back(*iter);
		for (simple_set<GLushort>::iterator iter = e2s.begin(); iter != e2s.end(); ++iter)
			loop_subd_index.push_back(*iter);
		loop_subd_index.push_back(12);
		loop_subd_index.push_back(v0);
		loop_subd_index.push_back(v1);
		loop_subd_index.push_back(v2);
		loop_subd_index.push_back(e00);
		loop_subd_index.push_back(e10);
		loop_subd_index.push_back(e20);
		loop_subd_index.push_back(e01);
		loop_subd_index.push_back(e11);
		loop_subd_index.push_back(e21);
		loop_subd_index.push_back(e0p);
		loop_subd_index.push_back(e1p);
		loop_subd_index.push_back(e2p);
		
		valence_array[v0] = (GLfloat)e0s.size();
		valence_array[v1] = (GLfloat)e1s.size();
		valence_array[v2] = (GLfloat)e2s.size();
	}
	*num_subd_index = (GLuint)loop_subd_index.size();
	GLushort* ret = (GLushort*)malloc(sizeof(GLushort) * loop_subd_index.size());
	memcpy(ret, &loop_subd_index[0], sizeof(GLushort) * loop_subd_index.size());
	
	#if 0
	/* dump index */
	{
		for (GLuint i = 0; i < *num_subd_index; )
		{
			int patch_size = ret[i];
			printf("%d: ", ret[i]);
			i++;
			for (int j = 0; j < patch_size; j++)
			{
				printf("%d ", ret[i]);
				i++;
			}
			printf("\n");
		}
	}
	#endif
	
	return ret;
}

static void load_objects(void)
{
	GLfloat vertex[] =
	{
		-1.2f,	 0.0f,	 1.2f,
		 1.2f,	 0.0f,	 1.2f,
		 1.2f,	 0.0f,	-1.2f,
		-1.2f,	 0.0f,	-1.2f,
		 0.0f,	 1.2f,	 0.0f,
		 0.0f,	-1.2f,	 0.0f,
	};

	GLfloat color[] =
	{
		1.f, 0.f, 0.f, 1.f,
		0.f, 1.f, 0.f, 1.f,
		0.f, 0.f, 1.f, 1.f,
		1.f, 1.f, 0.f, 1.f,
		1.f, 1.f, 1.f, 1.f,
		0.f, 0.f, 0.f, 1.f,
	};

	GLushort index[] =
	{
		0, 1, 4,
		1, 2, 4,
		2, 3, 4,
		3, 0, 4,
		1, 0, 5,
		2, 1, 5,
		3, 2, 5,
		0, 3, 5,
	};
	
	GLfloat* valence_array;
	GLushort* loopsubd_index_array;
	
	valence_array = (GLfloat*)malloc(sizeof(GLfloat) * (sizeof(vertex) / (sizeof(GLfloat) * 3)));
	g_num_index = sizeof(index) / sizeof(GLushort);
	loopsubd_index_array = make_loopsubdivision_info(index, g_num_index, &g_num_subd_index, valence_array);

	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	
	glBufferData(GL_ARRAY_BUFFER, sizeof(vertex) + sizeof(color) + (sizeof(vertex) / 3), 0, GL_STATIC_DRAW);
	glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(vertex), vertex);
	glBufferSubData(GL_ARRAY_BUFFER, sizeof(vertex), sizeof(color), color);
	glBufferSubData(GL_ARRAY_BUFFER, sizeof(vertex) + sizeof(color), (sizeof(vertex) / 3), valence_array);

	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);
	glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(sizeof(vertex)));
	glVertexAttribPointer(2, 1, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(sizeof(vertex) + sizeof(color)));
	
	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);
	glEnableVertexAttribArray(2);
	
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLushort) * g_num_subd_index, loopsubd_index_array, GL_STATIC_DRAW);
	
	free(valence_array);
	free(loopsubd_index_array);
}

static int frame = 0;

int drawframe(void)
{
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	mat4_t modelview;
	GLfloat m[16];
	
	/* projection setting */
	/* modelview setting */
	modelview = mat4_t::lookAt(0.f, 0.f, 7.f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	modelview = modelview * mat4_t::rotate(-(float)frame, 1.f, 1.f, 1.f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	/* draw subdivision level 0 */
	glViewport(0, 0, WIDTH, HEIGHT/2);
	glUniform1f(glGetUniformLocation(pgid, "dmp_Subdivision.level"), 0.f);
	glDrawElements(GL_GEOMETRY_PRIMITIVE_DMP, g_num_subd_index, GL_UNSIGNED_SHORT, 0);

	/* draw subdivision level 2 */
	glViewport(0, HEIGHT/2, WIDTH, HEIGHT/2);
	glUniform1f(glGetUniformLocation(pgid, "dmp_Subdivision.level"), 2.f);
	glDrawElements(GL_GEOMETRY_PRIMITIVE_DMP, g_num_subd_index, GL_UNSIGNED_SHORT, 0);
	glFinish();

	/* it is possible to save the content of the buffer */
	/*
	char fname[256];
	sprintf(fname, "frame-%04d.tga", f);
	outputImage(WIDTH, HEIGHT, fname);
	*/

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid[0] = glCreateShader(GL_VERTEX_SHADER);
	shid[1] = glCreateShader(GL_GEOMETRY_SHADER_DMP);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(2, shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid[0]);
	glAttachShader(pgid, shid[1]);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aColor");
	glBindAttribLocation(pgid, 2, "aValence");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	load_objects();
	
	/* set fragment lighting disabling */
	glUniform1i(glGetUniformLocation(pgid, "dmp_Subdivision.fragmentLightingEnabled"), GL_FALSE);
	
	mat4_t proj;
	GLfloat m[16];
	
	/* projection setting */
	proj = mat4_t::frustum(-0.03f, 0.03f, -0.03f * HEIGHT / WIDTH, 0.03f * HEIGHT / WIDTH, 0.2f, 200.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	glDeleteBuffers(1, &array_buffer_id);
	glDeleteBuffers(1, &element_array_buffer_id);
	glDeleteProgram(pgid);
	glDeleteShader(shid[0]);
	glDeleteShader(shid[1]);
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

