/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "SubdivisionLoop"
#define WIDTH 240
#define HEIGHT 400

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif
#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace SubdivisionLoop{

/* program id */
GLuint pgid;

/* shader id */
GLuint shid[2];

/* panda data */
dat_t panda;

/* diffuse texture name */
GLuint diff_tex_name;

static void load_objects(void)
{
	bool use_alpha;

	loadDAT(FILE_APP_ROOT "pandasub.dat", &panda);

	/* load texture */
	glGenTextures(1, &diff_tex_name);
	glActiveTexture(GL_TEXTURE0);
	/* stored in the texture collection */
	glBindTexture(GL_TEXTURE_2D, diff_tex_name);
	/* limit texture sampling to the top level only */
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_LOD, 0);
	loadTexture(FILE_APP_ROOT "panda_d.tga", GL_TEXTURE_2D, 0, use_alpha, true);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
}

static void unload_objects(void)
{
	unloadDAT(&panda);
	
	return;
}

static int frame = 0;

int drawframe(void)
{
	float step = 1.f;

	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	mat4_t modelview;
	GLfloat m[16];
	
	/* modelview setting */
	modelview = mat4_t::lookAt (
		1.5f * sinf(-step * frame * (float)M_PI / 180.f),
		0.f,
		1.5f * cosf(-step * frame * (float)M_PI / 180.f), 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	modelview = modelview * mat4_t::scale(0.8f, 0.8f, 0.8f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	glDrawElements(GL_GEOMETRY_PRIMITIVE_DMP, panda.patch[0].elm_size,
		GL_UNSIGNED_SHORT, (GLvoid*)(panda.patch[0].elm_offset + panda.obj[0].elm_offset));
	glFinish();

	/* it is possible to save the content of the buffer */
	/*
	char fname[256];
	sprintf(fname, "frame-%04d.tga", f);
	outputImage(WIDTH, HEIGHT, fname);
	*/

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid[0] = glCreateShader(GL_VERTEX_SHADER);
	shid[1] = glCreateShader(GL_GEOMETRY_SHADER_DMP);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(2, shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid[0]);
	glAttachShader(pgid, shid[1]);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aTexcoord0");
    glBindAttribLocation(pgid, 2, "aValence");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	load_objects();

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);
	glEnableVertexAttribArray(2);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	mat4_t proj;
	GLfloat m[16];
	
	/* projection setting */
	proj = mat4_t::frustum(-0.08f, 0.08f, -0.08f * HEIGHT / WIDTH, 0.08f * HEIGHT / WIDTH, 0.2f, 1000.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	glBindBuffer(GL_ARRAY_BUFFER, panda.posVB);
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);
	glBindBuffer(GL_ARRAY_BUFFER, panda.texVB);
	glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, 0);
	glBindBuffer(GL_ARRAY_BUFFER, panda.midxVB);
	glVertexAttribPointer(2, 4, GL_UNSIGNED_BYTE, GL_FALSE, 0, 0);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, panda.idxVB);

    glBindTexture(GL_TEXTURE_2D, diff_tex_name);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE0, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_TEXTURE0, GL_CONSTANT, GL_CONSTANT);

	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[1].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[1].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcRgb"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);

    glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_TEXTURE_2D);

    glUniform1i(glGetUniformLocation(pgid, "dmp_Subdivision.fragmentLightingEnabled"), GL_FALSE);
    glUniform1f(glGetUniformLocation(pgid, "dmp_Subdivision.level"), 2.f);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	unload_objects();
	glDeleteProgram(pgid);
	glDeleteShader(shid[0]);
	glDeleteShader(shid[1]);
	glDeleteTextures(1, &diff_tex_name);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

