/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "SilhouetteSimple"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)
#ifndef PI
#define PI DMP_PI
#endif

using namespace gputest::common;

namespace gputest{
namespace SilhouetteSimple{

/* buffer id */
GLuint array_buffer_id;
GLuint element_array_buffer_id;

/* program id */
GLuint pgid;

/* shader id */
GLuint shid[2];

/* index count */
GLint index_count;

enum primitive_type_t
{
	TRIANGLES,
	TRIANGLE_STRIP,
	LINE_STRIP,
	SUBD_PRIM_DMP,
	TRIANGLES_SILHOUETTE_DMP,
	TRIANGLE_STRIP_SILHOUETTE_DMP,
	LINES,
	POINTS
};

/* difinitions to specify use_arrays parameter */
#define USE_ARRAY_POSITION			0x0001
#define USE_ARRAY_COLOR				0x0002
#define USE_ARRAY_COLOR_BYTE		0x0004
#define USE_ARRAY_TEXCOORD0			0x0008
#define USE_ARRAY_TEXCOORD1			0x0010
#define USE_ARRAY_TEXCOORD2			0x0020
#define USE_ARRAY_TEXCOORD			(USE_ARRAY_TEXCOORD0 | USE_ARRAY_TEXCOORD1 | USE_ARRAY_TEXCOORD2)
#define USE_ARRAY_NORMAL			0x0040
#define USE_ARRAY_TANGENT			0x0080
#define USE_ARRAY_PALETTE			0x0100
#define USE_ARRAY_TEXCOORD0_3COMP	0x0200

/* difinitions to specify array_type */
typedef enum tag_array_type_t
{
	INDEPENDENT_ARRAY,
	INTERLEAVED_ARRAY
} array_type_t;

typedef struct tag_position_t
{
	GLfloat		x, y, z;
} position_t;

typedef struct tag_color_t
{
	GLfloat		r, g, b, a;
} color_t;

typedef struct tag_byte_color_t
{
	GLubyte		r, g, b, a;
} byte_color_t;

typedef struct tag_texcoord_t
{
	GLfloat		u, v;
} texcoord_t;

typedef struct tag_texcoord3_t
{
	GLfloat		u, v, w;
} texcoord3_t;

typedef struct tag_normal_t
{
	GLfloat		x, y, z;
} normal_t;

typedef struct tag_tangent_t
{
	GLfloat		x, y, z;
} tangent_t;

typedef struct tag_idx_t
{
	GLubyte		i[4];
} idx_t;

typedef struct tag_weight_t
{
	GLfloat		w[4];
} weight_t;

static void make_torus(GLenum geometry_mode, unsigned use_arrays, array_type_t array_type, int palette_size,
			int horizontal_polygon_count, int virtical_polygon_count, GLint* index_count)
{
	int vx, vy;
	int vxnum = horizontal_polygon_count;
	int vynum = virtical_polygon_count;
	
	int vertex_size = 0;
	unsigned char* vertex;
	int pos_offs, col_offs, tex_offs[3], nrm_offs, tang_offs, idx_offs, wgt_offs;

	pos_offs = col_offs = tex_offs[0] = tex_offs[1] = tex_offs[2] = nrm_offs = tang_offs = idx_offs = wgt_offs = 0;

	/* check vertex size and array offset */
	if (use_arrays & USE_ARRAY_POSITION)
	{
		pos_offs = vertex_size;
		vertex_size += sizeof(position_t);
	}
	if (use_arrays & USE_ARRAY_COLOR)
	{
		col_offs = vertex_size;
		vertex_size += sizeof(color_t);
	}
	else if (use_arrays & USE_ARRAY_COLOR_BYTE)
	{
		col_offs = vertex_size;
		vertex_size += sizeof(byte_color_t);
	}
	if (use_arrays & USE_ARRAY_NORMAL)
	{
		nrm_offs = vertex_size;
		vertex_size += sizeof(normal_t);
	}
	if (use_arrays & USE_ARRAY_TANGENT)
	{
		tang_offs = vertex_size;
		vertex_size += sizeof(tangent_t);
	}
	if (use_arrays & USE_ARRAY_PALETTE)
	{
		idx_offs = vertex_size;
		vertex_size += sizeof(idx_t);
		wgt_offs = vertex_size;
		vertex_size += sizeof(weight_t) / 4 * palette_size;
	}
	if (use_arrays & USE_ARRAY_TEXCOORD0)
	{
		tex_offs[0] = vertex_size;
		vertex_size += sizeof(texcoord_t);
	}
	if (use_arrays & USE_ARRAY_TEXCOORD1)
	{
		tex_offs[1] = vertex_size;
		vertex_size += sizeof(texcoord_t);
	}
	if (use_arrays & USE_ARRAY_TEXCOORD2)
	{
		tex_offs[2] = vertex_size;
		vertex_size += sizeof(texcoord_t);
	}
	
	/* make torus arrays */
	vertex = (unsigned char*)malloc(vertex_size * (vxnum + 1) * (vynum + 1));
	memset(vertex, 0, vertex_size * (vxnum + 1) * (vynum + 1));
	{
		const float rad_x = 0.4f;
		const float rad_y = 1.f;
		
		for (vy = 0; vy <= vynum; vy++)
		{
			float cosx, cosy, sinx, siny;
			
			cosy = (float)cos((2 * PI / vynum) * (vy % vynum));
			siny = (float)sin((2 * PI / vynum) * (vy % vynum));

			for (vx = 0; vx <= vxnum; vx++)
			{
				cosx = (float)cos((2 * PI / vxnum) * (vx % vxnum));
				sinx = (float)sin((2 * PI / vxnum) * (vx % vxnum));

				if (array_type == INTERLEAVED_ARRAY)
				{
					int offset = vertex_size * (vy * (vxnum + 1) + vx);
					
					/* position */
					if (use_arrays & USE_ARRAY_POSITION)
					{
						((position_t*)&(vertex[offset]))->x = cosy * (rad_y + cosx * rad_x);
						((position_t*)&(vertex[offset]))->y = siny * (rad_y + cosx * rad_x);
						((position_t*)&(vertex[offset]))->z = - sinx * rad_x;
						offset += sizeof(position_t);
					}
					
					/* color */
					if (use_arrays & USE_ARRAY_COLOR)
					{
						((color_t*)&(vertex[offset]))->r = 0.5f + cosx * 0.5f;
						((color_t*)&(vertex[offset]))->g = 0.5f - cosx * 0.5f;
						((color_t*)&(vertex[offset]))->b = 0.5f + cosy * 0.5f;
						((color_t*)&(vertex[offset]))->a = 1.f;
						offset += sizeof(color_t);
					}
					else if (use_arrays & USE_ARRAY_COLOR_BYTE)
					{
						((byte_color_t*)&(vertex[offset]))->r = (GLubyte)(255 * (0.5f + cosx * 0.5f));
						((byte_color_t*)&(vertex[offset]))->g = (GLubyte)(255 * (0.5f - cosx * 0.5f));
						((byte_color_t*)&(vertex[offset]))->b = (GLubyte)(255 * (0.5f + cosy * 0.5f));
						((byte_color_t*)&(vertex[offset]))->a = 255;
						offset += sizeof(byte_color_t);
					}
					
					/* normal */
					if (use_arrays & USE_ARRAY_NORMAL)
					{
						float norm[4];
						norm[0] = cosy * (rad_y + cosx * rad_x) - rad_y * cosy;
						norm[1] = siny * (rad_y + cosx * rad_x) - rad_y * siny;
						norm[2] = - sinx * rad_x;
						norm[3] = (float)sqrt(norm[0] * norm[0] + norm[1] * norm[1] + norm[2] * norm[2]);
						((normal_t*)&(vertex[offset]))->x = norm[0] / norm[3];
						((normal_t*)&(vertex[offset]))->y = norm[1] / norm[3];
						((normal_t*)&(vertex[offset]))->z = norm[2] / norm[3];
						offset += sizeof(normal_t);
					}
					
					/* tangent */
					if (use_arrays & USE_ARRAY_TANGENT)
					{
						((tangent_t*)&(vertex[offset]))->x = -siny;
						((tangent_t*)&(vertex[offset]))->y = cosy;
						((tangent_t*)&(vertex[offset]))->z = 0;
						offset += sizeof(tangent_t);
					}
					
					if (use_arrays & USE_ARRAY_PALETTE)
					{
						float ratio;
						
						/* palette index */
						((idx_t*)&(vertex[offset]))->i[0] = 0;
						((idx_t*)&(vertex[offset]))->i[1] = 1;
						((idx_t*)&(vertex[offset]))->i[2] = 2;
						((idx_t*)&(vertex[offset]))->i[3] = 3;
						offset += sizeof(idx_t);
						
						/* palette weight */
						
						ratio = (float)vy / (float)vynum;
						switch (palette_size)
						{
							case 4:
								((weight_t*)&(vertex[offset]))->w[0] = 0.01f;
								((weight_t*)&(vertex[offset]))->w[1] = 0.01f;
								((weight_t*)&(vertex[offset]))->w[2] = 0.01f;
								((weight_t*)&(vertex[offset]))->w[3] = 0.01f;
								if (ratio <= 0.25f)
								{
									((weight_t*)&(vertex[offset]))->w[0] = cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[1] = 1 - cosy - 0.01f;
								}
								else if (ratio <= 0.5f)
								{
									((weight_t*)&(vertex[offset]))->w[1] = 1 + cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[2] = -cosy - 0.01f;
								}
								else if (ratio <= 0.75f)
								{
									((weight_t*)&(vertex[offset]))->w[2] = -cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[3] = 1 + cosy - 0.01f;
								}
								else
								{
									((weight_t*)&(vertex[offset]))->w[3] = 1 - cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[0] = cosy - 0.01f;
								}
								break;
							case 3:
								if (ratio < 0.3333f)
								{
									((weight_t*)&(vertex[offset]))->w[0] = 1.f - ratio * 3.f - 0.005f;
									((weight_t*)&(vertex[offset]))->w[1] = ratio * 3.f - 0.005f;
								}
								else if (ratio < 0.6666f)
								{
									ratio -= 0.3333f;
									((weight_t*)&(vertex[offset]))->w[1] = 1.f - ratio * 3.f - 0.005f;
									((weight_t*)&(vertex[offset]))->w[2] = ratio * 3.f - 0.005f;
								}
								else
								{
									ratio -= 0.6666f;
									((weight_t*)&(vertex[offset]))->w[2] = 1.f - ratio * 3.f - 0.005f;
									((weight_t*)&(vertex[offset]))->w[0] = ratio * 3.f - 0.005f;
								}
								break;
							case 2:
								((weight_t*)&(vertex[offset]))->w[0] = (cosy + 1.f) / 2.f;
								((weight_t*)&(vertex[offset]))->w[1] = 1.f - (cosy + 1.f) / 2.f;
								break;
							case 1:
								((weight_t*)&(vertex[offset]))->w[0] = 1.f;
								break;
						}
						if (vy == 0 || vy == vynum)
						{
							switch (palette_size)
							{
								case 4:
									((weight_t*)&(vertex[offset]))->w[3] = 0.f;
								case 3:
									((weight_t*)&(vertex[offset]))->w[2] = 0.f;
								case 2:
									((weight_t*)&(vertex[offset]))->w[1] = 0.f;
								case 1:
									((weight_t*)&(vertex[offset]))->w[0] = 1.f;
									break;
							}
						}
						offset += sizeof(weight_t) / 4 * palette_size;
					}

					/* texture coord */
					if (use_arrays & USE_ARRAY_TEXCOORD)
					{
						if (use_arrays & USE_ARRAY_TEXCOORD0)
						{
							((texcoord_t*)&(vertex[offset]))->u = (float)vx * (float)(1.f / vxnum);
							((texcoord_t*)&(vertex[offset]))->v = (float)vy * (float)(1.f / vynum);
							offset += sizeof(texcoord_t);
						}
						if (use_arrays & USE_ARRAY_TEXCOORD1)
						{
							((texcoord_t*)&(vertex[offset]))->u = (float)vx * (float)(1.f / vxnum);
							((texcoord_t*)&(vertex[offset]))->v = (float)vy * (float)(1.f / vynum);
							offset += sizeof(texcoord_t);
						}
						if (use_arrays & USE_ARRAY_TEXCOORD2)
						{
							((texcoord_t*)&(vertex[offset]))->u = (float)vx * (float)(1.f / vxnum);
							((texcoord_t*)&(vertex[offset]))->v = (float)vy * (float)(1.f / vynum);
							offset += sizeof(texcoord_t);
						}
					}
					
				}
				else if (array_type == INDEPENDENT_ARRAY)
				{
					int offset;
					
					/* position */
					if (use_arrays & USE_ARRAY_POSITION)
					{
						offset = (vy * (vxnum + 1) + vx) * sizeof(position_t) + pos_offs * (vxnum + 1) * (vynum + 1);
						((position_t*)&(vertex[offset]))->x = cosy * (rad_y + cosx * rad_x);
						((position_t*)&(vertex[offset]))->y = siny * (rad_y + cosx * rad_x);
						((position_t*)&(vertex[offset]))->z = - sinx * rad_x;
					}
					
					/* color */
					if (use_arrays & USE_ARRAY_COLOR)
					{
						offset = (vy * (vxnum + 1) + vx) * sizeof(color_t) + col_offs * (vxnum + 1) * (vynum + 1);
						((color_t*)&(vertex[offset]))->r = 0.5f + cosx * 0.5f;
						((color_t*)&(vertex[offset]))->g = 0.5f - cosx * 0.5f;
						((color_t*)&(vertex[offset]))->b = 0.5f + cosy * 0.5f;
						((color_t*)&(vertex[offset]))->a = 1.f;
					}
					else if (use_arrays & USE_ARRAY_COLOR_BYTE)
					{
						offset = (vy * (vxnum + 1) + vx) * sizeof(byte_color_t) + col_offs * (vxnum + 1) * (vynum + 1);
						((byte_color_t*)&(vertex[offset]))->r = (GLubyte)(255 * (0.5f + cosx * 0.5f));
						((byte_color_t*)&(vertex[offset]))->g = (GLubyte)(255 * (0.5f - cosx * 0.5f));
						((byte_color_t*)&(vertex[offset]))->b = (GLubyte)(255 * (0.5f + cosy * 0.5f));
						((byte_color_t*)&(vertex[offset]))->a = 255;
					}
					
					/* normal */
					if (use_arrays & USE_ARRAY_NORMAL)
					{
						float norm[4];
						norm[0] = cosy * (rad_y + cosx * rad_x) - rad_y * cosy;
						norm[1] = siny * (rad_y + cosx * rad_x) - rad_y * siny;
						norm[2] = - sinx * rad_x;
						norm[3] = (float)sqrt(norm[0] * norm[0] + norm[1] * norm[1] + norm[2] * norm[2]);
						offset = (vy * (vxnum + 1) + vx) * sizeof(normal_t) + nrm_offs * (vxnum + 1) * (vynum + 1);
						((normal_t*)&(vertex[offset]))->x = norm[0] / norm[3];
						((normal_t*)&(vertex[offset]))->y = norm[1] / norm[3];
						((normal_t*)&(vertex[offset]))->z = norm[2] / norm[3];
					}
					
					/* tangent */
					if (use_arrays & USE_ARRAY_TANGENT)
					{
						offset = (vy * (vxnum + 1) + vx) * sizeof(tangent_t) + tang_offs * (vxnum + 1) * (vynum + 1);
						((tangent_t*)&(vertex[offset]))->x = -siny;
						((tangent_t*)&(vertex[offset]))->y = cosy;
						((tangent_t*)&(vertex[offset]))->z = 0;
					}
					
					if (use_arrays & USE_ARRAY_PALETTE)
					{
						float ratio;
						
						/* palette index */
						offset = (vy * (vxnum + 1) + vx) * (sizeof(idx_t) / 4 * palette_size) + idx_offs * (vxnum + 1) * (vynum + 1);
						switch (palette_size)
						{
							case 4:
								((idx_t*)&(vertex[offset]))->i[3] = 3;
							case 3:
								((idx_t*)&(vertex[offset]))->i[2] = 2;
							case 2:
								((idx_t*)&(vertex[offset]))->i[1] = 1;
							case 1:
								((idx_t*)&(vertex[offset]))->i[0] = 0;
								break;
						}
						
						/* palette weight */
						offset = (vy * (vxnum + 1) + vx) * (sizeof(weight_t) / 4 * palette_size) + wgt_offs * (vxnum + 1) * (vynum + 1);
						ratio = (float)vy / (float)vynum;
						switch (palette_size)
						{
							case 4:
								((weight_t*)&(vertex[offset]))->w[0] = 0.01f;
								((weight_t*)&(vertex[offset]))->w[1] = 0.01f;
								((weight_t*)&(vertex[offset]))->w[2] = 0.01f;
								((weight_t*)&(vertex[offset]))->w[3] = 0.01f;
								if (ratio <= 0.25f)
								{
									((weight_t*)&(vertex[offset]))->w[0] = cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[1] = 1 - cosy - 0.01f;
								}
								else if (ratio <= 0.5f)
								{
									((weight_t*)&(vertex[offset]))->w[1] = 1 + cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[2] = -cosy - 0.01f;
								}
								else if (ratio <= 0.75f)
								{
									((weight_t*)&(vertex[offset]))->w[2] = -cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[3] = 1 + cosy - 0.01f;
								}
								else
								{
									((weight_t*)&(vertex[offset]))->w[3] = 1 - cosy - 0.01f;
									((weight_t*)&(vertex[offset]))->w[0] = cosy - 0.01f;
								}
								break;
							case 3:
								if (ratio < 0.3333f)
								{
									((weight_t*)&(vertex[offset]))->w[0] = 1.f - ratio * 3.f - 0.005f;
									((weight_t*)&(vertex[offset]))->w[1] = ratio * 3.f - 0.005f;
								}
								else if (ratio < 0.6666f)
								{
									ratio -= 0.3333f;
									((weight_t*)&(vertex[offset]))->w[1] = 1.f - ratio * 3.f - 0.005f;
									((weight_t*)&(vertex[offset]))->w[2] = ratio * 3.f - 0.005f;
								}
								else
								{
									ratio -= 0.6666f;
									((weight_t*)&(vertex[offset]))->w[2] = 1.f - ratio * 3.f - 0.005f;
									((weight_t*)&(vertex[offset]))->w[0] = ratio * 3.f - 0.005f;
								}
								break;
							case 2:
								((weight_t*)&(vertex[offset]))->w[0] = (cosy + 1.f) / 2.f;
								((weight_t*)&(vertex[offset]))->w[1] = 1.f - (cosy + 1.f) / 2.f;
								break;
							case 1:
								((weight_t*)&(vertex[offset]))->w[0] = 1.f;
								break;
						}
						if (vy == 0 || vy == vynum)
						{
							switch (palette_size)
							{
								case 4:
									((weight_t*)&(vertex[offset]))->w[3] = 0.f;
								case 3:
									((weight_t*)&(vertex[offset]))->w[2] = 0.f;
								case 2:
									((weight_t*)&(vertex[offset]))->w[1] = 0.f;
								case 1:
									((weight_t*)&(vertex[offset]))->w[0] = 1.f;
									break;
							}
						}
					}
					
					/* texture coord */
					if (use_arrays & USE_ARRAY_TEXCOORD)
					{
						if (use_arrays & USE_ARRAY_TEXCOORD0)
						{
							offset = (vy * (vxnum + 1) + vx) * sizeof(texcoord_t) + tex_offs[0] * (vxnum + 1) * (vynum + 1);
							((texcoord_t*)&(vertex[offset]))->u = (float)vx * (float)(1.f / vxnum);
							((texcoord_t*)&(vertex[offset]))->v = (float)vy * (float)(1.f / vynum);
						}
						if (use_arrays & USE_ARRAY_TEXCOORD1)
						{
							offset = (vy * (vxnum + 1) + vx) * sizeof(texcoord_t) + tex_offs[1] * (vxnum + 1) * (vynum + 1);
							((texcoord_t*)&(vertex[offset]))->u = (float)vx * (float)(1.f / vxnum);
							((texcoord_t*)&(vertex[offset]))->v = (float)vy * (float)(1.f / vynum);
						}
						if (use_arrays & USE_ARRAY_TEXCOORD2)
						{
							offset = (vy * (vxnum + 1) + vx) * sizeof(texcoord_t) + tex_offs[2] * (vxnum + 1) * (vynum + 1);
							((texcoord_t*)&(vertex[offset]))->u = (float)vx * (float)(1.f / vxnum);
							((texcoord_t*)&(vertex[offset]))->v = (float)vy * (float)(1.f / vynum);
						}
					}
				}
			}
		}
	}
	
	/* Update vertex data */
	glBufferData(GL_ARRAY_BUFFER, vertex_size * (vxnum + 1) * (vynum + 1), vertex, GL_STATIC_DRAW);
	free(vertex);
	
	/* Specify the array pointer */
	if (array_type == INTERLEAVED_ARRAY)
	{
		int attr_number = 0;
		if (use_arrays & USE_ARRAY_POSITION)
			glVertexAttribPointer(attr_number++, 3, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)pos_offs);
		if (use_arrays & USE_ARRAY_COLOR)
			glVertexAttribPointer(attr_number++, 4, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)col_offs);
		else if (use_arrays & USE_ARRAY_COLOR_BYTE)
			glVertexAttribPointer(attr_number++, 4, GL_UNSIGNED_BYTE, GL_FALSE, vertex_size, (GLvoid*)col_offs);
		if (use_arrays & USE_ARRAY_NORMAL)
			glVertexAttribPointer(attr_number++, 3, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)nrm_offs);
		if (use_arrays & USE_ARRAY_TANGENT)
			glVertexAttribPointer(attr_number++, 3, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)tang_offs);
		if (use_arrays & USE_ARRAY_PALETTE)
		{
			glVertexAttribPointer(attr_number++, palette_size, GL_UNSIGNED_BYTE, GL_FALSE, vertex_size, (GLvoid*)idx_offs);
			glVertexAttribPointer(attr_number++, palette_size, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)wgt_offs);
		}
		if (use_arrays & USE_ARRAY_TEXCOORD)
		{
			if (use_arrays & USE_ARRAY_TEXCOORD0)
				glVertexAttribPointer(attr_number++, 2, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)tex_offs[0]);
			if (use_arrays & USE_ARRAY_TEXCOORD1)
				glVertexAttribPointer(attr_number++, 2, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)tex_offs[1]);
			if (use_arrays & USE_ARRAY_TEXCOORD2)
				glVertexAttribPointer(attr_number++, 2, GL_FLOAT, GL_FALSE, vertex_size, (GLvoid*)tex_offs[2]);
		}
	}
	else if (array_type == INDEPENDENT_ARRAY)
	{
		int attr_number = 0;
		if (use_arrays & USE_ARRAY_POSITION)
			glVertexAttribPointer(attr_number++, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(pos_offs * (vxnum + 1) * (vynum + 1)));
		if (use_arrays & USE_ARRAY_COLOR)
			glVertexAttribPointer(attr_number++, 4, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(col_offs * (vxnum + 1) * (vynum + 1)));
		else if (use_arrays & USE_ARRAY_COLOR_BYTE)
			glVertexAttribPointer(attr_number++, 4, GL_UNSIGNED_BYTE, GL_FALSE, 0, (GLvoid*)(col_offs * (vxnum + 1) * (vynum + 1)));
		if (use_arrays & USE_ARRAY_NORMAL)
			glVertexAttribPointer(attr_number++, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(nrm_offs * (vxnum + 1) * (vynum + 1)));
		if (use_arrays & USE_ARRAY_TANGENT)
			glVertexAttribPointer(attr_number++, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(tang_offs * (vxnum + 1) * (vynum + 1)));
		if (use_arrays & USE_ARRAY_PALETTE)
		{
			glVertexAttribPointer(attr_number++, palette_size, GL_UNSIGNED_BYTE, GL_FALSE, 0, (GLvoid*)(idx_offs * (vxnum + 1) * (vynum + 1)));
			glVertexAttribPointer(attr_number++, palette_size, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(wgt_offs * (vxnum + 1) * (vynum + 1)));
		}
		if (use_arrays & USE_ARRAY_TEXCOORD)
		{
			if (use_arrays & USE_ARRAY_TEXCOORD0)
				glVertexAttribPointer(attr_number++, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(tex_offs[0] * (vxnum + 1) * (vynum + 1)));
			if (use_arrays & USE_ARRAY_TEXCOORD1)
				glVertexAttribPointer(attr_number++, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(tex_offs[1] * (vxnum + 1) * (vynum + 1)));
			if (use_arrays & USE_ARRAY_TEXCOORD2)
				glVertexAttribPointer(attr_number++, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(tex_offs[2] * (vxnum + 1) * (vynum + 1)));
		}
	}

	/* Make triangle index */
	{
		int idxnum = 0;
		unsigned short* index_array = 0;
		
		/* Make triangles index */
		if (geometry_mode == TRIANGLES)
		{
			index_array = (unsigned short*)malloc(sizeof(unsigned short) * vxnum * vynum * 6);
			memset(index_array, 0, sizeof(unsigned short) * vxnum * vynum * 6);
			for (vy = 0; vy < vynum; vy++)
			{
				for (vx = 0; vx < vxnum; vx++)
				{
					unsigned short rect[4];
					rect[0] = vx + vy * (vxnum + 1);
					rect[1] = (vx + 1) + vy * (vxnum + 1);
					rect[2] = vx + (vy + 1) * (vxnum + 1);
					rect[3] = (vx + 1) + (vy + 1) * (vxnum + 1);
					
					index_array[idxnum++] = rect[2];
					index_array[idxnum++] = rect[0];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[0];
					index_array[idxnum++] = rect[1];
				}
			}
		}

		/* Make triangle strip index */
		else if (geometry_mode == TRIANGLE_STRIP || geometry_mode == LINE_STRIP)
		{
			index_array = (unsigned short*)malloc(sizeof(unsigned short) * (vxnum + 1) * 2 * vynum);
			memset(index_array, 0, sizeof(unsigned short) * (vxnum + 1) * 2 * vynum);
			for (vy = vynum - 1; vy >= 0; vy--)
			{
				for (vx = 0; vx <= vxnum; vx++)
				{
					index_array[idxnum++] = vx + (vy + 1) * (vxnum + 1);
					index_array[idxnum++] = vx + vy * (vxnum + 1);
				}
			}
		}

		/* Make subdivision index */
		else if (geometry_mode == SUBD_PRIM_DMP)
		{
			index_array = (unsigned short*)malloc(sizeof(unsigned short) * vxnum * vynum * 17);
			memset(index_array, 0, sizeof(unsigned short) * vxnum * vynum * 17);
			for (vy = 0; vy < vynum; vy++)
			{
				for (vx = 0; vx < vxnum; vx++)
				{
					index_array[idxnum +  0] = 16;
					index_array[idxnum +  1] = vy * (vxnum + 1) + vx;
					index_array[idxnum +  2] = vy * (vxnum + 1) + vx + 1;
					index_array[idxnum +  3] = (vy + 1) * (vxnum + 1) + vx + 1;
					index_array[idxnum +  4] = (vy + 1) * (vxnum + 1) + vx;
					index_array[idxnum +  5] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx;
					index_array[idxnum +  6] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx + 1;
					index_array[idxnum +  7] = ((vy + vynum -1) % vynum) * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum);
					index_array[idxnum +  8] = vy * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum);
					index_array[idxnum +  9] = (vy + 1) * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum);
					index_array[idxnum + 10] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum);
					index_array[idxnum + 11] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + vx + 1;
					index_array[idxnum + 12] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + vx;
					index_array[idxnum + 13] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + ((vx - 1 + vxnum) % vxnum);
					index_array[idxnum + 14] = (vy + 1) * (vxnum + 1) + ((vx - 1 + vxnum) % vxnum);
					index_array[idxnum + 15] = vy * (vxnum + 1) + ((vx - 1 + vxnum) % vxnum);
					index_array[idxnum + 16] = ((vy + vynum -1) % vynum) * (vxnum + 1) + ((vx - 1 + vxnum) % vxnum);
					idxnum += 17;
				}
			}
		}

		/* Make silhouette triangle index */
		else if (geometry_mode == TRIANGLES_SILHOUETTE_DMP)
		{
			index_array = (unsigned short*)malloc(sizeof(unsigned short) * vxnum * vynum * 12);
			memset(index_array, 0, sizeof(unsigned short) * vxnum * vynum * 12);
			for (vy = 0; vy < vynum; vy++)
			{
				for (vx = 0; vx < vxnum; vx++)
				{
					int rect[8];
					rect[0] = vy * (vxnum + 1) + vx;
					rect[1] = vy * (vxnum + 1) + vx + 1;
					rect[2] = (vy + 1) * (vxnum + 1) + vx;
					rect[3] = (vy + 1) * (vxnum + 1) + vx + 1;
					rect[4] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx;
					rect[5] = (vy + 1) * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum);
					rect[6] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + vx + 1;
					rect[7] = vy * (vxnum + 1) + ((vx - 1 + vxnum) % vxnum);
					
					index_array[idxnum++] = rect[0];
					index_array[idxnum++] = rect[1];
					index_array[idxnum++] = rect[4];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[2];
					index_array[idxnum++] = rect[5];
					
					index_array[idxnum++] = rect[0];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[1];
					index_array[idxnum++] = rect[2];
					index_array[idxnum++] = rect[7];
					index_array[idxnum++] = rect[6];
				}
			}
		}

		/* Make silhouette strip index */
		else if (geometry_mode == TRIANGLE_STRIP_SILHOUETTE_DMP)
		{
			index_array = (unsigned short*)malloc(sizeof(unsigned short) * (vxnum * vynum * 4 + 4));
			memset(index_array, 0, sizeof(unsigned short) * (vxnum * vynum * 4 + 4));
			for (vy = vynum - 1; vy >= 0; vy--)
			{
				for (vx = 0; vx < vxnum; vx++)
				{
					if (vx == 0 && (vy == (vynum - 1)))
					{
						index_array[idxnum++] = (vy + 1) * (vxnum + 1) + vx;
						index_array[idxnum++] = vy * (vxnum + 1) + vx;
						index_array[idxnum++] = vy * (vxnum + 1) + ((vx - 1 + vxnum) % vxnum);
						index_array[idxnum++] = (vy + 1) * (vxnum + 1) + vx + 1;
						index_array[idxnum++] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + vx + 1;
						index_array[idxnum++] = vy * (vxnum + 1) + vx + 1;
						index_array[idxnum++] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx;
						index_array[idxnum++] = (vy + 1) * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum);
					}
					else if (vx == vxnum - 2)
					{
						index_array[idxnum++] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + vx + 1; /* 11 */
						index_array[idxnum++] = vy * (vxnum + 1) + vx + 1; /* 2 */
						index_array[idxnum++] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx; /* 5 */
						index_array[idxnum++] = vy * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum); /* 8 */
					}
					else if (vx == vxnum - 1)
					{
						index_array[idxnum++] = (vy + 1) * (vxnum + 1) + vx + 1; /* 3 */
						index_array[idxnum++] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx + 1; /* 6 */
						index_array[idxnum++] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx; /* 5 */
						index_array[idxnum++] = vy * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum); /* 8 */
					}
					else
					{
						index_array[idxnum++] = ((vy + 2 + vynum) % vynum) * (vxnum + 1) + vx + 1; /* 11 */
						index_array[idxnum++] = vy * (vxnum + 1) + vx + 1; /* 2 */
						index_array[idxnum++] = ((vy + vynum -1) % vynum) * (vxnum + 1) + vx; /* 5 */
						index_array[idxnum++] = (vy + 1) * (vxnum + 1) + ((vx + 2 + vxnum) % vxnum); /* 9 */
					}
				}
			}
		}

		/* Make lines index (wireframe) */
		else if (geometry_mode == LINES)
		{
			index_array = (unsigned short*)malloc(sizeof(unsigned short) * vxnum * vynum * 6 * 2);
			memset(index_array, 0, sizeof(unsigned short) * vxnum * vynum * 6 * 2);
			for (vy = 0; vy < vynum; vy++)
			{
				for (vx = 0; vx < vxnum; vx++)
				{
					unsigned short rect[4];
					rect[0] = vx + vy * (vxnum + 1);
					rect[1] = (vx + 1) + vy * (vxnum + 1);
					rect[2] = vx + (vy + 1) * (vxnum + 1);
					rect[3] = (vx + 1) + (vy + 1) * (vxnum + 1);
					
					index_array[idxnum++] = rect[0];
					index_array[idxnum++] = rect[1];
					index_array[idxnum++] = rect[1];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[0];
					index_array[idxnum++] = rect[0];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[3];
					index_array[idxnum++] = rect[2];
					index_array[idxnum++] = rect[2];
					index_array[idxnum++] = rect[0];
				}
			}
		}

		else if (geometry_mode == POINTS)
		{
			index_array = (unsigned short*)malloc(sizeof(unsigned short) * vxnum * vynum);
			memset(index_array, 0, sizeof(unsigned short) * vxnum * vynum);
			for (vy = 0; vy < vynum; vy++)
			{
				for (vx = 0; vx < vxnum; vx++)
				{
					index_array[idxnum++] = vx + vy * (vxnum + 1);
				}
			}
		}

		/* No other primitive */
		else
			assert(0);

		*index_count = idxnum;
		glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(unsigned short) * idxnum, index_array, GL_STATIC_DRAW);
		free(index_array);
	}

	return;
}	/* void make_torus(GLenum geometry_mode, unsigned use_arrays, array_type_t array_type,
			int horizontal_polygon_count, int virtical_polygon_count, GLint* index_count) */

/* generate simple object */
static void load_objects(void)
{
	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	
	make_torus(TRIANGLE_STRIP_SILHOUETTE_DMP,
		USE_ARRAY_POSITION | USE_ARRAY_COLOR | USE_ARRAY_NORMAL, INTERLEAVED_ARRAY, 0, 40, 40, &index_count);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);
	glEnableVertexAttribArray(2);
}

static int frame = 0;

int drawframe(void)
{
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	GLfloat m[16];

	mat4_t proj = mat4_t::frustum(-0.02f, 0.02f, -0.02f*HEIGHT/WIDTH, 0.02f*HEIGHT/WIDTH, 0.1f, 100.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	mat4_t mv = mat4_t::lookAt(0.f, 0.4f, 7.5f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	mv = mv * mat4_t::rotate(-6.f*frame, 0.f, 1.f, 0.f);
	mv.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	glDrawElements(GL_GEOMETRY_PRIMITIVE_DMP, index_count, GL_UNSIGNED_SHORT, 0);
	glFinish();

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid[0] = glCreateShader(GL_VERTEX_SHADER);
	shid[1] = glCreateShader(GL_GEOMETRY_SHADER_DMP);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;

	glShaderBinary(2, shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid[0]);
	glAttachShader(pgid, shid[1]);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aColor");
	glBindAttribLocation(pgid, 2, "aNormal");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	load_objects();
	
	GLfloat silhwidth[2];
	silhwidth[0] = 0.15f;
	silhwidth[1] = silhwidth[0] * (float)WIDTH / (float)HEIGHT;
	glUniform2fv(glGetUniformLocation(pgid, "dmp_Silhouette.width"), 1, silhwidth);
	
	GLfloat silhcolor[4] = {0.f, 0.f, 1.f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_Silhouette.color"), 1, silhcolor);
	
	glUniform1i(glGetUniformLocation(pgid, "dmp_Silhouette.frontFaceCCW"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Silhouette.acceptEmptyTriangles"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Silhouette.scaleByW"), GL_TRUE);
	
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	glDeleteBuffers(1, &array_buffer_id);
	glDeleteBuffers(1, &element_array_buffer_id);
	glDeleteProgram(pgid);
	glDeleteShader(shid[0]);
	glDeleteShader(shid[1]);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

