/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "ShadowKnot"
#define WIDTH 240
#define HEIGHT 400

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif
#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace ShadowKnot{

/* program id */
GLuint lrProg, sdwProg, sdwSilhProg;

/* shader id */
GLuint shaders[4];

/* obj loader class objet */
dat_t knot, knotsilh, plane;

/* GL object name */
GLuint shadow_tex_name;
GLuint fbo_name;
GLuint lutids[3];

/* frame number */
unsigned int frame_no = 0;
/* 1st pass (shadow accumulation pass) frustum parameter */
const GLfloat n = 0.1f, f = 10.f, r = 0.1f, t = 0.1f;
/* light position */
const GLfloat lpos0[] = {1.f, 2.f, -1.f, 1.f};

/* shadow setting */
#define SBWIDTH 512
#define SBHEIGHT 512
/* shadow artifact suppression SETTINGS 0 - lowest 2 - highest */
#define SDW_SETTINGS 0
const float fSZBiasScale[] = {1.2f, 1.3f, 1.3f};
const float fSZScale[] = {1.f, 2.f, 4.f};

/* if this is defined, soft shadow can be used */
#define SOFT_SHADOW

/* draw hard shadow to shadow buffer */
static void render_objects_shadow(dat_t& _objs)
{
	glDisableVertexAttribArray(1);

	for (int i = 0; i < _objs.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, _objs.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)_objs.obj[i].vtx_offset);

		for (unsigned j = _objs.obj[i].patch_offset; j < _objs.obj[i].patch_size + _objs.obj[i].patch_offset; j++)
		{
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _objs.idxVB);
			glDrawElements(GL_TRIANGLES, _objs.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(_objs.patch[j].elm_offset + _objs.obj[i].elm_offset));
		}
	}
}

/* draw soft shadow to shadow buffer */
static void render_objects_shadow_silhouette(dat_t& _objs)
{
	glEnableVertexAttribArray(1);

	glDisable(GL_CULL_FACE);
	for (int i = 0; i < _objs.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, _objs.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)_objs.obj[i].vtx_offset);
		glBindBuffer(GL_ARRAY_BUFFER, _objs.normVB);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (void*)_objs.obj[i].nrm_offset);

		for (unsigned j = _objs.obj[i].patch_offset; j < _objs.obj[i].patch_size + _objs.obj[i].patch_offset; j++)
		{
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _objs.idxVB);
			glDrawElements(GL_GEOMETRY_PRIMITIVE_DMP, _objs.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(_objs.patch[j].elm_offset + _objs.obj[i].elm_offset));
		}
	}
	glEnable(GL_CULL_FACE);
}

static void render_objects_shadowed(dat_t& _objs)
{
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, shadow_tex_name);
	glEnableVertexAttribArray(1);

	for (int i = 0; i < _objs.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, _objs.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)_objs.obj[i].vtx_offset);
		glBindBuffer(GL_ARRAY_BUFFER, _objs.normVB);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (void*)_objs.obj[i].nrm_offset);

		for (unsigned j = _objs.obj[i].patch_offset; j < _objs.obj[i].patch_size + _objs.obj[i].patch_offset; j++)
		{
			GLfloat ma[4] = {0.f, 0.f, 0.f, 1.f};
			GLfloat md[4] = {0.f, 0.f, 0.f, 1.f};
			GLfloat ms[4] = {0.f, 0.f, 0.f, 1.f};
			for (int col = 0; col < 3; col++)
			{
				ma[col] = _objs.patch[j].ambient[col];
				md[col] = _objs.patch[j].diffuse[col];
				ms[col] = _objs.patch[j].specular[col];
			}
			glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentMaterial.ambient"), 1, ma);
			glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentMaterial.diffuse"), 1, md);
			glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentMaterial.specular0"), 1, ms);
			glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentMaterial.specular1"), 1, ms);

			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _objs.idxVB);
			glDrawElements(GL_TRIANGLES, _objs.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(_objs.patch[j].elm_offset + _objs.obj[i].elm_offset));
		}
	}
}

/* load objects */
static void load_objects(void)
{
	loadDAT(FILE_APP_ROOT "knot.dat", &knot);
	loadDAT(FILE_APP_ROOT "knotsilh.dat", &knotsilh);
	loadDAT(FILE_APP_ROOT "plane.dat", &plane);
}

/* unload objects */
static void unload_objects(void)
{
	unloadDAT(&knot);
	unloadDAT(&knotsilh);
	unloadDAT(&plane);
}

/* draw object depth to shadow buffer */
static void draw_1st_pass(void)
{
	/* set shadow accumulation render state */
	glBindFramebuffer(GL_FRAMEBUFFER, fbo_name);

	/* set camera position and view frustum in light position */
	glViewport(0, 0, SBWIDTH, SBHEIGHT);
	glBindTexture(GL_TEXTURE_2D, 0);

	/* shadow buffer clear color needs to be (1, 1, 1)color */
	glClearColor(1.f, 1.f, 1.f, 1.f);
	glClear(GL_COLOR_BUFFER_BIT);

	glUseProgram(sdwProg);
	
	GLfloat m[16];
	mat4_t modelview;
	modelview = mat4_t::lookAt(lpos0[0], lpos0[1], lpos0[2], 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(sdwProg, "uModelView"), 1, GL_FALSE, m);

	render_objects_shadow(plane);

	modelview = modelview * mat4_t::rotate((float)frame_no, 1.f, 1.f, 1.f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(sdwProg, "uModelView"), 1, GL_FALSE, m);

	render_objects_shadow(knot);

	glFinish();

#ifdef SOFT_SHADOW
	glUseProgram(sdwSilhProg);

	glUniformMatrix4fv(glGetUniformLocation(sdwSilhProg, "uModelView"), 1, GL_FALSE, m);

	render_objects_shadow_silhouette(knotsilh);
#endif
	glFinish();

	/* for debug purposes, it is possible to save the content of the shadow buffer */
	/*
	{
		char fnamesb[256];
		sprintf(fnamesb, "sb-%04d.tga", frame_no);
		outputImage(SBWIDTH, SBHEIGHT, fnamesb);
	}
	*/

	/* restore original(GL) fragment operation render state */
	//glBindFramebuffer(GL_FRAMEBUFFER, 0);
	cmn_setRenderTarget(0);
}

static void draw_2nd_pass(void)
{
	glViewport(0, 0, WIDTH, HEIGHT);

	glClearColor(0.36f+(frame_no%100)*0.0064f, 0.42f, 0.5f, 1.f);
	glClearDepthf(1.f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glUseProgram(lrProg);

	GLfloat m[16] ;
	mat4_t modelview;
	modelview = mat4_t::lookAt(
		3.f * sinf(frame_no * (float)M_PI/180.f),
		1.f,
		3.f * cosf(frame_no * (float)M_PI/180.f), 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(lrProg, "uModelView"), 1, GL_FALSE, m);

	vec4_t l(lpos0);
	l = modelview * l;
	glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentLightSource[0].position"), 1, &l.x);

	mat4_t tex = mat4_t::frustum(r / n, -3.f * r / n, t / n, -3.f * t / n, 1.f, 0.f);
	tex = tex * mat4_t::scale(-1.f / (f - n), -1.f / (f - n), -1.f / (f - n));
	tex = tex * mat4_t::lookAt(lpos0[0], lpos0[1], lpos0[2], 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	tex.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(lrProg, "uTexture"), 1, GL_FALSE, m);

	render_objects_shadowed(plane);

	glFinish();

	modelview = modelview * mat4_t::rotate((float)frame_no, 1.f, 1.f, 1.f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(lrProg, "uModelView"), 1, GL_FALSE, m);

	tex = tex * mat4_t::rotate((float)frame_no, 1.f, 1.f, 1.f);
	tex.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(lrProg, "uTexture"), 1, GL_FALSE, m);

	render_objects_shadowed(knot);

	glFinish();
}

int drawframe(void)
{
	/* shadow accumulation pass */
	draw_1st_pass();
	/* shadow test pass */
	draw_2nd_pass();

	/* it is possible to save the content of the buffer */
	/*
	char fname[256];
	sprintf(fname, "frame-%04d.tga", frame_no);
	outputImage(WIDTH, HEIGHT, fname);
	*/

	swap_buffer();

	frame_no++;

	return !glGetError();
}

static void initialize_shadowed_program(void)
{
	/* Initialize shadowed draw program */
	lrProg = glCreateProgram();

	glAttachShader(lrProg, shaders[2]);
	glAttachShader(lrProg, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(lrProg, 0, "aPosition");
	glBindAttribLocation(lrProg, 1, "aNormal");

	glLinkProgram(lrProg);
    glUseProgram(lrProg);

    glUniform1i(glGetUniformLocation(lrProg, "dmp_Texture[0].samplerType"), GL_TEXTURE_SHADOW_2D_DMP);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_FragmentLightSource[0].shadowed"), GL_TRUE);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_LightEnv.lutInputRR"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_LightEnv.lutInputFR"), GL_LIGHT_ENV_LN_DMP);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_LightEnv.lutEnabledRefl"), GL_TRUE);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG1_DMP);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_LightEnv.fresnelSelector"), GL_LIGHT_ENV_PRI_SEC_ALPHA_FRESNEL_DMP);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_LightEnv.shadowAlpha"), GL_TRUE);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_LightEnv.invertShadow"), GL_TRUE);

	glUniform1i(glGetUniformLocation(lrProg, "dmp_FragmentMaterial.samplerRR"), 0);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_FragmentMaterial.samplerFR"), 1);

	glGenTextures(3, lutids);
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, lutids[0]);
	
	GLfloat lut[512];
	memset(lut, 0, sizeof(lut));

	int j;

	for (j = 1; j < 128; j++)
	{
		lut[j] = powf((float)j/127.f, 30.f);
		lut[j+255] = lut[j] - lut[j-1];
	}

	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	memset(lut, 0, sizeof(lut));
	for (j = 1; j < 128; j++)
	{
		lut[j] = powf((float)j / 127.f, 2.0f);
		lut[j+255] = lut[j] - lut[j-1];
	}

	glBindTexture(GL_LUT_TEXTURE1_DMP, lutids[2]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	GLfloat ld0[] = {1.f,1.f,1.f,1.f};
	GLfloat ls0[] = {0.f,0.f,0.f,0.f};
	GLfloat ls1[] = {1.f,1.f,1.f,1.f};
	GLfloat la0[] = {1.f,1.f,1.f,1.f};

	glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentLightSource[0].diffuse"), 1, ld0);
	glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentLightSource[0].specular0"), 1, ls0);
	glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentLightSource[0].specular1"), 1, ls1);
	glUniform4fv(glGetUniformLocation(lrProg, "dmp_FragmentLightSource[0].ambient"), 1, la0);

	glUniform1f(glGetUniformLocation(lrProg, "dmp_Texture[0].shadowZScale"), fSZScale[SDW_SETTINGS] / SBWIDTH);
	glUniform1f(glGetUniformLocation(lrProg, "dmp_Texture[0].shadowZBias"), n * fSZBiasScale[SDW_SETTINGS] / (f - n));
	glUniform1i(glGetUniformLocation(lrProg, "dmp_Texture[0].perspectiveShadow"), GL_TRUE);

	glUniform1i(glGetUniformLocation(lrProg, "dmp_TexEnv[0].combineRgb"), GL_ADD_MULT_DMP);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);

	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_ONE_MINUS_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_FRAGMENT_PRIMARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[0].srcAlpha"), GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_CONSTANT, GL_CONSTANT);

	glUniform1i(glGetUniformLocation(lrProg, "dmp_TexEnv[1].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_TexEnv[1].combineAlpha"), GL_REPLACE);

	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[1].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[1].srcRgb"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(lrProg, "dmp_TexEnv[1].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform1f(glGetUniformLocation(lrProg, "dmp_FragOperation.wScale"), 0.f);
	glUniform1i(glGetUniformLocation(lrProg, "dmp_FragOperation.mode"), GL_FRAGOP_MODE_GL_DMP);

	GLfloat m[16];
	mat4_t proj = mat4_t::frustum(-0.1f, 0.1f, -0.1f, 0.1f, 0.2f, 10.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(lrProg, "uProjection"), 1, GL_FALSE, m);

	glValidateProgram(lrProg);
}

static void initialize_shadow_program(void)
{
	/* Initialize shadow draw program */
	sdwProg = glCreateProgram();

	glAttachShader(sdwProg, shaders[0]);
	glAttachShader(sdwProg, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(sdwProg, 0, "aPosition");

	glLinkProgram(sdwProg);
	glUseProgram(sdwProg);

    glUniform1i(glGetUniformLocation(sdwProg, "dmp_Texture[0].samplerType"), GL_FALSE);

	glUniform1i(glGetUniformLocation(sdwProg, "dmp_FragmentLighting.enabled"), GL_FALSE);
	glUniform1i(glGetUniformLocation(sdwProg, "dmp_FragmentLightSource[0].enabled"), GL_FALSE);

	glUniform1i(glGetUniformLocation(sdwProg, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(sdwProg, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[0].srcRgb"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);

	glUniform1i(glGetUniformLocation(sdwProg, "dmp_TexEnv[1].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(sdwProg, "dmp_TexEnv[1].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[1].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[1].srcRgb"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(sdwProg, "dmp_TexEnv[1].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform1f(glGetUniformLocation(sdwProg, "dmp_FragOperation.wScale"), 1.f / f);
	glUniform1i(glGetUniformLocation(sdwProg, "dmp_FragOperation.mode"), GL_FRAGOP_MODE_SHADOW_DMP);

	GLfloat m[16];
	mat4_t proj = mat4_t::frustum(-r, r, -t, t, n, f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(sdwProg, "uProjection"), 1, GL_FALSE, m);

	glValidateProgram(sdwProg);
}

static void initialize_silhouette_program(void)
{
	/* Initialize silhouette draw program */
	sdwSilhProg = glCreateProgram();

	glAttachShader(sdwSilhProg, shaders[1]);
	glAttachShader(sdwSilhProg, shaders[3]);
	glAttachShader(sdwSilhProg, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(sdwSilhProg, 0, "aPosition");
	glBindAttribLocation(sdwSilhProg, 1, "aNormal");

	glLinkProgram(sdwSilhProg);
    glUseProgram(sdwSilhProg);

    glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_Texture[0].samplerType"), GL_FALSE);

	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_FragmentLighting.enabled"), GL_FALSE);
	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_FragmentLightSource[0].enabled"), GL_FALSE);

	glUniform1f(glGetUniformLocation(sdwSilhProg, "dmp_FragOperation.penumbraScale"), 0.0f);
	glUniform1f(glGetUniformLocation(sdwSilhProg, "dmp_FragOperation.penumbraBias"), 1.0f);

	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[1].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[1].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[1].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[1].srcRgb"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(sdwSilhProg, "dmp_TexEnv[1].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);

	glUniform1f(glGetUniformLocation(sdwSilhProg, "dmp_FragOperation.wScale"), 1.f / f);
	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_FragOperation.mode"), GL_FRAGOP_MODE_SHADOW_DMP);

	GLfloat silhwidth[2];
	silhwidth[0] = 0.05f;
	silhwidth[1] = silhwidth[0] * (float)WIDTH / (float)HEIGHT;
	glUniform2fv(glGetUniformLocation(sdwSilhProg, "dmp_Silhouette.width"), 1, silhwidth);
	
	GLfloat silhcolor[4] = {1.f, 1.f, 1.f, 1.f};
	glUniform4fv(glGetUniformLocation(sdwSilhProg, "dmp_Silhouette.color"), 1, silhcolor);
	
	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_Silhouette.frontFaceCCW"), GL_TRUE);
	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_Silhouette.scaleByW"), GL_TRUE);
	glUniform1i(glGetUniformLocation(sdwSilhProg, "dmp_Silhouette.acceptEmptyTriangles"), GL_FALSE);

	GLfloat m[16];
	mat4_t proj = mat4_t::frustum(-r, r, -t, t, n, f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(sdwSilhProg, "uProjection"), 1, GL_FALSE, m);

	glValidateProgram(sdwSilhProg);
}

/* initialization */
static int initialize(void)
{
	frame_no = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	shaders[0] = glCreateShader(GL_VERTEX_SHADER);			/* for 1st pass hard shadow rendering */
	shaders[1] = glCreateShader(GL_VERTEX_SHADER);			/* for 1st pass soft shadow rendering */
	shaders[2] = glCreateShader(GL_VERTEX_SHADER);			/* for 2nd pass shadowed rendering */
	shaders[3] = glCreateShader(GL_GEOMETRY_SHADER_DMP);	/* for 1st pass soft shadow rendering */

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(4, shaders, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);
	
	initialize_shadowed_program();
	initialize_shadow_program();
	initialize_silhouette_program();

	load_objects();

	glEnableVertexAttribArray(0);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);

	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	/* generate shadow texture */
	glGenTextures(1, &shadow_tex_name);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, shadow_tex_name);
	GLfloat bcolor[] = {1.f, 1.f, 1.f, 1.f};
	glTexParameterfv(GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, bcolor);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_LOD, 0);

	glTexImage2D(
		GL_TEXTURE_2D, 0, GL_SHADOW_DMP, SBWIDTH, SBHEIGHT,
		0, GL_SHADOW_DMP, GL_UNSIGNED_INT, 0);
	glBindTexture(GL_TEXTURE_2D, 0);

	/* generate frame buffer object name */
	glGenFramebuffers(1, &fbo_name);

	/* set shadow accumulation render state */
	glBindFramebuffer(GL_FRAMEBUFFER, fbo_name);
	/* note attachment point of shadow texture */
	glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, shadow_tex_name, 0);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	unload_objects();
	glDeleteTextures(3, lutids);
	glDeleteTextures(1, &shadow_tex_name);
	glDeleteFramebuffers(1, &fbo_name);
	glDeleteProgram(lrProg);
	glDeleteProgram(sdwProg);
	glDeleteProgram(sdwSilhProg);
	glDeleteShader(shaders[0]);
	glDeleteShader(shaders[1]);
	glDeleteShader(shaders[2]);
	glDeleteShader(shaders[3]);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}
