/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "ProctexFFunction"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace ProctexFFunction{

/* buffer id */
GLuint array_buffer_id;
GLuint element_array_buffer_id;

/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

/* Specific to the sample */
GLuint	texid[9][8];

/* generate simple object */
static void load_objects(void)
{
	/* Setting Vertex Coordinates */
	GLfloat vertex[16] =
	{
		-1.0f,-1.0f,0.0f,1.0f,
		-1.0f, 1.0f,0.0f,1.0f,
		 1.0f, 1.0f,0.0f,1.0f,
		 1.0f,-1.0f,0.0f,1.0f
	};
	/* Setting Texture Coordinates */
	GLfloat t1 = -1.0f, t2 = 1.0f;
	GLfloat texCoord[8] =
	{
		t1, t1,
		t1, t2,
		t2, t2,
		t2, t1,
	}; 
	/* Setting Indeces */
	GLushort  _quadIndex[6] = {0, 2, 1, 0, 3, 2};

	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBufferData(GL_ARRAY_BUFFER, 28 * sizeof(GLfloat), 0, GL_STATIC_DRAW);
	glBufferSubData(GL_ARRAY_BUFFER, 0, 16 * sizeof(GLfloat), vertex);
	glBufferSubData(GL_ARRAY_BUFFER, 16 * sizeof(GLfloat), 8 * sizeof(GLfloat), texCoord);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, 6 * sizeof(GLfloat), _quadIndex, GL_STATIC_DRAW);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0) ;
	glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(16 * sizeof(GLfloat)));
}

int set_1D_tex(GLfloat* RR, GLfloat* GG, GLfloat* BB, GLfloat* AA, int N)
{
	memset(RR, 0, 512 * sizeof(GLfloat));
	memset(GG, 0, 512 * sizeof(GLfloat));
	memset(BB, 0, 512 * sizeof(GLfloat));
	memset(AA, 0, 512 * sizeof(GLfloat));

	RR[7*N]=0x00/255.0f;	GG[7*N]=0x00/255.0f;	BB[7*N]=0x00/255.0f;	AA[7*N]=0xFF/255.0f;
	RR[6*N]=0x00/255.0f;	GG[6*N]=0x00/255.0f;	BB[6*N]=0xFF/255.0f;	AA[6*N]=0xFF/255.0f;
	RR[5*N]=0xFF/255.0f;	GG[5*N]=0x00/255.0f;	BB[5*N]=0x00/255.0f;	AA[5*N]=0xFF/255.0f;
	RR[4*N]=0xFF/255.0f;	GG[4*N]=0x00/255.0f;	BB[4*N]=0xFF/255.0f;	AA[4*N]=0xFF/255.0f;
	RR[3*N]=0x00/255.0f;	GG[3*N]=0xFF/255.0f;	BB[3*N]=0x00/255.0f;	AA[3*N]=0xFF/255.0f;
	RR[2*N]=0x00/255.0f;	GG[2*N]=0xFF/255.0f;	BB[2*N]=0xFF/255.0f;	AA[2*N]=0xFF/255.0f;
	RR[1*N]=0xFF/255.0f;	GG[1*N]=0xFF/255.0f;	BB[1*N]=0x00/255.0f;	AA[1*N]=0xFF/255.0f;
	RR[0*N]=0xFF/255.0f;	GG[0*N]=0xFF/255.0f;	BB[0*N]=0xFF/255.0f;	AA[0*N]=0xFF/255.0f;

	for (int x = 0; x < 255; x++)
	{
		RR[256+x] = RR[x+1] - RR[x];
		GG[256+x] = GG[x+1] - GG[x];
		BB[256+x] = BB[x+1] - BB[x];
		AA[256+x] = AA[x+1] - AA[x];
	}
	RR[255] = 0.0f;
	GG[255] = 0.0f;
	BB[255] = 0.0f;
	AA[255] = 0.0f;

	return N * 8;
}

void LoadSin(float map_1D2D_LUT[256], GLfloat f)
{
	for (int i = 0; i < 128; i++)
		map_1D2D_LUT[i] = sin(fmod(f * DMP_PI * i/128.f, DMP_PI));

	for (int i = 0; i < 127; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];

	map_1D2D_LUT[255] = sin(fmod(f * DMP_PI, DMP_PI)) - map_1D2D_LUT[127];
}

void LoadPow(float map_1D2D_LUT[256], GLfloat v)
{
	for (int i = 0; i< 128; i++)
		map_1D2D_LUT[i] = pow(i / 128.f, v);

	for (int i = 0; i < 127; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];

	map_1D2D_LUT[255] = 1.f - map_1D2D_LUT[127];
}

void LoadHat(float map_1D2D_LUT[256], GLfloat h)
{
	for (int i = 0; i < 128; i++)
	{
		if (i < 64)
			map_1D2D_LUT[i] = h * (i / 64.f);
		else
			map_1D2D_LUT[i] = h * ((128 - i) / 64.f);
	}
	for (int i = 0; i < 127; i++)
	{
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];
	}
	map_1D2D_LUT[255] = 0.f - map_1D2D_LUT[127];
}

void LoadLamp(float map_1D2D_LUT[256], GLfloat a)
{
	if (a < 0)
	{
		for (int i = 0; i < 128; i++)
			map_1D2D_LUT[i] = -a * (1.f - i / 128.f);
	}
	else
	{
		for (int i = 0; i < 128; i++)
			map_1D2D_LUT[i] = a * (i / 128.f);
	}

	for (int i = 0; i < 127; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];

	map_1D2D_LUT[255] = ((a < 0) ? 0.f : a)  - map_1D2D_LUT[127];
}

void LoadFlat(float map_1D2D_LUT[256], float a)
{
	for (int i = 0; i < 128; i++)
	{
		map_1D2D_LUT[i] = a;
		map_1D2D_LUT[i + 128] = 0;
	}
}

void load_ProceduralTexture(int kid, float* F_FunctionRGB)
{
	/* Binding of the procedural texture as a texture collection */
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, texid[kid][0]);
	
	/* Declaration */
	float F_FunctionA[256], F_Noise[256];
	LoadFlat(F_FunctionA, 1.0f);
	memset(F_Noise, 0, 256 * sizeof(float));

	/* Texture collection */
	glBindTexture(GL_LUT_TEXTURE0_DMP, texid[kid][1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionRGB);

	glBindTexture(GL_LUT_TEXTURE1_DMP, texid[kid][2]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionA);
	
	glBindTexture(GL_LUT_TEXTURE2_DMP, texid[kid][3]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_Noise);

	const int NN = 512;
	GLfloat RR[NN], GG[NN], BB[NN], AA[NN];
	int texSize = set_1D_tex(RR, GG, BB, AA, 1);

	glBindTexture(GL_LUT_TEXTURE3_DMP, texid[kid][4]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, RR);

	glBindTexture(GL_LUT_TEXTURE4_DMP, texid[kid][5]);
	glTexImage1D(GL_LUT_TEXTURE4_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, GG);

	glBindTexture(GL_LUT_TEXTURE5_DMP, texid[kid][6]);
	glTexImage1D(GL_LUT_TEXTURE5_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, BB);

	glBindTexture(GL_LUT_TEXTURE6_DMP, texid[kid][7]);
	glTexImage1D(GL_LUT_TEXTURE6_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, AA);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerRgbMap"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerAlphaMap"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerNoiseMap"), 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerR"), 3);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerG"), 4);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerB"), 5);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerA"), 6);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].samplerType"), GL_TEXTURE_PROCEDURAL_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].texcoord"), GL_TEXTURE0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptMinFilter"), GL_LINEAR);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexWidth"), texSize);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexOffset"), 0);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptRgbMap"), GL_PROCTEX_ADDSQRT2_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaMap"), GL_PROCTEX_ADDSQRT2_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaSeparate"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampU"), GL_SYMMETRICAL_REPEAT_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampV"), GL_CLAMP_TO_EDGE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftU"), GL_NONE_DMP);
    glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftV"), GL_NONE_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE3, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_TEXTURE3, GL_CONSTANT, GL_CONSTANT);
}

static int frame = 0;

int drawframe(void)
{
	/* Setting Screen */
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	GLfloat m[16];
	mat4_t proj = mat4_t::perspective(45.0f, (float)WIDTH / (float)HEIGHT, 1.0f, 200.0f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	mat4_t mv = mat4_t::lookAt(0.0f, 0.0f, 25.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	
	glDisable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE);
	
	/* Preparing texture collection */
	/* One collection for each procedural texture */
	GLfloat F_FunctionRGB[256];
	GLfloat t = frame / 50.f;
	/* Setting Lower 3 Textures */
	LoadLamp(F_FunctionRGB, 0.6f + sin(DMP_PI * t) * 0.4f);
	load_ProceduralTexture(0, F_FunctionRGB);
	LoadLamp(F_FunctionRGB, -0.6f + sin(DMP_PI * t) * 0.4f);
	load_ProceduralTexture(1, F_FunctionRGB);
	LoadHat(F_FunctionRGB, 0.7f + sin(DMP_PI * t) * 0.2f);
	load_ProceduralTexture(2, F_FunctionRGB);

	/* Setting Middle 3 Textures */
	LoadPow(F_FunctionRGB, 0.7f + sin(DMP_PI * t) * 0.2f);
	load_ProceduralTexture(3, F_FunctionRGB);
	LoadPow(F_FunctionRGB, 1.5f + sin(DMP_PI * t) * 0.5f);
	load_ProceduralTexture(4, F_FunctionRGB);
	LoadPow(F_FunctionRGB, 2.0f + sin(DMP_PI * t) * 1.2f);
	load_ProceduralTexture(5, F_FunctionRGB);

	/* Setting Upper 3 Textures */
	LoadSin(F_FunctionRGB, 1.f + sin(DMP_PI * t) * 0.2f);
	load_ProceduralTexture(6, F_FunctionRGB);
	LoadSin(F_FunctionRGB, 2.f + sin(DMP_PI * t) * 0.5f);
	load_ProceduralTexture(7, F_FunctionRGB);
	LoadSin(F_FunctionRGB, 3.f + sin(DMP_PI * t) * 1.5f);
	load_ProceduralTexture(8, F_FunctionRGB);

	/* Drawing 9 Rectangles */
	for(int i = 0; i < 9; i++)
	{
		GLfloat x = ((GLfloat)(i % 3) - 1.0f) * 4.f;
		GLfloat y = ((GLfloat)(i / 3) - 1.0f) * 4.f;
		mat4_t mv_tr;
		mv_tr = mv * mat4_t::translate(x, y, 0.0f);
		mv_tr = mv_tr * mat4_t::rotate(6.f * frame, 0.f, 0.f, 1.f);
		mv_tr = mv_tr * mat4_t::scale(1.7f, 1.7f, 1.7f);
		mv_tr.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);
		
		glBindTexture(GL_TEXTURE_COLLECTION_DMP, texid[i][0]);
		glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, 0);
	}

	glFinish();
	swap_buffer();

	frame++;
	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aTexCoord");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);
	
	load_objects();

	glGenTextures(72, &texid[0][0]);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	glDeleteBuffers(1, &array_buffer_id);
	glDeleteBuffers(1, &element_array_buffer_id);
	glDeleteTextures(72, &texid[0][0]);
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}


