/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>

#include "Memory.h"

#define APP_NAME "ProctexClamp"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace ProctexClamp{

// buffer id
GLuint array_buffer_id;
GLuint element_array_buffer_id;

// program id
GLuint pgid;

// shader id
GLuint shid;

/*Specific to the sample*/
GLuint	texid[8];

/* generate simple object */
static void load_objects(void)
{
		/* Setting Vertex Coordinates */
	GLfloat vertex[16] =
	{
		-1.0f,-1.0f,0.0f,1.0f,
		-1.0f, 1.0f,0.0f,1.0f,
		 1.0f, 1.0f,0.0f,1.0f,
		 1.0f,-1.0f,0.0f,1.0f
	};
	/* Setting Texture Coordinates */
	GLfloat t1 = 0.0f, t2 = 4.0f;
	GLfloat texCoord[8] =
	{
		t1, t1,
		t1, t2,
		t2, t2,
		t2, t1,
	};
	/* Setting Indeces */
	GLushort _quadIndex[6] = {0, 2, 1, 0, 3, 2};

	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBufferData(GL_ARRAY_BUFFER, 28 * sizeof(GLfloat), 0, GL_STATIC_DRAW);
	glBufferSubData(GL_ARRAY_BUFFER, 0, 16 * sizeof(GLfloat), vertex);
	glBufferSubData(GL_ARRAY_BUFFER, 16 * sizeof(GLfloat), 8 * sizeof(GLfloat), texCoord);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, 6 * sizeof(GLfloat), _quadIndex, GL_STATIC_DRAW);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0) ;
	glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(16 * sizeof(GLfloat)));
}

void set_1D_tex(GLfloat* RR, GLfloat* GG, GLfloat* BB, GLfloat* AA)
{
	memset(RR, 0, 256 * sizeof(GLfloat));
	memset(GG, 0, 256 * sizeof(GLfloat));
	memset(BB, 0, 256 * sizeof(GLfloat));
	memset(AA, 0, 256 * sizeof(GLfloat));

	RR[7]=0xFF/255.0f;	GG[7]=0xFF/255.0f;	BB[7]=0xFF/255.0f;	AA[7]=0xFF/255.0f;
	RR[6]=0xFF/255.0f;	GG[6]=0xFF/255.0f;	BB[6]=0x00/255.0f;	AA[6]=0xFF/255.0f;
	RR[5]=0x00/255.0f;	GG[5]=0xFF/255.0f;	BB[5]=0xFF/255.0f;	AA[5]=0xFF/255.0f;
	RR[4]=0x00/255.0f;	GG[4]=0xFF/255.0f;	BB[4]=0x00/255.0f;	AA[4]=0xFF/255.0f;
	RR[3]=0xFF/255.0f;	GG[3]=0x00/255.0f;	BB[3]=0xFF/255.0f;	AA[3]=0xFF/255.0f;
	RR[2]=0xFF/255.0f;	GG[2]=0x00/255.0f;	BB[2]=0x00/255.0f;	AA[2]=0xFF/255.0f;
	RR[1]=0x00/255.0f;	GG[1]=0x00/255.0f;	BB[1]=0xFF/255.0f;	AA[1]=0xFF/255.0f;
	RR[0]=0x00/255.0f;	GG[0]=0x00/255.0f;	BB[0]=0x00/255.0f;	AA[0]=0xFF/255.0f;

	/*RR[7]=0xFFFFFFFF;	GG[7]=0xFFFFFFFF;	BB[7]=0xFFFFFFFF;	AA[7]=0xFFFFFFFF;
	RR[6]=0xFF00FFFF;	GG[6]=0xFF00FFFF;	BB[6]=0xFF00FFFF;	AA[6]=0xFF00FFFF;
	RR[5]=0xFFFFFF00;	GG[5]=0xFFFFFF00;	BB[5]=0xFFFFFF00;	AA[5]=0xFFFFFF00;
	RR[4]=0xFF00FF00;	GG[4]=0xFF00FF00;	BB[4]=0xFF00FF00;	AA[4]=0xFF00FF00;
	RR[3]=0xFFFF00FF;	GG[3]=0xFFFF00FF;	BB[3]=0xFFFF00FF;	AA[3]=0xFFFF00FF;
	RR[2]=0xFF0000FF;	GG[2]=0xFF0000FF;	BB[2]=0xFF0000FF;	AA[2]=0xFF0000FF;
	RR[1]=0xFFFF0000;	GG[1]=0xFFFF0000;	BB[1]=0xFFFF0000;	AA[1]=0xFFFF0000;
	RR[0]=0xFF000000;	GG[0]=0xFF000000;	BB[0]=0xFF000000;	AA[0]=0xFF000000;*/

	for (int i = 0; i < 7; i++)
	{
		RR[i + 256] = RR[i + 1] - RR[i];
		GG[i + 256] = GG[i + 1] - GG[i];
		BB[i + 256] = BB[i + 1] - BB[i];
		AA[i + 256] = AA[i + 1] - AA[i];
	}
	RR[7 + 256] = 0;
	GG[7 + 256] = 0;
	BB[7 + 256] = 0;
	AA[7 + 256] = 0;
}

void LoadLamp(float map_1D2D_LUT[256])
{
	for (int i = 0; i < 128; i++)
		map_1D2D_LUT[i] = 1.f - i / 128.f;

	for (int i = 0; i < 127; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];
	map_1D2D_LUT[127 + 128] = (1.f - 128.f / 128.f) - map_1D2D_LUT[127];
}

void LoadFlat(float map_1D2D_LUT[256], float a)
{
	for (int i = 0; i < 128; i++)
	{
		map_1D2D_LUT[i] = a;
		map_1D2D_LUT[i + 128] = 0;
	}
}

void load_ProceduralTexture(void)
{
	glGenTextures(8, &texid[0]);

	/*Binding of the procedural texture as a texture collection*/
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, texid[0]);
	
	/* Declaration */
	float F_FunctionRGB[256], F_FunctionA[256], F_Noise[256];
	LoadLamp(F_FunctionRGB);
	LoadFlat(F_FunctionA, 1.0f);
	memset(F_Noise, 0, 256 * sizeof(float));

	//Texture collection
	glBindTexture(GL_LUT_TEXTURE0_DMP, texid[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionRGB);

	glBindTexture(GL_LUT_TEXTURE1_DMP, texid[2]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionA);
	
	glBindTexture(GL_LUT_TEXTURE2_DMP, texid[3]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_Noise);

	const int NN = 8 + 256;
	GLfloat RR[NN], GG[NN], BB[NN], AA[NN];
	set_1D_tex(RR, GG, BB, AA);

	glBindTexture(GL_LUT_TEXTURE3_DMP, texid[4]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP, 0, GL_LUMINANCEF_DMP, NN, 0, GL_LUMINANCEF_DMP, GL_FLOAT, RR);

	glBindTexture(GL_LUT_TEXTURE4_DMP, texid[5]);
	glTexImage1D(GL_LUT_TEXTURE4_DMP, 0, GL_LUMINANCEF_DMP, NN, 0, GL_LUMINANCEF_DMP, GL_FLOAT, GG);

	glBindTexture(GL_LUT_TEXTURE5_DMP, texid[6]);
	glTexImage1D(GL_LUT_TEXTURE5_DMP, 0, GL_LUMINANCEF_DMP, NN, 0, GL_LUMINANCEF_DMP, GL_FLOAT, BB);

	glBindTexture(GL_LUT_TEXTURE6_DMP, texid[7]);
	glTexImage1D(GL_LUT_TEXTURE6_DMP, 0, GL_LUMINANCEF_DMP, NN, 0, GL_LUMINANCEF_DMP, GL_FLOAT, AA);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerRgbMap"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerAlphaMap"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerNoiseMap"), 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerR"), 3);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerG"), 4);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerB"), 5);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerA"), 6);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].samplerType"), GL_TEXTURE_PROCEDURAL_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].texcoord"), GL_TEXTURE0);
  	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptMinFilter"), GL_LINEAR);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexWidth"), 8);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexOffset"), 0);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptRgbMap"), GL_PROCTEX_MAX_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaMap"), GL_PROCTEX_MAX_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaSeparate"), GL_FALSE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE3, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_TEXTURE3, GL_CONSTANT, GL_CONSTANT);

	float black[4]={0.3f, 0.2f, 0.2f, 1.0f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_TexEnv[0].constRgba"), 1, black);
}

void test_clamp(GLint fclamp, GLint shiftU, GLint shiftV)
{
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampU"), fclamp);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampV"), fclamp);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftU"), shiftU);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftV"), shiftV);
}

static int frame = 0;

int drawframe(void)
{
	GLenum clamp;

	/* Setting Screen */
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	GLfloat m[16];
	mat4_t proj = mat4_t::perspective(45.0f, (float)WIDTH / (float)HEIGHT, 1.0f, 200.0f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	mat4_t mv = mat4_t::lookAt(0.0f, 0.0f, 25.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	
	glDisable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE);
	
	/* variable clamp must be GL_SYMMETRICAL_REPEAT_DMP or GL_MIRRORED_REPEAT */
	clamp = ((frame / 5) % 2) ? GL_SYMMETRICAL_REPEAT_DMP : GL_MIRRORED_REPEAT;
	
	GLenum _shift[9][2]=
	{
		{ GL_NONE_DMP,	GL_NONE_DMP },
		{ GL_NONE_DMP,	GL_ODD_DMP	},
		{ GL_NONE_DMP,	GL_EVEN_DMP },
		{ GL_ODD_DMP,	GL_NONE_DMP },
		{ GL_ODD_DMP,	GL_ODD_DMP	},
		{ GL_ODD_DMP,	GL_EVEN_DMP },
		{ GL_EVEN_DMP,	GL_NONE_DMP },
		{ GL_EVEN_DMP,	GL_ODD_DMP	},
		{ GL_EVEN_DMP,	GL_EVEN_DMP }
	};
	
	/* Drawing 9 Rectangles */
	for(int i = 0; i < 9; i++)
	{
		GLfloat x = ((GLfloat)(i % 3) - 1.0f) * 4.f;
		GLfloat y = ((GLfloat)(i / 3) - 1.0f) * 4.f;
		mat4_t mv_tr;
		mv_tr = mv * mat4_t::translate(x, y, 0.0f);
		mv_tr = mv_tr * mat4_t::rotate(6.f * frame, 0.f, 0.f, 1.f);
		mv_tr = mv_tr * mat4_t::scale(1.7f, 1.7f, 1.7f);
		mv_tr.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);
		
		test_clamp(clamp, _shift[i][0], _shift[i][1]);
		glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, 0);
	}

	glFinish();
	swap_buffer();

	frame++;
	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aTexCoord");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	load_ProceduralTexture();
	load_objects();
	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	glDeleteBuffers(1, &array_buffer_id);
	glDeleteBuffers(1, &element_array_buffer_id);
	glDeleteTextures(8, &texid[0]);
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}


}
}

