/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <nn/gx/CTR/gx_CTR.h>

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "PipelineTest"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace PipelineTest{


/* buffer id */
static GLuint array_buffer_id = 0;
static GLuint element_array_buffer_id = 0;

/* program id */
static GLuint pgid = 0;

/* shader id */
static GLuint shid = 0;

static unsigned	g_index_count = 0;

/* texture id */
#define TEXID_NUM		3
#define PROC_TEXID_NUM	7
#define LIGHT_TEXID_NUM	2
static GLuint texid[TEXID_NUM] = {0};
static GLuint texid_proc[PROC_TEXID_NUM] = {0};
static GLuint texid_foglut = 0;
static GLuint texid_lightlut[LIGHT_TEXID_NUM] = {0};

/* fog mode settings */
static GLfloat fogStart = 7.f;		/* GL_FOG_START in OpenGL ES1.1 compatible settings */
static GLfloat fogEnd = 20.f;		/* GL_FOG_END in OpenGL ES1.1 compatible settings */
static GLfloat fogDensity = 1.0f;	/* GL_FOG_DENSITY in OpenGL ES1.1 compatible settings */
#define FOG_MODE			(FOG_MODE_LINEAR)	/* Selection of fog mode */
#define FOG_MODE_LINEAR		(0)					/* GL_LINEAR mode in OpenGL ES1.1 compatible */
#define FOG_MODE_EXP		(1)					/* GL_EXP mode in OpenGL ES1.1 compatible */

/* Lut number */
#define LUT_NUM_PROC_BASE		(0)
#define LUT_NUM_FOG				(7)
#define LUT_NUM_FRAGMENT_LIGHT	(8)

/* rectangle setting */
#define RECTANGLE_COUNT	1
#define VERTEX_NUM_X	15
#define VERTEX_NUM_Y	30
#define RECT_SIZE	2.0f
#define DRAW_COUNT	60

#define SHADER_FILE_NAME	"shader.shbin"
#define TEX_LOAD_TARGET		(GL_TEXTURE_2D | NN_GX_MEM_VRAMB)

//#define CHECK_PERFORMANCE

#ifdef CHECK_PERFORMANCE
#include <nn/gxlow/CTR/gxlow_CTR.h>
#endif


static int set_1D_tex(GLfloat* RR, GLfloat* GG, GLfloat* BB, GLfloat* AA, int N)
{
	memset(RR, 0, 512 * sizeof(GLfloat));
	memset(GG, 0, 512 * sizeof(GLfloat));
	memset(BB, 0, 512 * sizeof(GLfloat));
	memset(AA, 0, 512 * sizeof(GLfloat));

	RR[7*N]=0x00/255.0f;	GG[7*N]=0x00/255.0f;	BB[7*N]=0x00/255.0f;	AA[7*N]=0xFF/255.0f;
	RR[6*N]=0x00/255.0f;	GG[6*N]=0x00/255.0f;	BB[6*N]=0xff/255.0f;	AA[6*N]=0xFF/255.0f;
	RR[5*N]=0xff/255.0f;	GG[5*N]=0x00/255.0f;	BB[5*N]=0x00/255.0f;	AA[5*N]=0xFF/255.0f;
	RR[4*N]=0xff/255.0f;	GG[4*N]=0x00/255.0f;	BB[4*N]=0xff/255.0f;	AA[4*N]=0xFF/255.0f;
	RR[3*N]=0x00/255.0f;	GG[3*N]=0xff/255.0f;	BB[3*N]=0x00/255.0f;	AA[3*N]=0xFF/255.0f;
	RR[2*N]=0x00/255.0f;	GG[2*N]=0xff/255.0f;	BB[2*N]=0xff/255.0f;	AA[2*N]=0xFF/255.0f;
	RR[1*N]=0xff/255.0f;	GG[1*N]=0xff/255.0f;	BB[1*N]=0x00/255.0f;	AA[1*N]=0xFF/255.0f;
	RR[0*N]=0xff/255.0f;	GG[0*N]=0xff/255.0f;	BB[0*N]=0xff/255.0f;	AA[0*N]=0xFF/255.0f;

	for (int x = 0; x < 255; x++)
	{
		RR[256+x] = RR[x+1] - RR[x];
		GG[256+x] = GG[x+1] - GG[x];
		BB[256+x] = BB[x+1] - BB[x];
		AA[256+x] = AA[x+1] - AA[x];
	}
	RR[255] = 0.0f;
	GG[255] = 0.0f;
	BB[255] = 0.0f;
	AA[255] = 0.0f;

	return N * 8;
}

static void LoadSin(float map_1D2D_LUT[256], GLfloat f)
{
	for (int i = 0; i < 128; i++)
		map_1D2D_LUT[i] = sin(fmod(f * DMP_PI * i/128.f, DMP_PI));

	for (int i = 0; i < 127; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];

	map_1D2D_LUT[255] = sin(fmod(f * DMP_PI, DMP_PI)) - map_1D2D_LUT[127];
}

static void LoadFlat(float map_1D2D_LUT[256], float a)
{
	for (int i = 0; i < 128; i++)
	{
		map_1D2D_LUT[i] = a;
		map_1D2D_LUT[i + 128] = 0;
	}
}

static void load_ProceduralTexture(float* F_FunctionRGB)
{
	/* Declaration */
	float F_FunctionA[256], F_Noise[256];
	LoadFlat(F_FunctionA, 1.0f);
	memset(F_Noise, 0, 256 * sizeof(float));

	/* Texture collection */
	glBindTexture(GL_LUT_TEXTURE0_DMP + LUT_NUM_PROC_BASE, texid_proc[0]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP  + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionRGB);

	glBindTexture(GL_LUT_TEXTURE1_DMP + LUT_NUM_PROC_BASE, texid_proc[1]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionA);
	
	glBindTexture(GL_LUT_TEXTURE2_DMP + LUT_NUM_PROC_BASE, texid_proc[2]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_Noise);

	const int NN = 512;
	GLfloat RR[NN], GG[NN], BB[NN], AA[NN];
	int texSize = set_1D_tex(RR, GG, BB, AA, 1);

	glBindTexture(GL_LUT_TEXTURE3_DMP + LUT_NUM_PROC_BASE, texid_proc[3]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, RR);

	glBindTexture(GL_LUT_TEXTURE4_DMP + LUT_NUM_PROC_BASE, texid_proc[4]);
	glTexImage1D(GL_LUT_TEXTURE4_DMP + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, GG);

	glBindTexture(GL_LUT_TEXTURE5_DMP + LUT_NUM_PROC_BASE, texid_proc[5]);
	glTexImage1D(GL_LUT_TEXTURE5_DMP + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, BB);

	glBindTexture(GL_LUT_TEXTURE6_DMP + LUT_NUM_PROC_BASE, texid_proc[6]);
	glTexImage1D(GL_LUT_TEXTURE6_DMP + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, AA);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerRgbMap"), LUT_NUM_PROC_BASE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerAlphaMap"), LUT_NUM_PROC_BASE+1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerNoiseMap"), LUT_NUM_PROC_BASE+2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerR"), LUT_NUM_PROC_BASE+3);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerG"), LUT_NUM_PROC_BASE+4);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerB"), LUT_NUM_PROC_BASE+5);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptSamplerA"), LUT_NUM_PROC_BASE+6);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].samplerType"), GL_TEXTURE_PROCEDURAL_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].texcoord"), GL_TEXTURE2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptMinFilter"), GL_LINEAR);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexWidth"), texSize);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptTexOffset"), 0);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptRgbMap"), GL_PROCTEX_ADDSQRT2_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaMap"), GL_PROCTEX_ADDSQRT2_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptAlphaSeparate"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampU"), GL_SYMMETRICAL_REPEAT_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptClampV"), GL_SYMMETRICAL_REPEAT_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftU"), GL_NONE_DMP);
    glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[3].ptShiftV"), GL_NONE_DMP);
}

static unsigned make_rectangle(GLfloat *coords, GLfloat *normals, GLfloat *texcoords, GLfloat *proc_texcoords,GLushort *indexs,
						   GLfloat coord_x_base, GLfloat coord_y_base, GLfloat coord_z_base, GLushort index_base,  unsigned x_num, unsigned y_num
						   )
{
	unsigned c_offset = 0;
	unsigned t_offset = 0;
	unsigned n_offset = 0;
	int i_offset = 0;
	GLfloat coord_step_x = RECT_SIZE / x_num;
	GLfloat coord_step_y = RECT_SIZE / y_num;
	GLfloat tex_step_x = 2.f / x_num;
	GLfloat tex_step_y = 2.f / y_num;
	GLfloat coord_x = coord_x_base;
	GLfloat coord_y = coord_y_base;
	GLfloat tex_x = -1.f;
	GLfloat tex_y = -1.f;
	unsigned i , j;

	for (i = 0; i <= y_num; i++) {
		coord_x = coord_x_base;
		tex_x = -1.f;
		for (j = 0; j <= x_num; j++) {
			coords[c_offset++] = coord_x;
			coords[c_offset++] = coord_y;
			coords[c_offset++] = coord_z_base;
			coords[c_offset++] = 1.0f;

			proc_texcoords[t_offset] = tex_x;
			texcoords[t_offset++] = (tex_x +1.f)/2;
			proc_texcoords[t_offset] = tex_y;
			texcoords[t_offset++] = (tex_y +1.f)/2;

			normals[n_offset++] = 0.f;
			normals[n_offset++] = 0.f;
			normals[n_offset++] = 1.f;

			coord_x += coord_step_x;
			tex_x += tex_step_x;
		}
		coord_y += coord_step_y;
		tex_y += tex_step_y;
	}
	for (i = 0; i < y_num; i++) {
		unsigned base = i * (x_num + 1);
		for (j = 0 ; j <= x_num; j++) {
			indexs[i_offset++] = base + (x_num + 1) + j + index_base;
			indexs[i_offset++] = base + j + index_base;
		}
	}

	return i_offset;
}

static void load_rectangle_objects(void)
{
#define COORDS_COUNT		(4 * (VERTEX_NUM_X + 1)* (VERTEX_NUM_Y + 1))
#define NORMALS_COUNT		(3 * (VERTEX_NUM_X + 1)* (VERTEX_NUM_Y + 1))
#define TEXCOORDS_COUNT		(2 * (VERTEX_NUM_X + 1)* (VERTEX_NUM_Y + 1))
#define INDEXS_COUNT		((VERTEX_NUM_X+1) * 2 * (VERTEX_NUM_Y))
#define SIZEOF_COORDS()		(sizeof(GLfloat) * COORDS_COUNT * RECTANGLE_COUNT)
#define SIZEOF_NORMALS()	(sizeof(GLfloat) * NORMALS_COUNT * RECTANGLE_COUNT)
#define SIZEOF_TEXCOORDS()	(sizeof(GLfloat) * TEXCOORDS_COUNT * RECTANGLE_COUNT)
#define SIZEOF_INDEXS()		(sizeof(GLushort) * INDEXS_COUNT * RECTANGLE_COUNT)

	GLfloat *coords;
	GLfloat *normals;
	GLfloat *texcoords;
	GLfloat *proc_texcoords;
	GLushort *indexs;

	coords = (GLfloat*)malloc(SIZEOF_COORDS());
	normals = (GLfloat*)malloc(SIZEOF_NORMALS());
	texcoords = (GLfloat*)malloc(SIZEOF_TEXCOORDS());
	proc_texcoords = (GLfloat*)malloc(SIZEOF_TEXCOORDS());
	indexs = (GLushort*)malloc(SIZEOF_INDEXS());

	GLfloat base_x = -1.0f;
	GLfloat base_y = -1.0f;
	GLfloat base_z = -0.f;
	for (int i = 0; i < RECTANGLE_COUNT; i++) {
		g_index_count += make_rectangle(
			coords + COORDS_COUNT * i,
			normals + NORMALS_COUNT * i, 
			texcoords + TEXCOORDS_COUNT * i, 
			proc_texcoords + TEXCOORDS_COUNT * i,
			indexs + g_index_count,
			base_x , base_y, base_z , 
			i * ((VERTEX_NUM_X + 1) * (VERTEX_NUM_Y + 1) ), 
			VERTEX_NUM_X, VERTEX_NUM_Y);
	}

	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBufferData(GL_ARRAY_BUFFER, SIZEOF_COORDS() + SIZEOF_NORMALS() + SIZEOF_TEXCOORDS() * 2, 0, GL_STATIC_DRAW);
	glBufferSubData(GL_ARRAY_BUFFER, 0, SIZEOF_COORDS(), coords);
	glBufferSubData(GL_ARRAY_BUFFER, SIZEOF_COORDS(), SIZEOF_NORMALS(), normals);
	glBufferSubData(GL_ARRAY_BUFFER, SIZEOF_COORDS() + SIZEOF_NORMALS(), SIZEOF_TEXCOORDS(), texcoords);
	glBufferSubData(GL_ARRAY_BUFFER, SIZEOF_COORDS() + SIZEOF_NORMALS() + SIZEOF_TEXCOORDS(), SIZEOF_TEXCOORDS(), proc_texcoords);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, SIZEOF_INDEXS(), indexs, GL_STATIC_DRAW);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);
	glEnableVertexAttribArray(2);
	glEnableVertexAttribArray(3);

	glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0) ;
	glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(SIZEOF_COORDS()));
	glVertexAttribPointer(2, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid*)( SIZEOF_COORDS() + SIZEOF_NORMALS()));
	glVertexAttribPointer(3, 2, GL_FLOAT, GL_FALSE, 0, (GLvoid*)( SIZEOF_COORDS() + SIZEOF_NORMALS() + SIZEOF_TEXCOORDS()));
	glVertexAttrib4f(4, 1.f, 1.f, 1.f, 0.15f);

	free(coords);
	free(normals);
	free(texcoords);
	free(proc_texcoords);
	free(indexs);

}

static void load_textures(void)
{
	bool bUseAlpha ;

	/* Texture 0 */
	glActiveTexture(GL_TEXTURE0);
	glGenTextures( TEXID_NUM, &texid[0] );
	glBindTexture( GL_TEXTURE_2D, texid[0] );

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );

	loadTexture(FILE_APP_ROOT "dmp0128.tga", TEX_LOAD_TARGET, 0, bUseAlpha);
	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[0].samplerType"), GL_TEXTURE_2D );
	
	/* Texture 1 */
	glActiveTexture(GL_TEXTURE1);
	glBindTexture( GL_TEXTURE_2D, texid[0] );

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );

	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[1].samplerType"), GL_TEXTURE_2D );

	/* Texture 2 */
	glActiveTexture(GL_TEXTURE2);
	glBindTexture( GL_TEXTURE_2D, texid[0] );

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );

	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[2].samplerType"), GL_TEXTURE_2D );
	glUniform1i( glGetUniformLocation( pgid, "dmp_Texture[2].texcoord"), GL_TEXTURE1 );

	/* load procedural texture */
	GLfloat F_FunctionRGB[256];
	GLfloat t = 0 / 100.f;
	glGenTextures( PROC_TEXID_NUM, &texid_proc[0] );
	LoadSin(F_FunctionRGB, 3.f + sin(DMP_PI * t) * 1.5f);
	load_ProceduralTexture( F_FunctionRGB);

	/* Texture Combiner0 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE0, GL_TEXTURE3, GL_FRAGMENT_SECONDARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[0].combineRgb"), GL_MULT_ADD_DMP );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE );
	/* Texture Combiner1 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcRgb"), GL_PREVIOUS, GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_FRAGMENT_SECONDARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[1].combineRgb"), GL_MULT_ADD_DMP );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[1].combineAlpha"), GL_REPLACE );
	/* Texture Combiner2 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].srcRgb"), GL_PREVIOUS, GL_TEXTURE1, GL_FRAGMENT_SECONDARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[2].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[2].combineRgb"), GL_MULT_ADD_DMP );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[2].combineAlpha"), GL_REPLACE );
	/* Texture Combiner3 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[3].srcRgb"), GL_PREVIOUS, GL_TEXTURE3, GL_FRAGMENT_SECONDARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[3].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[3].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[3].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[3].combineRgb"), GL_MULT_ADD_DMP  );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[3].combineAlpha"), GL_REPLACE  );
	/* Texture Combiner4 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[4].srcRgb"), GL_PREVIOUS, GL_TEXTURE2, GL_FRAGMENT_SECONDARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[4].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[4].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[4].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[4].combineRgb"), GL_MULT_ADD_DMP  );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[4].combineAlpha"), GL_REPLACE  );
	/* Texture Combiner5 */
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[5].srcRgb"), GL_PREVIOUS, GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_FRAGMENT_SECONDARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[5].srcAlpha"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[5].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[5].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[5].combineRgb"), GL_MULT_ADD_DMP  );
	glUniform1i( glGetUniformLocation( pgid, "dmp_TexEnv[5].combineAlpha"), GL_REPLACE  );
}

static unsigned frame_num = 0;

static int drawframe(void)
{	
	GLfloat lpos0[] = {0.f,0.f,1.f,1.f};
	
	/* proc texture */
	GLfloat F_FunctionRGB[256];
	GLfloat t = frame_num / 100.f;

	glClear( GL_COLOR_BUFFER_BIT );
	LoadSin(F_FunctionRGB, 3.f + sin(DMP_PI * t) * 1.5f);
	glTexImage1D(GL_LUT_TEXTURE0_DMP + LUT_NUM_PROC_BASE, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionRGB);

	for (int i = 0; i < DRAW_COUNT; i++) {
		glDrawElements(GL_TRIANGLE_STRIP, g_index_count, GL_UNSIGNED_SHORT, 0);
	}

	glFinish();

	lpos0[0] = 0.1f * (frame_num % 200);
	vec4_t lpos = vec4_t(lpos0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].position"), 1, &lpos.x);

#ifdef CHECK_PERFORMANCE
	nngxlowPerfLog perfLog;
	if (!(frame_num % 100)) {
		nngxSplitDrawCmdlist();
		nngxRunCmdlist();
		nngxWaitCmdlistDone();
		nngxStopCmdlist();
		nngxlowGetPerfLog( &perfLog );
	}
#endif

	swap_buffer();


#ifdef CHECK_PERFORMANCE
	static s64 currentTick = 0;

	if (!(frame_num % 100)) {
		s64 startTick_ = nn::os::Tick::GetSystemCurrent();
		int pastmillisec = (int)nnosTickConvertToMilliSeconds(startTick_ - currentTick);
		currentTick = nn::os::Tick::GetSystemCurrent();
		NN_LOG("frame:%d msec\n", pastmillisec / 100);
		NN_LOG("3d:%d\n", perfLog.p3d);
	}
#endif

	frame_num++;

	return !glGetError();
}

static void set_lighting_state(void)
{
	GLfloat la[] = {1.0f, 0.8f, 0.8f, 1.f};
	GLfloat ls0[] = {0.8f, 0.8f, 0.8f, 1.f};
	GLfloat ls20[] = {0.2f, 0.2f, 0.2f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular0"), 1, ls0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].ambient"), 1, la);

	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular1"), 1, ls20);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_FALSE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD1"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG2_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);

	/* bind lookup table */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD1"), LUT_NUM_FRAGMENT_LIGHT);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), LUT_NUM_FRAGMENT_LIGHT+1);

	/* setup lookup table */
	glGenTextures(LIGHT_TEXID_NUM, texid_lightlut);

	GLfloat lut[512];
	int j;
	memset(lut, 0, sizeof(lut));

	/* set layer1 distribution function */
	for (j = 0; j < 256; j++) {
		lut[j] = gaussian((float)j/255.9375f, 0.149f);
	}
	for (j = 0; j < 255; j++) {
		lut[j + 256] = lut[j + 1] - lut[j];
	}
	lut[255 + 256] = 1.f - lut[255];

	glBindTexture(GL_LUT_TEXTURE0_DMP+LUT_NUM_FRAGMENT_LIGHT, texid_lightlut[0]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP+LUT_NUM_FRAGMENT_LIGHT, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* set layer2 distribution function */
	memset(lut, 0, sizeof(lut));

	for (j = 0; j < 256; j++) {
		lut[j] = gaussian((float)j/255.9375f, 1.f);
	}
	for (j = 0; j < 255; j++) {
		lut[j + 256] = lut[j + 1] - lut[j];
	}
	lut[255 + 256] = 1.f - lut[255];

	glBindTexture(GL_LUT_TEXTURE1_DMP+LUT_NUM_FRAGMENT_LIGHT, texid_lightlut[1]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP+LUT_NUM_FRAGMENT_LIGHT, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

}

/* This function returns fog coefficent for input value c */
static float fog_coef(float fog_c)
{
	/* GL_LINEAR compatible */
	if (FOG_MODE == FOG_MODE_LINEAR)
	{
		if (fog_c < fogStart)
			return 1.0f;
		else if (fog_c > fogEnd)
			return 0.0f;
		else
			return (fogEnd - fog_c) / (fogEnd - fogStart);
	}
	/* GL_EXP compatible */
	else if (FOG_MODE == FOG_MODE_EXP)
	{
		return exp(-fogDensity * fog_c);
	}
	/* error */
	else
		return -1.0;
}

static void initialize_fog_table(mat4_t *MP)
{
	static float Fog_LUT[256];		/* Fog lut contents */
	static float Fog_c[128 + 1];	/* c (distance from origin in eye coordinate */

	int i;
	mat4_t invPM;
	vec4_t v_clip(0.0f, 0.0f, 0.0f, 1.0f);
	vec4_t v_eye;
	
	/* Convert Fog table input value to OpenGL compatible        */
	/* fog coordinate (distance from origin in eye coordinate    */
	/* Here, assumed that projection matrix is OpenGL compatible */
	invPM = MP->invert();
	for (i = 0; i <= 128; i++)
	{
		v_clip.z = ((float)(i * 2 - 128)) / 128;
		v_eye = invPM * v_clip;
		Fog_c[i] = -(v_eye.z / v_eye.w);
	}

	for (i = 0; i < 128; i++)
	{
		Fog_LUT[i] = fog_coef(Fog_c[i]);
		Fog_LUT[128 + i]= fog_coef(Fog_c[i + 1]) - fog_coef(Fog_c[i]);
	}
	
	glGenTextures(1, &texid_foglut);
	glBindTexture(GL_LUT_TEXTURE0_DMP + LUT_NUM_FOG, texid_foglut);
	glTexImage1D(GL_LUT_TEXTURE0_DMP + LUT_NUM_FOG, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, &Fog_LUT);
}

/* initialization */
static int initialize(void)
{
	frame_num = 0;
	g_index_count = 0;

	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT SHADER_FILE_NAME, &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation( pgid, 0, "aPosition" );
	glBindAttribLocation( pgid, 1, "aNormal" );
	glBindAttribLocation( pgid, 2, "aTexCoord" );
	glBindAttribLocation( pgid, 3, "aTexCoord_proc" );
	glBindAttribLocation( pgid, 4, "aColor" );

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	GLfloat m[16] ;
	mat4_t proj = mat4_t::ortho(-1.f, 1.f, -1.f, 1.f, -1.f, 1.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	glViewport( 0, 0, WIDTH, HEIGHT );

	load_rectangle_objects();
	load_textures();

	glEnable( GL_CULL_FACE );
	glFrontFace( GL_CCW );

	glClearColor( 0.8f, 0.8f, 1.f, 1.f );
	glClearDepthf( 1.f );
	glClearStencil( 1 );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

	/* depth test */
//    glEnable(GL_DEPTH_TEST);
//    glDepthFunc(GL_ALWAYS);

	/* enable stencil test */
//	glEnable( GL_STENCIL_TEST );
//	glStencilOp( GL_REPLACE, GL_REPLACE, GL_REPLACE );
//	glStencilFunc( GL_ALWAYS, 0x00, 0xff );


	/* material */
	GLfloat ma[4] = {1.0f, 0.0f, 0.0f, 1.f};
	GLfloat md[4] = {0.0f, 1.0f, 0.0f, 1.f};
	GLfloat ms[4] = {0.0f, 0.0f, 1.0f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.diffuse"), 1, md);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1, ms);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.ambient"), 1, ma); 
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular1"), 1, ms);

	/* Fog */
	glUniform1i(glGetUniformLocation( pgid, "dmp_Fog.mode"), GL_FOG);
	GLfloat fog_color[3] = {0.0f, 0.0f, 1.0f};
	glUniform3fv(glGetUniformLocation( pgid, "dmp_Fog.color"), 1, fog_color);
	glUniform1i(glGetUniformLocation( pgid, "dmp_Fog.sampler"), LUT_NUM_FOG);
	initialize_fog_table(&proj);

	/* Fragment light */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);
	set_lighting_state();
	
	/* enable alpha test */
	glUniform1i( glGetUniformLocation( pgid, "dmp_FragOperation.enableAlphaTest" ), GL_TRUE );
	glUniform1i( glGetUniformLocation( pgid, "dmp_FragOperation.alphaTestFunc" ), GL_ALWAYS );

	/* modelview */
	mat4_t mv;
	mv.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	glEnable( GL_BLEND );
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
#ifdef CHECK_PERFORMANCE
	nngxlowSetPerfLogMode( true );
#endif

	return 0;
}

/* finalization */
static void finalize()
{
	if (texid[0]) {
		glDeleteTextures( TEXID_NUM, texid );
		memset(texid, 0, sizeof(GLuint) * TEXID_NUM);
	}
	if (texid_proc[0]) {
		glDeleteTextures( PROC_TEXID_NUM, texid_proc );
		memset(texid_proc, 0, sizeof(GLuint) * PROC_TEXID_NUM);
	}
	if (texid_foglut) {
		glDeleteTextures( 1, &texid_foglut );
		texid_foglut = 0;
	}
	if (texid_lightlut[0]) {
		glDeleteTextures( 2, texid_lightlut );
		memset(texid_lightlut, 0, sizeof(GLuint) * LIGHT_TEXID_NUM);
	}
	if (array_buffer_id) {
		glDeleteBuffers(1, &array_buffer_id);
		array_buffer_id = 0;
	}
	if (element_array_buffer_id) {
		glDeleteBuffers(1, &element_array_buffer_id);
		element_array_buffer_id = 0;
	}
	if (shid) {
		glDeleteShader(shid);
		shid = 0;
	}
	if (pgid) {
		glDeleteProgram(pgid);
		pgid = 0;
	}

}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	/* finalization */
	finalize();
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}
}
}
