/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

/*
 * Comments
 * --------
 *
 * 
 * DMP Particle system sample. 
 * This sample show various pattern that can be obtained
 * with Particle system
 *  
 */


#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>

#include "Memory.h"

#define APP_NAME "PartsysSimple"
#define REAL_WIDTH 240
#define REAL_HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace PartsysSimple{

/* buffer id */
GLuint array_buffer_id;
GLuint element_array_buffer_id;

/* program id */
GLuint pgid[1];
GLuint shaders[2] ;

/* Procedural texture */
GLuint	procTexid[8];
int		procTexSize;

/* 2D texture */
GLuint	texture2D_name;

/* Structure containing particle system settings*/
struct _particleSys
{
	mat4_t		m_partsys_center;
	mat4_t		m_partsys_color;
	mat4_t		m_partsys_radius;
	mat4_t		m_partsys_aspect;

	float		m_speed;
	float		m_particleCountMax;
	float		m_size_min_max[2];
	float		m_prng[4];						/* constant for the Pseudo Random Number Generator (modulo based) */
	float		m_random_seed[4];				/* seeds for the PRNG */

	float		simulationTime;					/* application side. Holds the current time for particle system */
	float		dTime;							/* application side. Holds the time interval between each frame */
	int			NFrame;							/* application side. Number of frames for this particle system */
};
}
}

#include "PartsysPattern.h"

namespace gputest{
namespace PartsysSimple{

struct _particleSys pSys[PARTSYS_LAST];
int CurSys = 2;


/* Load a 2D texture (DMP Letters)*/
void loadTexture2D(void)
{
	int _lev = 0;
	glGenTextures(1, &texture2D_name);
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, texture2D_name);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_LOD, 0);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	bool bUseAlpha;
	loadTexture(FILE_APP_ROOT "dmp.tga", GL_TEXTURE_2D, _lev, bUseAlpha);
}

/* Procedural texture: RGBA texture definition
 a simple white gradient defined as 2 colors */
int set_1D_tex(GLfloat* RR, GLfloat* GG, GLfloat* BB, GLfloat* AA)
{
	memset(RR, 0, 512 * sizeof(GLfloat));
	memset(GG, 0, 512 * sizeof(GLfloat));
	memset(BB, 0, 512 * sizeof(GLfloat));
	memset(AA, 0, 512 * sizeof(GLfloat));

	RR[0] = 0xFF / 255.0f;	GG[0] = 0xFF / 255.0f;	BB[0] = 0xFF / 255.0f;	AA[0] = 0xFF / 255.0f;
	RR[1] = 0x00 / 255.0f;	GG[1] = 0x00 / 255.0f;	BB[1] = 0x00 / 255.0f;	AA[1] = 0x00 / 255.0f;

	for (int x = 0; x < 255; x++)
	{
		RR[256 + x] = RR[x + 1] - RR[x];
		GG[256 + x] = GG[x + 1] - GG[x];
		BB[256 + x] = BB[x + 1] - BB[x];
		AA[256 + x] = AA[x + 1] - AA[x];
	}
	RR[255] = 0.0f;
	GG[255] = 0.0f;
	BB[255] = 0.0f;
	AA[255] = 0.0f;
	
	return 2;
}

/* Procedural texture: mapping function
 bias function in [0.0.5]. zero otherwise */
void LoadBias(float *map_1D2D_LUT, float p)
{
	float x = 0.0f;
	float dx = 2.0f / 127.0f;
	float aa = 1.0f / p;
	float t;
	for (unsigned int i = 0; i < 128; i++)
	{
		t = x;
		if (t >= 1.0f)
			map_1D2D_LUT[i] = 1.0f;
		else
			map_1D2D_LUT[i] = t / ((aa - 2.0f) * (1.0f - t) + 1.0f);
		x += dx;
	}
	for (unsigned int i = 0; i < 127; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];
	map_1D2D_LUT[255] = 0.0f;
}

/* Procedural texture: mapping function
 potential function in [0.0.5]. zero otherwise */
void LoadPotential(float *map_1D2D_LUT, float p)
{
	float x = 0.0f;
	float dx = 2.0f / 127.0f;
	float f = 0, d2;
	
	for (unsigned int i = 0; i < 128; i++)
	{
		f = 0;
		d2 = x;
		if (d2 < 1.0f)
		{
			if (d2 < 0.25f)
				f = 1.0f - ((3.0f * d2) * (3.0f * d2)) / (p + (4.5f - 4.0f * p) * d2);
			else 
				f = ((1.0f - d2) * (1.0f - d2)) / (0.75f - p + (1.5f + 4.0f * p) * d2);
		}
		
		map_1D2D_LUT[i] = f;
		x += dx;
	}
	
	for (unsigned int i = 0; i < 126; i++)
		map_1D2D_LUT[i + 128] = map_1D2D_LUT[i + 1] - map_1D2D_LUT[i];
	map_1D2D_LUT[255] = 0.0f;
}

void loadProceduralTexture(GLuint pgID)
{
	(void)pgID;
	glGenTextures(8, &procTexid[0]);
	
	/* Binding of the procedural texture as a texture collection */
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, procTexid[0]);
	
	/* Declaration */
	float F_FunctionRGB[256], F_FunctionZero[256];
	LoadBias(F_FunctionRGB, 0.8f);
	memset(F_FunctionZero, 0, 256 * sizeof(float));

	/* Texture collection */
	glBindTexture(GL_LUT_TEXTURE0_DMP, procTexid[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionRGB);

	glBindTexture(GL_LUT_TEXTURE1_DMP, procTexid[2]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionZero);
	
	glBindTexture(GL_LUT_TEXTURE2_DMP, procTexid[3]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, F_FunctionZero);

	const int NN = 512;
	GLfloat RR[NN], GG[NN], BB[NN], AA[NN];
	procTexSize = set_1D_tex(RR,GG,BB,AA);

	glBindTexture(GL_LUT_TEXTURE3_DMP, procTexid[4]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, RR);

	glBindTexture(GL_LUT_TEXTURE4_DMP, procTexid[5]);
	glTexImage1D(GL_LUT_TEXTURE4_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, GG);

	glBindTexture(GL_LUT_TEXTURE5_DMP, procTexid[6]);
	glTexImage1D(GL_LUT_TEXTURE5_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, BB);

	glBindTexture(GL_LUT_TEXTURE6_DMP, procTexid[7]);
	glTexImage1D(GL_LUT_TEXTURE6_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, AA);
}

void validateParticleSystem(GLuint pgID, struct _particleSys *pp, unsigned int _width, unsigned int _height)
{
	GLfloat mPos[16];
	pp->m_partsys_center.toFloatArr(mPos);
	glUniformMatrix4fv(glGetUniformLocation(pgID, "uCenter"), 1, GL_FALSE, mPos);

	GLfloat mSize[16];
	pp->m_partsys_radius.toFloatArr(mSize);
	glUniformMatrix4fv(glGetUniformLocation(pgID, "uRadii"), 1, GL_FALSE, mSize);

	GLfloat mCol[16];
	pp->m_partsys_color.toFloatArr(mCol);
	glUniformMatrix4fv(glGetUniformLocation(pgID, "dmp_PartSys.color"), 1, GL_FALSE, mCol);

	GLfloat mAspect[16];
	pp->m_partsys_aspect.toFloatArr(mAspect);
	glUniformMatrix4fv(glGetUniformLocation(pgID, "dmp_PartSys.aspect"), 1, GL_FALSE, mAspect);

	GLfloat viewport_size[2] = {0.0f, 0.0f};
	viewport_size[0] = 1.f / (GLfloat)_width;
	viewport_size[1] = 1.f / (GLfloat)_height;
	glUniform2fv(glGetUniformLocation(pgID, "dmp_PartSys.viewport"), 1, &viewport_size[0]);
	
	GLfloat distance_attenuation[3] = {0.0f, 0.0f, 0.0f};
	distance_attenuation[0] = 0.0f;
	distance_attenuation[1] = 0.0f;
	distance_attenuation[2] = 1.0f / (float)(_width * _height);
	glUniform3fv(glGetUniformLocation(pgID, "dmp_PartSys.distanceAttenuation"), 1, &distance_attenuation[0]);
	
	glUniform1fv(glGetUniformLocation(pgID, "dmp_PartSys.countMax"), 1, &pp->m_particleCountMax);
	glUniform2fv(glGetUniformLocation(pgID, "dmp_PartSys.pointSize"), 1, &pp->m_size_min_max[0]);
	glUniform1fv(glGetUniformLocation(pgID, "dmp_PartSys.speed"), 1, &pp->m_speed);

	glUniform4fv(glGetUniformLocation(pgID, "dmp_PartSys.randomCore"), 1, &pp->m_prng[0]);
	glUniform4fv(glGetUniformLocation(pgID, "dmp_PartSys.randSeed"), 1, &pp->m_random_seed[0]);
}

void validateProceduralTexture(GLuint pgID)
{
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptSamplerRgbMap"), 0);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptSamplerAlphaMap"), 1);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptSamplerNoiseMap"), 2);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptSamplerR"), 3);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptSamplerG"), 4);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptSamplerB"), 5);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptSamplerA"), 6);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptMinFilter"), GL_LINEAR);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptTexWidth"), procTexSize);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptTexOffset"), 0);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptAlphaSeparate"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptClampU"), GL_CLAMP_TO_EDGE);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptClampV"), GL_CLAMP_TO_EDGE);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptShiftU"), GL_NONE_DMP);
    glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptShiftV"), GL_NONE_DMP);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptRgbMap"), GL_PROCTEX_MAX_DMP);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].ptAlphaMap"), GL_PROCTEX_MAX_DMP);
}

void validateTextureBlender_ProcTex(GLuint pgID)
{
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, procTexid[0]);
	glUniform1i ( glGetUniformLocation(pgID, "dmp_Texture[3].samplerType"), GL_TEXTURE_PROCEDURAL_DMP);
	glUniform1i ( glGetUniformLocation(pgID, "dmp_Texture[3].texcoord"), GL_TEXTURE2);
	glUniform1i ( glGetUniformLocation(pgID, "dmp_Texture[0].samplerType"), GL_FALSE);
	glUniform1i ( glGetUniformLocation(pgID, "dmp_TexEnv[0].combineRgb"), GL_MODULATE);
	glUniform1i ( glGetUniformLocation(pgID, "dmp_TexEnv[0].combineAlpha"), GL_MODULATE);
	glUniform3i ( glGetUniformLocation(pgID, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i ( glGetUniformLocation(pgID, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i ( glGetUniformLocation(pgID, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE3, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i ( glGetUniformLocation(pgID, "dmp_TexEnv[0].srcAlpha"), GL_TEXTURE3, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
}

void validateTextureBlender_TX0(GLuint pgID)
{
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, texture2D_name);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[3].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgID, "dmp_Texture[0].samplerType"), GL_TEXTURE_2D);
	glUniform1i(glGetUniformLocation(pgID, "dmp_TexEnv[0].combineRgb"), GL_MODULATE);
	glUniform1i(glGetUniformLocation(pgID, "dmp_TexEnv[0].combineAlpha"), GL_MODULATE);
	glUniform3i(glGetUniformLocation(pgID, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgID, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgID, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE0, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgID, "dmp_TexEnv[0].srcAlpha"), GL_TEXTURE0, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
}

void drawParticleSystem(GLuint pgID, struct _particleSys *pp, float _Psys_time)
{
	(void)pp;
	glUniform1fv(glGetUniformLocation(pgID, "dmp_PartSys.time"), 1, &_Psys_time);
	glDrawElements(GL_GEOMETRY_PRIMITIVE_DMP, 4, GL_UNSIGNED_SHORT, 0);
}

static int frame = 0;
int drawframe(void)
{
	static bool switchTexture = false;
	static bool useProctex = false;
	static float t = 0.f;
	GLfloat m[16];
	mat4_t proj = mat4_t::perspective(45.0f, (float)REAL_WIDTH / (float)REAL_HEIGHT, 1.0f, 200.0f);
	mat4_t mv = mat4_t::lookAt(0.0f, 0.0f, 50.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);

	if (frame == 0)
	{
		/* reset uniforms */
		validateParticleSystem(pgid[0], &pSys[CurSys], REAL_WIDTH, REAL_HEIGHT);

		proj.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pgid[0], "uProjection"), 1, GL_FALSE, m);
		
		mv.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pgid[0], "uModelView"), 1, GL_FALSE, m);
		t = 0.f;
	}
	
	glClearColor(0.0f+(frame%100)*0.004f, 0.0f, 0.0f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	if (CurSys == PARTSYS_EXPLOSION2)
	{
		/* As an illustration, we draw 2 times the particle system. */
		mv.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pgid[0], "uModelView"), 1, GL_FALSE, m);
		
		drawParticleSystem(pgid[0], &pSys[CurSys], t);
		
		mat4_t mv_rot = mv * mat4_t::rotate(45.0f, 0.0f, 0.0f, 1.0f);
		mv_rot.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pgid[0], "uModelView"), 1, GL_FALSE, m);
		drawParticleSystem(pgid[0], &pSys[CurSys], t);
	}
	else
		drawParticleSystem(pgid[0], &pSys[CurSys], t);
	
	swap_buffer();
	
	if (++frame == pSys[CurSys].NFrame)
	{
		/* Changing 2D texture / procedural texture */
		if (switchTexture)
		{
			if (!useProctex)
			{
				validateTextureBlender_TX0(pgid[0]);
				useProctex = true;
			}
			else
			{
				validateProceduralTexture(pgid[0]);
				validateTextureBlender_ProcTex(pgid[0]);
				useProctex = false;
			}

			switchTexture = false;
		}
		/* change to next particle system */
		CurSys = ++CurSys % PARTSYS_LAST;
		frame = 0;
		t = 0.f;

		/* Changing blending mode for the spinming pattern */
		glBlendFunc(GL_SRC_ALPHA, (CurSys == PARTSYS_SPIN) ? GL_ONE_MINUS_SRC_ALPHA : GL_ONE);
		switchTexture = (CurSys == PARTSYS_LAST - 1);
	}
	else
		t += pSys[CurSys].dTime;
	
	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(REAL_WIDTH, REAL_HEIGHT, APP_NAME, drawframe);

	glViewport(0, 0, REAL_WIDTH, REAL_HEIGHT);

	glClearDepthf(1.f);

	glDisable(GL_DEPTH_TEST);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE);
	glDisable(GL_CULL_FACE);

    /* create program and load & attach vertex shader */
	shaders[0] = glCreateShader(GL_VERTEX_SHADER);
	shaders[1] = glCreateShader(GL_GEOMETRY_SHADER_DMP);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(2, shaders, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	pgid[0] = glCreateProgram();
	glAttachShader(pgid[0], shaders[0]);
	glAttachShader(pgid[0], shaders[1]);
	glAttachShader(pgid[0], GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid[0], 0, "attrCtrPointIndex");

	glLinkProgram(pgid[0]);
	glValidateProgram(pgid[0]);
	glUseProgram(pgid[0]);
	
	loadProceduralTexture(pgid[0]);
	loadTexture2D();
	loadParticlePattern(&pSys[PARTSYS_DEFAULT], PARTSYS_DEFAULT);
	loadParticlePattern(&pSys[PARTSYS_EXPLOSION1], PARTSYS_EXPLOSION1);
	loadParticlePattern(&pSys[PARTSYS_EXPLOSION2], PARTSYS_EXPLOSION2);
	loadParticlePattern(&pSys[PARTSYS_SNOW], PARTSYS_SNOW);
	loadParticlePattern(&pSys[PARTSYS_SPIN], PARTSYS_SPIN);
	
	/* make vertex buffer */
	GLfloat vertex[4] = {0.0f, 1.0f, 2.0f, 3.0f};
	GLushort index[4] = {0, 1, 2, 3};
	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBufferData(GL_ARRAY_BUFFER, 4 * sizeof(GLfloat), &vertex[0], GL_STATIC_DRAW);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0, 1, GL_FLOAT, GL_FALSE, 0, 0);
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, 4 * sizeof(GLushort), &index[0], GL_STATIC_DRAW);

	/* set texture configuration */
	//validateTextureBlender_TX0(pgid[0]);
	validateProceduralTexture(pgid[0]);
	validateTextureBlender_ProcTex(pgid[0]);
	
	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	glDeleteTextures(1, &texture2D_name);
	glDeleteTextures(8, &procTexid[0]);
	glDeleteBuffers(1, &array_buffer_id);
	glDeleteBuffers(1, &element_array_buffer_id);
	glDeleteProgram(pgid[0]);
	glDeleteShader(shaders[0]);
	glDeleteShader(shaders[1]);

	/* shutdown_display */
	shutdown_display();

	return 0;
}


}
}

