/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "Gas.h"
#include "File.h"

#include <string.h>

#include "Memory.h"

using namespace gputest::common;

namespace gputest{
namespace PartsysGas{

/* program id */
GLuint pGeoId;		/* for normal geometry */
GLuint pAccId;		/* for gas accumulation pass */
GLuint pPostId;		/* for gas shading/post synthesis pass */

GLuint shaders[4] ;

/* Standard object (triangles, not gaseous object) */
dat_t plane;
struct gas_data gas;

/*=======================================================*/
/* Sample specific. Different color variation for gas	 */
/*=======================================================*/
#define NUM_COLOR_SAMPLE  6
static float gasColorTable[NUM_COLOR_SAMPLE][24]=
{
	{	/* red flame */
		0.0f,	0.0f,	0.0f,
		0.2f,	0.15f,	0.05f,
		0.6f,	0.25f,	0.15f,
		0.9f,	0.35f,	0.2f,
		0.92f,	0.6f,	0.15f,
		0.95f,	0.85f,	0.05f,
		1.0f,	0.95f,	0.0f,
		1.0f,	1.0f,	1.0f
	},
	{	/* blue flame */
		0.00f,	0.00f,	0.00f,
		0.05f,	0.15f,	0.20f,
		0.15f,	0.25f,	0.60f,
		0.20f,	0.35f,	0.90f,
		0.15f,	0.60f,	0.92f,
		0.05f,	0.85f,	0.95f,
		0.00f,	0.95f,	1.00f,
		1.00f,	1.00f,	1.00f
	},
	{	/* Grey Gradient */
		0.000000f,	 0.000000f,	0.000000f,
		0.140000f,	 0.140000f,	0.140000f,
		0.280000f,	 0.280000f,	0.280000f,
		0.420000f,	 0.420000f,	0.420000f,
		0.560000f,	 0.560000f,	0.560000f,
		0.700000f,	 0.700000f,	0.700000f,
		0.840000f,	 0.840000f,	0.840000f,
		1.000000f,	 1.000000f,	1.000000f
	},	
	{	/* Pastel color */
		255.0f/255.0f,		191.0f/255.0f,	191.0f/255.0f,
		255.0f/255.0f,		238.0f/255.0f,	191.0f/255.0f,
		222.0f/255.0f,		255.0f/255.0f,	191.0f/255.0f,
		191.0f/255.0f,		255.0f/255.0f,	207.0f/255.0f,
		191.0f/255.0f,		254.0f/255.0f,	255.0f/255.0f,
		191.0f/255.0f,		206.0f/255.0f,	255.0f/255.0f,
		223.0f/255.0f,		191.0f/255.0f,	255.0f/255.0f,
		255.0f/255.0f,		191.0f/255.0f,	238.0f/255.0f
	},
	{	/* Greenish */
		125.0f/255.0f,	255.0f/255.0f,	1.0f/255.0f,
		173.0f/255.0f,	255.0f/255.0f,	35.0f/255.0f,
		189.0f/255.0f,	255.0f/255.0f,	46.0f/255.0f,
		204.0f/255.0f,	255.0f/255.0f,	57.0f/255.0f,
		219.0f/255.0f,	255.0f/255.0f,	68.0f/255.0f,
		229.0f/255.0f,	247.0f/255.0f,	65.0f/255.0f,
		234.0f/255.0f,	230.0f/255.0f,	53.0f/255.0f,
		239.0f/255.0f,	212.0f/255.0f,	40.0f/255.0f
	},
	{	/* Red strips */
		 0.000000f,		0.000000f,	 0.000000f,
		 1.000000f,		0.500000f,	 0.000000f,
		 0.500000f,		0.000000f,	 0.000000f,
		 1.000000f,		0.500000f,	 0.000000f,
		 0.500000f,		0.000000f,	 0.000000f,
		 1.000000f,		0.50000f,	 0.000000f,
		 0.500000f,		0.000000f,	 0.000000f,
		 1.000000f,		0.200000f,	 0.300000f
	}
};

/*=======================================================*/
/* Static function declaration                           */
/*=======================================================*/
static int drawframe(void);

/*=======================================================*/
/* initialization                                        */
/*=======================================================*/

static void initialize(void)
{
	/* Initialize display */
	init_display(DISPLAY_WIDTH, DISPLAY_HEIGHT, APP_NAME, drawframe);

	/* Initialization for gas */
	gas_initialize(cmn_getRenderbufferObjectID(0, GL_DEPTH_STENCIL_ATTACHMENT));
}

static void loadParticleSystem_1(struct _particleSys *pp)
{
	GLfloat vertex[4] = {0.0f, 1.0f, 2.0f, 3.0f};

	glGenBuffers(1, &pp->m_pSysPositionID) ;
	glBindBuffer(GL_ARRAY_BUFFER, pp->m_pSysPositionID);
	glBufferData(GL_ARRAY_BUFFER, 4 * sizeof(GLfloat), &vertex[0], GL_STATIC_DRAW);

	/*------------------------------------------------------------------------------------*/
	/*--------Definition of the 4 control point positions---------------------------------*/
	/*------------------------------------------------------------------------------------*/
	/*-----------Control Point 1---------------------Control Point 2----------------------*/
	pp->m_partsys_center[0][0] = 0.0f;		/*|*/	pp->m_partsys_center[1][0] = -5.0f;
	pp->m_partsys_center[0][1] = -20.0f;	/*|*/	pp->m_partsys_center[1][1] = -20.0f;
	pp->m_partsys_center[0][2] = 0.0f;		/*|*/	pp->m_partsys_center[1][2] = 0.0f;
	pp->m_partsys_center[0][3] = 1.0f;		/*|*/	pp->m_partsys_center[1][3] = 1.0f;
	/*-----------Control Point 3---------------------Control Point 4----------------------*/
	pp->m_partsys_center[2][0] = 5.0f;		/*|*/	pp->m_partsys_center[3][0] = 0.0f;
	pp->m_partsys_center[2][1] = 5.0f;		/*|*/	pp->m_partsys_center[3][1] = 15.0f;
	pp->m_partsys_center[2][2] = 0.0f;		/*|*/	pp->m_partsys_center[3][2] = 0.0f;
	pp->m_partsys_center[2][3] = 1.0f;		/*|*/	pp->m_partsys_center[3][3] = 1.0f;
	/*------------------------------------------------------------------------------------*/
	/*------------------------------------------------------------------------------------*/

	/*------------------------------------------------------------------------------------*/
	/*--------Definition of the 4 control point colors------------------------------------*/
	/*------------------------------------------------------------------------------------*/
	/*Color is not used in this sample.                                                   */
	/*To modify the alpha value, please use pp->m_partsys_aspect[k][3]                    */
	/*-----------Control Point 1---------------------Control Point 2----------------------*/
	pp->m_partsys_color[0][0] = 0.0f;		/*|*/	pp->m_partsys_color[1][0] = 1.0f;
	pp->m_partsys_color[0][1] = 0.0f;		/*|*/	pp->m_partsys_color[1][1] = 0.0f;
	pp->m_partsys_color[0][2] = 0.0f;		/*|*/	pp->m_partsys_color[1][2] = 0.0f;
	pp->m_partsys_color[0][3] = 1.0f;		/*|*/	pp->m_partsys_color[1][3] = 1.0f;
	/*-----------Control Point 3---------------------Control Point 4----------------------*/
	pp->m_partsys_color[2][0] = 0.0f;		/*|*/	pp->m_partsys_color[3][0] = 0.0f;
	pp->m_partsys_color[2][1] = 1.0f;		/*|*/	pp->m_partsys_color[3][1] = 0.0f;
	pp->m_partsys_color[2][2] = 0.0f;		/*|*/	pp->m_partsys_color[3][2] = 1.0f;
	pp->m_partsys_color[2][3] = 0.5f;		/*|*/	pp->m_partsys_color[3][3] = 0.0f;
	/*------------------------------------------------------------------------------------*/
	/*------------------------------------------------------------------------------------*/

	/*------------------------------------------------------------------------------------*/
	/*--------Definition of the 4 bounding boxes------------------------------------------*/
	/*--------Radii in X, Y and Z direction. W not used ----------------------------------*/
	/*------------------------------------------------------------------------------------*/
	/*-----------Control Point 1---------------------Control Point 2----------------------*/
	pp->m_partsys_radius[0][0] = 0.0f;		/*|*/	pp->m_partsys_radius[1][0] = 0.2f;
	pp->m_partsys_radius[0][1] = 0.0f;		/*|*/	pp->m_partsys_radius[1][1] = 2.0f;
	pp->m_partsys_radius[0][2] = 0.0f;		/*|*/	pp->m_partsys_radius[1][2] = 0.2f;
	pp->m_partsys_radius[0][3] = 1.0f;		/*|*/	pp->m_partsys_radius[1][3] = 1.0f;
	/*-----------Control Point 3---------------------Control Point 4----------------------*/
	pp->m_partsys_radius[2][0] = 1.0f;		/*|*/	pp->m_partsys_radius[3][0] = 5.0f;
	pp->m_partsys_radius[2][1] = 10.0f;		/*|*/	pp->m_partsys_radius[3][1] = 0.0f;
	pp->m_partsys_radius[2][2] = 1.0f;		/*|*/	pp->m_partsys_radius[3][2] = 5.0f;
	pp->m_partsys_radius[2][3] = 1.0f;		/*|*/	pp->m_partsys_radius[3][3] = 1.0f;
	/*------------------------------------------------------------------------------------*/
	/*------------------------------------------------------------------------------------*/

	/*------------------------------------------------------------------------------------*/
	/*--------Definition of the particle aspect-------------------------------------------*/
	/*--------Particle size, texture coordinates !!!??TBD	------------------------------*/
	/*------------------------------------------------------------------------------------*/
	/*Rotation disabled: Size, unused,unused,unused/alpha                                 */
	/*Rotation Enabled:  Size, Angle, Radius, unused/alpha                                */
	/*-----------Control Point 1---------------------Control Point 2----------------------*/
	pp->m_partsys_aspect[0][0] = 0.0f;		/*|*/	pp->m_partsys_aspect[1][0] = 4.0f;
	pp->m_partsys_aspect[0][1] = 0.0f;		/*|*/	pp->m_partsys_aspect[1][1] = 0.0f;
	pp->m_partsys_aspect[0][2] = 0.9f;		/*|*/	pp->m_partsys_aspect[1][2] = 0.9f;
	pp->m_partsys_aspect[0][3] = 1.0f;		/*|*/	pp->m_partsys_aspect[1][3] = 1.0f;
	/*-----------Control Point 3---------------------Control Point 4----------------------*/
	pp->m_partsys_aspect[2][0] = 6.0f;		/*|*/	pp->m_partsys_aspect[3][0] = 30.0f;
	pp->m_partsys_aspect[2][1] = 0.0f;		/*|*/	pp->m_partsys_aspect[3][1] = 0.0f;
	pp->m_partsys_aspect[2][2] = 0.9f;		/*|*/	pp->m_partsys_aspect[3][2] = 0.9f;
	pp->m_partsys_aspect[2][3] = 0.0f;		/*|*/	pp->m_partsys_aspect[3][3] = 0.0f;
	/*------------------------------------------------------------------------------------*/
	/*------------------------------------------------------------------------------------*/
	pp->m_size_min_max[0] = 1.0f;
	pp->m_size_min_max[1] = (GLfloat)DISPLAY_HEIGHT / 2.0f;

	pp->m_random_seed[0] = 45.0f;
	pp->m_random_seed[1] = 2.564f + pp->m_random_seed[0];
	pp->m_random_seed[2] = 1.15f + pp->m_random_seed[0];
	pp->m_random_seed[3] = 0.9875f + pp->m_random_seed[0];

	/* Core value for random
	 Algorithm is based Pseudo-random number generators: X(n+1) = (a*Xn+b) mod m
	 Expected values for uniform are
	 [0] = a, [1] = b, [2] = m, [3] = 1/m
	*/
	pp->m_prng[0] = 17.0f;					/* a */
	pp->m_prng[1] = 37.0f;					/* b */
	pp->m_prng[2] = 65535.0f;				/* m */
	pp->m_prng[3] = 1.0f / (pp->m_prng[2]);	/* 1/m */

	pp->m_speed = 1.0f;				/*  emitter->particle_speed  */
	pp->m_particleCountMax = 60.0f;	/*  particle number          */

	/* Application only */
	pp->simulationTime = 2.0f;
	pp->NFrame = 512;
	pp->dTime = pp->simulationTime / pp->NFrame;
}

static void sceneObject_initialize(void)
{
	/* Load object (non-gaseous object) */
	loadDAT(FILE_APP_ROOT "planeXY3.dat", &plane);

	/* Load Gaseous object */
	defaultGasObject(&gas, gasColorTable[0]);

	/* Define the gasesous object geometry as particle system */
	loadParticleSystem_1(&gas.pSys);
}

static void sceneObject_finalize(void)
{
	unloadDAT(&plane);
	glDeleteBuffers(1, &gas.pSys.m_pSysPositionID) ;
	
	return;
}

/*=======================================================*/
/* Initialization  of shaders                            */
/*=======================================================*/
int shader_initialize(void)
{
	shaders[0] = glCreateShader(GL_VERTEX_SHADER);
	shaders[1] = glCreateShader(GL_VERTEX_SHADER);
	shaders[2] = glCreateShader(GL_VERTEX_SHADER);
	shaders[3] = glCreateShader(GL_GEOMETRY_SHADER_DMP);
	
	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(4, shaders, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	/* for normal geometry */
	pGeoId = glCreateProgram();
	glAttachShader(pGeoId, shaders[0]);
	glAttachShader(pGeoId, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pGeoId, 0, "aPosition");

	glLinkProgram(pGeoId);
	glValidateProgram(pGeoId);
	glUseProgram(pGeoId);

	/* for gas accumulation pass */
	pAccId = glCreateProgram();
	glAttachShader(pAccId, shaders[1]);
	glAttachShader(pAccId, shaders[3]);
	glAttachShader(pAccId, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pAccId, 0, "attrCtrPointIndex");
	
	glLinkProgram(pAccId);
	glValidateProgram(pAccId);
	glUseProgram(pAccId);

	/* for gas shading/pos synthesis pass */
	pPostId = glCreateProgram();
	glAttachShader(pPostId, shaders[2]);
	glAttachShader(pPostId, GL_DMP_FRAGMENT_SHADER_DMP);
	glBindAttribLocation(pPostId, 0, "aPosition");
	glBindAttribLocation(pPostId, 1, "aTexCoord");
	glBindAttribLocation(pPostId, 2, "aColor");
	glLinkProgram(pPostId);
	glValidateProgram(pPostId);
	glUseProgram(pPostId);

	return 0;
}

/*=======================================================*/
/* draw object                                           */
/*=======================================================*/
static void drawObject(void)
{
	/* This function draws only one quadrangle. */
	glUseProgram(pGeoId);
	glClearColor(0.2f, 0.4f, 0.7f, 1.0f);
	glClearDepthf(1.f);

	/* Set fragment operation mode to GL_FRAGOP_MODE_GL_DMP */
	glUniform1i(glGetUniformLocation(pGeoId, "dmp_FragOperation.mode"), GL_FRAGOP_MODE_GL_DMP);

	/*
	 * Setup texture and blending unit
	 * Color of the object is defined as constant color of blending unit 0
	*/
	glUniform1i(glGetUniformLocation(pGeoId, "dmp_Texture[0].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pGeoId, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pGeoId, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pGeoId, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pGeoId, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pGeoId, "dmp_TexEnv[0].srcRgb"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pGeoId, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);
	
	float almostblack[4] = {0.15f, 0.15f, 0.15f, 0.0f};
	glUniform4fv(glGetUniformLocation(pGeoId, "dmp_TexEnv[0].constRgba"), 1, almostblack);

	/*
	 * misc settings
	*/
	glDisable(GL_BLEND);
	glEnable(GL_DEPTH_TEST);
	glDepthMask(GL_TRUE);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	glEnableVertexAttribArray(0);	/* enable position */
	glDisableVertexAttribArray(1);	/* disable texture coordinate */
	glDisableVertexAttribArray(2);	/* disable color */

	for (int i = 0; i < plane.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, plane.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)plane.obj[i].vtx_offset);
		for (unsigned j = plane.obj[i].patch_offset; j < plane.obj[i].patch_size + plane.obj[i].patch_offset; j++)
		{
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, plane.idxVB);
			glDrawElements(GL_TRIANGLES, plane.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(plane.patch[j].elm_offset + plane.obj[i].elm_offset));
		}
	}

	/*
	 * Finalization
	*/
	glDisableVertexAttribArray(0);
}

static int frame = 0;
static int pattern = 0;

/*=======================================================*/
/* draw frame                                            */
/*=======================================================*/
static int drawframe(void)
{
	/* Clear display buffer */
	//glBindFramebuffer(GL_FRAMEBUFFER, DISPLAY_BUFFER);
	cmn_setRenderTarget(0);

	glUseProgram(pGeoId);
	GLfloat m[16];
	mat4_t proj = mat4_t::perspective(35.0f, (float)DISPLAY_WIDTH / (float)DISPLAY_HEIGHT, 1.0f, 200.f);
	mat4_t mv = mat4_t::lookAt(0.f, 0.f, 80.0f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	
	if (frame == 0)
	{
		/* set next gas table */
		set_gasColorTable(&gas,gasColorTable[pattern % NUM_COLOR_SAMPLE]);
		gas.pSys.simulationTime = 0.f;
		pattern++;
	}
	
	//glBindFramebuffer(GL_FRAMEBUFFER, DISPLAY_BUFFER);
	glUseProgram(pGeoId);
	glClearColor(0.2f+(frame%100)*0.008f, 0.4f, 0.7f, 1.0f);
	glClearDepthf(1.f);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	glViewport(0, 0, DISPLAY_WIDTH, DISPLAY_HEIGHT);

	{
		/* Draw the Object */
		glUseProgram(pGeoId);
		proj.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pGeoId, "uProjection"), 1, GL_FALSE, m);
		mat4_t mv_rot = mv * mat4_t::rotate(45.0f, 0.f, 1.f, 0.f);
		mv_rot = mv_rot * mat4_t::scale(0.5f,0.5f,0.5f);
		mv_rot.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pGeoId, "uModelView"), 1, GL_FALSE, m);
		drawObject();
	}
	
	{
		/* Gas accumulation pass */
		glUseProgram(pAccId);
		
		glDisable(GL_CULL_FACE);
		glFrontFace(GL_CCW);
		glCullFace(GL_BACK);

		proj.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pAccId, "uProjection"), 1, GL_FALSE, m);
		mv.toFloatArr(m);
		mat4_t mv_tr = mv ;//* mat4_t::translate((float)trans, 0.f, 0.f);
		mv_tr.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pAccId, "uModelView"), 1, GL_FALSE, m);
		gas_accumulation();
	}

	cmn_setRenderTarget(0);
	/* Gas shading pass */
	gas_shading();

	gas.pSys.simulationTime += gas.pSys.dTime;
	/* swap buffer */
	glFinish();
	swap_buffer();

	if (++frame == gas.pSys.NFrame)
		frame = 0;	/* go to next gas pattern */

	/* return 0 when 16 frames done */
	return !glGetError();
}

/* ======================================================= */
/* main function                                           */
/* ======================================================= */
#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	frame = 0;
	pattern = 0;
	
	/* initialization of frame buffer and gas buffer*/
	initialize();

	/* Initialization of shaders */ 
	if (shader_initialize() >= 0)
	{
		/*standard and gasesous object initializations */
		sceneObject_initialize();

		/* Enter loop */
		draw_loop();
	}

	sceneObject_finalize();
	gas_finalize();
	glDeleteProgram(pGeoId);
	glDeleteProgram(pAccId);
	glDeleteProgram(pPostId);
	glDeleteShader(shaders[0]);
	glDeleteShader(shaders[1]);
	glDeleteShader(shaders[2]);
	glDeleteShader(shaders[3]);
	/* shutdown_display */
	shutdown_display();

	return 0;
}


}
}

