/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>

#include "Memory.h"

#define APP_NAME "LogicOp"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace LogicOp{

/* buffer id */
static GLuint array_buffer_id = 0;
static GLuint element_array_buffer_id = 0;

/* program id */
static GLuint pgid = 0;

/* shader id */
static GLuint shid = 0;

static GLfloat coords[] = {
	-0.20f, -0.15f, -0.5f, 1.f,
	 0.20f, -0.15f, -0.5f, 1.f,
	 0.20f,  0.15f, -0.5f, 1.f,
	-0.20f,  0.15f, -0.5f, 1.f,

	-0.20f, -0.15f, -0.5f, 1.f,
	 0.20f, -0.15f, -0.5f, 1.f,
	 0.20f,  0.15f, -0.5f, 1.f,
	-0.20f,  0.15f, -0.5f, 1.f,
};

static GLfloat color[] = {
	1.f, 1.f, 1.f, 1.f,
	1.f, 1.f, 1.f, 1.f,
	0.f, 0.f, 0.f, 0.f,
	0.f, 0.f, 0.f, 0.f,

	1.f, 1.f, 1.f, 1.f,
	0.f, 0.f, 0.f, 0.f,
	0.f, 0.f, 0.f, 0.f,
	1.f, 1.f, 1.f, 1.f,
};

static GLushort idxs[] = {
	0, 1, 2, 3,
	4, 5, 6, 7
};


/* generate simple object */
static void load_objects(void)
{
	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBufferData(GL_ARRAY_BUFFER, sizeof(coords) + sizeof(color), 0, GL_STATIC_DRAW);
	glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(coords), coords);
	glBufferSubData(GL_ARRAY_BUFFER, sizeof(coords), sizeof(color), color);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(idxs), idxs, GL_STATIC_DRAW);
}

static unsigned frame_num = 0;

static int drawframe(void)
{
	glClearColor(0.1f , 0.008f * (frame_num % 100), 0.8f, 1.f );
	glClearDepthf( 1.f );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	
	float x = -0.6f;
	float y = -0.75f;

	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0) ;
	glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, 0, (GLvoid*)sizeof(coords));

	glEnable( GL_COLOR_LOGIC_OP );

	mat4_t r = mat4_t::rotate((GLfloat)frame_num, 0, 0, 1);

	unsigned int op[] = {
		GL_CLEAR,
		GL_AND,
		GL_AND_REVERSE,
		GL_COPY,
		GL_AND_INVERTED,
		GL_NOOP,
		GL_XOR,
		GL_OR,
		GL_NOR,
		GL_EQUIV,
		GL_INVERT,
		GL_OR_REVERSE,
		GL_COPY_INVERTED,
		GL_OR_INVERTED,
		GL_NAND,
		GL_SET,
	};

	for ( int j = 0; j < 4; j++ ) {
		x = -0.75f;
		for ( int i = 0; i < 4; i++ ) {
			GLfloat mv[16];
			mat4_t m = mat4_t::translate( x, y, 0.f ) * r;
			m.toFloatArr(mv);
			glUniformMatrix4fv( glGetUniformLocation( pgid, "uModelView" ), 1, GL_FALSE, mv );
			
			glLogicOp( GL_COPY );
			glDrawElements( GL_TRIANGLE_FAN, 4, GL_UNSIGNED_SHORT, 0 );
			glLogicOp( op[4*j + i] );
			glDrawElements( GL_TRIANGLE_FAN, 4, GL_UNSIGNED_SHORT, (void*)(sizeof(GLushort)*4) );

			x += 0.5f;
		}
		y += 0.5f;
	}

	glFinish();

	swap_buffer();

	frame_num++;

	glDisable( GL_COLOR_LOGIC_OP );

	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame_num = 0;

	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation( pgid, 0, "aPosition" );
	glBindAttribLocation( pgid, 1, "aColor" );

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	GLfloat proj[16], mv[16];
	mat4_t m = mat4_t::ortho(-1.f, 1.f, -1.f, 1.f, -1.f, 1.f);
	m.toFloatArr(proj);
	glUniformMatrix4fv( glGetUniformLocation( pgid, "uProjection" ), 1, GL_FALSE, proj );

	mat4_t m2;
	m2.toFloatArr(mv);
	glUniformMatrix4fv( glGetUniformLocation( pgid, "uModelView" ), 1, GL_FALSE, mv );

	glViewport( 0, 0, WIDTH, HEIGHT );

	load_objects();

	glEnable( GL_CULL_FACE );
	glFrontFace( GL_CCW );

	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	return 0;
}

/* finalization */
static void finalize()
{
	if (array_buffer_id) {
		glDeleteBuffers(1, &array_buffer_id);
		array_buffer_id = 0;
	}
	if (element_array_buffer_id) {
		glDeleteBuffers(1, &element_array_buffer_id);
		element_array_buffer_id = 0;
	}
	if (shid) {
		glDeleteShader(shid);
		shid = 0;
	}
	if (pgid) {
		glDeleteProgram(pgid);
		pgid = 0;
	}
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	/* finalization */
	finalize();

	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

