/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

/*
 * Comments
 * --------
 *
 * VN masking means that light reflection is set to zero when VN is smaller
 * than some threshold value 'outline.' This method works poorly when an 
 * object has concave places (as a center part of the inside of a palm). 
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "LightingToonApple"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace LightingToonApple{

/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

/* env texture id */
GLuint env_tex_name;
/* obj file loader class object */
dat_t sphere;

/* texture collection id */
GLuint tex_coll;
GLuint luts[4]; 

/* if USE_LN is not defined, half-vector is used as the vector controlling both diffuse illumination and highlight position */
#define USE_LN
/* if WINDOW is defined, a window highlight is used */
//#define WINDOW

static void set_lut_table(void)
{
	glGenTextures(4, luts); 

	const float outline = 0.15f; 
	const float highlight_eps = 0.01f; 
	float delta[] = {1.f, 0.7f, 0.5f, -1.f};

	GLfloat lut[512];
	int j;
	memset(lut, 0, sizeof(lut));
	float previous = 0, LN; 
	int i = 0; 

	/* f(D) = D is used as the initial shading function - see "Illumination Models in Maestro" for details
	first specify the steps in the [0, 1) interval */
	for (j = 127; j >= 0; j--)
	{
		LN = (float)j/127.9375f;
		if (LN > delta[i])
			lut[j] = previous; 
		else 
		{
			lut[j] = LN; 
			previous = lut[j]; 
			i++; 
		}
	}
	for (j = 0; j < 127; j++)
		lut[j + 256] = lut[j+1] - lut[j];
	lut[127 + 256] = 0.f; 

	/* now specify the steps in the [-1, 0) interval */
	for (j = 255; j  >= 128; j--)
	{
		LN = (float)(j - 256) /128.f;
		if (LN > delta[i])
			lut[j] = previous; 
		else 
		{
			lut[j] = LN; 
			previous = lut[j]; 
			i++; 
		}
	}

	/* now calculate differences (some are 0, some are not) */
	for (j = 128; j < 255; j++)
			lut[j + 256] = lut[j+1] - lut[j];
    lut[255 + 256] = lut[0] - lut[255];

	glBindTexture(GL_LUT_TEXTURE0_DMP, luts[0]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	for (j = 0; j < 256; j++)
		if ((float)j/255.9375f <= 1.f - highlight_eps)
			lut[j] = 0.f; 
		else 
			lut[j] = 1.f; 
	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j+1] - lut[j];
	lut[255 + 256] = 0.f;

	glBindTexture(GL_LUT_TEXTURE1_DMP, luts[1]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	for (j = 0; j < 256; j++)
		if ((float)j/255.9375f < outline) 
			lut[j] = 0.f; 
		else 
			lut[j] = 1.f; 

	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j+1] - lut[j];
	lut[255 + 256] = 1.f - lut[255];

	glBindTexture(GL_LUT_TEXTURE2_DMP, luts[2]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	for (j = 0; j < 256; j++)
		lut[j] = 1.f; 

	for (j = 0; j < 256; j++)
		lut[j + 256] = 0.f; 

	glBindTexture(GL_LUT_TEXTURE3_DMP, luts[3]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);
}

/* set lighting render state */
static void set_render_state(void)
{
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputFR"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputRR"), GL_FALSE);

#ifdef USE_LN
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRR"), GL_LIGHT_ENV_LN_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_LN_DMP);
#else
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRR"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_NH_DMP);
#endif
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD1"), GL_LIGHT_ENV_NV_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputFR"), GL_LIGHT_ENV_NV_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].shadowed"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].twoSideDiffuse"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_FALSE);

	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD0"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD1"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleSP"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleFR"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRB"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRG"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRR"), 1.f);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].spotEnabled"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG6_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.bumpMode"), GL_LIGHT_ENV_BUMP_NOT_USED_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.fresnelSelector"), GL_LIGHT_ENV_PRI_SEC_ALPHA_FRESNEL_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerRR"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD1"), 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerFR"), 3);

	set_lut_table();

	GLfloat ma[] = {0.f, 0.f, 0.f, 1.f};	/* material ambient */
	GLfloat md[] = {0.f, 0.f, 0.f, 1.f};	/* material diffuse */
	GLfloat ms[] = {1.f, 1.f, 1.f, 1.f};	/* material specular */
#ifdef WINDOW
	ms[0] = 0.f;
	ms[1] = 0.f;
	ms[2] = 0.f;
#endif
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.diffuse"), 1, md);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1, ms);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.ambient"), 1, ma);

	GLfloat ld[] = {1.f, 1.f, 1.f, 1.f};	/* light diffuse */
	GLfloat ls[] = {1.f, 1.f, 1.f, 1.f};	/* light specular */
	GLfloat ls2[] = {1.f, 0.2f, 0.f, 1.f};	/* light specular2 */
	GLfloat la[] = {1.f, 1.f, 1.f, 1.f};	/* light ambient */

	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].diffuse"), 1, ld);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular0"), 1, ls);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular1"), 1, ls2);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].ambient"), 1, la);

	GLfloat m[16] ;
    /* set projection matrix */
    mat4_t proj = mat4_t::frustum(-0.015f, 0.015f, -0.015f*HEIGHT/WIDTH, 0.015f*HEIGHT/WIDTH, 0.2f, 200.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	/* set texture setting */
#ifdef WINDOW
	/* output color is EnvTexture_rgb * PriColor_alpha + SecColor_rgb
	EnvTexture_rgb is environment highlight specular, PriColor_alpha is frensel reflection,
	and SecColor_rgb is toon shading diffuse color. */
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_TEXTURE_CUBE_MAP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_MULT_ADD_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_ALPHA, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE0, GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_FRAGMENT_SECONDARY_COLOR_DMP);
#else
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_CONSTANT, GL_CONSTANT);
#endif
}

/* load objects */
static void load_objects(void)
{	
	/* load obj file geometory and diffuse texture */
	loadDAT(FILE_APP_ROOT "sphere.dat", &sphere);

	/* if you environment highlight specular, load environment cube texture */
#ifdef WINDOW
	char *env_tex[] =
	{
		FILE_APP_ROOT "wright.tga",
		FILE_APP_ROOT "wleft.tga",
		FILE_APP_ROOT "wtop.tga",
		FILE_APP_ROOT "wbottom.tga",
		FILE_APP_ROOT "wback.tga",
		FILE_APP_ROOT "wfront.tga",
	};
	glGenTextures(1, &env_tex_name);
	glBindTexture(GL_TEXTURE_CUBE_MAP, env_tex_name);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_LOD, 0);

	for (int face = 0; face < 6; face++)
	{
		bool use_alpha;
		loadTexture(env_tex[face], GL_TEXTURE_CUBE_MAP_POSITIVE_X+face, 0, use_alpha, true, 0, GL_RGBA);
	}
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
#endif
}

static void unload_objects(void)
{
	unloadDAT(&sphere);
	
	return;
}

static int frame = 0;

int drawframe(void)
{
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

	GLfloat m[16] ;
	mat4_t id;
	id.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uWorld"), 1, GL_FALSE, m);

	/* modelview setting */
	mat4_t modelview = mat4_t::lookAt(0.f, 5.75f, 22.5f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);

	/* light direction setting */
	GLfloat lpos0[4] = {0.f, 0.f, 2.f, 1.f};
	vec4_t lpos = modelview * vec4_t(lpos0) ;
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].position"), 1, &lpos.x);

//	modelview = modelview * mat4_t::rotate(-(float)frame, 0.f, 1.f, 0.f);
	modelview = modelview * mat4_t::translate(0.f, 1.8f * cos(DMP_PI / 90.f * frame), 0.f);
	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	/* draw objects */
	for (int i = 0; i < sphere.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, sphere.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)sphere.obj[i].vtx_offset);
		glBindBuffer(GL_ARRAY_BUFFER, sphere.normVB);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (void*)sphere.obj[i].nrm_offset);

		for (unsigned j = sphere.obj[i].patch_offset; j < sphere.obj[i].patch_size + sphere.obj[i].patch_offset; j++)
		{
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, sphere.idxVB);
			glDrawElements(GL_TRIANGLES, sphere.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(sphere.patch[j].elm_offset + sphere.obj[i].elm_offset));
		}
	}

	glFinish();

	/* it is possible to save the content of the buffer */
	/*
	char fname[256];
	sprintf(fname, "frame-%04d.tga", f);
	outputImage(WIDTH, HEIGHT, f);
	*/

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);

	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);

    /* attach fixed-function fragment shader */
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aNormal");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
    /* set program as current one to enable setting its uniforms */
	glUseProgram(pgid);

    /* enable arrays that will be mapped to position and normal */
    glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	/* create and bind texture collection object to capture subsequent modifications in the texture state
	 both lookup table-related one and ordinary (2D, cube) textures */
	glGenTextures(1, &tex_coll) ;
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, tex_coll);

	load_objects();

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* set another render state */
		set_render_state();

		/* Enter loop */
		draw_loop();
	}
	
	unload_objects();
	
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	glDeleteTextures(4, luts); 
#ifdef WINDOW
	glDeleteTextures(1, &env_tex_name);
#endif
	glDeleteTextures(1, &tex_coll) ;
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

