/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"


#define APP_NAME "LightingSpotAtte"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace LightingSpotAtte{

/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

GLuint lutids[3];

static void set_render_state(void)
{
	GLfloat ld_0[] = {1.f, 0.f, 0.f, 1.f};
	GLfloat ld_1[] = {0.f, 1.f, 0.f, 1.f};
	GLfloat ld_2[] = {0.f, 0.f, 1.f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].diffuse"), 1, ld_0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].diffuse"), 1, ld_1);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[2].diffuse"), 1, ld_2);

	/* setup spot lighting configuration */
	/* X̃Cgɂ͌ʂ̎Qƃe[u蓖Ă܂Qƃe[u̓͂ƃTvO̎dɂĂ
	   ׂẴCgŋʂ̐ݒ肪Kp邱Ƃɒӂĉ */
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputSP"), GL_LIGHT_ENV_SP_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputSP"), GL_TRUE);

	float lut[512];
	unsigned int j;
	glGenTextures(3, lutids);

#define COS_45 0.71f
#define COS_40 0.77f
#define COS_35 0.82f
#define COS_30 0.87f
#define COS_25 0.91f
#define COS_20 0.94f
#define COS_15 0.97f
#define COS_10 0.98f

	memset( lut, 0, sizeof( lut ) );
	for (j = 0; j < 256; j++)
	{
		float x = ( float )j / ( float )256;
		if ( x > COS_30 )
			lut[j] = 1.f;
	}

	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[0]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].samplerSP"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].spotEnabled"), GL_TRUE);

	memset( lut, 0, sizeof( lut ) );
	for (j = 0; j < 256; j++)
	{
		float x = ( float )j / ( float )256;
		if ( x > COS_20 )
			lut[j] = 1.f;
	}

	glBindTexture(GL_LUT_TEXTURE1_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].samplerSP"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].spotEnabled"), GL_TRUE);

	memset( lut, 0, sizeof( lut ) );
	for (j = 0; j < 256; j++)
	{
		float x = ( float )j / ( float )256;
		if ( x > COS_10 )
			lut[j] = 1.f;
	}

	glBindTexture(GL_LUT_TEXTURE2_DMP, lutids[2]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[2].samplerSP"), 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[2].spotEnabled"), GL_TRUE);
}

static void draw_plane()
{
	/* tOgCeBOł͕r|Sfł_CeBOƈقȂȃX|bgCgł܂
	@ ̃Tvł̓X|bgCgKpĂ郂f͂킸Q|Ŝ݂ō\Ă܂Ǎʂ𒸓_CeBO
	   Ŏꍇ́AȂׂfgpKv܂ */
	GLfloat coords[] =
	{
		-3.f, 0.f, 3.f,
		3.f, 0.f, 3.f,
		3.f, 0.f, -3.f,
		-3.f, 0.f, -3.f,
	};
	
	GLfloat norm[] =
	{
		0.f, 1.f, 0.f,
		0.f, 1.f, 0.f,
		0.f, 1.f, 0.f,
		0.f, 1.f, 0.f,
	};

	unsigned short index[] =
	{
		0, 1, 2, 0, 2, 3
	};

	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, coords);
	glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, norm);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, index);
}

static int frame = 0;

int drawframe(void)
{
	GLfloat m[16];
	mat4_t proj, mv;

	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* setup projection matrix */
	proj = mat4_t::frustum(-0.06f, 0.06f, -0.06f*HEIGHT/WIDTH, 0.06f*HEIGHT/WIDTH, 0.2f, 200.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	mv = mat4_t::rotate(-90.f, 0.f, 0.f, 1.f);
	mv = mv * mat4_t::lookAt(0.f, 7.f, 7.f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);

	/* setup light direction and spot direction (light0) */
	/* Cg̈ʒuуX|bgCg͎̕_Wnɂlɕϊ̂
	@ \tOgVF[_̃jtH[ɓn܂
	   _Wnւ̕ϊ͐΂Ȃ̂ł͂܂
	   ̃Tvł́AtOgCeBǑvZŕKvɂȂ@юxNg
   @@_VF[_Ŏ_Wn̂̂ɕϊĂ܂
	 @̂߃CgʒuƃX|bgCg_Wn̂̂KvɂȂ܂
	@ ܂X|bg̕xNg͐K̂nKv邱ƂɒӂĂ */
	GLfloat lpos0[] = {0.f, 1.f, 0.f, 1.f};
	lpos0[1] = 3.f * cosf(2.f * frame / 60.f) + 5.f;	
	vec4_t lpos = mv * vec4_t(lpos0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].position"), 1, &lpos.x);
	GLfloat lpos0_dir[] = {0.f, -1.f, 0.f, 0.f};
	lpos0_dir[0] = 0.5f * cosf(2.f * frame / 60.f);
	lpos0_dir[2] = 0.5f * sinf(2.f * frame / 60.f);
	vec3_t lpos_dir = mv * vec4_t(lpos0_dir);
	lpos_dir /= lpos_dir.norm();
	glUniform3fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].spotDirection"), 1, &lpos_dir.x);

	/* setup light direction and spot direction (light1) */
	GLfloat lpos1[] = {0.f, 1.f, 0.f, 1.f};
	lpos1[1] = 3.f * cosf(2.5f * frame / 60.f) + 4.f;	
	lpos = mv * vec4_t(lpos1);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].position"), 1, &lpos.x);
	GLfloat lpos1_dir[] = {0.f, -1.f, 0.f, 0.f};
	lpos1_dir[0] = 0.5f * cosf(2.5f * frame / 60.f);
	lpos1_dir[2] = 0.5f * sinf(2.5f * frame / 60.f);
	lpos_dir = mv * vec4_t(lpos1_dir);
	lpos_dir /= lpos_dir.norm();
	glUniform3fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].spotDirection"), 1, &lpos_dir.x);

	/* setup light direction and spot direction (light2) */
	GLfloat lpos2[] = {0.f, 1.f, 0.f, 1.f};
	lpos2[1] = 3.f * cosf(3.f * frame / 60.f) + 4.f;	
	lpos = mv * vec4_t(lpos2);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[2].position"), 1, &lpos.x);
	GLfloat lpos2_dir[] = {0.f, -1.f, 0.f, 0.f};
	lpos2_dir[0] = 0.5f * cosf(3.f * frame / 60.f);
	lpos2_dir[2] = 0.5f * sinf(3.f * frame / 60.f);
	lpos_dir = mv * vec4_t(lpos2_dir);
	lpos_dir /= lpos_dir.norm();
	glUniform3fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[2].spotDirection"), 1, &lpos_dir.x);

	/* setup modelview matrix */
	mv.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	/* draw objects */
	draw_plane();

	glFinish();

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aNormal");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	/* set another render state */
	set_render_state();

	/* enable DMP fragment lighting */
	/* this sample use 3 lights */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[2].enabled"), GL_TRUE);

	/* this sample use only fragment primary color */
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);
	
	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	glDeleteTextures(3, lutids);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}
