/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "LightingMicrofacet"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

#define STATE_SETTINGS 0 /* 0 - simple Blinn-Phong, 1 - generalized Blinn-Phong, 2 - Cook-Torrance */

using namespace gputest::common;

namespace gputest{
namespace LightingMicrofacet{

static void state0(void);

static void state1(void);

static void state2(void);

void (*render_state_func[])(void) =
{
	state0, state1, state2,
};

/* obj file loader class object */
dat_t robot;

/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

GLuint lutids[6];
int lutidnum = 0;

static void state0(void)
{
	GLfloat ls0[] = {1.f, 1.f, 0.f, 1.f};
	GLfloat ms[] = {1.f, 1.f, 1.f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular0"), 1, ls0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1, ms);

	/* to ignore 2nd specular term, set 0 to ms2.
	use of const reflection allows material specular2(ms2). */
	GLfloat ms2[] = {0.f, 0.f, 0.f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular1"), 1, ms2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_FALSE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"), GL_TRUE);

	/* becase 2nd specular term is not used, only D0 table is used
	in configuration 0. */
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG0_DMP);

	/* geometry factor is not considered. */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_FALSE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);

	/* bind lookup table */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), 0);

	/* setup lookup table */
	glGenTextures(2, lutids);
	lutidnum = 2;
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, lutids[0]);

	GLfloat lut[512];
	int j;
	memset(lut, 0, sizeof(lut));

	for (j = 0; j < 256; j++)
		/* shininess is 10.0 */
		lut[j] = powf((float)j/255.9375f, 10.f);
		/* shininess is 1.0 */
		/* lut[j] = powf((float)j/255.9375f, 1.f); */
	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j + 1] - lut[j];
	lut[255 + 256] = 1.f - lut[255];

	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);
}

/* generaized Blinn-Pnong reflection */
static void state1(void)
{
	GLfloat ls0[] = {0.f, 1.f, 0.f, 1.f};
	GLfloat ms[] = {1.f, 1.f, 1.f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular0"), 1, ls0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1, ms);

	GLfloat ls20[] = {1.f, 0.01f, 0.25f, 1.f};
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular1"), 1, ls20);
	/* use of const reflection allows material specular2(ms2). */
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_FALSE);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular1"), 1, ms);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD1"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG2_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);

	/* bind lookup table */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD1"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), 1);

	/* setup lookup table */
	glGenTextures(3, lutids);
	lutidnum = 3;
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, lutids[0]);

	GLfloat lut[512];
	int j;
	memset(lut, 0, sizeof(lut));

	/* set layer1 distribution function */
	for (j = 0; j < 256; j++)
		/* when s is equal to 2.0/m^2, highlight specular is similar between
		power function and gaussian function.
		when s is 90.0, m is 0.149, and when s is 2.0,
		m is 1.0. */
		/* lut[j] = powf((float)j/255.9375f, 90.f); */
		lut[j] = gaussian((float)j/255.9375f, 0.149f);
	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j + 1] - lut[j];
	lut[255 + 256] = 1.f - lut[255];

	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* set layer2 distribution function */
	memset(lut, 0, sizeof(lut));

	for (j = 0; j < 256; j++)
		/* lut[j] = powf((float)j/255.9375f, 2.f); */
		lut[j] = gaussian((float)j/255.9375f, 1.f);
	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j + 1] - lut[j];
	lut[255 + 256] = 1.f - lut[255];

	glBindTexture(GL_LUT_TEXTURE1_DMP, lutids[2]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);
}


/* full-scope Cook-Torannce reflection */
static void state2(void)
{
	GLfloat ls0[] = {1.f, 1.f, 1.f, 1.f};
	GLfloat disable[] = {0.f, 0.f, 0.f, 0.f};

	/* to ignore specular 1st term, set 0. */
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1, disable);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular1"), 1, ls0);

	/* to use Rr, Rg, Rb table */
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRR"), GL_LIGHT_ENV_VH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRG"), GL_LIGHT_ENV_VH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRB"), GL_LIGHT_ENV_VH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD1"), GL_LIGHT_ENV_NH_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputRR"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputRG"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputRB"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD1"), GL_FALSE);

	/* below we make bit shift after sampling table of Fresnel reflection
	we have to do this because we do not have HDR(sky has high intensity) */
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRR"), 2.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRG"), 2.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRB"), 2.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD0"), 2.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD1"), 2.f);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG4_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);

	/* geometory factor is considered in Cook-Torrance model */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_TRUE);

	/* bind lookup table */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerRR"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerRG"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerRB"), 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD1"), 3);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), 4);

	/* setup lookup table */
	glGenTextures(6, lutids);
	lutidnum = 6;
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, lutids[0]);

	GLfloat lut[512];
	int j;
	memset(lut, 0, sizeof(lut));
	glBindTexture(GL_LUT_TEXTURE4_DMP, lutids[5]);
	glTexImage1D(GL_LUT_TEXTURE4_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* in this sample, gold is used as fresnel factor,
	and real and imaginary part of refractive index of gold
	is following.
	    n        k
	---------------------------------
	0.183521, 2.959155  at 630 nm
	0.516924, 2.276178  at 525 nm
	1.464924, 1.860113  at 455 nm */
	/* setup Rr table */
	for (j = 0; j < 128; j++)
		lut[j] = nk_fresnel((float)j/128.f, 0.183521f, 2.959155f);

	for (j = 0; j < 127; j++)
		lut[j + 256] = lut[j+1] - lut[j];

	lut[127 + 256] = nk_fresnel(1.f, 0.183521f, 2.959155f) - lut[127];

	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* setup Rg table */
	memset(lut, 0, sizeof(lut));
	for (j = 0; j < 128; j++)
		lut[j] = nk_fresnel((float)j/128.f, 0.516924f, 2.276178f);

	for (j = 0; j < 127; j++)
		lut[j + 256] = lut[j+1] - lut[j];

	lut[127 + 256] = nk_fresnel(1.f, 0.516924f, 2.276178f) - lut[127];

	glBindTexture(GL_LUT_TEXTURE1_DMP, lutids[2]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* setup Rb table */
	memset(lut, 0, sizeof(lut));
	for (j = 0; j < 128; j++)
		lut[j] = nk_fresnel((float)j/128.f, 1.464924f, 1.860113f);

	for (j = 0; j < 127; j++)
		lut[j + 256] = lut[j+1] - lut[j];

	lut[127 + 256] = nk_fresnel(1.f, 1.464924f, 1.860113f) - lut[127];

	glBindTexture(GL_LUT_TEXTURE2_DMP, lutids[3]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* don't use D0 table, so setup D1 table only */
	memset(lut, 0, sizeof(lut));
	/* beckmann function is used in m=1.0 and m=0.5 */
	for (j = 1; j < 128; j++)
		lut[j] = beckmann((float)j/128.f, 1.f);	/* m=1.0 */
		/* lut[j] = beckmann((float)j/128.f, 0.5f); */ /* m=0.5 */

	for (j = 0; j < 127; j++)
		lut[j + 256] = lut[j+1] - lut[j];
	lut[127 + 256] = 1.f - lut[127];

	glBindTexture(GL_LUT_TEXTURE3_DMP, lutids[4]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_CONSTANT, GL_CONSTANT);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);
}

/* generate simple object */
static void load_objects(void)
{
	/* load obj file geometory and diffuse texture */
	loadDAT(FILE_APP_ROOT "robot1.dat", &robot);
}

static void unload_objects(void)
{
	unloadDAT(&robot);
	
	return;
}

static int frame = 0;

int drawframe(void)
{
	GLfloat m[16];
	mat4_t proj, mv;

	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* setup projection matrix */
	proj = mat4_t::frustum(-0.06f, 0.06f, -0.06f*HEIGHT/WIDTH, 0.06f*HEIGHT/WIDTH, 0.2f, 200.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	mv = mat4_t::rotate(-90.f, 0.f, 0.f, 1.f);
	mv = mv * mat4_t::lookAt(0.f, 0.0f, 6.f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);

	/* setup light direction */
	GLfloat lpos0[] = {0.f, 1.f, 1.f, 0.f};
	vec4_t lpos = mv * vec4_t(lpos0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].position"), 1, &lpos.x);

	/* setup modelview matrix */
	mv = mv * mat4_t::rotate(-(float)frame, 0.f, 1.f, 0.f);
	mv.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	/* draw objects */
	for (int i = 0; i < robot.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, robot.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)robot.obj[i].vtx_offset);
		glBindBuffer(GL_ARRAY_BUFFER, robot.normVB);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (void*)robot.obj[i].nrm_offset);

		for (unsigned j = robot.obj[i].patch_offset; j < robot.obj[i].patch_size + robot.obj[i].patch_offset; j++)
		{
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, robot.idxVB);
			glDrawElements(GL_TRIANGLES, robot.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(robot.patch[j].elm_offset + robot.obj[i].elm_offset));
		}
	}

	glFinish();

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aNormal");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	load_objects();

	/* set another render state */
	(*render_state_func[STATE_SETTINGS])();

	/* enable DMP fragment lighting */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);
	
	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	unload_objects();
	glDeleteProgram(pgid);
	glDeleteShader(pgid);
	glDeleteTextures(lutidnum, lutids);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

