/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

/*
 * Comments
 * --------
 *
 * Light reflection in this sample consists of two parts - diffuse one
 * (simulating reflection from inside of cloth) modulated by texture and 
 * anisotropic one (simulating reflection of the air-fiber interface). 
 *
 * To render anisotropic appearance, we use Schlick's model described in 
 * the following paper 
 *
 * Christophe Schlick, An Inexpensive BRDF Model for Physically-Based 
 * Rendering, Comput. Graph. Forum 13(3), 233-246 (1994).
 */


#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>

#include "Memory.h"

#define APP_NAME "LightingFresnel"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace LightingFresnel{

/* buffer id */
GLuint array_buffer_id;
GLuint element_array_buffer_id;

/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

/* lut texture ID */
GLuint lutids[4];

/* env texture ID */
GLuint env_tex_name;

/* bump texture ID */
GLuint bump_tex_name;

/* obj file loader class object */
dat_t car;

static void set_lut_table()
{
	/* Beckmann function is used as distribution function. see Microfacet Reflection
	section in "ILLUMINATION MODELS IN MAESTRO LIGHTING" for details.
	distribution function means distribution ratio of maicrofacets normal(local normal)
	against surface normal(macroscopic normal). */
	GLfloat lut[512];
	int j;

	glGenTextures ( 4, lutids ) ;
	glBindTexture ( GL_TEXTURE_COLLECTION_DMP, lutids[0] ) ;

	memset(lut, 0, sizeof(lut));
	for (j = 1; j < 128; j++)
		lut[j] = 0.85f * beckmann((float)j/128.f, 0.3f);

	for (j = 0; j < 127; j++)
		lut[j + 256] = lut[j+1] - lut[j];
	lut[127 + 256] = 1.f - lut[127];

	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	for (j = 1; j < 128; j++)
		lut[j] = beckmann((float)j/128.f, 1.4f);

	for (j = 0; j < 127; j++)
		lut[j + 256] = lut[j+1] - lut[j];
	lut[127 + 256] = 1.f - lut[127];
	glBindTexture(GL_LUT_TEXTURE1_DMP, lutids[2]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	for (j = 0; j < 256; j++)
		lut[j] = r_fresnel((float)j/256.f);

	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j+1] - lut[j];

	lut[255 + 256] = r_fresnel(1.f) - lut[255];
	glBindTexture(GL_LUT_TEXTURE2_DMP, lutids[3]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD1"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerFR"), 2);
}

/* set lighting render state */
static void set_render_state(void)
{
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"),	GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD1"),	GL_FALSE);
	/* below we use GL_TRUE to have non-zero for negative NV because negative NV values happen on sillouette
	we need to do this trick because we use normal map(Change to GL_FALSE if normal map is not used) */
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputFR"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD1"), GL_LIGHT_ENV_NH_DMP);
	/* fresnel function is looked up by light incidence. an angle of incidence is
	be equivalent to angle of reflectance. so fresnel looked up by
	dot product of light vector and view vector. */
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputFR"), GL_LIGHT_ENV_NV_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].twoSideDiffuse"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);

	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD0"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD1"), 1.f);
	/* below we make bit shift after sampling table of Fresnel reflection
	we have to do this because we do not have HDR(and sky has high intensity) */
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleFR"), 2.f);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.fresnelSelector"), GL_LIGHT_ENV_PRI_SEC_ALPHA_FRESNEL_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG3_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD1"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.bumpMode"), GL_LIGHT_ENV_BUMP_AS_BUMP_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.bumpSelector"), GL_TEXTURE1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.bumpRenorm"), GL_TRUE);

	set_lut_table();

	GLfloat ms2[] = {0.7f, 0.01f, 0.25f, 1.f};
	GLfloat ld0[] = {1.f, 1.f, 1.f, 1.f};
	GLfloat ls0[] = {1.f, 1.f, 1.f, 1.f};

	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular1"), 1, ms2);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].diffuse"), 1, ld0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular0"), 1, ls0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular1"), 1, ls0);

	{
		/* Loading bump texture */
		char filename[]= FILE_APP_ROOT "car_tsp0.tga";
		int _lev = 0;
		glGenTextures(1, &bump_tex_name);
		glActiveTexture(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D, bump_tex_name);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_LOD, 0);
		bool bUseAlpha;
		loadTexture(filename, GL_TEXTURE_2D, _lev, bUseAlpha);
	}

	{
		/* loading environement */
		char *env_tex[] =
		{
			FILE_APP_ROOT "tpx0002.tga",
			FILE_APP_ROOT "tnx0002.tga",
			FILE_APP_ROOT "tpy0002.tga",
			FILE_APP_ROOT "tny0002.tga",
			FILE_APP_ROOT "tpz0002.tga",
			FILE_APP_ROOT "tnz0002.tga"
		};

		glGenTextures(1, &env_tex_name);
		glActiveTexture(GL_TEXTURE0);
		glBindTexture(GL_TEXTURE_CUBE_MAP, env_tex_name);
    	demo::NotifyCubeTexStart();
		for (int face = 0; face < 6; face++)
		{
			bool use_alpha;
			loadTexture(env_tex[face], GL_TEXTURE_CUBE_MAP_POSITIVE_X+face,0, use_alpha, true, 0, GL_RGBA);
		}
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_LOD, 0);
	}
	
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_TEXTURE_CUBE_MAP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_ADD);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_CONSTANT);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[1].samplerType"), GL_TEXTURE_2D);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[1].combineRgb"), GL_INTERPOLATE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[1].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcRgb"), GL_TEXTURE0, GL_PREVIOUS, GL_FRAGMENT_SECONDARY_COLOR_DMP);
}

/* load objects */
static void load_objects(void)
{	
	loadDAT(FILE_APP_ROOT "supercar.dat", &car);
	
	return;
}

static void unload_objects(void)
{
	unloadDAT(&car);
	
	return;
}

static int frame = 0;

int drawframe(void)
{
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	GLfloat m[16];

	mat4_t proj = mat4_t::frustum(-0.025f, 0.025f, -0.025f*HEIGHT/WIDTH, 0.025f*HEIGHT/WIDTH, 0.2f, 200.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	float step = 0.5f;

	mat4_t id;
	id.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uWorld"), 1, GL_FALSE, m);
	
	mat4_t mv = mat4_t::lookAt(0.f, 5.75f, 22.5f, 0.f, 0.f, 0.f, -1.f, 0.f, 0.f);
	
	GLfloat lpos0[] = {8.f, 8.5f, 1.f, 1.f};
	vec4_t lpos = mv * vec4_t(lpos0);
	glUniform4fv ( glGetUniformLocation ( pgid, "dmp_FragmentLightSource[0].position" ), 1, &lpos.x ) ;

	mv = mv * mat4_t::rotate(-step*frame, 0.f, 1.f, 0.f);
	mv.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	for (int i = 0; i < car.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, car.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)car.obj[i].vtx_offset);
		glBindBuffer(GL_ARRAY_BUFFER, car.normVB);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (void*)car.obj[i].nrm_offset);
		glBindBuffer(GL_ARRAY_BUFFER, car.tangVB);
		glVertexAttribPointer(2, 3, GL_FLOAT, GL_FALSE, 0, (void*)car.obj[i].tgt_offset);
		glBindBuffer(GL_ARRAY_BUFFER, car.texVB);
		glVertexAttribPointer(3, 3, GL_FLOAT, GL_FALSE, 0, (void*)car.obj[i].tex_offset);

		for (unsigned j = car.obj[i].patch_offset; j < car.obj[i].patch_size + car.obj[i].patch_offset; j++)
		{
			GLfloat ma[4] = {0.f, 0.f, 0.f, 1.f};
			GLfloat md[4] = {0.f, 0.f, 0.f, 1.f};
			GLfloat ms[4] = {0.f, 0.f, 0.f, 1.f};
			for (int col = 0; col < 3; col++)
			{
				ma[col] = car.patch[j].ambient[col];
				md[col] = car.patch[j].diffuse[col];
				ms[col] = car.patch[j].specular[col];
			}
			glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.diffuse"), 1, md);
			glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1, ms);
			glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.ambient"), 1, ma);
			
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, car.idxVB);
			glDrawElements(GL_TRIANGLES, car.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(car.patch[j].elm_offset + car.obj[i].elm_offset));
		}
	}


	glFinish();

	/* it is possible to save the content of the buffer */
	/*
	char fname[256];
	sprintf(fname, "frame-%04d.tga", f);
	outputImage(WIDTH, HEIGHT, fname);
	*/

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);

	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

    /* create program and load & attach vertex shader */
	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);

    /* attach fixed-function fragment shader */
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aNormal");
	glBindAttribLocation(pgid, 2, "aTang");
	glBindAttribLocation(pgid, 3, "aTexCoord");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	/* set program as current one to enable setting its uniforms */
	glUseProgram(pgid);

	load_objects();

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);
	glEnableVertexAttribArray(2);
	glEnableVertexAttribArray(3);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* set another render state */
		set_render_state();

		/* Enter loop */
		draw_loop();
	}
	unload_objects();
	glDeleteTextures ( 4, lutids ) ;
	glDeleteTextures(1, &bump_tex_name);
	demo::NotifyCubeTexStart();
	glDeleteTextures(1, &env_tex_name);
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}


}
}
