/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

/*
 * Comments
 * --------
 *
 * Light reflection in this sample consists of two parts - diffuse one
 * (simulating reflection from inside of cloth) modulated by texture and 
 * anisotropic one (simulating reflection of the air-fiber interface). 
 *
 * To render anisotropic appearance, we use Schlick's model described in 
 * the following paper 
 *
 * Christophe Schlick, An Inexpensive BRDF Model for Physically-Based 
 * Rendering, Comput. Graph. Forum 13(3), 233-246 (1994).
 *
 * (c) DMP, Inc.
 * konstantin.kolchin@dmprof.com
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>
#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>

#include "Memory.h"

#define APP_NAME "LightingAniso"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace LightingAniso{

/* buffer id */
GLuint array_buffer_id;
GLuint element_array_buffer_id;

/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

/* diffuse texture ID */
GLuint diff_tex_name;

/* bump texture ID */
GLuint bump_tex_name;

/* obj file loader class object */
dat_t kimono;

/* texture collection name */
GLuint tex_coll ;
GLuint lutids[5] ;

static void set_lut_table()
{
	/* Schlick suggested that distribution function of anisotropic reflection
	is composed of two functions, one dependent on dot product NH, and other dependent on
	dot product of the angle between the projection of H on the tangent plane and the
	tangent vector. in microfacets theory, distribution function is not
	depended on wave length(R,G,B) essentially. but, this sample distribution term 
	looked up by NH is depended on wave length. therefore, specular has a little blue. */

	glGenTextures(5, lutids);

	GLfloat lut[512];
	int j;
	memset(lut, 0, sizeof(lut));

	/* to ignore 1st specular term, set 0. */
	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[0]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* set term depended on NH */
	for (j = 0; j < 256; j++)
		lut[j] = 0.7f * z_schlick(0.7f, (float)j/256.f, true);

	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j+1] - lut[j];

	lut[255 + 256] = 0.7f * z_schlick(0.7f, 1.f, true) - lut[255];
	glBindTexture(GL_LUT_TEXTURE1_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE1_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	for (j = 0; j < 256; j++)
		lut[j] = 0.7f * z_schlick(0.7f, (float)j/256.f, true);

	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j+1] - lut[j];

	lut[255 + 256] = 0.7f * z_schlick(0.7f, 1.f, true) - lut[255];
	glBindTexture(GL_LUT_TEXTURE2_DMP, lutids[2]);
	glTexImage1D(GL_LUT_TEXTURE2_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	for (j = 0; j < 256; j++)
		lut[j] = z_schlick(0.5f, (float)j/256.f, true);

	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j+1] - lut[j];

	lut[255 + 256] = z_schlick(0.5f, 1.f, true) - lut[255];
	glBindTexture(GL_LUT_TEXTURE3_DMP, lutids[3]);
	glTexImage1D(GL_LUT_TEXTURE3_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	/* set term dependent on dot product between the projection of H on the tangent plane
	and the tangent vector */
	for (j = 0; j < 256; j++)
		lut[j] = a_schlick(0.015f, (float)j/256.f, true);

	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j + 1] - lut[j];

	lut[255 + 256] = a_schlick(0.015f, 1.f, true) - lut[255];
	glBindTexture(GL_LUT_TEXTURE4_DMP, lutids[4]);
	glTexImage1D(GL_LUT_TEXTURE4_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);
}

/* set lighting render state */
static void set_render_state(void)
{
	GLfloat lpos0[] = {-1.f,0.5f,1.f,1.f} ;
	GLfloat lpos1[] = { 1.f,0.5f,1.f,1.f} ;
	GLfloat ld0[] = {0.5f,0.5f,0.5f,1.f} ;
	GLfloat ls0[] = {0.7f,0.7f,0.7f,1.f} ;
	GLfloat ld1[] = {0.5f,0.5f,0.5f,1.f} ;
	GLfloat ls1[] = {0.7f,0.7f,0.7f,1.f} ;

	/* all light setting */
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD1"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputSP"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputFR"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputRB"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputRG"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputRR"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD1"), GL_LIGHT_ENV_CP_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRB"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRG"), GL_LIGHT_ENV_NH_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputRR"), GL_LIGHT_ENV_NH_DMP);

	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD0"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD1"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleSP"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleFR"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRB"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRG"), 1.f);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleRR"), 1.f);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_TRUE);

	/* when anisotropic lighting, distribution function is looked up by
	dot product between projection of half-vector onto tangent plane and
	tangent vector. HW specification require layer configuration setting7
	in this case. another layer configuration setting can not be used. */
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG7_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.bumpMode"), GL_LIGHT_ENV_BUMP_AS_BUMP_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.bumpSelector"), GL_TEXTURE1);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD1"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_FALSE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].twoSideDiffuse"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].geomFactor0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].geomFactor1"), GL_FALSE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].twoSideDiffuse"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), 0);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerRR"), 1);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerRG"), 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerRB"), 3);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD1"), 4);

	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].diffuse"), 1, ld0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular0"), 1, ls0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular1"), 1, ls0);

	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].diffuse"), 1, ld1);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].specular0"), 1, ls1);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].specular1"), 1, ls1);

    /* light position settings */
	GLfloat m[16];

	mat4_t modelview = mat4_t::lookAt(0.f, 0.f, 1.5f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	vec4_t lpos = modelview * vec4_t(lpos0);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].position"), 1, &lpos.x);
	lpos = modelview * vec4_t(lpos1);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[1].position"), 1, &lpos.x);

    /* set projection matrix */
    mat4_t proj = mat4_t::frustum(-0.01f, 0.01f, -0.01f*HEIGHT/WIDTH, 0.01f*HEIGHT/WIDTH, 0.2f, 10.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	/* texture and texture environment settings */
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_TEXTURE_2D);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[1].samplerType"), GL_TEXTURE_2D);

	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_MULT_ADD_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_TEXTURE0, GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_FRAGMENT_SECONDARY_COLOR_DMP);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_CONSTANT, GL_CONSTANT);

    /* lookup table settings */
	set_lut_table();
}

/* load objects */
static void load_objects(void)
{
	bool use_alpha;
	/* load obj file geometry and diffuse texture */
	loadDAT(FILE_APP_ROOT "kimono.dat", &kimono);
	
	/* load texture */
	glGenTextures(1, &diff_tex_name);
	glActiveTexture(GL_TEXTURE0);
	/* stored in the texture collection */
	glBindTexture(GL_TEXTURE_2D, diff_tex_name);
	/* limit texture sampling to the top level only */
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_LOD, 0);
	loadTexture(FILE_APP_ROOT "KIMONO01.tga", GL_TEXTURE_2D, 0, use_alpha, true);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	/* load bump texture */
	glGenTextures(1, &bump_tex_name);
	glActiveTexture(GL_TEXTURE1) ;
	/* stored in the texture collection */
	glBindTexture(GL_TEXTURE_2D, bump_tex_name);
	/* limit texture sampling to the top level only */
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_LOD, 0);
	loadTexture(FILE_APP_ROOT "KIMONOB0.tga", GL_TEXTURE_2D, 0, use_alpha, true);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	/* restore active texture to unit 0 to enable loading of object's diffuse texture */
	glActiveTexture(GL_TEXTURE0);
}

static void unload_objects(void)
{
	unloadDAT(&kimono);
	
	return;
}

static int frame = 0;

int drawframe(void)
{
	glClearColor(0.36f+(frame%100)*0.0064f, 0.42f, 0.5f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	GLfloat m[16] ;

	mat4_t modelview = mat4_t::lookAt(0.f, 0.f, 1.5f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);

	modelview = modelview * mat4_t::rotate(-(float)frame, 0.f, 1.f, 0.f);

	modelview.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	for (int i = 0; i < kimono.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, kimono.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)kimono.obj[i].vtx_offset);
		glBindBuffer(GL_ARRAY_BUFFER, kimono.normVB);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (void*)kimono.obj[i].nrm_offset);
		if (kimono.obj[i].tex_size)
		{
			glBindBuffer(GL_ARRAY_BUFFER, kimono.tangVB);
			glVertexAttribPointer(2, 3, GL_FLOAT, GL_FALSE, 0, (void*)kimono.obj[i].tgt_offset);
			glBindBuffer(GL_ARRAY_BUFFER, kimono.texVB);
			glVertexAttribPointer(3, 3, GL_FLOAT, GL_FALSE, 0, (void*)kimono.obj[i].tex_offset);
			glEnableVertexAttribArray(2);
			glEnableVertexAttribArray(3) ;
			glUniform1i(glGetUniformLocation(pgid, "uTangentEnabled"), GL_TRUE);
		}
		else
		{
			glDisableVertexAttribArray(2);
			glDisableVertexAttribArray(3);
			glUniform1i(glGetUniformLocation(pgid, "uTangentEnabled"), GL_FALSE);
		}

		for (unsigned j = kimono.obj[i].patch_offset; j < kimono.obj[i].patch_size + kimono.obj[i].patch_offset; j++)
		{
			GLfloat ma[4] = {0.f, 0.f, 0.f, 1.f};
			GLfloat md[4] = {0.f, 0.f, 0.f, 1.f};
			GLfloat ms[4] = {0.f, 0.f, 0.f, 1.f};
			for (int col = 0; col < 3; col++)
			{
				ma[col] = kimono.patch[j].ambient[col];
				md[col] = kimono.patch[j].diffuse[col];
				ms[col] = kimono.patch[j].specular[col];
			}
			glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.diffuse"), 1, md);
			glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1, ms);
			glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.ambient"), 1, ma);

			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, kimono.idxVB);
			glDrawElements(GL_TRIANGLES, kimono.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(kimono.patch[j].elm_offset + kimono.obj[i].elm_offset));
		}
	}

	glFinish();

	/* it is possible to save the content of the buffer */
	/*
	char fname[256];
	sprintf(fname, "frame-%04d.tga", f);
	outputImage(WIDTH, HEIGHT, fname);
	*/

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	glClearDepthf(1.f);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);

	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

    /* create program and load & attach vertex shader */
	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);

	/* attach fixed-function fragment shader */
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aNormal");
	glBindAttribLocation(pgid, 2, "aTang");
	glBindAttribLocation(pgid, 3, "aTexCoord");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	/* set program as current one to enable setting its uniforms */
	glUseProgram(pgid);

	/* create and bind texture collection object to capture subsequent modifications in the texture state
	 both lookup table-related one and ordinary (2D, cube) textures */
	glGenTextures(1, &tex_coll);
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, tex_coll);

	load_objects();

	/* enable arrays that will be mapped to position and normal */
	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* set another render state */
		set_render_state();

		/* Enter loop */
		draw_loop();
	}
	unload_objects();
	
	glDeleteTextures(5, lutids);
	glDeleteTextures(1, &diff_tex_name);
	glDeleteTextures(1, &bump_tex_name);
	glDeleteTextures(1, &tex_coll);
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}
