/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"
#include "Gas.h"

#include "Memory.h"

using namespace gputest::common;

namespace gputest{
namespace GasColorOptimal{

/* program id */
GLuint pAccId;
GLuint pPostId;

/* shader id */
GLuint shaders[2];

/* Standard object (triangles, not gaseous object) */
struct gas_data gas;
dat_t plane;

/*=======================================================*/
/* Static function declaration                           */
/*=======================================================*/
static int drawframe(void);

/*=======================================================*/
/* initialization                                        */
/*=======================================================*/

static void initialize(void)
{
	/* Initialize display */
	init_display(DISPLAY_WIDTH, DISPLAY_HEIGHT, APP_NAME, drawframe);

	/* Initialization for gas */
	gas_initialize();
}

static void sceneObject_finalize(void)
{
	unloadDAT(&plane);
	glDeleteBuffers(1, &gas.gasgeo_center_ID);
	glDeleteBuffers(1, &gas.gasgeo_density_ID);
	glDeleteBuffers(1, &gas.gasgeo_tx0_ID);
	glDeleteBuffers(1, &gas.gasgeo_tri_ID);
}

static void sceneObject_initialize(void)
{
	/* Load object (non-gaseous object) */
	loadDAT(FILE_APP_ROOT "planeXY3.dat", &plane);

	/* Load Gaseous object */
	float tr_flame[24] =
	{
		0.0f,	0.00f,	0.0f,	/* 0 */
		0.2f,	0.15f,	0.05f,	/* 1 */
		0.6f,	0.25f,	0.15f,	/* 2 */
		0.9f,	0.35f,	0.2f,	/* 3 */
		0.92f,	0.6f,	0.15f,	/* 4 */
		0.95f,	0.85f,	0.05f,	/* 5 */
		1.0f,	0.95f,	0.0f,	/* 6 */
		1.0f,	1.0f,	1.0f	/* 7 */
	}; 
	defaultGasObject(&gas, tr_flame);

	/* Define the gasesous object geometry */
	/* A simple quad in this sample */
	float cx = 0.0;
	float cy = 0.0;
	float rd = 8.0;
	float _vertex[16]=
	{
		cx - rd, cy - rd, 0.0f, 1.0f,
		cx - rd, cy + rd, 0.0f, 1.0f,
		cx + rd, cy + rd, 0.0f, 1.0f,
		cx + rd, cy - rd, 0.0f, 1.0f
	};

	float density = 1.0;
	float _color[4]= {density, density, density, density };
	
	float t0 = 0.0f, t1 = 1.0f;
	float _texture[8]=
	{
		t0, t0,
		t1, t0,
		t1, t1,
		t0, t1
	};

	glGenBuffers(1, &gas.gasgeo_center_ID);
	glGenBuffers(1, &gas.gasgeo_density_ID);
	glGenBuffers(1, &gas.gasgeo_tx0_ID);
	glGenBuffers(1, &gas.gasgeo_tri_ID);
	
	glBindBuffer(GL_ARRAY_BUFFER, gas.gasgeo_center_ID);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 16, &_vertex[0], GL_STATIC_DRAW);
	gas.gasgeo_size = 6;
	
	glBindBuffer(GL_ARRAY_BUFFER, gas.gasgeo_tx0_ID);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 8, &_texture[0], GL_STATIC_DRAW);

	glBindBuffer(GL_ARRAY_BUFFER, gas.gasgeo_density_ID);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 4, &_color[0], GL_STATIC_DRAW);

	GLushort idxs[6] = {0, 1, 2, 0, 2,  3};
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, gas.gasgeo_tri_ID);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, 6*sizeof(GLushort), idxs, GL_STATIC_DRAW);
}

/*=======================================================*/
/* Initialization  of shaders                            */
/*=======================================================*/
int shader_initialize(void)
{
	shaders[0] = glCreateShader(GL_VERTEX_SHADER);
	shaders[1] = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(2, shaders, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	pAccId = glCreateProgram();
	glAttachShader(pAccId, shaders[0]);
	glAttachShader(pAccId, GL_DMP_FRAGMENT_SHADER_DMP);
	glBindAttribLocation(pAccId, 0, "aPosition");
	glBindAttribLocation(pAccId, 1, "aTexCoord");
	glBindAttribLocation(pAccId, 2, "aColor");

	glLinkProgram(pAccId);
	glValidateProgram(pAccId);
	glUseProgram(pAccId);

	pPostId = glCreateProgram();
	glAttachShader(pPostId, shaders[1]);
	glAttachShader(pPostId, GL_DMP_FRAGMENT_SHADER_DMP);
	glBindAttribLocation(pPostId, 0, "aPosition");
	glBindAttribLocation(pPostId, 1, "aTexCoord");
	glBindAttribLocation(pPostId, 2, "aColor");

	glLinkProgram(pPostId);
	glValidateProgram(pPostId);
	glUseProgram(pPostId);

	return 0;
}

/*=======================================================*/
/* draw object                                           */
/*=======================================================*/
static void drawObject(void)
{
	/* This function draws only one quadrangle. */
	glUseProgram(pAccId);
	glClearColor(0.2f, 0.4f, 0.7f, 1.0f);
	glClearDepthf(1.f);

	/* Set fragment operation mode to GL_FRAGOP_MODE_GL_DMP */
	glUniform1i(glGetUniformLocation(pAccId, "dmp_FragOperation.mode"), GL_FRAGOP_MODE_GL_DMP);

	/*
	 * Setup texture and blending unit
	 * Color of the object is defined as constant color of blending unit 0
	*/
	glUniform1i(glGetUniformLocation(pAccId, "dmp_Texture[0].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pAccId, "dmp_TexEnv[0].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pAccId, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pAccId, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pAccId, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pAccId, "dmp_TexEnv[0].srcRgb"), GL_CONSTANT, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pAccId, "dmp_TexEnv[0].srcAlpha"), GL_CONSTANT, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	
	float almostblack[4] = {0.15f, 0.15f, 0.15f, 0.0f};
	glUniform4fv(glGetUniformLocation(pAccId, "dmp_TexEnv[0].constRgba"), 1, almostblack);

	/*
	 * misc settings
	*/
	glDisable(GL_BLEND);
	glEnable(GL_DEPTH_TEST);
	glDepthMask(GL_TRUE);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	glEnableVertexAttribArray(0);	/* enable position */
	glDisableVertexAttribArray(1);	/* disable texture coordinate */
	glDisableVertexAttribArray(2);	/* disable color */

	for (int i = 0; i < plane.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, plane.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)plane.obj[i].vtx_offset);
		for (unsigned j = plane.obj[i].patch_offset; j < plane.obj[i].patch_size + plane.obj[i].patch_offset; j++)
		{
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, plane.idxVB);
			glDrawElements(GL_TRIANGLES, plane.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(plane.patch[j].elm_offset + plane.obj[i].elm_offset));
		}
	}

	/*
	 * Finalization
	*/
	glDisableVertexAttribArray(0);
}

static int frame = 0;

/*=======================================================*/
/* draw frame                                            */
/*=======================================================*/
static int drawframe(void)
{
	static const int step = 60;

	/* Clear display buffer */
	//glBindFramebuffer(GL_FRAMEBUFFER, DISPLAY_BUFFER);
	cmn_setRenderTarget(0);
	glClearColor(0.2f+(frame%100)*0.008f, 0.4f, 0.7f, 1.0f);
	glClearDepthf(1.f);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	glViewport(0, 0, DISPLAY_WIDTH, DISPLAY_HEIGHT);

	glUseProgram(pAccId);
	GLfloat m[16];
	mat4_t proj = mat4_t::perspective(45.0f, (float)DISPLAY_WIDTH / (float)DISPLAY_HEIGHT, 1.0f, 200.f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pAccId, "uProjection"), 1, GL_FALSE, m);

	mat4_t mv = mat4_t::lookAt(0.f, 0.f, 80.0f, 0.f, 0.f, 0.f, 0.f, 1.f, 0.f);
	mv  = mv * mat4_t::rotate(-90.f, 0.f, 0.f, 1.f);

	{
		/* Draw the Object */
		mat4_t mv_rot = mv * mat4_t::rotate(45.0f, 0.f, 1.f, 0.f);
		mv_rot.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pAccId, "uModelView"), 1, GL_FALSE, m);
		drawObject();
	}

	/* Draw Z buffer for accumulation */
	gas_render_accumulationZ_buffer(drawObject);
	
	{
		/* Gas accumulation pass */
		mv.toFloatArr(m);
		mat4_t mv_tr = mv * mat4_t::translate((float)(frame - 20), 0.f, 0.f);
		mv_tr.toFloatArr(m);
		glUniformMatrix4fv(glGetUniformLocation(pAccId, "uModelView"), 1, GL_FALSE, m);
		gas_accumulation();
	}

	/* Gas shading pass */
	gas_shading();

	/* swap buffer */
	glFinish();
	swap_buffer();

	if (++frame == step)
		frame = 0;

	return !glGetError();
}

/*=======================================================*/
/* main function                                         */
/*=======================================================*/
#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	frame = 0;
	
	/* initialization of frame buffer and gas buffer*/
	initialize();

	/* Initialization of shaders */ 
	if (shader_initialize() >= 0)
	{
		/*standard and gasesous object initializations */
		sceneObject_initialize();

		/* Enter loop */
		draw_loop();
	}
	sceneObject_finalize();
	gas_finalize();
	glDeleteProgram(pAccId);
	glDeleteProgram(pPostId);
	glDeleteShader(shaders[0]);
	glDeleteShader(shaders[1]);

	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}
