/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <math.h>

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "Loader.h"
#include "File.h"

#include <string.h>

#include "Gas.h"

#include "Memory.h"

using namespace gputest::common;

namespace gputest{
namespace GasCessna{

extern GLfloat particleData[];
extern float gasColorTable[6][24];
#define NUM_COLOR_SAMPLE 6

/* program id */
GLuint pGeoId;		/* for normal geometry */
GLuint pAccId;		/* for gas accumulation pass */
GLuint pPostId;		/* for gas shading/post synthesis pass */
GLuint shaders[4] ;

/* Standard object (triangles, not gaseous object) */
dat_t cessna;
struct gas_data gas;

/* Standard object (triangles, not gaseous object) */
GLuint lutids[2];

static struct tagParticeInfo
{
	GLint		cnt;				/* particle count          */
	GLfloat*	position;			/* particle position array */
	GLfloat*	size;				/* particle size array     */
	GLfloat*	density;			/* particle density array  */
	GLushort*	index;
} prtclinfo;


/*=======================================================*/
/* Static function declaration                           */
/*=======================================================*/
static int drawframe(void);

/*=======================================================*/
/* initialization                                        */
/*=======================================================*/
static void initialize(void)
{
	/* Initialize display */
	init_display(DISPLAY_WIDTH, DISPLAY_HEIGHT, APP_NAME, drawframe);

	/* Initialization for gas */
	gas_initialize(cmn_getRenderbufferObjectID(0, GL_DEPTH_STENCIL_ATTACHMENT));
}

/*=======================================================*/
/* finalization                                          */
/*=======================================================*/
void sceneObject_finalize(void)
{
	free(prtclinfo.position);
	free(prtclinfo.size);
	free(prtclinfo.density);
	free(prtclinfo.index);
	unloadDAT(&cessna);
	
	glDeleteBuffers(1, &gas.gasgeo_center_ID);
	glDeleteBuffers(1, &gas.gasgeo_density_ID);
	glDeleteBuffers(1, &gas.gasgeo_size_ID);
	glDeleteBuffers(1, &gas.gasgeo_ind_ID);
}

/*=======================================================*/
/* initialization of the geometry and gaseous object	 */
/*=======================================================*/
static void sceneObject_initialize(void)
{
	/* Load object (non-gaseous object) */
	loadDAT(FILE_APP_ROOT "cessna.dat", &cessna);

	/* Load Gaseous object */
	defaultGasObject(&gas, gasColorTable[0]);

	/* Define the gasesous object geometry */
	/* A simple quad in this sample */
	GLfloat* p = &particleData[1];
	prtclinfo.cnt = (GLuint)particleData[0];
	prtclinfo.position = (GLfloat*)malloc(sizeof(GLfloat) * prtclinfo.cnt * 4);
	prtclinfo.size = (GLfloat*)malloc(sizeof(GLfloat) * prtclinfo.cnt);
	prtclinfo.density = (GLfloat*)malloc(sizeof(GLfloat) * prtclinfo.cnt);
	prtclinfo.index = (GLushort*)malloc(sizeof(GLushort) * prtclinfo.cnt);
	for (int i = 0; i < prtclinfo.cnt; i++)
	{
		prtclinfo.position[i*4+0] = *p++ / 1.0f;		/* x       */
		prtclinfo.position[i*4+1] = *p++ / 1.0f;		/* y       */
		prtclinfo.position[i*4+2] = *p++ / 1.0f;		/* z       */
		prtclinfo.position[i*4+3] = 1.0f;				/* w       */
		prtclinfo.size[i] = *p++ / 1.0f;				/* size    */
		prtclinfo.density[i] = *p++  * 5.f;				/* density */

		/* Adjust particle size based on accumulation buffer size */
		prtclinfo.size[i] *= GAS_ACC_HEIGHT * 2;
		prtclinfo.index[i] = i;
	}
	
	glGenBuffers(1, &gas.gasgeo_center_ID);
	glGenBuffers(1, &gas.gasgeo_density_ID);
	glGenBuffers(1, &gas.gasgeo_size_ID);
	glGenBuffers(1, &gas.gasgeo_ind_ID);
	
	glBindBuffer(GL_ARRAY_BUFFER, gas.gasgeo_center_ID);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 4 * prtclinfo.cnt, &prtclinfo.position[0], GL_STATIC_DRAW);
	gas.gasgeo_size = 6;
	
	glBindBuffer(GL_ARRAY_BUFFER, gas.gasgeo_size_ID);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 1 *prtclinfo.cnt , &prtclinfo.size[0], GL_STATIC_DRAW);

	glBindBuffer(GL_ARRAY_BUFFER, gas.gasgeo_density_ID);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * 1 * prtclinfo.cnt, &prtclinfo.density[0], GL_STATIC_DRAW);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, gas.gasgeo_ind_ID);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLushort) * prtclinfo.cnt, &prtclinfo.index[0], GL_STATIC_DRAW);

	gas.gasgeo_size = prtclinfo.cnt;

	SetGasAccumulationUniform();
	SetGasShadingUniform();
}

/*=======================================================*/
/* Initialization  of material used for the goemrty		 */
/*=======================================================*/
void SetGeoMaterial(GLuint pgid)
{
	GLfloat lut[512];
	int j;

	glGenTextures(2, lutids);
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, lutids[0]);

	memset(lut, 0, sizeof(lut));
	for (j = 0; j < 256; j++)
		lut[j] = pow((float)j/256.0f, 15.f);
	for (j = 0; j < 255; j++)
		lut[j + 256] = lut[j+1] - lut[j];
	lut[255 + 256] = 1.f - lut[255];
	glBindTexture(GL_LUT_TEXTURE0_DMP, lutids[1]);
	glTexImage1D(GL_LUT_TEXTURE0_DMP, 0, GL_LUMINANCEF_DMP, 512, 0, GL_LUMINANCEF_DMP, GL_FLOAT, lut);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentMaterial.samplerD0"), 0);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLighting.enabled"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].enabled"), GL_TRUE);

	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.absLutInputD0"),	GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutInputD0"), GL_LIGHT_ENV_NH_DMP);

	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor0"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].geomFactor1"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].twoSideDiffuse"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledRefl"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.clampHighlights"), GL_FALSE);
	glUniform1f(glGetUniformLocation(pgid, "dmp_LightEnv.lutScaleD0"), 1.f);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.config"), GL_LIGHT_ENV_LAYER_CONFIG0_DMP);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.lutEnabledD0"), GL_TRUE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_LightEnv.bumpMode"), GL_LIGHT_ENV_BUMP_NOT_USED_DMP);


	GLfloat white[] = {1.0f, 1.0f, 1.0f, 1.0f};
	GLfloat black[] = {0.0f, 0.0f, 0.0f, 0.0f};
	GLfloat green[] = {0.2f, 0.7f, 0.4f, 1.0f};

	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular1"), 1, black);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].diffuse"), 1, white);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular0"), 1, white);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentLightSource[0].specular1"), 1, black);

	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.diffuse"), 1, green);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.specular0"), 1,white);
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragmentMaterial.ambient"), 1, black);


	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[0].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineRgb"), GL_ADD);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[0].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_FRAGMENT_PRIMARY_COLOR_DMP, GL_FRAGMENT_SECONDARY_COLOR_DMP, GL_CONSTANT);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Texture[1].samplerType"), GL_FALSE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[1].combineRgb"), GL_REPLACE);
	glUniform1i(glGetUniformLocation(pgid, "dmp_TexEnv[1].combineAlpha"), GL_REPLACE);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[1].srcRgb"), GL_PREVIOUS, GL_PREVIOUS, GL_PREVIOUS);
}

/*=======================================================*/
/* Initialization  of shaders                            */
/*=======================================================*/
int shader_initialize(void)
{
	shaders[0] = glCreateShader(GL_VERTEX_SHADER);
	shaders[1] = glCreateShader(GL_VERTEX_SHADER);
	shaders[2] = glCreateShader(GL_VERTEX_SHADER);
	shaders[3] = glCreateShader(GL_GEOMETRY_SHADER_DMP);
	
	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(4, shaders, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	pGeoId = glCreateProgram();
	glAttachShader(pGeoId, shaders[0]);
	glAttachShader(pGeoId, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pGeoId, 0, "aPosition");
	glBindAttribLocation(pGeoId, 1, "aNormal");

	glLinkProgram(pGeoId);
	glValidateProgram(pGeoId);
	glUseProgram(pGeoId);
	SetGeoMaterial(pGeoId);

	pAccId = glCreateProgram();
	glAttachShader(pAccId, shaders[1]);
	glAttachShader(pAccId, shaders[3]);
	glAttachShader(pAccId, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pAccId, 0, "aPosition");
	glBindAttribLocation(pAccId, 1, "aColor");
	glBindAttribLocation(pAccId, 2, "aPointSize");

	glLinkProgram(pAccId);
	glValidateProgram(pAccId);
	glUseProgram(pAccId);

	pPostId = glCreateProgram();
	glAttachShader(pPostId, shaders[2]);
	glAttachShader(pPostId, GL_DMP_FRAGMENT_SHADER_DMP);
	glBindAttribLocation(pPostId, 0, "aPosition");
	glBindAttribLocation(pPostId, 1, "aTexCoord");
	glBindAttribLocation(pPostId, 2, "aColor");
	glLinkProgram(pPostId);
	glValidateProgram(pPostId);
	glUseProgram(pPostId);

	return 0;
}

/*=======================================================*/
/* draw object                                           */
/*=======================================================*/
static void drawObject(void)
{
	/* This function draws only one quadrangle. */
	glUseProgram(pGeoId);
//	glClearColor(0.2f, 0.4f, 0.7f, 1.0f);
	glClearDepthf(1.f);

	/* Set fragment operation mode to GL_FRAGOP_MODE_GL_DMP */
	glUniform1i(glGetUniformLocation(pGeoId, "dmp_FragOperation.mode"), GL_FRAGOP_MODE_GL_DMP);

	/*
	 * misc settings
	*/
	glDisable(GL_BLEND);
	glEnable(GL_DEPTH_TEST);
	glDepthMask(GL_TRUE);
	glDepthFunc(GL_LEQUAL);

	glDisable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	glEnableVertexAttribArray(0);	/* enable position */
	glEnableVertexAttribArray(1);	/* disable texture coordinate */

	for (int i = 0; i < cessna.obj_num; i++)
	{
		glBindBuffer(GL_ARRAY_BUFFER, cessna.posVB);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, (void*)cessna.obj[i].vtx_offset);
		glBindBuffer(GL_ARRAY_BUFFER, cessna.normVB);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (void*)cessna.obj[i].nrm_offset);
		for (unsigned j = cessna.obj[i].patch_offset; j < cessna.obj[i].patch_size + cessna.obj[i].patch_offset; j++)
		{
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, cessna.idxVB);
			glDrawElements(GL_TRIANGLES, cessna.patch[j].elm_size,
				GL_UNSIGNED_SHORT, (GLvoid*)(cessna.patch[j].elm_offset + cessna.obj[i].elm_offset));
		}
	}

	/*
	 * Finalization
	*/
	glDisableVertexAttribArray(0);
	glDisableVertexAttribArray(1);
}

/*=======================================================*/
/* draw frame                                            */
/*=======================================================*/
static int frame = 0;
static int gas_pattern = 0;
static int drawframe(void)
{
	static const int step = 60;

	/* Clear display buffer */
	glUseProgram(pGeoId);
	//glBindFramebuffer(GL_FRAMEBUFFER, DISPLAY_BUFFER);
	cmn_setRenderTarget(0);
	glClearColor(0.2f+(frame%100)*0.008f, 0.4f, 0.7f, 1.0f);
	glClearDepthf(1.f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glViewport(0, 0, DISPLAY_WIDTH, DISPLAY_HEIGHT);

	glUseProgram(pGeoId);
	GLfloat m[16];
	mat4_t proj = mat4_t::perspective(65.0f, (float)DISPLAY_WIDTH / (float)DISPLAY_HEIGHT, 1.0f, 200.f);
	mat4_t mv = mat4_t::lookAt(0.f, 1.f, 16.0f, 0.f, 0.f, 0.f, -1.f, 0.f, 0.f);
	
	if (frame == 0)
		set_gasColorTable(&gas,gasColorTable[gas_pattern]);
	
	/* Draw the Object */
	glUseProgram(pGeoId);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pGeoId, "uProjection"), 1, GL_FALSE, m);
	mat4_t mv_rot = mv * mat4_t::rotate(-90.0f, 0.f, 1.f, 0.f);
	mv_rot = mv_rot * mat4_t::scale(0.1f, 0.1f, 0.1f);
	mv_rot = mv_rot * mat4_t::translate(100.0f, 0.0f, 0.0f);
	mv_rot.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pGeoId, "uModelView"), 1, GL_FALSE, m);
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, lutids[0]);

	GLfloat lpos0[] = {0.57735f, 0.57735f, 0.57735f, 0.f};
	vec4_t lpos = vec4_t(lpos0);
	glUniform4fv(glGetUniformLocation(pGeoId, "dmp_FragmentLightSource[0].position"), 1, &lpos.x);
	
	drawObject();
	
	/* Gas accumulation pass */
	glUseProgram(pAccId);
	glDisable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pAccId, "uProjection"), 1, GL_FALSE, m);
	mv.toFloatArr(m);
	mat4_t mv_tr = mv * mat4_t::translate(0.0f, -1.0f, (float)(frame - 20));
	mv_tr.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pAccId, "uModelView"), 1, GL_FALSE, m);
	gas_accumulation();
	
	cmn_setRenderTarget(0);
	/* Gas shading pass */
	gas_shading();

	/* swap buffer */
	glFinish();
	swap_buffer();

	if (++frame == step)
	{
		gas_pattern = ++gas_pattern % 3;
		frame = 0;
	}

	return !glGetError();
}

/* ======================================================= */
/* main function                                           */
/* ======================================================= */
#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	frame = 0;
	gas_pattern = 0;
	
	/* initialization of frame buffer and gas buffer*/
	initialize();

	/* Initialization of shaders */ 
	if (shader_initialize() >= 0)
	{
		/*standard and gasesous object initializations */
		sceneObject_initialize();

		/* Enter loop */
		draw_loop();
	}

	sceneObject_finalize();
	gas_finalize();

	glDeleteTextures(2, lutids);
	glDeleteProgram(pGeoId);
	glDeleteProgram(pAccId);
	glDeleteProgram(pPostId);
	glDeleteShader(shaders[0]);
	glDeleteShader(shaders[1]);
	glDeleteShader(shaders[2]);
	glDeleteShader(shaders[3]);

	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}
