/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

/*
 * Comments
 * --------
 *
 * Fog sample. 
 * This sample shows how to setup look table for linear and exponent fog.
 * Rendering display is splitted in 2 using scissor, and is relevant for fog linear.
 * In that case, the left side of the screen shows a quadrangle with fog enabled.
 * The right side shows three different quadrangles, corresponding to 3 regions:
 * i)before fog start, ii)between fog start and fog end and iii)after fog end
 */


#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <math.h>

#include "Memory.h"

#define APP_NAME "Fog"
#define WIDTH 240
#define HEIGHT 400

using namespace gputest::common;

namespace gputest{
namespace Fog{

/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

/* fog color */
GLfloat fogColor[4]={1.0f, 1.0f, 1.0f, 1.0f};

/* fog lut number */
#define FOG_LUT_NUM		(0)

/* fog mode settings */
#define FOG_MODE			(FOG_MODE_LINEAR)	/* Selection of fog mode */
#define FOG_MODE_LINEAR		(0)					/* GL_LINEAR mode in OpenGL ES1.1 compatible */
#define FOG_MODE_EXP		(1)					/* GL_EXP mode in OpenGL ES1.1 compatible */

float fogStart = 10.f;		/* GL_FOG_START in OpenGL ES1.1 compatible settings */
float fogEnd = 30.f;		/* GL_FOG_END in OpenGL ES1.1 compatible settings */
float fogDensity = 0.1f;	/* GL_FOG_DENSITY in OpenGL ES1.1 compatible settings */

float fogAfter = 500.f;

float width = 1000.f;
float height = 0.f;

/* Buffer ids */
GLuint quadFog_indexBuf_ID;
GLuint quadFog_vertBuf_ID;
GLuint quadFog_colBuf_ID;

GLuint quadAll_indexBuf_ID;
GLuint quadAll_vertBuf_ID;
GLuint quadAll_colBuf_ID;

/* Vertex attribute related */
GLfloat vertexFog[12];
const GLushort _quadIndexAll[18] =
{
	0, 1, 2, 0, 2, 3,
	4, 5, 6, 4, 6, 7,
	8, 9, 10, 8, 9, 11
};
const GLushort _quadIndexFog[6] = {0, 1, 2, 0, 2, 3};

GLfloat vertexAll[36];
GLfloat vertexAll_Color[48];
GLfloat vertexFog_Color[48];

GLuint FogLut_ID;
GLuint CollectionLUT_ID;


/* This function returns fog coefficent for input value c */
float fog_coef(float fog_c)
{
	/* GL_LINEAR compatible */
	if (FOG_MODE == FOG_MODE_LINEAR)
	{
		if (fog_c < fogStart)
			return 1.0f;
		else if (fog_c > fogEnd)
			return 0.0f;
		else
			return (fogEnd - fog_c) / (fogEnd - fogStart);
	}
	/* GL_EXP compatible */
	else if (FOG_MODE == FOG_MODE_EXP)
	{
		return exp(-fogDensity * fog_c);
	}
	/* error */
	else
		return -1.0;
}

void initialize_fog_table(mat4_t *MP)
{
	static float Fog_LUT[256];		/* Fog lut contents */
	static float Fog_c[128 + 1];	/* c (distance from origin in eye coordinate */

	int i;
	mat4_t invPM;
	vec4_t v_clip(0.0f, 0.0f, 0.0f, 1.0f);
	vec4_t v_eye;
	
	/* Convert Fog table input value to OpenGL compatible        */
	/* fog coordinate (distance from origin in eye coordinate    */
	/* Here, assumed that projection matrix is OpenGL compatible */
	invPM = MP->invert();
	for (i = 0; i <= 128; i++)
	{
		v_clip.z = ((float)(i * 2 - 128)) / 128;
		v_eye = invPM * v_clip;
		Fog_c[i] = -(v_eye.z / v_eye.w);
	}
	/* If projection matrix is DMPGL2.0 one, USE code following */
	/* commented code instead of one above                      */
	/*
	invPM = MP->invert();
	for (i=0; i<= 128; i++)
	{
		v_clip.z = -((float)i)/128;
		v_eye = invPM * v_clip;
		Fog_c[i] = -(v_eye.z / v_eye.w);
	}
	*/

	for (i = 0; i < 128; i++)
	{
		Fog_LUT[i] = fog_coef(Fog_c[i]);
		Fog_LUT[128 + i]= fog_coef(Fog_c[i + 1]) - fog_coef(Fog_c[i]);
	}
	
	glBindTexture(GL_TEXTURE_COLLECTION_DMP, CollectionLUT_ID);
	glBindTexture(GL_LUT_TEXTURE0_DMP + FOG_LUT_NUM, FogLut_ID);
	glTexImage1D(GL_LUT_TEXTURE0_DMP + FOG_LUT_NUM, 0, GL_LUMINANCEF_DMP, 256, 0, GL_LUMINANCEF_DMP, GL_FLOAT, Fog_LUT);
}

/* load objects */
static void load_objects(void)
{
	vertexAll[0] = -width;
	vertexAll[1] = height;
	vertexAll[2] = 0;

	vertexAll[3] = width;
	vertexAll[4] = height;
	vertexAll[5] = 0;

	vertexAll[6] = width;
	vertexAll[7] = height;
	vertexAll[8] = fogStart;

	vertexAll[9]  = -width;
	vertexAll[10] = height;
	vertexAll[11] = fogStart;

	for (int i = 0; i < 4; i++)
	{
		vertexAll_Color[4 * i + 0] = 1.0f;
		vertexAll_Color[4 * i + 1] = 0.0f;
		vertexAll_Color[4 * i + 2] = 0.0f;
		vertexAll_Color[4 * i + 3] = 1.0f;
	}

	vertexAll[12 + 0] = -width;
	vertexAll[12 + 1] = height;
	vertexAll[12 + 2] = fogStart;

	vertexAll[12 + 3] = width;
	vertexAll[12 + 4] = height;
	vertexAll[12 + 5] = fogStart;

	vertexAll[12 + 6] = width;
	vertexAll[12 + 7] = height;
	vertexAll[12 + 8] = fogEnd;

	vertexAll[12 + 9]  = -width;
	vertexAll[12 + 10] = height;
	vertexAll[12 + 11] = fogEnd;
	for (int i = 0; i < 4; i++)
	{
		vertexAll_Color[16 + 4 * i + 0] = 1.0f;
		vertexAll_Color[16 + 4 * i + 1] = 1.0f;
		vertexAll_Color[16 + 4 * i + 2] = 0.0f;
		vertexAll_Color[16 + 4 * i + 3] = 1.0f;
	}

	vertexAll[24 + 0] = -width;
	vertexAll[24 + 1] = height;
	vertexAll[24 + 2] = fogEnd;

	vertexAll[24 + 3] = width;
	vertexAll[24 + 4] = height;
	vertexAll[24 + 5] = fogEnd;

	vertexAll[24 + 6] = width;
	vertexAll[24 + 7] = height;
	vertexAll[24 + 8] = fogEnd+fogAfter;

	vertexAll[24 + 9]  = -width;
	vertexAll[24 + 10] = height;
	vertexAll[24 + 11] = fogEnd+fogAfter;
	for (int i = 0; i < 4; i++)
	{
		vertexAll_Color[32 + 4 * i + 0] = 1.0f;
		vertexAll_Color[32 + 4 * i + 1] = 0.0f;
		vertexAll_Color[32 + 4 * i + 2] = 1.0f;
		vertexAll_Color[32 + 4 * i + 3] = 1.0f;
	}

	vertexFog[0] = -width;
	vertexFog[1] = height;
	vertexFog[2] = 0;

	vertexFog[3] = width;
	vertexFog[4] = height;
	vertexFog[5] = 0;

	vertexFog[6] = width;
	vertexFog[7] = height;
	vertexFog[8] = fogEnd + fogAfter;

	vertexFog[9]  = -width;
	vertexFog[10] = height;
	vertexFog[11] = fogEnd + fogAfter;

	for (int i = 0; i < 4; i++)
	{
		vertexFog_Color[4 * i + 0] = 0.0f;
		vertexFog_Color[4 * i + 1] = 0.0f;
		vertexFog_Color[4 * i + 2] = 1.0f;
		vertexFog_Color[4 * i + 3] = 1.0f;
	}

	glGenBuffers(1, &quadFog_indexBuf_ID);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, quadFog_indexBuf_ID);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, 6 * sizeof(GLushort), &_quadIndexFog, GL_STATIC_DRAW);
	
	glGenBuffers(1, &quadAll_indexBuf_ID);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, quadAll_indexBuf_ID);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, 18 * sizeof(GLushort), &_quadIndexAll, GL_STATIC_DRAW);
	
	glGenBuffers(1, &quadFog_vertBuf_ID);
	glBindBuffer(GL_ARRAY_BUFFER, quadFog_vertBuf_ID);
	glBufferData(GL_ARRAY_BUFFER, 12 * sizeof(GLfloat), &vertexFog, GL_STATIC_DRAW);
	glGenBuffers(1, &quadFog_colBuf_ID);
	glBindBuffer(GL_ARRAY_BUFFER, quadFog_colBuf_ID);
	glBufferData(GL_ARRAY_BUFFER, 16*sizeof(GLfloat), &vertexFog_Color, GL_STATIC_DRAW);

	glGenBuffers(1, &quadAll_vertBuf_ID);
	glBindBuffer(GL_ARRAY_BUFFER, quadAll_vertBuf_ID);
	glBufferData(GL_ARRAY_BUFFER, 36 * sizeof(GLfloat), &vertexAll, GL_STATIC_DRAW);
	glGenBuffers(1, &quadAll_colBuf_ID);
	glBindBuffer(GL_ARRAY_BUFFER, quadAll_colBuf_ID);
	glBufferData(GL_ARRAY_BUFFER, 48 * sizeof(GLfloat), &vertexAll_Color, GL_STATIC_DRAW);
}

void draw_scene()
{
	glViewport(0, 0, WIDTH, HEIGHT);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	glDisable(GL_CULL_FACE);
	glEnable(GL_SCISSOR_TEST);
	glScissor(0, 0, WIDTH, HEIGHT / 2);
	
	glBindBuffer(GL_ARRAY_BUFFER, quadAll_vertBuf_ID);
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);

	glBindBuffer(GL_ARRAY_BUFFER, quadAll_colBuf_ID);
	glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, 0, 0);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, quadAll_indexBuf_ID);
	glDrawElements(GL_TRIANGLES, 18, GL_UNSIGNED_SHORT, 0);

	glScissor(0, HEIGHT / 2, WIDTH, HEIGHT / 2);
	glUniform1i(glGetUniformLocation(pgid, "dmp_Fog.mode"), GL_FOG);

	glBindBuffer(GL_ARRAY_BUFFER, quadFog_vertBuf_ID);
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);

	glBindBuffer(GL_ARRAY_BUFFER, quadFog_colBuf_ID);
	glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, 0, 0);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, quadFog_indexBuf_ID);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT,0);

	glUniform1i(glGetUniformLocation(pgid, "dmp_Fog.mode"), 0);

	glDisable(GL_SCISSOR_TEST);
}

static int frame = 0;

int drawframe(void)
{
	static const int step = 128;
	
	glClearColor(0.2f+(frame%100)*0.008f, 0.4f, 0.6f, 1.0f);
	/* fog_sampler 0 is exponential fog */
	/* fog_sampler 1 is linear fog */
	glUniform1i(glGetUniformLocation(pgid, "dmp_Fog.sampler"), FOG_LUT_NUM);
	float dx = frame * (1.0f / (float)(step - 1));
	fogColor[0] = 0.2f + dx * (1.0f - 0.2f);
	fogColor[1] = 0.4f + dx * (1.0f - 0.4f);
	fogColor[2] = 0.6f + dx * (1.0f - 0.6f);
	glUniform3fv(glGetUniformLocation(pgid, "dmp_Fog.color"), 1, fogColor);
	draw_scene();
	swap_buffer();

	if (frame == step - 1)
		frame=0;
	else
		frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);
	glClearDepthf(1.f);

	glViewport(0, 0, WIDTH, HEIGHT);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);

	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

	/* create program and load & attach vertex shader */
	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);

	/* attach fixed-function fragment shader */
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aColor");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	/* set program as current one to enable setting its uniforms */
	glUseProgram(pgid);

	load_objects();

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	GLfloat m[16] ;
	mat4_t proj = mat4_t::perspective(65.0f,(float)WIDTH / (float)HEIGHT, 1.0f, 500.0f);
	proj.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	mat4_t mv = mat4_t::lookAt(0.0f, 2.0f, 0.0f, 0.0f, 2.0f, 10.0f, 1.0f, 0.0f, 0.0f);
	mv.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	glGenTextures(1, &CollectionLUT_ID);
	glGenTextures(1, &FogLut_ID);

	initialize_fog_table(&proj);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	glDeleteTextures(1, &CollectionLUT_ID);
	glDeleteTextures(1, &FogLut_ID);
	glDeleteBuffers(1, &quadFog_indexBuf_ID);
	glDeleteBuffers(1, &quadAll_indexBuf_ID);
	glDeleteBuffers(1, &quadFog_vertBuf_ID);
	glDeleteBuffers(1, &quadFog_colBuf_ID);
	glDeleteBuffers(1, &quadAll_vertBuf_ID);
	glDeleteBuffers(1, &quadAll_colBuf_ID);
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}
