/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#ifndef _VEC_ALG_H_
#define _VEC_ALG_H_

#include <assert.h>
#include <GLES2/gl2.h>

namespace gputest{
namespace common{

struct vec4_t ;

struct vec3_t
{
	GLfloat x, y, z ;

	vec3_t () {}
	vec3_t ( const GLfloat& _x, const GLfloat& _y, const GLfloat& _z ) : x(_x), y(_y), z(_z) {} 
	vec3_t ( const vec3_t& _v ) : x(_v.x), y(_v.y), z(_v.z) {}
	vec3_t ( const vec4_t& _v ) ;
	void operator= ( const GLfloat& _f )
	{
		x = y = z = _f ;
	}

	void operator= ( const vec3_t& _v )
	{
		x = _v.x ; y = _v.y ; z = _v.z ;
	}

	void operator= ( const vec4_t& _v ) ;

	GLfloat& operator[] ( int _idx )
	{
		switch ( _idx )
		{
		case 0:
			return x ;
		case 1:
			return y ;
		case 2:
			return z ;
		default:
			assert ( false ) ;
			return z ;
		}
	}
	const GLfloat& operator[] ( int _idx ) const
	{
		switch ( _idx )
		{
		case 0:
			return x ;
		case 1:
			return y ;
		case 2:
			return z ;
		default:
			assert ( false ) ;
			return z ;
		}
	}

	vec3_t operator* ( const GLfloat& _f ) const
	{
		return vec3_t ( x*_f, y*_f, z*_f );
	}

	GLfloat operator* ( const vec3_t& _rhs ) const
	{
		return x*_rhs.x + y*_rhs.y + z*_rhs.z ;
	}

	vec3_t operator^ ( const vec3_t& _rhs ) const
	{
		return vec3_t ( y*_rhs.z - z*_rhs.y, z*_rhs.x - x*_rhs.z, x*_rhs.y - y*_rhs.x );
	}

	vec3_t vmul ( const vec3_t& _rhs ) const
	{
		return vec3_t ( x*_rhs.x, y*_rhs.y, z*_rhs.z );
	}

	vec3_t operator+ ( const vec3_t& _rhs ) const
	{
		return vec3_t ( x+_rhs.x, y+_rhs.y, z+_rhs.z );
	}

	vec3_t operator- ( const vec3_t& _rhs ) const
	{
		return vec3_t ( x-_rhs.x, y-_rhs.y, z-_rhs.z );
	}

	vec3_t operator/ ( const GLfloat& _f ) const
	{
		GLfloat f = 1.f/_f;
		return vec3_t ( x*f, y*f, z*f );
	}

	void operator+= ( const vec3_t& _rhs )
	{
		x += _rhs.x ; y += _rhs.y ; z += _rhs.z ;
	}

	void operator-= ( const vec3_t& _rhs )
	{
		x -= _rhs.x ; y -= _rhs.y ; z -= _rhs.z ;
	}

	void operator*= ( const GLfloat& _f )
	{
		x *= _f ; y *= _f ; z *= _f ;
	}

	void operator/= ( const GLfloat& _f )
	{
		GLfloat f = 1.f/_f ;
		x *= f ; y *= f ; z *= f ;
	}

	vec3_t operator- () const
	{
		return vec3_t(-x,-y,-z) ;
	}

	GLfloat norm() const ;

	void clamp()
	{
		if ( x > 1.f ) x = 1.f ; if ( x < 0.f ) x = 0 ;
		if ( y > 1.f ) y = 1.f ; if ( y < 0.f ) y = 0 ;
		if ( z > 1.f ) z = 1.f ; if ( z < 0.f ) z = 0 ;
	}
} ;

vec3_t operator* ( GLfloat _arg0, const vec3_t& _arg1 ) ;

struct vec4_t
{
	GLfloat x, y, z, w ;

	vec4_t () {}
	vec4_t ( const GLfloat& _x, const GLfloat& _y, const GLfloat& _z, const GLfloat& _w ) :
		x(_x), y(_y), z(_z), w(_w) {} 

	vec4_t ( const GLfloat _v[4] ) :
		x(_v[0]), y(_v[1]), z(_v[2]), w(_v[3]) {} 

	vec4_t ( const vec3_t& _v ) : x(_v.x), y(_v.y), z(_v.z), w(1.f) {}
	vec4_t ( const vec4_t& _v ) : x(_v.x), y(_v.y), z(_v.z), w(_v.w) {}
	void operator= ( const GLfloat& _f )
	{
		x = y = z = w = _f ;
	}

	void operator= ( const vec4_t& _v )
	{
		x = _v.x ; y = _v.y ; z = _v.z ; w = _v.w ;
	}

	void operator= ( const vec3_t& _v )
	{
		x = _v.x ; y = _v.y ; z = _v.z ; w = 1.f ;
	}

	GLfloat& operator[] ( int _idx )
	{
		switch ( _idx )
		{
		case 0:
			return x ;
		case 1:
			return y ;
		case 2:
			return z ;
		case 3:
			return w ;
		default:
			assert ( false ) ;
			return w ;
		}
	}
	const GLfloat& operator[] ( int _idx ) const
	{
		switch ( _idx )
		{
		case 0:
			return x ;
		case 1:
			return y ;
		case 2:
			return z ;
		case 3:
			return w ;
		default:
			assert ( false ) ;
			return w ;
		}
	}

	GLfloat operator* ( const vec4_t& _rhs ) const
	{
		return x*_rhs.x + y*_rhs.y + z*_rhs.z + w*_rhs.w ;
	}

	vec4_t vmul ( const vec4_t& _rhs ) const
	{
		return vec4_t ( x*_rhs.x, y*_rhs.y, z*_rhs.z, w*_rhs.w );
	}

	vec4_t operator+ ( const vec4_t& _rhs ) const
	{
		return vec4_t ( x+_rhs.x, y+_rhs.y, z+_rhs.z, w+_rhs.w );
	}

	vec4_t operator- ( const vec4_t& _rhs ) const
	{
		return vec4_t ( x-_rhs.x, y-_rhs.y, z-_rhs.z, w-_rhs.w );
	}

	vec4_t operator* ( const GLfloat& _f ) const
	{
		return vec4_t ( x*_f, y*_f, z*_f, w*_f );
	}

	vec4_t operator/ ( const GLfloat& _f ) const
	{
		GLfloat f = 1.f/_f;
		return vec4_t ( x*f, y*f, z*f, w*f );
	}

	void operator+= ( const vec4_t& _rhs )
	{
		x += _rhs.x ; y += _rhs.y ; z += _rhs.z ; w += _rhs.w ;
	}

	void operator-= ( const vec4_t& _rhs )
	{
		x -= _rhs.x ; y -= _rhs.y ; z -= _rhs.z ; w -= _rhs.w ;
	}

	void operator*= ( const GLfloat& _f )
	{
		x *= _f ; y *= _f ; z *= _f ; w *= _f ;
	}

	void operator/= ( const GLfloat& _f )
	{
		GLfloat f = 1.f/_f ;
		x *= f ; y *= f ; z *= f ; w *= _f ;
	}

	vec4_t operator- () const
	{
		return vec4_t(-x,-y,-z,-w) ;
	}

	void clamp()
	{
		if ( x > 1.f ) x = 1.f ; if ( x < 0.f ) x = 0 ;
		if ( y > 1.f ) y = 1.f ; if ( y < 0.f ) y = 0 ;
		if ( z > 1.f ) z = 1.f ; if ( z < 0.f ) z = 0 ;
		if ( w > 1.f ) w = 1.f ; if ( w < 0.f ) w = 0 ;
	}

	vec3_t unitdiff ( const vec4_t& _rhs ) const
	{
		vec3_t res ;
		if ( w == 0 && _rhs.w == 0 )
			res = *this - _rhs ;
		else if ( w == 0 )
			res = *this ;
		else if ( _rhs.w == 0 )
			res = -_rhs ;
		else
			res = *this - _rhs ;

		res /= res.norm() ;

		return res ;
	}
} ;

vec4_t operator* ( GLfloat _arg0, const vec4_t& _arg1 ) ;

struct mat4_t
{
	vec4_t m[4] ;

	mat4_t ()
	{
		m[0] = vec4_t(1.f,0,0,0) ; m[1] = vec4_t(0,1.f,0,0) ;
		m[2] = vec4_t(0,0,1.f,0) ; m[3] = vec4_t(0,0,0,1.f) ;
	}

	mat4_t ( const GLfloat* _m )
	{
		m[0] = vec4_t(_m[0],_m[4],_m[8],_m[12]) ; m[1] = vec4_t(_m[1],_m[5],_m[9],_m[13]) ;
		m[2] = vec4_t(_m[2],_m[6],_m[10],_m[14]) ; m[3] = vec4_t(_m[3],_m[7],_m[11],_m[15]) ;
	}

	mat4_t ( const mat4_t& _m )
	{
        m[0] = _m[0] ; m[1] = _m[1] ; m[2] = _m[2] ; m[3] = _m[3] ;
	}

	mat4_t ( const vec4_t& _v0, const vec4_t& _v1, const vec4_t& _v2, const vec4_t& _v3 )
	{
		m[0] = _v0 ; m[1] = _v1 ; m[2] = _v2 ; m[3] = _v3 ;
	}

	void operator= ( const mat4_t& _m )
	{
		 m[0] = _m[0] ; m[1] = _m[1] ; m[2] = _m[2] ; m[3] = _m[3] ;
	}

	mat4_t operator* ( GLfloat _f ) const
	{
		return mat4_t ( m[0]*_f, m[1]*_f, m[2]*_f, m[3]*_f ) ;
	}

	void operator+= ( const mat4_t& _rhs )
	{
		*this = mat4_t ( m[0]+_rhs[0], m[1]+_rhs[1], m[2]+_rhs[2], m[3]+_rhs[3] ) ;
	}

	mat4_t transpose () const
	{
		mat4_t res(*this) ;
		GLfloat t = res[0][1] ; res[0][1] = res[1][0] ; res[1][0] = t ;
		t = res[0][2] ; res[0][2] = res[2][0] ; res[2][0] = t ;
		t = res[0][3] ; res[0][3] = res[3][0] ; res[3][0] = t ;
		t = res[1][2] ; res[1][2] = res[2][1] ; res[2][1] = t ;
		t = res[1][3] ; res[1][3] = res[3][1] ; res[3][1] = t ;
		t = res[2][3] ; res[2][3] = res[3][2] ; res[3][2] = t ;

		return res ;
	}

	mat4_t operator* ( const mat4_t& _rhs ) const
	{
		mat4_t rhs = _rhs.transpose () ;
		mat4_t res ;
		res[0][0] = m[0]*rhs[0] ; res[0][1] = m[0]*rhs[1] ; res[0][2] = m[0]*rhs[2] ; res[0][3] = m[0]*rhs[3] ;
		res[1][0] = m[1]*rhs[0] ; res[1][1] = m[1]*rhs[1] ; res[1][2] = m[1]*rhs[2] ; res[1][3] = m[1]*rhs[3] ;
		res[2][0] = m[2]*rhs[0] ; res[2][1] = m[2]*rhs[1] ; res[2][2] = m[2]*rhs[2] ; res[2][3] = m[2]*rhs[3] ;
		res[3][0] = m[3]*rhs[0] ; res[3][1] = m[3]*rhs[1] ; res[3][2] = m[3]*rhs[2] ; res[3][3] = m[3]*rhs[3] ;

		return res ;
	}

	vec4_t operator* ( const vec4_t& _rhs ) const
	{
		return vec4_t( m[0]*_rhs, m[1]*_rhs, m[2]*_rhs, m[3]*_rhs ) ;
	}

	void operator*= ( const mat4_t& _rhs )
	{
		*this = operator*(_rhs) ;
	}

	vec4_t& operator[] ( int _idx )
	{
		switch ( _idx )
		{
		case 0:
			return m[0] ;
		case 1:
			return m[1] ;
		case 2:
			return m[2] ;
		case 3:
			return m[3] ;
		default:
			assert ( false ) ;
			return m[3] ;
		}
	}
	const vec4_t& operator[] ( int _idx ) const
	{
		switch ( _idx )
		{
		case 0:
			return m[0] ;
		case 1:
			return m[1] ;
		case 2:
			return m[2] ;
		case 3:
			return m[3] ;
		default:
			assert ( false ) ;
			return m[3] ;
		}
	}

	void toFloatArr ( GLfloat _m[16] )
	{
		for ( int i = 0 ; i < 16 ; i++ )
			_m[i] = m[i&0x3][i>>2] ;
	}

	static mat4_t rotate ( GLfloat _th, GLfloat _x, GLfloat _y, GLfloat _z ) ;

	static mat4_t translate ( GLfloat _x, GLfloat _y, GLfloat _z )
	{
		mat4_t res ;
		res[0][3] = _x ; res[1][3] = _y ; res[2][3] = _z ;

		return res ;
	}

	static mat4_t scale ( GLfloat _x, GLfloat _y, GLfloat _z )
	{
		mat4_t res ;
		res[0][0] = _x ; res[1][1] = _y ; res[2][2] = _z ;

		return res ;
	}

	static mat4_t lookAt(GLfloat eyex, GLfloat eyey, GLfloat eyez, GLfloat centerx,
		GLfloat centery, GLfloat centerz, GLfloat upx, GLfloat upy, GLfloat upz) ;

	static mat4_t perspective (GLfloat fovy, GLfloat aspect, GLfloat znear, GLfloat zfar);

	static mat4_t frustum ( GLfloat _l, GLfloat _r, GLfloat _b, GLfloat _t, GLfloat _n, GLfloat _f )
	{
		mat4_t res ;
		res[0][0] = 2.f*_n/(_r-_l) ; res[1][1] = 2.f*_n/(_t-_b) ;
		res[2][2] = (_f+_n)/(_n-_f) ; res[3][3] = 0 ;
		res[0][2] = (_r+_l)/(_r-_l) ; res[1][2] = (_t+_b)/(_t-_b) ;
		res[2][3] = 2.f*_f*_n/(_n-_f) ; res[3][2] = -1.f ;

		return res ;
	}

	static mat4_t ortho ( GLfloat _l, GLfloat _r, GLfloat _b, GLfloat _t, GLfloat _n, GLfloat _f )
	{
		mat4_t res ;
		res[0][0] = 2.f/(_r-_l) ; res[1][1] = 2.f/(_t-_b) ; res[2][2] = 2.f/(_n-_f) ;
		res[0][3] = (_r+_l)/(_l-_r) ; res[1][3] = (_t+_b)/(_b-_t) ; res[2][3] = (_f+_n)/(_n-_f) ;

		return res ;
	}

	GLfloat minor ( int _i, int _j ) const ;

	mat4_t invert () const
	{
		mat4_t res ;

		GLfloat rdet = 1.f/(m[0][0]*minor(0,0) + m[0][1]*minor(0,1) + m[0][2] * minor(0,2) * m[0][3] * minor(0,3)) ;

		for ( int i = 0 ; i < 4 ; i++ )
			for ( int j = 0 ; j < 4 ; j++ )
				res[i][j] = minor(j,i)*rdet ;

		//mat4_t test = res.transpose() * (*this) ;

		return res ;
	}


} ; 

struct mat3_t
{
	vec3_t m[3] ;

	mat3_t ()
	{
		m[0] = vec4_t(1.f,0,0,0) ; m[1] = vec4_t(0,1.f,0,0) ;
		m[2] = vec4_t(0,0,1.f,0) ;
	}

	mat3_t ( const mat3_t& _m )
	{
		m[0] = _m[0] ; m[1] = _m[1] ; m[2] = _m[2] ;
	}
	mat3_t ( const mat4_t& _m )
	{
		m[0] = _m[0] ; m[1] = _m[1] ; m[2] = _m[2] ;
	}
	mat3_t ( const vec3_t& _v0, const vec3_t& _v1, const vec3_t& _v2 )
	{
		m[0] = _v0 ; m[1] = _v1 ; m[2] = _v2 ;
	}

	void operator= ( const mat3_t& _m )
	{
		 m[0] = _m[0] ; m[1] = _m[1] ; m[2] = _m[2] ;
	}

	mat3_t transpose () const
	{
		mat3_t res(*this) ;
		GLfloat t = res[0][1] ; res[0][1] = res[1][0] ; res[1][0] = t ;
		t = res[0][2] ; res[0][2] = res[2][0] ; res[2][0] = t ;
		t = res[1][2] ; res[1][2] = res[2][1] ; res[2][1] = t ;

		return res ;
	}

	GLfloat minor ( int _i, int _j ) const
	{
		int i0 = _i == 0 ? 1 : 0 ;
		int i1 = _i == 2 ? 1 : 2 ;
		int j0 = _j == 0 ? 1 : 0 ;
		int j1 = _j == 2 ? 1 : 2 ;

		GLfloat res = m[i0][j0]*m[i1][j1]-m[i0][j1]*m[i1][j0] ;
		if ( (_i+_j)&1 ) res = -res ;

		return res ;
	}

	GLfloat det () const
	{
		return (m[0][0]*minor(0,0) + m[0][1]*minor(0,1) + m[0][2] * minor(0,2)) ;
	}

	mat3_t invert () const
	{
		mat3_t res ;

		GLfloat rdet = 1.f/(m[0][0]*minor(0,0) + m[0][1]*minor(0,1) + m[0][2] * minor(0,2)) ;

		for ( int i = 0 ; i < 3 ; i++ )
			for ( int j = 0 ; j < 3 ; j++ )
				res[i][j] = minor(j,i)*rdet ;

		return res ;
	}

	mat3_t operator* ( const mat3_t& _rhs ) const
	{
		mat3_t rhs = _rhs.transpose () ;
		mat3_t res ;
		res[0][0] = m[0]*rhs[0] ; res[0][1] = m[0]*rhs[1] ; res[0][2] = m[0]*rhs[2] ;
		res[1][0] = m[1]*rhs[0] ; res[1][1] = m[1]*rhs[1] ; res[1][2] = m[1]*rhs[2] ;
		res[2][0] = m[2]*rhs[0] ; res[2][1] = m[2]*rhs[1] ; res[2][2] = m[2]*rhs[2] ;

		return res ;
	}

	vec3_t operator* ( const vec3_t& _rhs ) const
	{
		return vec3_t( m[0]*_rhs, m[1]*_rhs, m[2]*_rhs ) ;
	}

	void operator*= ( const mat3_t& _rhs )
	{
		*this = operator*(_rhs) ;
	}

	vec3_t& operator[] ( int _idx )
	{
		switch ( _idx )
		{
		case 0:
			return m[0] ;
		case 1:
			return m[1] ;
		case 2:
			return m[2] ;
		default:
			assert ( false ) ;
			return m[2] ;
		}
	}
	const vec3_t& operator[] ( int _idx ) const
	{
		switch ( _idx )
		{
		case 0:
			return m[0] ;
		case 1:
			return m[1] ;
		case 2:
			return m[2] ;
		default:
			assert ( false ) ;
			return m[2] ;
		}
	}

	mat3_t operator* ( GLfloat _f ) const
	{
		return mat3_t ( m[0]*_f, m[1]*_f, m[2]*_f ) ;
	}

	void operator+= ( const mat3_t& _rhs )
	{
		*this = mat3_t ( m[0]+_rhs[0], m[1]+_rhs[1], m[2]+_rhs[2] ) ;
	}

} ; 

}
}
#endif
