/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */
#include <math.h>

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>
#include "Util.h"
#include "Vecalg.h"
#include "Tga.h"

#include "Memory.h"

namespace gputest{
namespace common{

void loadTexture(const char * _name, unsigned _target, int _level, bool& _useAlpha, bool _is2D, int *_width, unsigned _forceformat)
{
	unsigned char* pixels;
	unsigned width, height, format, type, orientation;
	
	(void)_useAlpha;
	pixels = (unsigned char*)dmpLoadTGA(_name, &width, &height, &format, &type, &orientation);
	if (!pixels)
		return;
	
	switch (format)
	{
		case IMAGE_RGB:		format = GL_RGB;	break;
		case IMAGE_RGBA:	format = GL_RGBA;	break;
		default:
			if (pixels)
				free(pixels);
			return;
	}
	
	/* force to convert format */
	if (_forceformat != 0 && format != _forceformat)
	{
		unsigned i;
		unsigned char* newpixels = 0;
		
		switch (format)
		{
			case GL_RGB:
				switch (_forceformat)
				{
					case GL_RGBA:
						newpixels = (unsigned char*)malloc(width * height * 4);
						if (!newpixels)
						{
							free(pixels);
							return;
						}
						for (i = 0; i < width * height; i++)
						{
							newpixels[i * 4 + 0] = pixels[i * 3 + 0];
							newpixels[i * 4 + 1] = pixels[i * 3 + 1];
							newpixels[i * 4 + 2] = pixels[i * 3 + 2];
							newpixels[i * 4 + 3] = 0xff;
						}
						break;
					case GL_RGBA4:
						newpixels = (unsigned char*)malloc(width * height * 2);
						if (!newpixels)
						{
							free(pixels);
							return;
						}
						for (i = 0; i < width * height; i++)
						{
							newpixels[i * 2 + 0] = (pixels[i * 3 + 2] & 0xf0) | 0xf;
							newpixels[i * 2 + 1] = (pixels[i * 3 + 0] & 0xf0) | (pixels[i * 3 + 1] >> 4);
						}
						_forceformat = GL_RGBA;
						type = GL_UNSIGNED_SHORT_4_4_4_4;
						break;
					case GL_RGB5_A1:
						newpixels = (unsigned char*)malloc(width * height * 2);
						if (!newpixels)
						{
							free(pixels);
							return;
						}
						for (i = 0; i < width * height; i++)
						{
							unsigned short pixel = ((pixels[i * 3 + 0] >> 3) << 11) | ((pixels[i * 3 + 1] >> 3) << 6) | ((pixels[i * 3 + 2] >> 3) << 1) | 1;
							newpixels[i * 2 + 0] = pixel & 0xff;
							newpixels[i * 2 + 1] = pixel >> 8;
						}
						_forceformat = GL_RGBA;
						type = GL_UNSIGNED_SHORT_5_5_5_1;
						break;
					case GL_RGB565:
						newpixels = (unsigned char*)malloc(width * height * 2);
						if (!newpixels)
						{
							free(pixels);
							return;
						}
						for (i = 0; i < width * height; i++)
						{
							unsigned short pixel = ((pixels[i * 3 + 0] >> 3) << 11) | ((pixels[i * 3 + 1] >> 2) << 5) | ((pixels[i * 3 + 2] >> 3) << 0);
							newpixels[i * 2 + 0] = pixel & 0xff;
							newpixels[i * 2 + 1] = pixel >> 8;
						}
						_forceformat = GL_RGB;
						type = GL_UNSIGNED_SHORT_5_6_5;
						break;
				}
				break;
			case GL_RGBA:
				switch (_forceformat)
				{
					case GL_RGB:
						newpixels = (unsigned char*)malloc(width * height * 3);
						if (!newpixels)
						{
							free(pixels);
							return;
						}
						for (i = 0; i < width * height; i++)
						{
							newpixels[i * 3 + 0] = pixels[i * 4 + 0];
							newpixels[i * 3 + 1] = pixels[i * 4 + 1];
							newpixels[i * 3 + 2] = pixels[i * 4 + 2];
						}
						break;
				}
				break;
		}
		
		if (newpixels)
		{
			format = _forceformat;
			free(pixels);
			pixels = newpixels;
		}
	}
	
	switch (type)
	{
		case IMAGE_UNSIGNED_BYTE:	type = GL_UNSIGNED_BYTE;	break;
		default:
			break;
	}
	
	if (_is2D)
		glTexImage2D(_target, _level, format, width, height, 0, format, type, pixels);
	else
		glTexImage1D(_target, _level, format, width, 0, format, type, pixels);
	
	if (pixels)
		free(pixels);
	
	if (_width)
		(*_width) = width;
}

float z_schlick(float r, float t, bool normalize)
{
	float ret ;
	ret = 1.f + (r - 1.f) * t * t ;
	ret *= ret ;
	if (!ret)
		return 1000.f ;
	if (normalize)
		ret = r * r / ret ;
	else
		ret = r / ret ;
	return ret ;
}

float a_schlick(float p, float w, bool normalize)
{
	float ret ;
	w *= w ;
	ret = p * p * (1.f - w) + w ;
	if (!ret)
		return 1000.f ;
	if (normalize)
		ret = p * sqrt(1.f / ret) ;
	else
		ret = sqrt(p / ret) ;
	return ret ;
}

float gaussian(float _c, float m)
{
	GLfloat a = acos(_c) ;
	return exp( - (a * a) / (m * m)) ;
}

float beckmann(float _c, float m)
{
	float tan2 ;
	if (!_c)
		return 0.f ;
	tan2 = (_c * _c - 1.f) / (_c * _c) ;
	return exp(tan2 / (m * m)) ;
}

float beckmann2(float _c, float m)
{
	if (!_c)
		return 0.f ;
	GLfloat tan2 ;
	GLfloat cos2 = _c * _c;
	tan2 = (cos2 - 1.f) / cos2 ;
	return exp(tan2 / (m * m)) / (cos2 * cos2) ;
}

float r_fresnel(float _c, float _m, float _r0, float _r1 )
{
	// we get Schlick's approximation of the Fresnel formula
	return _r1 + (_r0 - _r1) * pow(1.f - _c, _m) ;
}

float nk_fresnel(float _c, float _n, float _k)
{
	// we get Schlick's approximation of the Fresnel formula
	GLfloat r1, r0, m ;
	r1 = ((_n-1) * (_n-1) + _k * _k) / ((_n+1)*(_n+1) + _k*_k) ;
	r0 = 1.f ;
	m  = 5.0f ;
	return r_fresnel(_c, m, r0, r1) ;
}


}
}

