/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     Tga.cpp

  Copyright (C)2009 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 7298 $
 *---------------------------------------------------------------------------*/

/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */
#include <string.h>
#include <cstdlib>
#include "Tga.h"
#include "Memory.h"

#define _TGA_HEADER_SIZE    18

namespace gputest{
namespace common{

/*
 * TGA image header
 */
typedef struct {
    unsigned char   id_len;
    unsigned char   colormap_type;
    unsigned char   image_type;
    unsigned char   padding0;

    unsigned short  colormap_first;
    unsigned short  colormap_len;
    unsigned char   colormap_entry;
    unsigned char   padding1;

    unsigned short  x, y;
    unsigned short  width, height;
    unsigned char   depth;
    unsigned char   descriptor;

    /* */

    unsigned int    bpp;
} _TGA_HEADER;

static void _setTGAHeader(
    unsigned char data[_TGA_HEADER_SIZE],
    _TGA_HEADER *tga)
{
    tga->id_len         = data[0];
    tga->colormap_type  = data[1];
    tga->image_type     = data[2];

    tga->colormap_first = data[4] * 256 + data[3];
    tga->colormap_len   = data[6] * 256 + data[5];
    tga->colormap_entry = data[7];

    tga->x              = data[9] * 256 + data[8];
    tga->y              = data[11] * 256 + data[10];
    tga->width          = data[13] * 256 + data[12];
    tga->height         = data[15] * 256 + data[14];
    tga->depth          = data[16];
    tga->descriptor     = data[17];

    tga->bpp = tga->depth / 8;
}

#if 0
static void _getTGAHeader(
    unsigned char data[_TGA_HEADER_SIZE],
    _TGA_HEADER *tga)
{
    data[0] = tga->id_len;
    data[1] = tga->colormap_type;
    data[2] = tga->image_type;

    data[3] = tga->colormap_first & 0xff;
    data[4] = (tga->colormap_first & 0xff00) >> 8;
    data[5] = tga->colormap_len & 0xff;
    data[6] = (tga->colormap_len & 0xff00) >> 8;
    data[7] = tga->colormap_entry;

    data[8] = tga->x & 0xff;
    data[9] = (tga->x & 0xff00) >> 8;
    data[10] = tga->y & 0xff;
    data[11] = (tga->y & 0xff00) >> 8;
    data[12] = tga->width & 0xff;
    data[13] = (tga->width & 0xff00) >> 8;
    data[14] = tga->height & 0xff;
    data[15] = (tga->height & 0xff00) >> 8;
    data[16] = tga->depth;
    data[17] = tga->descriptor;
}
#endif

static unsigned int _LoadTGAScanline(
    _TGA_HEADER *tga,
    unsigned char *dst, 
    unsigned char *src)
{
    int x;
    unsigned int l;

    l = 0;
    for(x = 0; x < tga->width; x++) {
        if (tga->bpp == 3) {
            dst[0] = src[2];
            dst[1] = src[1];
            dst[2] = src[0];
        } else { 
            /* 32BIT */
            dst[0] = src[2];
            dst[1] = src[1];
            dst[2] = src[0];
            dst[3] = src[3];
        }
        src += tga->bpp;
        dst += tga->bpp;
        l += tga->bpp;
    }

    return l;
}

static unsigned int _LoadTGAScanlineRLE(
    _TGA_HEADER *tga,
    unsigned char *dst, 
    unsigned char *src)
{
    int x;
    unsigned int l;
    int errFlag;


    l = 0;
    x = 0;
    errFlag = 0;
    while(x < tga->width && !errFlag) {
        int count = (*src & 0x7f) + 1;
        if (*src & 0x80) {
            int i;

            src++;
            l++;
            for(i = 0; i < count; i++) {
                if (x >= tga->width) {
                    errFlag = 1;
                    break;
                }
                if (tga->bpp == 3) {
                    dst[0] = src[2];
                    dst[1] = src[1];
                    dst[2] = src[0];
                } else { 
                    /* 32BIT */
                    dst[0] = src[2];
                    dst[1] = src[1];
                    dst[2] = src[0];
                    dst[3] = src[3];
                }
                dst += tga->bpp;
                x++;
            }
            src += tga->bpp;
            l += tga->bpp;
        } else {
            int i;

            src++;
            l++;
            for (i = 0; i < count; i++) {
                if (x >= tga->width) {
                    errFlag = 1;
                    break;
                }
                if (tga->bpp == 3) {
                    dst[0] = src[2];
                    dst[1] = src[1];
                    dst[2] = src[0];
                } else { 
                    /* 32BIT */
                    dst[0] = src[2];
                    dst[1] = src[1];
                    dst[2] = src[0];
                    dst[3] = src[3];
                }
                src += tga->bpp;
                l += tga->bpp;
                dst += tga->bpp;
                x++;
            }
        }
    }

    if (errFlag) {
        return 0;
    }

    return l;
}

void *dmpLoadTGA(
    const char *filename,
    unsigned int *width,
    unsigned int *height,
    unsigned int *format,
    unsigned int *type,
    unsigned int *orientation)
{
    unsigned char *header = NULL;
    unsigned char *pixels = NULL;
    long long int filesize = 0;
    _TGA_HEADER tga;
    unsigned int _rle = 0;
    unsigned int _upper = 0;
    int y;
    unsigned char *src, *srchead, *dst;
    nn::fs::FileReader file;
    
    src = srchead = dst = NULL;
    
    /* init */
    if (width) {
        *width = 0;     /* unknown */
    }
    if (height) {
        *height = 0;    /* unknown */
    }
    if (format) {
        *format = IMAGE_UNKNOWN;
    }
    if (type) {
        *type = IMAGE_UNSIGNED_BYTE;
    }
    if (orientation) {
        *orientation = IMAGE_LOWER_LEFT;
    }

    /* open */
    wchar_t* wfilename = static_cast<wchar_t*>(malloc( (strlen(filename)+1)*2 ));
    (void)std::mbstowcs( wfilename, filename, strlen(filename)+1 );
    file.Initialize(wfilename);

    filesize = file.GetSize();
    if (filesize < _TGA_HEADER_SIZE) {
        /* ERROR */
        free(wfilename);
        return NULL;
    }

    header = static_cast<unsigned char*>(malloc(filesize));
    file.Read(header, _TGA_HEADER_SIZE);

	memset(&tga, 0, sizeof(tga));
    _setTGAHeader(header, &tga);
    
    free(header);
    free(wfilename);

    _upper = tga.descriptor & 0x20 ? 1 : 0;

    if (tga.depth == 24 && (tga.descriptor & 0x0f) == 0x00) {
        /* 24BIT */
    } else if (tga.depth == 32 && (tga.descriptor & 0x0f) == 0x08) {
        /* 32BIT */
    } else if (tga.depth == 32 && (tga.descriptor & 0x0f) == 0x00) {
        /* 32BIT */
    } else {
        /* ERROR */
        file.Finalize();
        return NULL;
    }

    if (tga.colormap_type == 0x00 && tga.image_type == 0x02) {
        /* TRUE-COLOR IMAGE */
        _rle = 0;
    } else if (tga.colormap_type == 0x00 && tga.image_type == 0x0a) {
        /* RLE TRUE-COLOR IMAGE */
        _rle = 1;
    } else {
        /* ERROR */
        file.Finalize();
        return NULL;
    }


    if ((pixels = static_cast<unsigned char*>(malloc(tga.width * tga.height * tga.bpp))) == NULL) {
        /* ERROR */
        file.Finalize();
        return NULL;
    }

    if ((srchead = static_cast<unsigned char *>(malloc(filesize - _TGA_HEADER_SIZE))) == NULL)
    {
        file.Finalize();
        return NULL;
    }
    
    src = srchead;
    file.Read(src, filesize - _TGA_HEADER_SIZE);
    for(y = 0; y < tga.height; y++) {
        unsigned int l;

        dst = pixels + tga.width * tga.bpp * 
                (_upper ? tga.height - y - 1 : y);

        if (!_rle) {
            if ((l = _LoadTGAScanline(&tga, dst, src)) == 0) {
                /* ERROR */
                file.Finalize();
                free(srchead);
                free(pixels);
                return NULL;
            }
        } else {
            if ((l = _LoadTGAScanlineRLE(&tga, dst, src)) == 0) {
                /* ERROR */
                file.Finalize();
                free(srchead);
                free(pixels);
                return NULL;
            }
        }
        src += l;
    }

    file.Finalize();
    free(srchead);

    if (width) {
        *width = tga.width;
    }
    if (height) {
        *height = tga.height;
    }
    if (format) {
        if (tga.depth == 24) {
            *format = IMAGE_RGB;
        } else {
            *format = IMAGE_RGBA;
        }
    }

    return (void *)pixels;
}

#if 0
static int _cmpPixel(
    _TGA_HEADER *tga,
    unsigned char *p1, 
    unsigned char *p2)
{
    int cmp = 1;
    
    if (p1[0] == p2[0] && p1[1] == p2[1] && p1[2] == p2[2]) {
        if (tga->depth == 32) {
            if (p1[3] == p2[3]) {
                cmp = 0;
            }
        } else {
            cmp = 0;
        }
    }

    return cmp;
}

static int _cmpLine(
    _TGA_HEADER *tga,
    unsigned char *data, 
    int maxSize,
    int maxHit)
{
    int i;
    int count = 0;

    for(i = 1; i < maxSize; i++) {
        if (!_cmpPixel(tga, data, data + tga->bpp * i)) {
            count++;
            if (count == maxHit) {
                break;
            }
        } else {
            break;
        }
    }

    return count;
}

static int _uncmpLine(
    _TGA_HEADER *tga,
    unsigned char *data, 
    int maxSize,
    int maxHit)
{
    int i;
    int count = 0;
    
    for(i = 1; i < maxSize; i++) {
        if (_cmpPixel(tga, data + tga->bpp * (i - 1), data + tga->bpp * i)) {
            count++;
            if (count > maxHit + 1) {
                break;
            }
        } else {
            if (count - 1 >= 0) {
                count--;
            }
            break;
        }   
    }

    if (count > maxHit) {
        count = maxHit;
    }

    return count;
}

static unsigned int _SaveTGAScanlineRLE(
    _TGA_HEADER *tga,
    unsigned char *data,
    unsigned char *fbuf)
{
    /* fbuf is buffer to write file, return value is the incremented size. */
    int x;
    unsigned char *p1;
    unsigned ret = 0;

    p1 = data;
    x = 0;
    while(x < tga->width) {
        int mode;
        int count;
        unsigned char code;

        if (x == tga->width - 1) {
            mode = 1;
            count = 0;
        } else {
            mode = 0;
            count = _cmpLine(tga, p1, tga->width - x, 127);

            if (count == 0) {
                mode = 1;
                count = _uncmpLine(tga, p1, tga->width - x, 127);
            }
        }

        code = (unsigned char)(count + (mode ? 0x00 : 0x80));
        memcpy(&fbuf[ret], &code, 1);
        ret += 1;

        if (!mode) {
            memcpy(&fbuf[ret], p1, tga->bpp);
            ret += tga->bpp;
        } else {
            memcpy(&fbuf[ret], p1, tga->bpp * (count + 1));
            ret += tga->bpp * (count + 1);
        }

        p1 += (count + 1) * tga->bpp;
        x += count + 1;
    }

    return ret;
}
#endif

}
}

