/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

/*
 * Comments
 * --------
 *
 * User clipping plane (fragment). 
 * This sample displays a cube with lower left vertex (-10,-10,-10) and upper right at (10,10,10)
 * Clipping plane is enabled and rotating around the z axis
 * Clipping plane is defined by the equation -1.x +0.y+ 0.z +5 = 0
 */


#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include "Memory.h"

#define APP_NAME "ClipEquation"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace ClipEquation{


/* program id */
GLuint pgid;

/* shader id */
GLuint shid;

GLuint _quadindexID;
GLuint _quadvertID;
GLuint _quadcolID;

/* load objects */
static void load_objects(void)
{
	const float vertex[24] =
	{
		-10.0f, -10.0f, -10.0f,
		-10.0f,  10.0f, -10.0f,
		 10.0f,  10.0f, -10.0f,
		 10.0f, -10.0f, -10.0f,
		-10.0f, -10.0f, 10.0f,
		-10.0f,  10.0f, 10.0f,
		 10.0f,  10.0f, 10.0f,
		 10.0f, -10.0f, 10.0f
	};
	
	const float color[32] =
	{
		1.0f, 0.0f, 0.0f, 1.0f,
		0.0f, 1.0f, 0.0f, 1.0f,
		0.0f, 0.0f, 1.0f, 1.0f,
		1.0f, 1.0f, 1.0f, 1.0f,
		1.0f, 0.0f, 0.0f, 1.0f,
		0.0f, 1.0f, 0.0f, 1.0f,
		0.0f, 0.0f, 1.0f, 1.0f,
		1.0f, 1.0f, 1.0f, 1.0f
	};
	
	const GLushort _quadIndex[36] =
	{
		0, 1, 2,	/* back face 1 */
		0, 2, 3,	/* back face 2 */
		4, 5, 6,	/* front face 1 */
		4, 6, 7,	/* front face 2 */
		4, 0, 3,	/* bottom face 1 */
		4, 3, 7,	/* bottom face 2 */
		5, 1, 2,	/* top face 1 */
		5, 2, 6,	/* top face 2 */
		4, 5, 1,	/* left face 1 */
		4, 1, 0,	/* left face 2 */
		7, 6, 2,	/* right face 1 */
		7, 2, 3,	/* right face 2 */
	};

	glGenBuffers(1, &_quadindexID) ;
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _quadindexID);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, 36 * sizeof(GLushort), &_quadIndex, GL_STATIC_DRAW);
	
	glGenBuffers(1, &_quadvertID);
	glBindBuffer(GL_ARRAY_BUFFER, _quadvertID);
	glBufferData(GL_ARRAY_BUFFER, 24 * sizeof(GLfloat), &vertex, GL_STATIC_DRAW);

	glGenBuffers(1, &_quadcolID);
	glBindBuffer(GL_ARRAY_BUFFER, _quadcolID);
	glBufferData(GL_ARRAY_BUFFER, 32 * sizeof(GLfloat), &color, GL_STATIC_DRAW);
}

static int frame = 0;

int drawframe(void)
{
	glViewport(0, 0, WIDTH, HEIGHT);
	glClearColor(0.2f+(frame%100)*0.008f, 0.4f, 0.6f, 1.0f);
	glClear ( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT ) ;
	
	/* simple disable culling to see properly clipped and non clipped faces clipped by clipping plane */
	glDisable(GL_CULL_FACE);
	
	GLfloat m[16];
	mat4_t mp = mat4_t::perspective(50.0f, (float)WIDTH / (float)HEIGHT, 1.0f, 2000.0f);
	mp.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uProjection"), 1, GL_FALSE, m);

	mat4_t mv = mat4_t::lookAt(0.0f, 0.0f, 50.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	/* rotation of the object */
	/* mv = mv * mat4_t::rotate(f, 1.0f, 1.0f, 1.0f); */
	mv.toFloatArr(m);
	glUniformMatrix4fv(glGetUniformLocation(pgid, "uModelView"), 1, GL_FALSE, m);

	/* rotation of the clipping plane */
	mv = mv * mat4_t::rotate((float)frame, 0.0f, 0.0f, 1.0f);
	vec4_t eq(1.0f, 0.0f, 0.0f, -5.0f);
	vec4_t p = vec4_t(eq);
	mat4_t mmp(mp);
	mmp[2] = (mp[2] + mp[3]) * -0.5f;
	p = (mmp * mv).invert().transpose() * p;
	
	glUniform4fv(glGetUniformLocation(pgid, "dmp_FragOperation.clippingPlane"), 1, &p[0]);
	glUniform1i(glGetUniformLocation(pgid, "dmp_FragOperation.enableClippingPlane"), GL_TRUE);

	glBindBuffer(GL_ARRAY_BUFFER, _quadvertID);
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);

	glBindBuffer(GL_ARRAY_BUFFER, _quadcolID);
	glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, 0, 0);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, _quadindexID);
	glDrawElements(GL_TRIANGLES, 36, GL_UNSIGNED_SHORT, 0);

	swap_buffer();

	frame++;

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame = 0;
	
	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);
	glClearDepthf(1.f);

	glViewport(0, 0, WIDTH, HEIGHT);

	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);

	glEnable(GL_CULL_FACE);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);

    /* create program and load & attach vertex shader */
	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);

    /* attach fixed-function fragment shader */
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation(pgid, 0, "aPosition");
	glBindAttribLocation(pgid, 1, "aColor");

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	/* set program as current one to enable setting its uniforms */
	glUseProgram(pgid);

	load_objects();

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	return 0;
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	
	glDeleteBuffers(1, &_quadindexID);
	glDeleteBuffers(1, &_quadvertID);
	glDeleteBuffers(1, &_quadcolID);
	glDeleteProgram(pgid);
	glDeleteShader(shid);
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}
}
}
