/*
 *------------------------------------------------------------
 * Copyright(c) 2009-2010 by Digital Media Professionals Inc.
 * All rights reserved.
 *------------------------------------------------------------
 * This source code is the confidential and proprietary
 * of Digital Media Professionals Inc.
 *------------------------------------------------------------
 */

#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "Display.h"
#include "Util.h"
#include "Vecalg.h"
#include "File.h"

#include <string.h>
#include <math.h>

#include "Memory.h"

#define APP_NAME "AlphaTest"
#define WIDTH 240
#define HEIGHT 400

#define DMP_PI	(3.1415926f)
#define REV_PI	(1.0f/DMP_PI)

using namespace gputest::common;

namespace gputest{
namespace AlphaTest{

/* buffer id */
static GLuint array_buffer_id = 0;
static GLuint element_array_buffer_id = 0;

/* program id */
static GLuint pgid = 0;

/* shader id */
static GLuint shid = 0;

static GLfloat coords[] = {
	-0.16f, -0.08f, 0.f, 1.f,
	 0.16f, -0.08f, 0.f, 1.f,
	 0.16f,  0.08f, 0.f, 1.f,
	-0.16f,  0.08f, 0.f, 1.f
};

static GLfloat color[] = {
	1.f, 0.f, 0.f, 0.5f,
	0.f, 1.f, 0.f, 0.5f,
	0.f, 0.f, 1.f, 0.5f,
	1.f, 1.f, 1.f, 0.5f
};

static GLushort idxs[] = {
	0, 1, 2, 3
};


/* generate simple object */
static void load_objects(void)
{
	glGenBuffers(1, &array_buffer_id);
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBufferData(GL_ARRAY_BUFFER, sizeof(coords) + sizeof(color), 0, GL_STATIC_DRAW);
	glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(coords), coords);
	glBufferSubData(GL_ARRAY_BUFFER, sizeof(coords), sizeof(color), color);
	
	glGenBuffers(1, &element_array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(idxs), idxs, GL_STATIC_DRAW);
}

static unsigned frame_num = 0;

static int drawframe(void)
{
	glClearColor(0.1f , 0.008f * (frame_num % 100), 0.8f, 1.f );
	glClearDepthf( 1.f );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	
	glBindBuffer(GL_ARRAY_BUFFER, array_buffer_id);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, element_array_buffer_id);

	glEnableVertexAttribArray(0);
	glEnableVertexAttribArray(1);

	glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0) ;
	glVertexAttribPointer(1, 4, GL_FLOAT, GL_FALSE, 0, (GLvoid*)sizeof(coords));

	//
	// draw back ground
	//
	glUniform1i( glGetUniformLocation( pgid, "dmp_FragOperation.enableAlphaTest" ), GL_FALSE );

	glVertexAttrib4f( 1, 1.f, 1.f, 0.f, 1.f );
	glDisableVertexAttribArray( 1 );

	int i, j;
	float x = - 0.8f;
	float y = - 0.85f;
	
	mat4_t r = mat4_t::rotate((GLfloat)frame_num, 0 , 0, 1);
	for ( i = 0; i < 8; i++ ) {
		for ( j = 0; j < 5; j++ ) {
			GLfloat mv[16];
			mat4_t m = mat4_t::translate(x, y, 0.f) * r;
			m.toFloatArr(mv);
			glUniformMatrix4fv( glGetUniformLocation( pgid, "uModelView" ), 1, GL_FALSE, mv );
			glDrawElements( GL_TRIANGLE_FAN, 4, GL_UNSIGNED_SHORT, 0 );
			x += 0.4f;
		}
		x = - 0.8f;
		y += 0.24f;
	}

	//
	// draw test polygons
	//
	unsigned int func[] =
	{
		GL_NEVER,	
		GL_ALWAYS,
		GL_EQUAL,
		GL_NOTEQUAL,
		GL_LESS,
		GL_LEQUAL,
		GL_GREATER,
		GL_GEQUAL,
	};
	GLfloat aref[] = {
		0.f, 0.25f, 0.5f, 0.75f, 1.f,
	};

	glUniform1i( glGetUniformLocation( pgid, "dmp_FragOperation.enableAlphaTest" ), GL_TRUE );
	glEnableVertexAttribArray( 1 );

	x = - 0.8f;
	y = - 0.85f;
	for ( i = 0; i < 8; i++ ) {
		for ( j = 0; j < 5; j++ ) {
			glUniform1f( glGetUniformLocation( pgid, "dmp_FragOperation.alphaRefValue" ), aref[j] );
			glUniform1i( glGetUniformLocation( pgid, "dmp_FragOperation.alphaTestFunc" ), func[i] );
			GLfloat mv[16];
			mat4_t m = mat4_t::translate(x, y, 0.f) * r;
			m.toFloatArr(mv);
			glUniformMatrix4fv( glGetUniformLocation( pgid, "uModelView" ), 1, GL_FALSE, mv );
			glDrawElements( GL_TRIANGLE_FAN, 4, GL_UNSIGNED_SHORT, 0 );
			x += 0.4f;
		}
		x = - 0.8f;
		y += 0.24f;
	}

	glFinish();

	swap_buffer();

	frame_num++;

	glUniform1i( glGetUniformLocation( pgid, "dmp_FragOperation.enableAlphaTest" ), GL_FALSE );
	glBindBuffer(GL_ARRAY_BUFFER, 0);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	return !glGetError();
}

/* initialization */
static int initialize(void)
{
	frame_num = 0;

	/* Initialize display */
	init_display(WIDTH, HEIGHT, APP_NAME, drawframe);

	pgid = glCreateProgram();
	shid = glCreateShader(GL_VERTEX_SHADER);

	int fsize;
	unsigned char* binary = ReadFile(FILE_APP_ROOT "shader.shbin", &fsize);
	if (!binary)
		return -1;
	
	glShaderBinary(1, &shid, GL_PLATFORM_BINARY_DMP, binary, fsize);
	free(binary);

	glAttachShader(pgid, shid);
	glAttachShader(pgid, GL_DMP_FRAGMENT_SHADER_DMP);

	glBindAttribLocation( pgid, 0, "aPosition" );
	glBindAttribLocation( pgid, 1, "aColor" );

	glLinkProgram(pgid);
	glValidateProgram(pgid);
	glUseProgram(pgid);

	GLfloat proj[16], mv[16];
	mat4_t m = mat4_t::ortho(-1.f, 1.f, -1.f, 1.f, -1.f, 1.f);
	m.toFloatArr(proj);
	glUniformMatrix4fv( glGetUniformLocation( pgid, "uProjection" ), 1, GL_FALSE, proj );

	mat4_t m2;
	m2.toFloatArr(mv);
	glUniformMatrix4fv( glGetUniformLocation( pgid, "uModelView" ), 1, GL_FALSE, mv );

	glViewport( 0, 0, WIDTH, HEIGHT );

	load_objects();

	glEnable( GL_CULL_FACE );
	glFrontFace( GL_CCW );

	glEnable( GL_BLEND );
	/*
	glBlendEquationSeparate( GL_FUNC_ADD, GL_FUNC_ADD );
	glBlendFuncSeparate( GL_ONE, GL_ZERO, GL_ONE, GL_ZERO );	
	*/
	glBlendFunc( GL_ONE, GL_ZERO );
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcRgb"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);
	glUniform3i(glGetUniformLocation(pgid, "dmp_TexEnv[0].srcAlpha"), GL_PRIMARY_COLOR, GL_PRIMARY_COLOR, GL_PRIMARY_COLOR);

	return 0;
}

/* finalization */
static void finalize()
{
	if (array_buffer_id) {
		glDeleteBuffers(1, &array_buffer_id);
		array_buffer_id = 0;
	}
	if (element_array_buffer_id) {
		glDeleteBuffers(1, &element_array_buffer_id);
		element_array_buffer_id = 0;
	}
	if (shid) {
		glDeleteShader(shid);
		shid = 0;
	}
	if (pgid) {
		glDeleteProgram(pgid);
		pgid = 0;
	}
}

#ifdef _NO_OS
int main(int argc, char* argv[])
#else
int sample_main(void)
#endif
{
	/* initialization */
	if (initialize() >= 0)
	{
		/* Enter loop */
		draw_loop();
	}
	/* finalization */
	finalize();
	
	/* shutdown_display */
	shutdown_display();

	return 0;
}

}
}

