/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     y2r_manager.h

  Copyright (C)2009 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 20433 $
 *---------------------------------------------------------------------------*/

#ifndef Y2R_MANAGER_H_
#define Y2R_MANAGER_H_

#include <nn/camera.h>
#include <nn/fnd.h>
#include <nn/y2r.h>
#include <nn/gx.h>
#include <nn/math.h>
#include <string.h>

namespace camera_exp {

    //==============================================================================
    /** Y2r̉摜TCYύXAPI
     */
    void SetY2rSize(
        size_t imageWidth,
        size_t imageHeight,
        nn::y2r::StandardCoefficient coefficient,
        nn::y2r::OutputFormat format,
        nn::y2r::BlockAlignment alignment );

    //==============================================================================
    /** Y2RNX : static
     */
    class Y2rManager_cl
    {
    public:
        enum
        {
            e_EVENT_END,

            e_EVENT_NUM
        };

        // eNX`摜̍őTCY (obt@̃TCYŊmۂ)
        static const s32 s_MaxY2rTextureWidth  = 640;
        static const s32 s_MaxY2rTextureHeight = 640;

    public:
        Y2rManager_cl(){}

    public:
        /** 
         *  @param [in] heap          : obt@蓖ėp̃q[v 
         *  @param [in] imageWidth    : 摜̕ 
         *  @param [in] imageHeight   : 摜̍
         *  @param [in] textureWidth  : eNX`̕  (摜̕ɍ킹邱)
         *  @param [in] textureHeight : eNX`̍(摜̍ȏł邱)
         *  @param [in] isCenter      : eNX`摜傫ƂƂRGBϊʂeNX`obt@̐^ɒuǂ 
         *  @param [in] format        : RGBϊʂ̃tH[}bg
         */
        static void Initialize( 
            nn::fnd::ExpHeap & heap,
            size_t imageWidth, size_t imageHeight,
            size_t textureWidth, size_t textureHeight,
            bool   isCenter,
            nn::y2r::StandardCoefficient coefficient,
            nn::y2r::OutputFormat format,
            nn::y2r::BlockAlignment alignment )
        {
            SetY2rSize( imageWidth, imageHeight, coefficient, format, alignment );

            size_t rgbBytes    = nn::y2r::GetOutputFormatBytes( format );
            m_Coefficient      = coefficient;
            m_Format           = format;
            m_Alignment        = alignment;
            m_ImageWidth       = imageWidth;
            m_ImageHeight      = imageHeight;
            m_TextureWidth     = textureWidth;
            m_TextureHeight    = textureHeight;
            m_SendTransferUnit = nn::camera::GetLineBytes( imageWidth );
            m_SendSize         = nn::camera::GetFrameBytes( imageWidth, imageHeight );
            m_RecvTransferUnit = imageWidth * rgbBytes * 8;
            m_RecvSize         = imageWidth * imageHeight * rgbBytes;
            m_RecvOffset       = (isCenter) ? ( ((textureHeight - imageHeight) / 2) * textureWidth * rgbBytes ) : 0;
            m_WorkIndex        = 0;
            m_TextureScale     = nn::math::VEC3( 1.0f, 1.0f, 1.0f );

            for( s32 i=0; i < 2; i++ )
            {
                for( s32 j=0; j < 2; j++ )
                {
                    // eNX`̃TCYŊmۂ
                    // (摜TCYeNX`TCY傫Ƃꍇ)
                    mpa_RgbBuffers[ i ][ j ] 
                        = reinterpret_cast<u8*>( heap.Allocate( s_MaxY2rTextureWidth * s_MaxY2rTextureHeight * rgbBytes, 64 ) );
                    memset( mpa_RgbBuffers[ i ][ j ], 0, s_MaxY2rTextureWidth * s_MaxY2rTextureHeight * rgbBytes );
                }
                m_RgbRp[ i ] = 0;
            }

            nn::y2r::GetTransferEndEvent( &ma_Events[ e_EVENT_END ] );
            ma_Events[ e_EVENT_END ].ClearSignal();

            InitializeInternalWork();
        }

        static void Finalize( nn::fnd::ExpHeap & heap )
        {
            FinalizeInternalWork();

            ma_Events[ e_EVENT_END ].Finalize();

            for( s32 i=0; i < 2; i++ )
            {
                for( s32 j=0; j < 2; j++ )
                {
                    heap.Free( mpa_RgbBuffers[ i ][ j ] );
                }
            }
        }

    private:
        /** wb_ɂ͏ȂO[oϐ̏
         */
        static void InitializeInternalWork( void );
        static void FinalizeInternalWork( void );

    public:
        /** TCYύX
         */
        static void SetSize(
            size_t imageWidth,
            size_t imageHeight,
            size_t textureWidth,
            size_t textureHeight,
            bool   isCenter,
            nn::y2r::StandardCoefficient coefficient,
            nn::y2r::OutputFormat format,
            nn::y2r::BlockAlignment alignment )
        {
            SetY2rSize( imageWidth, imageHeight, coefficient, format, alignment );

            // eNX`ύXƂɂ
            // eNX`\|SȂƈ΂ĉfĂ܂
            // |SXP[
            f32 scale          = m_TextureScale.x * static_cast< f32 >( textureWidth ) / static_cast< f32 >( m_TextureWidth );
            m_TextureScale     = nn::math::VEC3( scale, scale, 1.0f );

            size_t rgbBytes    = nn::y2r::GetOutputFormatBytes( format );
            m_Coefficient      = coefficient;
            m_Format           = format;
            m_Alignment        = alignment;
            m_ImageWidth       = imageWidth;
            m_ImageHeight      = imageHeight;
            m_TextureWidth     = textureWidth;
            m_TextureHeight    = textureHeight;
            m_SendTransferUnit = nn::camera::GetLineBytes( imageWidth );
            m_SendSize         = nn::camera::GetFrameBytes( imageWidth, imageHeight );
            m_RecvTransferUnit = imageWidth * rgbBytes * 8;
            m_RecvSize         = imageWidth * imageHeight * rgbBytes;
            m_RecvOffset       = (isCenter) ? ( ((textureHeight - imageHeight) / 2) * textureWidth * rgbBytes ) : 0;
        }

        /** ݂̃TCYݒԂ
         */
        static void GetSize(
            size_t * p_ImageWidth,
            size_t * p_ImageHeight,
            size_t * p_TextureWidth,
            size_t * p_TextureHeight,
            bool   * p_IsCenter,
            nn::y2r::StandardCoefficient *p_Coefficient,
            nn::y2r::OutputFormat   * p_Format,
            nn::y2r::BlockAlignment * p_Alignment )
        {
            *p_ImageWidth    = m_ImageWidth;
            *p_ImageHeight   = m_ImageHeight;
            *p_TextureWidth  = m_TextureWidth;
            *p_TextureHeight = m_TextureHeight;
            *p_IsCenter      = ( m_RecvOffset == 0 ) ? false : true;
            *p_Coefficient   = m_Coefficient;
            *p_Format        = m_Format;
            *p_Alignment     = m_Alignment;
        }

        /** YUVobt@؂ւ
         */
        static void SwapBuffer( void );

        /** YUVobt@؂ւ (CfbNXw)
         */
        static void SwapBuffer( s32 index );

        /** Y2R̓]obt@̃|C^Ԃ
         */
        static u8 * GetRecvBuffer( s32 index );

        /** eNX`obt@̃|C^Ԃ 
         */
        static u8 * GetTextureBuffer( s32 index );

        /** eNX`̃XP[ʂԂ
         */
        static nn::math::VEC3 GetTextureScale( void )
        {
            return m_TextureScale;
        }

        /** DMA]PʂԂ
         */
        static size_t GetSendTransferUnit( void )
        {
            return m_SendTransferUnit;
        }
        static size_t GetRecvTransferUnit( void )
        {
            return m_RecvTransferUnit;
        }

        /** DMA]TCYԂ
         */
        static size_t GetSendSize( void )
        {
            return m_SendSize;
        }
        static size_t GetRecvSize( void )
        {
            return m_RecvSize;
        }

        /** eNX`TCYԂ
         */
        static size_t GetTextureWidth( void )
        {
            return m_TextureWidth;
        }
        static size_t GetTextureHeight( void )
        {
            return m_TextureHeight;
        }

        /** eCxgւ̎QƂԂ
         */
        static nn::os::Event & GetEvent( s32 index )
        {
            return ma_Events[ index ];
        }

        /** eCxgւ̃|C^Ԃ
         */
        static nn::os::Event * GetEventPointer( s32 index )
        {
            return &ma_Events[ index ];
        }

        /** obt@NA
         */
        static void ClearBuffer( void )
        {
            nn::y2r::OutputFormat format = nn::y2r::GetOutputFormat();
            size_t rgbBytes = nn::y2r::GetOutputFormatBytes( format );

            for( s32 i=0; i < 2; i++ )
            {
                for( s32 j=0; j < 2; j++ )
                {
                    memset( mpa_RgbBuffers[ i ][ j ], 0, m_TextureWidth * m_TextureHeight * rgbBytes );
                }
            }
        }

        /** Y2R gpĂ邱ƂL
         */
        static void SetWorkIndex( s32 index );

        /** DMAgĕϊJn
         *  @param[in] index       : ǂ̃J(CAM1:0, CAM2:1)
         *  @param[in] p_YuvBuffer : obt@
         */
        static void StartDmaConversion( s32 index, u8 * p_YuvBuffer );

        /** DMAgϊ̏I
         */
        static void EndDmaConversion( void );

    private:
        // Y2RϊW
        static nn::y2r::StandardCoefficient m_Coefficient;

        // Y2RtH[}bg
        static nn::y2r::OutputFormat   m_Format;
        static nn::y2r::BlockAlignment m_Alignment;

        // DMA]TCY
        static size_t m_SendTransferUnit;
        static size_t m_SendSize;
        static size_t m_RecvTransferUnit;
        static size_t m_RecvSize;

        // DMA]̃ItZbg
        // eNX`TCY摜TCY傫\̂
        // Y2R ̓]ƂĂ͒[̗̈XLbvobt@̈ʒuԂ悤ɂ
        static s32    m_RecvOffset;

        // 摜TCY
        static size_t m_ImageWidth;
        static size_t m_ImageHeight;

        // eNX`TCY
        static size_t m_TextureWidth;
        static size_t m_TextureHeight;

        // eNX`\摜̃XP[
        static nn::math::VEC3   m_TextureScale;

        // ]̔ԍ
        // 2̃J狣̂łǂ]Ă邩LĂ
        static s32    m_WorkIndex;

        // RGBobt@
        static u8   * mpa_RgbBuffers[ 2 ][ 2 ];   // Y2R̃CgGPŨ[hՓ˂Ȃ悤Ƀ_uobt@ɂ
        static s32    m_RgbRp[ 2 ];               // GPŨ[hw(Y2Rϊɐ؂ւ)

        // ݒ˗Cxg
        static nn::os::Event  ma_Events[ e_EVENT_NUM ];

    }; //class Y2rManager_cl

} //namespace camera_exp

#endif //#ifndef Y2R_MANAGER_H_
