/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     key_manager.cpp

  Copyright (C)2009 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 18253 $
 *---------------------------------------------------------------------------*/

#include <nn/hid.h>

#include "key_manager.h"

using namespace camera_exp;

nn::hid::PadReader * KeyManager_cl::mp_PadReader;
nn::hid::PadStatus * KeyManager_cl::mp_PadStatus;
f32 KeyManager_cl::m_AkX;
f32 KeyManager_cl::m_AkY;
f32 KeyManager_cl::m_AkX_Back;
f32 KeyManager_cl::m_AkY_Back;
f32 KeyManager_cl::m_AkThreshold;
u32 KeyManager_cl::m_Trigger;
u32 KeyManager_cl::m_Release;
u32 KeyManager_cl::m_Repeat;
u32 KeyManager_cl::m_Hold;
u32 KeyManager_cl::m_Hold_Back;
u8  KeyManager_cl::ma_RepeatCount[ KeyManager_cl::e_KEY_MAX ];
u8  KeyManager_cl::m_RepeatStart;
u8  KeyManager_cl::m_RepeatInterval;

//==============================================================================
void KeyManager_cl::Read( void )
{
    m_AkX_Back  = m_AkX;
    m_AkY_Back  = m_AkY;
    m_Hold_Back = m_Hold;

    mp_PadReader->ReadLatest( mp_PadStatus );

    m_AkX = mp_PadReader->NormalizeStick( mp_PadStatus->stick.x );
    m_AkY = mp_PadReader->NormalizeStick( mp_PadStatus->stick.y );

    UpdateTrigger();
    UpdateRelease();
    UpdateHold();
    UpdateRepeat();
}

//==============================================================================
bool KeyManager_cl::IsTrigger( u32 key )
{
    return ((m_Trigger & key) != 0);
}

//==============================================================================
bool KeyManager_cl::IsRelease( u32 key )
{
    return ((m_Release & key) != 0);
}

//==============================================================================
bool KeyManager_cl::IsRepeat( u32 key )
{
    return ((m_Repeat & key) != 0);
}

//==============================================================================
bool KeyManager_cl::IsHold( u32 key )
{
    return ((m_Hold & key) != 0);
}

//==============================================================================
void KeyManager_cl::UpdateTrigger( void )
{
    m_Trigger = mp_PadStatus->trigger;
    if( (m_AkX_Back <= m_AkThreshold) && (m_AkX > m_AkThreshold) )
    {
        m_Trigger |= e_AK_RIGHT;
    }
    if( (m_AkX_Back >= -m_AkThreshold) && (m_AkX < -m_AkThreshold) )
    {
        m_Trigger |= e_AK_LEFT;
    }
    if( (m_AkY_Back <= m_AkThreshold) && (m_AkY > m_AkThreshold) )
    {
        m_Trigger |= e_AK_UP;
    }
    if( (m_AkY_Back >= -m_AkThreshold) && (m_AkY < -m_AkThreshold) )
    {
        m_Trigger |= e_AK_DOWN;
    }
}

//==============================================================================
void KeyManager_cl::UpdateRelease( void )
{
    m_Release = mp_PadStatus->release;
    if( (m_AkX <= m_AkThreshold) && (m_AkX_Back > m_AkThreshold) )
    {
        m_Release |= e_AK_RIGHT;
    }
    if( (m_AkX >= -m_AkThreshold) && (m_AkX_Back < -m_AkThreshold) )
    {
        m_Release |= e_AK_LEFT;
    }
    if( (m_AkY <= m_AkThreshold) && (m_AkY_Back > m_AkThreshold) )
    {
        m_Release |= e_AK_UP;
    }
    if( (m_AkY >= -m_AkThreshold) && (m_AkY_Back < -m_AkThreshold) )
    {
        m_Release |= e_AK_DOWN;
    }
}

//==============================================================================
void KeyManager_cl::UpdateHold( void )
{
    m_Hold = mp_PadStatus->hold;
    if( m_AkX > m_AkThreshold )
    {
        m_Hold |= e_AK_RIGHT;
    }
    if( m_AkX < -m_AkThreshold )
    {
        m_Hold |= e_AK_LEFT;
    }
    if( m_AkY > m_AkThreshold )
    {
        m_Hold |= e_AK_UP;
    }
    if( m_AkY < -m_AkThreshold )
    {
        m_Hold |= e_AK_DOWN;
    }
}

//==============================================================================
void KeyManager_cl::UpdateRepeat( void )
{
    m_Repeat = 0;

    // evXÂĂ邩𒲂ׂ
    for( s32 i=0; i < e_KEY_MAX; i++ )
    {
        u32  key  = 1 << i;
        u32  prev = m_Hold_Back & key;
        u32  curr = m_Hold & key;
        
        if( m_Trigger & key )
        {
            ma_RepeatCount[i] = 0;
        }
        if( (curr != 0) && (prev == curr) )
        {
            ma_RepeatCount[i]++;
            
            if( ma_RepeatCount[i] >= (m_RepeatStart + m_RepeatInterval) )
            {
                m_Repeat |= key;
                ma_RepeatCount[i]  = m_RepeatStart;
            }
        }
    }
}

