/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     graphics.cpp

  Copyright (C)2009 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 36187 $
 *---------------------------------------------------------------------------*/

#include <nn.h>

#include "graphics.h"
#include "camera_manager.h"
#include "y2r_manager.h"
#include "key_manager.h"
#include "model_manager.h"

//==============================================================================
// gpt[[N
demo::RenderSystemExt s_RenderSystem;

// 3D/2D؂ւ
s32 s_DisplayMode = 1;

//==============================================================================
//namespace {

    const int MEMORY_SIZE_FCRAM_GX = 0x800000;    // OtBbNXpɊmۂ FCRAM ̃TCY

    /* buffer id */
    GLuint s_ArrayBufferID;
    GLuint s_ElementArrayBufferID;

    /* program id */
    GLuint s_ProgramID;

    /* shader id */
    GLuint s_ShaderID;

    /* q[v */
    uptr s_AddrForGxHeap;

    // eNX`
    GLuint s_Texture[2] = { 0, 0 };

//} // namespace

//==============================================================================
namespace {

    // Lf[^
    nn::camera::StereoCameraCalibrationData * sp_CalData = NULL;

    // ␳Ɏ܂߂邩
    bool s_IsParallax = false;

    // sg킹鋗
    f32  s_Distance = 500.0f;

    // Eɂ炷ǂ
    bool s_IsShiftRight = false;

    // eNX`ƂȂ摜̃tH[}bg
    nn::y2r::OutputFormat   s_Format    = nn::y2r::OUTPUT_RGB_24;
    nn::y2r::BlockAlignment s_Alignment = nn::y2r::BLOCK_8_BY_8;

    // RGB 摜\Ƃ̉摜
    u8      * spa_RgbImage[ 2 ]  = { NULL, NULL };
    size_t  s_RgbTextureWidth   = 1024;
    size_t  s_RgbTextureHeight  = 1024;
    size_t  s_RgbOriginalWidth  = 640;
    size_t  s_RgbOriginalHeight = 480;


    // Jʒu
    camera_exp::ModelManager_cl s_CameraModelManager;

} //namespace

//==============================================================================
void LoadObjects(void);
void DeleteObjects( void );
void ReadyObjects(void);
void SetTextureCombiner(void);
void SetTextureCombinerBlend(void);
void DrawDisplay0Common( nn::camera::CameraSelect currentCamera );
void DrawDisplay0ExtCommon( void );

//==============================================================================
int InitializeGx( nn::fnd::ExpHeap & heap )
{
    s_AddrForGxHeap = reinterpret_cast<uptr>(heap.Allocate(MEMORY_SIZE_FCRAM_GX));
    s_RenderSystem.Initialize(s_AddrForGxHeap, MEMORY_SIZE_FCRAM_GX);

    s_ProgramID = glCreateProgram();
    s_ShaderID = glCreateShader(GL_VERTEX_SHADER);

//    nn::fs::FileReader file(L"rom:/shader.shbin");
    nn::fs::FileReader file(L"rom:/camera/shader.shbin");
    size_t fileSize = file.GetSize();
    void* buf = heap.Allocate(fileSize);
    s32 read = file.Read(buf, fileSize);

    glShaderBinary(1, &s_ShaderID, GL_PLATFORM_BINARY_DMP, buf, read);

    file.Finalize();
    heap.Free(buf);

    glAttachShader(s_ProgramID, s_ShaderID);
    glAttachShader(s_ProgramID, GL_DMP_FRAGMENT_SHADER_DMP);

    glBindAttribLocation(s_ProgramID, 0, "aPosition");
    glBindAttribLocation(s_ProgramID, 1, "aTexCoord");

    glLinkProgram(s_ProgramID);
    glValidateProgram(s_ProgramID);
    glUseProgram(s_ProgramID);

    glGenTextures(2, s_Texture);
//    DEMO_ASSERT_GL_ERROR();

    glBindTexture(GL_TEXTURE_2D, s_Texture[0]);
    glBindTexture(GL_TEXTURE_2D, s_Texture[1]);
    glBindTexture(GL_TEXTURE_2D, 0);

    s_RenderSystem.SetClearColor(NN_GX_DISPLAY0, 0.36f, 0.42f, 0.5f, 1.0f);
    s_RenderSystem.SetClearColor(NN_GX_DISPLAY1, 0.0f, 0.0f, 0.0f, 1.0f);
    s_RenderSystem.SetClearColor(NN_GX_DISPLAY0_EXT, 0.36f, 0.42f, 0.5f, 1.0f);

    glClearDepthf(1.f);

    glEnable(GL_CULL_FACE);
    glFrontFace(GL_CCW);
    glCullFace(GL_BACK);

    LoadObjects();

    // Jʒȕ
    s_CameraModelManager.InitPosition();

    return 0;
}

//==============================================================================
void FinalizeGx( nn::fnd::ExpHeap & heap )
{
    DeleteObjects();

    glDeleteTextures( 2, s_Texture );
    glDetachShader( s_ProgramID, GL_DMP_FRAGMENT_SHADER_DMP );
    glDetachShader( s_ProgramID, s_ShaderID );
    glDeleteShader( s_ShaderID );
    glDeleteProgram( s_ProgramID );

    s_RenderSystem.Finalize();

    heap.Free( reinterpret_cast< void * >( s_AddrForGxHeap ) );
}

//==============================================================================
void LoadObjects(void)
{
    GLfloat coords[] = {
        -1.024f, 1.024f, 0.f, 1.f,
        -1.024f,-1.024f, 0.f, 1.f,
         1.024f, 1.024f, 0.f, 1.f,
         1.024f,-1.024f, 0.f, 1.f
    };
    GLfloat texcoords[] = {
        0.f, 1.f, 0.f,
        0.f, 0.f, 0.f,
        1.f, 1.f, 0.f,
        1.f, 0.f, 0.f,
    };

    GLushort idxs[] = {0, 1, 2, 3};

    glGenBuffers(1, &s_ArrayBufferID);
    glBindBuffer(GL_ARRAY_BUFFER, s_ArrayBufferID);
    glBufferData(GL_ARRAY_BUFFER, sizeof(coords) + sizeof(texcoords), 0, GL_STATIC_DRAW);
    glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(coords), coords);
    glBufferSubData(GL_ARRAY_BUFFER, sizeof(coords), sizeof(texcoords), texcoords);
    
    glGenBuffers(1, &s_ElementArrayBufferID);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, s_ElementArrayBufferID);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(idxs), idxs, GL_STATIC_DRAW);

    glEnableVertexAttribArray(0);
    glEnableVertexAttribArray(1);

    glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0) ;
    glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)sizeof(coords));
}

//==============================================================================
void DeleteObjects( void )
{
    glDeleteBuffers( 1, &s_ElementArrayBufferID );
    glDeleteBuffers( 1, &s_ArrayBufferID );
}

//==============================================================================
void ReadyObjects(void)
{
    glUseProgram(s_ProgramID);
    glBindBuffer(GL_ARRAY_BUFFER, s_ArrayBufferID);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, s_ElementArrayBufferID);

    glEnableVertexAttribArray(0);
    glEnableVertexAttribArray(1);

    glVertexAttribPointer(0, 4, GL_FLOAT, GL_FALSE, 0, 0);
    glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)(16*sizeof(GLfloat)));
}

//==============================================================================
// ɂȂ
void SetTextureCombiner(void)
{
    glUniform4f(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].constRgba"), 0.0f, 0.0f, 0.0f, 1.0f);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].srcRgb"), GL_TEXTURE0, GL_PREVIOUS, GL_PREVIOUS);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].srcAlpha"), GL_CONSTANT, GL_PREVIOUS, GL_PREVIOUS);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].operandAlpha"),GL_SRC_ALPHA , GL_SRC_ALPHA, GL_SRC_ALPHA);
    glUniform1i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].combineRgb"), GL_REPLACE);
    glUniform1i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].combineAlpha"), GL_REPLACE);
    glUniform1f(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].scaleRgb"), 1.0f);
    glUniform1f(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].scaleAlpha"), 1.0f);
}

//==============================================================================
// 
void SetTextureCombinerBlend(void)
{
    glUniform4f(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].constRgba"), 0.0f, 0.0f, 0.0f, 0.5f);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].srcRgb"), GL_TEXTURE0, GL_PREVIOUS, GL_PREVIOUS);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].srcAlpha"), GL_CONSTANT, GL_PRIMARY_COLOR, GL_PREVIOUS);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].operandRgb"), GL_SRC_COLOR, GL_SRC_COLOR, GL_SRC_COLOR);
    glUniform3i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].operandAlpha"), GL_SRC_ALPHA, GL_SRC_ALPHA, GL_SRC_ALPHA);
    glUniform1i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].combineRgb"), GL_REPLACE);
    glUniform1i(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].combineAlpha"), GL_REPLACE);
    glUniform1f(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].scaleRgb"), 1.0f);
    glUniform1f(glGetUniformLocation(s_ProgramID, "dmp_TexEnv[2].scaleAlpha"), 1.0f);
}

//==============================================================================
void SetDrawingStereoCamera(
        bool isDrawRgbImage,
        nn::camera::StereoCameraCalibrationData * p_CalData,
        bool isParallax,
        f32  distance,
        bool isShiftRight,
        nn::y2r::OutputFormat format,
        nn::y2r::BlockAlignment alignment )
{
//    s_IsDrawRgbImage = isDrawRgbImage;
    static_cast<void>( isDrawRgbImage );// warning΍
    sp_CalData       = p_CalData;
    s_IsParallax     = isParallax;
    s_Distance       = distance;
    s_IsShiftRight   = isShiftRight;
    s_Format         = format;
    s_Alignment      = alignment;
} //SetDrawingStereoCamera()

//==============================================================================
/** XeI摜\鏈ʉ
 *  XeIJ摜BMP摜̗ɑΉ邽
 */
void DrawStereoTexture( s32 index, bool isBlend,
                        void   * image,
                        size_t textureWidth,  size_t textureHeight,
                        size_t originalWidth, size_t originalHeight NN_IS_UNUSED_VAR )
{
    GLenum type = GL_UNSIGNED_BYTE;
    switch( s_Format )
    {
    case nn::y2r::OUTPUT_RGB_32:
    case nn::y2r::OUTPUT_RGB_24:
        {
            type = GL_UNSIGNED_BYTE;
        }
        break;
    case nn::y2r::OUTPUT_RGB_16_555:
        {
            type = GL_UNSIGNED_SHORT_5_5_5_1;
        }
        break;
    case nn::y2r::OUTPUT_RGB_16_565:
        {
            type = GL_UNSIGNED_SHORT_5_6_5;
        }
        break;
    }

    glBindTexture(GL_TEXTURE_2D, s_Texture[index]);
    if( s_Alignment == nn::y2r::BLOCK_8_BY_8)
    {
        glTexImage2D(
            GL_TEXTURE_2D | NN_GX_MEM_FCRAM | GL_NO_COPY_FCRAM_DMP,
            0,
            GL_RGB_NATIVE_DMP,
            textureWidth,
            textureHeight,
            0,
            GL_RGB_NATIVE_DMP,
            type,
            image );
    }
    else
    {
        glTexImage2D(
            GL_TEXTURE_2D,
            0,
            GL_RGB,
            textureWidth,
            textureHeight,
            0,
            GL_RGB,
            type,
            image );
    }

    ReadyObjects();
    
    // eNX`RoCiݒ
    glUniform1i(glGetUniformLocation(s_ProgramID, "dmp_Texture[0].samplerType"), GL_TEXTURE_2D);

    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, s_Texture[index]);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

    // ULCD\ȂƂɂ͉摜dˍ킹ĕ\
    // ɂKv邪摜̃At@lύX̂ł͂Ȃ
    // eNX`RoCi̐ݒύX邱ƂőΉ
    if( isBlend )
    {
        SetTextureCombinerBlend();
    }
    else
    {
        SetTextureCombiner();
    }

    nn::math::Matrix44 proj, m;

    // e̎̐ς肷
    // ctȂ̂
    // DISPLAY0_HEIGHT : 400
    // DISPLAY0_WIDTH  : 240
    // ƂȂĂ邱Ƃɒ

    nn::math::MTX44Frustum(
        &proj,
        -0.02f,
         0.02f,
        -0.02f* static_cast<f32>(nn::gx::DISPLAY0_HEIGHT) / static_cast<f32>(nn::gx::DISPLAY0_WIDTH),
         0.02f* static_cast<f32>(nn::gx::DISPLAY0_HEIGHT) / static_cast<f32>(nn::gx::DISPLAY0_WIDTH),
         0.2f,
         10.f);
    nn::math::MTX44Transpose(&m, &proj);
    glUniformMatrix4fv(glGetUniformLocation(s_ProgramID, "uProjection"), 1, GL_FALSE, static_cast<f32*>(m));

    // J(3DfBeĴ. JW[ł͂Ȃ)̈ʒu, 
    // J(̓JW[̂)摜 pixel by pixel ŕ\łʒuɒu
    // 400 * 240 ̍オ (-0.8, -0.48) 
    // jANbsOʂ̍オ (-0.02*400/240, -0.02) , 
    // jANbsOʂJ z  0.2 Ă̂
    // J̃|S z ̈ʒu
    // Opɂ -0.48 : 0.02 = z : 0.2  z = 4.8 ƂȂ
    // ă|S z W 0 ̂ƂJ̈ʒu 4.8 ƂȂ

    nn::math::Matrix34 eye;
    nn::math::Vector3 camPos(0.f, 0.f, 4.8f);
    nn::math::Vector3 camUp(0.f, 1.f, 0.f);
    nn::math::Vector3 target(0.f, 0.f, 0.f);

    nn::math::MTX34LookAt(&eye, &camPos, &camUp, &target);

    // _J𓮂
    nn::math::Matrix34 camMove( s_CameraModelManager.GetTransMatrix34() );
    nn::math::MTX34Mult( &eye, &camMove, &eye );

    nn::math::MTX34  tmp(eye);
    {
        nn::math::MTX34 move;
        nn::math::MTX34Identity( &move );
        
        // eNX`TCYƂɈL΂Ȃ悤ɃXP[OĂ
        // ŏɃXP[O邱Ƃŕ␳⎋ł̂炵ʂ̌vZŃeNX`΂lȂĂ悢
        f32  sizeScaleF = static_cast< f32 >( textureWidth ) / 512.0f;
        nn::math::VEC3 sizeScaleVec = nn::math::VEC3( sizeScaleF, sizeScaleF, 0.0f );
        nn::math::MTX34 sizeScale;
        nn::math::MTX34Scale( &sizeScale, &sizeScaleVec );
        nn::math::MTX34Mult( &move, &sizeScale, &move );

        // XeIJ̃Lu[V(ʒuY␳)
        if( index == 1 )        // 
        {
            // 摜̃TCYɂĂ炷ʂ͈قȂ
            f32 step = 0.004f * static_cast< f32 >( originalWidth ) / 640.0f;

            // Lu[V̂߂̍s擾
            // 摜1sNZ߂ɕKvȈړʂ VGA ̂Ƃ 0.96 / 240.0 = 0.004 (0.96 ͈̔͂ 240 sNZʂ)
            nn::math::MTX34 cal;
            nn::camera::GetStereoCameraCalibrationMatrix( &cal, *sp_CalData, step, s_IsParallax );
            
            // J摜ɈʒuY␳
            nn::math::MTX34Mult( &move, &cal, &move );
        }

        // ʑ̂яoČ邩Ɍ邩͉摜̐̈ړʂɂĈقȂ
        // ł͎w肳ꂽɂʑ̂dȂ悤ɁAs_Distance [mm] ̎߂āA
        // ̕J摜iړ
        if( ( index == 1 ) && ( s_Distance > 0.0f ) )
        {
            // 摜̃TCYɂĂ炷ʂ͈قȂ
            f32 step = 0.004f * static_cast< f32 >( originalWidth ) / 640.0f;

            f32 p = nn::camera::GetParallax( *sp_CalData, s_Distance * 0.001f );

            nn::math::VEC3  transVec( p * step, 0.0f, 0.0f );     // 3Dԏ̈ړʂɕϊ邽߂ 0.004 
            nn::math::MTX34 trans;
            nn::math::MTX34Translate( &trans, &transVec );
            
            nn::math::MTX34Mult( &move, &trans, &move );
        }

        // ^ɂƉE摜̐^ƍ摜̉E[r邱ƂɂȂ̂
        // 摜̍\悤ɉEɂ炷
        if( s_IsShiftRight
            && ( textureWidth > nn::gx::DISPLAY0_HEIGHT ) )
        {
            f32 right = static_cast< f32 >( ( textureWidth - nn::gx::DISPLAY0_HEIGHT ) / 2 ); 
            nn::math::MTX34 transRight;
            nn::math::VEC3  transRightVec( right * 0.004f, 0.0f, 0.0f );
            nn::math::MTX34Translate( &transRight, &transRightVec );
            nn::math::MTX34Mult( &move, &transRight, &move );
        }

        // Wn{̂Ƃ 90 xɉĂ悤ɒ`Ă̂ŉE 90 x
        nn::math::MTX34 rot90;
        nn::math::MTX34RotXYZDeg( &rot90, 0.0f, 0.0f, -90.0f );
        nn::math::MTX34Mult( &move, &rot90, &move );

        nn::math::MTX34Mult( &tmp, &tmp, &move );
    }
    nn::math::MTX44 mv(tmp);
    
    nn::math::MTX44Transpose(&mv, &mv);   // hCo̒̌`ɍ킹邽߂ɓ]u
    glUniformMatrix4fv(glGetUniformLocation(s_ProgramID, "uModelView"), 1, GL_FALSE, static_cast<f32*>(mv));

    if( isBlend )
    {
        // At@uh
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

        // Zrɂ
        glDisable(GL_DEPTH_TEST);
    }
    else
    {
        glDisable(GL_BLEND);
    }

    glDrawElements(GL_TRIANGLE_STRIP, 4, GL_UNSIGNED_SHORT, 0);
} //DrawStereoTexture

//==============================================================================
void DrawStereoCamera( s32 index, bool isBlend )
{
    DrawStereoTexture(
        index,
        isBlend,
        camera_exp::Y2rManager_cl::GetTextureBuffer( index ),
        camera_exp::Y2rManager_cl::GetTextureWidth(),
        camera_exp::Y2rManager_cl::GetTextureHeight(),
        camera_exp::OcamManager_cl::GetCameraManager( index ).GetWidth(),
        camera_exp::OcamManager_cl::GetCameraManager( index ).GetHeight() );
} //DrawStereoCamera

//==============================================================================
void DrawRgbImage( s32 index, bool isBlend )
{
    DrawStereoTexture(
        index,
        isBlend,
        spa_RgbImage[ index ],
        s_RgbTextureWidth,
        s_RgbTextureHeight,
        s_RgbOriginalWidth,
        s_RgbOriginalHeight );
}

//==============================================================================
void DrawDisplay0StereoCommon( nn::camera::CameraSelect currentCamera )
{
    // ULCD\邩ǂ̐؂ւ
    if(    !camera_exp::KeyManager_cl::IsHold( camera_exp::KeyManager_cl::e_BUTTON_A )
             && !camera_exp::KeyManager_cl::IsHold( camera_exp::KeyManager_cl::e_BUTTON_B )
             && !camera_exp::KeyManager_cl::IsHold( camera_exp::KeyManager_cl::e_BUTTON_Y )
             && !camera_exp::KeyManager_cl::IsHold( camera_exp::KeyManager_cl::e_BUTTON_X )
             && !camera_exp::KeyManager_cl::IsHold( camera_exp::KeyManager_cl::e_BUTTON_R )
             && !camera_exp::KeyManager_cl::IsHold( camera_exp::KeyManager_cl::e_BUTTON_L ) )
    {
        if( camera_exp::KeyManager_cl::IsTrigger( camera_exp::KeyManager_cl::e_AK_DOWN ) )
        {
            if( ++s_DisplayMode >= 4 )
            {
                s_DisplayMode = 0;
            }
        }
        if( camera_exp::KeyManager_cl::IsTrigger( camera_exp::KeyManager_cl::e_AK_UP ) )
        {
            if( --s_DisplayMode < 0 )
            {
                s_DisplayMode = 3;
            }
        }
    }

    if(    ( currentCamera == nn::camera::SELECT_OUT1_OUT2 )
        || ( currentCamera == nn::camera::SELECT_IN1_OUT2 ) )
    {
        switch( s_DisplayMode )
        {
        case 0:
            {
                // Vb^[ON
                if( s_RenderSystem.GetLcdMode() != NN_GX_DISPLAYMODE_STEREO )
                {
                    s_RenderSystem.SetLcdMode( NN_GX_DISPLAYMODE_STEREO );
                }

// TODO ͉ʕ`Kv
            }
            break;

        case 1:
        case 2:
        case 3:
            // Vb^[OFF
            {
                if( s_RenderSystem.GetLcdMode() != NN_GX_DISPLAYMODE_NORMAL )
                {
                    s_RenderSystem.SetLcdMode( NN_GX_DISPLAYMODE_NORMAL );
                }

// TODO ʕ`Kv
            }
            break;
        }
    }
    else
    {
//        s_RenderSystem.SetLcdMode( NN_GX_DISPLAYMODE_NORMAL );
    }

    // ڗp̊G`
    DrawDisplay0Common( currentCamera );

    // Eڗp̊G`
    if( s_RenderSystem.GetLcdMode() == NN_GX_DISPLAYMODE_STEREO )
    {
        DrawDisplay0ExtCommon();
    }

    s_RenderSystem.SwapBuffers();

} //DrawDisplay0StereoCommon()

//==============================================================================
void DrawDisplay0Common( nn::camera::CameraSelect currentCamera )
{
    // 摜
    // STEREO ł OUT2 \
    s_RenderSystem.SetRenderTarget(NN_GX_DISPLAY0);
    glViewport(0, 0, nn::gx::DISPLAY0_WIDTH, nn::gx::DISPLAY0_HEIGHT);
    s_RenderSystem.Clear();

    if( s_RenderSystem.GetLcdMode() == NN_GX_DISPLAYMODE_STEREO )
    {
        DrawStereoCamera( 1, false );      // J摜
    }
    else
    {
        switch( currentCamera )
        {
        case nn::camera::SELECT_OUT1_OUT2:
        case nn::camera::SELECT_IN1_OUT2:
            {
                switch( s_DisplayMode )
                {
                case 1:
                    {
                        // dˍ킹ĕ\
                        DrawStereoCamera( 0, true );
                        DrawStereoCamera( 1, true );
                    }
                    break;

                case 2:
                    {
                        DrawStereoCamera( 0, false );
                    }
                    break;

                case 3:
                    {
                        DrawStereoCamera( 1, false );
                    }
                    break;
                }
            }
            break;

        case nn::camera::SELECT_OUT1:
        case nn::camera::SELECT_IN1:
            {
                DrawStereoCamera( 0, false );
            }
            break;

        case nn::camera::SELECT_OUT2:
            {
                DrawStereoCamera( 1, false );
            }
            break;
        }
    }
    s_RenderSystem.Transfer();
} //DrawDisplay0Common()

//==============================================================================
void DrawDisplay0ExtCommon( void )
{
    // E摜
    // STEREO ł OUT1 \
    s_RenderSystem.SetRenderTarget(NN_GX_DISPLAY0_EXT);
    glViewport(0, 0, nn::gx::DISPLAY0_WIDTH, nn::gx::DISPLAY0_HEIGHT);
    s_RenderSystem.Clear();

    DrawStereoCamera( 0, false );      // EJ摜
    s_RenderSystem.Transfer();
} //DrawDisplay0ExtCommon()

//==============================================================================
void DrawLineFrame( const f32 sx, const f32 sy, const f32 width, const f32 height )
{
    f32 ex = sx + width  - 1.0f;
    f32 ey = sy + height - 1.0f;

    s_RenderSystem.DrawLine( sx, sy, ex, sy );
    s_RenderSystem.DrawLine( sx, sy, sx, ey );
    s_RenderSystem.DrawLine( sx, ey, ex, ey );
    s_RenderSystem.DrawLine( ex, sy, ex, ey );
}

//==============================================================================
void DrawImage(
    GLuint & textureId, const f32 sx, const f32 sy, const f32 width, const f32 height,
    void * p_Image, const s32 imageWidth, const s32 imageHeight )
{
    if( textureId == 0 )
    {
        s_RenderSystem.GenerateTexture(
            GL_TEXTURE_2D | NN_GX_MEM_FCRAM | GL_NO_COPY_FCRAM_DMP,
            GL_RGB_NATIVE_DMP,
            imageWidth,
            imageHeight,
            GL_RGB_NATIVE_DMP,
            GL_UNSIGNED_BYTE,
            p_Image,
            textureId );
    }
    else
    {
        glBindTexture( GL_TEXTURE_2D, textureId );
        glTexImage2D(
            GL_TEXTURE_2D | NN_GX_MEM_FCRAM | GL_NO_COPY_FCRAM_DMP,
            0,
            GL_RGB_NATIVE_DMP,
            imageWidth,
            imageHeight,
            0,
            GL_RGB_NATIVE_DMP,
            GL_UNSIGNED_BYTE,
            p_Image );
    }
    s_RenderSystem.FillTexturedRectangle(
        textureId,
        sx,
        sy,
        width,
        height,
        imageWidth,
        imageHeight,
        imageWidth,
        imageHeight );
} // DrawImage()

