/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     camera_manager.h

  Copyright (C)2009 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 19033 $
 *---------------------------------------------------------------------------*/

#ifndef CAMERA_MANAGER_H_
#define CAMERA_MANAGER_H_

#include <nn/camera.h>
#include <nn/fnd.h>
#include <nn/y2r.h>
#include <string.h>
#include "util.h"

namespace camera_exp {

    //==============================================================================
    /** JI/F̓]TCYύXAPI
     */
    void SetCameraSizeInterface(
            nn::camera::CameraSelect  camera,
            size_t        cameraWidth,
            size_t        cameraHeight,
            size_t        trimWidth,
            size_t        trimHeight );

    //==============================================================================
    /** J̉摜TCYύXAPI
     */
    void SetCameraSize(
            nn::camera::CameraSelect  camera,
            size_t        cameraWidth,
            size_t        cameraHeight,
            size_t        trimWidth,
            size_t        trimHeight );

    void SetCameraSize(
            nn::camera::CameraSelect  camera,
            nn::camera::Size          size,
            size_t        trimWidth,
            size_t        trimHeight );


    //==============================================================================
    /** JNX
     */
    class CameraManager_cl
    {
    public:
        enum
        {
            e_EVENT_DMA = 0,
            e_EVENT_ERROR,
            e_EVENT_VSYNC,

            e_EVENT_NUM
        };
        static const s32 sc_MaxBufferNum = 5;

        // J摜̍őTCY (obt@̃TCYŊmۂ)
        static const s32 s_MaxCameraWidth  = 640;
        static const s32 s_MaxCameraHeight = 480;

    public:
        CameraManager_cl(){}

    public:
        void Initialize(
            nn::camera::Port port,
            nn::fnd::ExpHeap & heap,
            s32    bufferNum,
            size_t cameraWidth, size_t cameraHeight,
            size_t trimWidth, size_t trimHeight )
        {
            SetCameraSize( GetCamera( port ),
                           cameraWidth, cameraHeight,
                           trimWidth, trimHeight );

            m_CsSet.Initialize();
            m_CsDma.Initialize();

            m_Port              = port;
            m_CameraWidth       = cameraWidth;
            m_CameraHeight      = cameraHeight;
            m_TrimWidth         = trimWidth;
            m_TrimHeight        = trimHeight;
            m_TransferUnit      = nn::camera::GetMaxBytes( trimWidth, trimHeight );
            m_OutputSize        = nn::camera::GetFrameBytes( trimWidth, trimHeight );
            m_IsCapturingFull   = false;
            m_TransferUnitFull  = nn::camera::GetMaxBytes( cameraWidth, cameraHeight );
            m_OutputSizeFull    = nn::camera::GetFrameBytes( cameraWidth, cameraHeight );
            m_BufferNum         = bufferNum;
            m_CapturingPos      = 0;
            m_LatestCapturedPos = bufferNum - 1;
            m_ReadingPos        = bufferNum - 1;

            // obt@̊m
            // Ƃ蓾ő̃TCYɂĂ
            s32 bufSize = nn::camera::GetFrameBytes( s_MaxCameraWidth, s_MaxCameraHeight );
            for( s32 i=0; i < bufferNum; i++ )
            {
                mpa_YuvBuffers[ i ]
                    = reinterpret_cast<u8*>( heap.Allocate( bufSize, 64 ) );
                memset( mpa_YuvBuffers[ i ], 0, bufSize );
            }
            mp_YuvBufferFull
                = reinterpret_cast<u8*>( heap.Allocate( bufSize, 64 ) );
            memset( mp_YuvBufferFull, 0, bufSize );

            ma_Events[ e_EVENT_DMA ].Initialize( false );
            GetBufferErrorInterruptEvent( &ma_Events[ e_EVENT_ERROR ], port );
            GetVsyncInterruptEvent( &ma_Events[ e_EVENT_VSYNC ], port );
            ma_Events[ e_EVENT_ERROR ].ClearSignal();
            ma_Events[ e_EVENT_VSYNC ].ClearSignal();
        }

        void Finalize( nn::fnd::ExpHeap & heap )
        {
            // obt@̉
            for( s32 i=0; i < m_BufferNum; i++ )
            {
                heap.Free( mpa_YuvBuffers[ i ] );
            }
            heap.Free( mp_YuvBufferFull );

            ma_Events[ e_EVENT_DMA ].Finalize();        // ZbgŏĂĂJvZXŎ蓮Zbgɂ
            ma_Events[ e_EVENT_ERROR ].Finalize();
            ma_Events[ e_EVENT_VSYNC ].Finalize();

            m_CsSet.Finalize();
            m_CsDma.Finalize();
        }

    public:
        /** YUVobt@؂ւ
         */
        void SwapBuffer( void );

        /** ݍς݂̍ŐṼobt@̃|C^Ԃ
         */
        u8 * GetLatestBuffer( void );

        /** ݒ̃obt@̃|C^Ԃ 
         */
        u8 * GetWorkBuffer( void );

        /** DMA]PʂԂ
         */
        size_t GetTransferUnit( void ) const
        {
            return m_TransferUnit;
        }

        /** DMA]TCYԂ
         */
        size_t GetOutputSize( void ) const
        {
            return m_OutputSize;
        }

        /** eCxgւ̎QƂԂ
         */
        nn::os::Event & GetEvent( s32 index )
        {
            return ma_Events[ index ];
        }

        /** eCxgւ̃|C^Ԃ
         */
        nn::os::Event * GetEventPointer( s32 index )
        {
            return &ma_Events[ index ];
        }

        /** DMAX^[g
         */
        void StartDmaRecv( void );

        /** 摜TCYύX
         */
        void SetSize( size_t cameraWidth, size_t cameraHeight,
                      size_t trimWidth,   size_t trimHeight )
        {
            SetCameraSize( GetCamera( m_Port ),
                           cameraWidth, cameraHeight,
                           trimWidth, trimHeight );

            m_CameraWidth      = cameraWidth;
            m_CameraHeight     = cameraHeight;
            m_TrimWidth        = trimWidth;
            m_TrimHeight       = trimHeight;
            m_TransferUnit     = nn::camera::GetMaxBytes( trimWidth, trimHeight );
            m_OutputSize       = nn::camera::GetFrameBytes( trimWidth, trimHeight );
            m_TransferUnitFull = nn::camera::GetMaxBytes( cameraWidth, cameraHeight );
            m_OutputSizeFull   = nn::camera::GetFrameBytes( cameraWidth, cameraHeight );
        }

        /** 摜TCYύX enum
         */
        void SetSize( nn::camera::Size size,
                      size_t trimWidth, size_t trimHeight )
        {
            SetCameraSize( GetCamera( m_Port ),
                           size,
                           trimWidth, trimHeight );

            m_CameraWidth      = GetCameraWidth( size );
            m_CameraHeight     = GetCameraHeight( size );
            m_TrimWidth        = trimWidth;
            m_TrimHeight       = trimHeight;
            m_TransferUnit     = nn::camera::GetMaxBytes( trimWidth, trimHeight );
            m_OutputSize       = nn::camera::GetFrameBytes( trimWidth, trimHeight );
            m_TransferUnitFull = nn::camera::GetMaxBytes( GetCameraWidth( size ), GetCameraHeight( size ) );
            m_OutputSizeFull   = nn::camera::GetFrameBytes( GetCameraWidth( size ), GetCameraHeight( size ) );
        }

        /** 摜TCYԂ
         */
        size_t GetWidth( void )
        {
            return m_CameraWidth;
        }
        size_t GetHeight( void )
        {
            return m_CameraHeight;
        }

        /** g~OTCYԂ
         */
        size_t GetTrimWidth( void )
        {
            return m_TrimWidth;
        }
        size_t GetTrimHeight( void )
        {
            return m_TrimHeight;
        }

        /** 摜̃tTCYLv`̏sȂ
         *  Lv`~܂Ă邱ƂO
         */
        void ReadyFullSizeCapture( void )
        {
            // Lv`TCYtTCY(ꎞI)ύX
            SetCameraSizeInterface(
                GetCamera( m_Port ),
                m_CameraWidth, m_CameraHeight,
                m_CameraWidth, m_CameraHeight );
            m_IsCapturingFull = true;
        }

        /** 摜̃tTCYLv`߂
         *  Lv`~܂Ă邱ƂO
         */
        void ExitFullSizeCapture( void )
        {
            // 摜TCYg~OTCY(ꎞI)ύX
            SetCameraSizeInterface(
                GetCamera( m_Port ),
                m_CameraWidth, m_CameraHeight,
                m_TrimWidth, m_TrimHeight );
            m_IsCapturingFull = false;
        }

        /** tTCYLv`ǂ擾
         *  (obt@G[ł̕Aǂ΂悢ʂ邽)
         */
        bool IsCapturingFullSize( void )
        {
            return m_IsCapturingFull;
        }

        /** tTCYLv`pDMA]
         *  ReadyFullSizeCapture()Ă΂Ă邱ƂO
         */
        void StartDmaRecvFullSize( void );

        /** tTCYLv`̃obt@Ԃ
         */
        u8 * GetFullSizeCaptureBuffer( void )
        {
            return mp_YuvBufferFull;
        }

        /** obt@NA
         */
        void ClearBuffer( void )
        {
            for( int i=0; i < m_BufferNum; i++ )
            {
                memset( mpa_YuvBuffers[ i ], 0, nn::camera::GetFrameBytes( m_TrimWidth, m_TrimHeight ) );
            }
        }

        /** t[[gv
         */
        void CalcFrameRate( void )
        {
            nn::os::Tick current = nn::os::Tick::GetSystemCurrent();
            ma_VsyncDiffTimes[ m_VsyncTimePos ] = current - m_VsyncPrevTime;
            m_VsyncTimePos  = (m_VsyncTimePos + 1) % 4;
            m_VsyncPrevTime = current;
            nn::os::Tick sum(0);
            s32 i=0;
            for( i=0; i < 4; i++ )
            {
                if( ma_VsyncDiffTimes[ i ].ToTimeSpan() == 0 )
                {
                    break;
                }
                sum += ma_VsyncDiffTimes[i];
            }
            m_Fps = static_cast<s64>( i ) * 1000000000LL / sum.ToTimeSpan().GetMicroSeconds();
        }

        /** t[[gԂ
         */
        s64 GetFrameRate( void ) const
        {
            return m_Fps;
        }

        /** Vsync̃^C~O擾
         */
        nn::fnd::TimeSpan * GetLatestVsyncTiming( size_t past = 5 )
        {
            nn::camera::GetLatestVsyncTiming( ma_VsyncTiming, m_Port, past );
            return ma_VsyncTiming;
        }

        /** Vsync̃^C~O擾ʂQƂ
         */
        nn::fnd::TimeSpan * ReferLatestVsyncTiming( void )
        {
            return ma_VsyncTiming;
        }

    private:
        // |[g
        nn::util::SizedEnum1<nn::camera::Port>   m_Port;
        NN_PADDING3;

        // XbhɘA邽߂̃Cxg
        nn::os::Event   ma_Events[ e_EVENT_NUM ];

        // obt@
        u8              * mpa_YuvBuffers[ sc_MaxBufferNum ];    // J̃CgY2R̃[hՓ˂Ȃ悤Ƀobt@𕡐
        s32             m_BufferNum;                            // obt@̐
        s32             m_CapturingPos;                         // ݓ]̃obt@ԍ
        s32             m_LatestCapturedPos;                    // ŐṼLv`ς݃obt@ԍ
        s32             m_ReadingPos;                           // ǂݍݒ̃obt@ԍ

        // DMA]TCY
        size_t          m_OutputSize;
        size_t          m_TransferUnit;

        // tTCYLv`pobt@
        u8              * mp_YuvBufferFull;

        // tTCYLv`ǂ
        bool            m_IsCapturingFull;
        NN_PADDING3;

        // tTCYLv`DMA]TCY
        size_t          m_OutputSizeFull;
        size_t          m_TransferUnitFull;

        // 摜TCY
        size_t          m_CameraWidth;
        size_t          m_CameraHeight;
        size_t          m_TrimWidth;
        size_t          m_TrimHeight;

        // r
        nn::os::CriticalSection m_CsSet;
        nn::os::CriticalSection m_CsDma;

        // t[[gvp
        NN_PADDING4;
        nn::os::Tick    ma_VsyncDiffTimes[ 4 ];
        nn::os::Tick    m_VsyncPrevTime;
        s64             m_Fps;
        s32             m_VsyncTimePos;
        NN_PADDING4;

        // Vsync̃^C~Oێp
        nn::fnd::TimeSpan  ma_VsyncTiming[ 5 ];
    }; //class CameraManager_cl

    //==============================================================================
    /** OCAMNX : static
     */
    class OcamManager_cl
    {
    public:
        enum
        {
            e_EVENT_WB_SYNC_START = 0,    // zCgoX̓Jn/ICxg
            e_EVENT_WB_SYNC_END,

            e_EVENT_NUM
        };

    public:
        OcamManager_cl(){}

    public:
        static void Initialize( void )
        {
            m_IsEnableManualVsyncSync = false;

            for( s32 i=0; i < e_EVENT_NUM; i++ )
            {
                ma_Events[ i ].Initialize( false );
            }
        }

        static void Finalize( void )
        {
            for( s32 i=0; i < e_EVENT_NUM; i++ )
            {
                ma_Events[ i ].Finalize();
            }
        }

    public:
        /** Jʂ̊ǗNXւ̎QƂԂ
         */
        static CameraManager_cl & GetCameraManager( s32 index )
        {
            return ma_CameraManagers[ index ];
        }

        /** eCxgւ̎QƂԂ
         */
        static nn::os::Event & GetEvent( s32 index )
        {
            return ma_Events[ index ];
        }

        /** eCxgւ̃|C^Ԃ
         */
        static nn::os::Event * GetEventPointer( s32 index )
        {
            return &ma_Events[ index ];
        }

        /** Vsync̃}jA̗L/߂
         */
        static void SetManualVsyncSync( bool enable )
        {
            m_IsEnableManualVsyncSync = enable;
        }

        /** VSync̃}jALǂԂ
         */
        static bool IsEnableManualVsyncSync( void )
        {
            return m_IsEnableManualVsyncSync;
        }

    private:
        // Vsync̃}jAݒF߂邩
        static bool m_IsEnableManualVsyncSync;

        // ݒ˗Cxg
        static nn::os::Event ma_Events[ e_EVENT_NUM ];

        // J2̐ݒ
        static CameraManager_cl ma_CameraManagers[ 2 ];

    }; //class OCamManager_cl

} //namespace camera_exp

#endif //#ifndef CAMERA_MANAGER_H_
