﻿/*---------------------------------------------------------------------------*
  Project:  Horizon
  File:     snd.cpp

  Copyright (C)2009 Nintendo Co., Ltd.  All rights reserved.

  These coded instructions, statements, and computer programs contain
  proprietary information of Nintendo of America Inc. and/or Nintendo
  Company Ltd., and are protected by Federal copyright law.  They may
  not be disclosed to third parties or copied or duplicated in any form,
  in whole or in part, without the prior written consent of Nintendo.

  $Rev: 12996 $
 *---------------------------------------------------------------------------*/

#include <nn.h>
#include <nn/os.h>
#include <nn/fnd.h>
#include <nn/math.h>

#include <nn/dsp.h>
#include <nn/snd.h>

#include "snd.h"

// アプリ用ヒープ
extern nn::fnd::ExpHeap s_AppHeap;

namespace
{
    nn::snd::Voice* pVoice;
    s16* pMemory;
    nn::snd::WaveBuffer waveBuffer;
    const int nBufferSize = 2048;
    bool isPlaying = false;
    f32 pitch = 1.0f;
#if 0
    // サイン波生成
    void MakeSineWave(s16* p, s32 n)
    {
        for (int i = 0; i < n; i++)
        {
            p[i] = (32767.0f * nn::math::SinFIdx(4.0f*i));
        }
    }
#endif
    // サウンドスレッド関連
    const int SOUND_THREAD_PRIORITY = 2;
    const int SOUND_THREAD_STACK_SIZE = 1024;
    nn::os::Thread threadSound;
    bool threadSoundFlag;
    void SoundThreadFunc(uptr arg)
    {
        (void)arg;
        threadSoundFlag = true;
        while (threadSoundFlag)
        {
            nn::snd::WaitForDspSync();  // DSP からのデータ受信を待つ。
            nn::snd::SendParameterToDsp();  // パラメータをDSP に送信。
        }
    }
}

void InitializeSnd()
{
    nn::Result result;

    // dsp, snd の初期化
    result = nn::dsp::Initialize();
    NN_UTIL_PANIC_IF_FAILED(result);
    result = nn::dsp::LoadDefaultComponent();
    NN_UTIL_PANIC_IF_FAILED(result);
    result = nn::snd::Initialize();
    NN_UTIL_PANIC_IF_FAILED(result);


    // サウンドスレッドを起動（DSP 割り込みイベント待ち）
    threadSound.StartUsingAutoStack(
        SoundThreadFunc,
        NULL,
        SOUND_THREAD_STACK_SIZE,
        SOUND_THREAD_PRIORITY
    );
    
    // マスターボリュームを設定
    nn::snd::SetMasterVolume(1.0f);

#if 0
    // 連続メモリ領域の取得、確認
    pMemory = reinterpret_cast<s16*>(s_AppHeap.Allocate(nBufferSize * sizeof(s16), 32));
    NN_TASSERT_(pMemory);
    MakeSineWave(pMemory, nBufferSize);
    nn::snd::FlushDataCache(reinterpret_cast<uptr>(pMemory), nBufferSize * sizeof(s16));
#endif

    pVoice = nn::snd::AllocVoice(128, NULL, NULL);
    NN_TASSERT_(pVoice);

    pVoice->SetChannelCount(1);
    pVoice->SetSampleFormat(nn::snd::SAMPLE_FORMAT_PCM16);

    // 音量の設定
    nn::snd::MixParam mix;
    mix.mainBus[nn::snd::CHANNEL_INDEX_FRONT_LEFT ] = 0.5f; // メインボリューム (L)
    mix.mainBus[nn::snd::CHANNEL_INDEX_FRONT_RIGHT] = 0.5f; // メインボリューム (R)
    pVoice->SetMixParam(mix);
    pVoice->SetVolume(1.0f);

    // pitch の設定
    pVoice->SetSampleRate(HW_I2S_CLOCK_32KHZ);
    pVoice->SetPitch(pitch);

    nn::snd::InitializeWaveBuffer(&waveBuffer);
    waveBuffer.bufferAddress = pMemory;
    waveBuffer.sampleLength  = nn::snd::GetSampleLength(nBufferSize * sizeof(s16), nn::snd::SAMPLE_FORMAT_PCM16, 1);
    waveBuffer.loopFlag = true;
    pVoice->AppendWaveBuffer(&waveBuffer);
    

}

void FinalizeSnd(void)
{
    // 再生終了
    pVoice->SetState(nn::snd::Voice::STATE_STOP);

    // サウンドスレッドの破棄
    threadSoundFlag = false;
    threadSound.Join();
    threadSound.Finalize();

    // SND の終了処理
    nn::Result result = nn::snd::Finalize();
    NN_UTIL_PANIC_IF_FAILED(result);

    // DSP の終了処理
    result = nn::dsp::UnloadComponent();
    NN_UTIL_PANIC_IF_FAILED(result);
    nn::dsp::Finalize();

    // ヒープの解放
    s_AppHeap.Free(pMemory);
}

void PlaySound(void)
{
	u32 te = 0;
    if (!isPlaying)
    {
        if( te == 0)
        {
        pVoice->SetState(nn::snd::Voice::STATE_PLAY);
        te = 1;
        }
    }
    isPlaying = true;

    pitch -= 0.02f;
    pVoice->SetPitch(pitch);
}

void StopSound(void)
{
    if (isPlaying)
    {
        pVoice->SetState(nn::snd::Voice::STATE_PAUSE);
    }
    isPlaying = false;

    pitch = 1.0f;
    pVoice->SetPitch(pitch);
}

/*---------------------------------------------------------------------------*
  End of file
 *---------------------------------------------------------------------------*/
